// Main Application Class
class AdminApp {
    constructor() {
        // Auto-detect API base path based on environment
        this.apiBase = window.location.hostname.includes('localhost') || window.location.hostname.includes('127.0.0.1') 
            ? '/student1/api' 
            : '/api';
        this.token = localStorage.getItem('admin_token');
        this.user = null;
        this.currentPage = 'dashboard';
        
        // Clean up any modal artifacts on load
        this.cleanupModalArtifacts();
        
        this.init();
    }
    
    cleanupModalArtifacts() {
        // Remove any lingering modal backdrops
        document.querySelectorAll('.modal-backdrop').forEach(backdrop => backdrop.remove());
        document.querySelectorAll('.custom-modal-backdrop').forEach(backdrop => backdrop.remove());
        
        // Remove modal-open class from body
        document.body.classList.remove('modal-open');
        
        // Hide any visible modals
        document.querySelectorAll('.modal.show').forEach(modal => {
            modal.style.display = 'none';
            modal.classList.remove('show');
        });
    }
    
    init() {
        // Ensure clean initial state
        this.showLoginScreen();
        
        // Check if already logged in
        if (this.token) {
            this.verifyToken();
        } else {
            // Clear any saved page state if not logged in
            localStorage.removeItem('currentPage');
        }
        
        // Bind events
        this.bindEvents();
    }
    
    bindEvents() {
        // Login form
        document.getElementById('loginForm').addEventListener('submit', (e) => {
            e.preventDefault();
            this.login();
        });
        
        // Navigation links
        document.addEventListener('click', (e) => {
            if (e.target.matches('[data-page]')) {
                e.preventDefault();
                const page = e.target.getAttribute('data-page');
                this.navigateTo(page);
            }
        });
        
        // Logout
        document.getElementById('logoutBtn').addEventListener('click', (e) => {
            e.preventDefault();
            this.logout();
        });
    }
    
    async login() {
        const username = document.getElementById('username').value;
        const password = document.getElementById('password').value;
        const errorDiv = document.getElementById('loginError');
        
        try {
            const response = await this.apiCall('POST', '/auth/admin/login', {
                username,
                password
            });
            
            if (response.success) {
                this.token = response.data.token;
                this.user = response.data.user;
                localStorage.setItem('admin_token', this.token);
                
                // Clear any saved page for fresh login
                localStorage.removeItem('currentPage');
                
                this.showMainApp();
                
                // Check URL for page parameter
                const urlParams = new URLSearchParams(window.location.search);
                const targetPage = urlParams.get('page') || 'dashboard';
                
                // Navigate to the requested page
                await this.navigateTo(targetPage);
            } else {
                this.showError(errorDiv, response.message);
            }
        } catch (error) {
            this.showError(errorDiv, 'Login failed. Please try again.');
        }
    }
    
    async verifyToken() {
        try {
            const response = await this.apiCall('GET', '/auth/me');
            if (response.success && response.data.type === 'admin') {
                this.user = response.data;
                this.showMainApp();
                
                // Restore saved page or default to dashboard
                const savedPage = localStorage.getItem('currentPage') || 'dashboard';
                await this.navigateTo(savedPage);
            } else {
                this.logout();
            }
        } catch (error) {
            console.log('Token verification failed:', error);
            this.logout();
        }
    }
    
    logout() {
        this.token = null;
        this.user = null;
        this.currentPage = 'dashboard';
        localStorage.removeItem('admin_token');
        localStorage.removeItem('currentPage');
        
        // Clear any existing page content
        const pageContent = document.getElementById('pageContent');
        if (pageContent) {
            pageContent.innerHTML = '';
        }
        
        // Show login screen
        this.showLoginScreen();
    }
    
    showLoginScreen() {
        const app = document.getElementById('app');
        app.classList.remove('logged-in');
        
        // Clear any error messages
        const errorDiv = document.getElementById('loginError');
        if (errorDiv) {
            errorDiv.style.display = 'none';
            errorDiv.textContent = '';
        }
        
        // Clear form
        const usernameField = document.getElementById('username');
        const passwordField = document.getElementById('password');
        if (usernameField) usernameField.value = '';
        if (passwordField) passwordField.value = '';
    }
    
    showMainApp() {
        const app = document.getElementById('app');
        app.classList.add('logged-in');
        
        // Update user info
        const userInfo = document.getElementById('userInfo');
        if (userInfo && this.user) {
            userInfo.textContent = `${this.user.full_name} (${this.user.role})`;
        }
        
        // Apply role-based navigation visibility
        this.applyRoleBasedNavigation();
        
        // Don't automatically navigate to dashboard here
        // Let the calling function handle navigation
    }
    
    applyRoleBasedNavigation() {
        if (!this.user || !this.user.role) {
            return;
        }
        
        const role = this.user.role;
        const navItems = document.querySelectorAll('[data-page]');
        
        // Define role permissions - based on AuthService permissions
        const rolePermissions = {
            'admin': ['*'], // Admin can access everything
            'registrar': [
                'dashboard', 'students', 'enrollments', 'academic-years', 
                'grades', 'classes', 'student-cards'
            ],
            'teacher': [
                'dashboard', 'exams', 'results', 'students', 'grades', 
                'classes', 'subjects', 'grade-subject-rules'
            ],
            'accountant': [
                'dashboard', 'payments', 'students', 'grades', 'classes'
            ]
        };
        
        const allowedPages = rolePermissions[role] || [];
        const hasWildcard = allowedPages.includes('*');
        
        navItems.forEach(navItem => {
            const page = navItem.getAttribute('data-page');
            const parentLi = navItem.closest('li');
            
            if (hasWildcard || allowedPages.includes(page)) {
                // Show the navigation item
                if (parentLi) {
                    parentLi.style.display = '';
                }
                navItem.style.display = '';
            } else {
                // Hide the navigation item
                if (parentLi) {
                    parentLi.style.display = 'none';
                }
                navItem.style.display = 'none';
            }
        });
        
        // Handle section headers - hide if all items in section are hidden
        this.handleSectionHeaders();
    }
    
    handleSectionHeaders() {
        const sectionHeaders = document.querySelectorAll('.sidebar-heading');
        
        sectionHeaders.forEach(header => {
            const headerLi = header.closest('li');
            if (!headerLi) return;
            
            // Find all navigation items after this header until the next header
            let nextElement = headerLi.nextElementSibling;
            let hasVisibleItems = false;
            
            while (nextElement && !nextElement.querySelector('.sidebar-heading')) {
                const navLink = nextElement.querySelector('[data-page]');
                if (navLink && nextElement.style.display !== 'none') {
                    hasVisibleItems = true;
                    break;
                }
                nextElement = nextElement.nextElementSibling;
            }
            
            // Hide section header if no visible items
            headerLi.style.display = hasVisibleItems ? '' : 'none';
        });
    }
    
    async navigateTo(page) {
        // Check if user has permission to access this page
        if (!this.hasPagePermission(page)) {
            this.showError(null, 'You do not have permission to access this page.');
            // Redirect to dashboard if they don't have permission
            if (page !== 'dashboard') {
                await this.navigateTo('dashboard');
            }
            return;
        }
        
        this.currentPage = page;
        
        // Save current page to localStorage
        localStorage.setItem('currentPage', page);
        
        // Update active nav
        document.querySelectorAll('.nav-link').forEach(link => {
            link.classList.remove('active');
        });
        const activeLink = document.querySelector(`[data-page="${page}"]`);
        if (activeLink) activeLink.classList.add('active');
        
        // Load page content
        await this.loadPage(page);
    }
    
    hasPagePermission(page) {
        if (!this.user || !this.user.role) {
            return false;
        }
        
        const role = this.user.role;
        
        // Define role permissions - same as in applyRoleBasedNavigation
        const rolePermissions = {
            'admin': ['*'], // Admin can access everything
            'registrar': [
                'dashboard', 'students', 'enrollments', 'academic-years', 
                'grades', 'classes', 'student-cards'
            ],
            'teacher': [
                'dashboard', 'exams', 'results', 'students', 'grades', 
                'classes', 'subjects', 'grade-subject-rules'
            ],
            'accountant': [
                'dashboard', 'payments', 'students', 'grades', 'classes'
            ]
        };
        
        const allowedPages = rolePermissions[role] || [];
        return allowedPages.includes('*') || allowedPages.includes(page);
    }
    
    async loadPage(page) {
        const content = document.getElementById('pageContent');
        content.innerHTML = '<div class="loading"><i class="bi bi-hourglass-split"></i> <span data-translate="dashboard.loading">Loading...</span></div>';
        
        try {
            switch (page) {
                case 'dashboard':
                    await this.loadDashboard();
                    break;
                case 'academic-years':
                    await this.loadAcademicYears();
                    break;
                case 'terms':
                    await this.loadTerms();
                    break;
                case 'grades':
                    await this.loadGrades();
                    break;
                case 'classes':
                    await this.loadClasses();
                    break;
                case 'subjects':
                    await this.loadSubjects();
                    break;
                case 'grade-subject-rules':
                    await this.loadGradeSubjectRules();
                    break;
                case 'students':
                    await this.loadStudents();
                    break;
                case 'enrollments':
                    await this.loadEnrollments();
                    break;
                case 'exams':
                    await this.loadExams();
                    break;
                case 'results':
                    await this.loadResults();
                    break;
                case 'payments':
                    await this.loadPayments();
                    break;
                case 'settings':
                    await this.loadSettings();
                    break;
                case 'users':
                    await this.loadUsers();
                    break;
                default:
                    content.innerHTML = `<h2>${page.replace('-', ' ').toUpperCase()}</h2><p>Page content coming soon...</p>`;
            }
        } catch (error) {
            content.innerHTML = `<div class="alert alert-danger">Error loading page: ${error.message}</div>`;
        }
    }
    
    async loadDashboard() {
        const content = `
            <div class="d-flex justify-content-between flex-wrap flex-md-nowrap align-items-center pt-3 pb-2 mb-3 border-bottom">
                <h1 class="h2" data-translate="dashboard.title">Dashboard</h1>
            </div>
            
            <div class="row">
                <div class="col-xl-3 col-md-6 mb-4">
                    <div class="card border-left-primary shadow h-100 py-2">
                        <div class="card-body">
                            <div class="row no-gutters align-items-center">
                                <div class="col mr-2">
                                    <div class="text-xs font-weight-bold text-primary text-uppercase mb-1" data-translate="dashboard.total_students">Total Students</div>
                                    <div class="h5 mb-0 font-weight-bold text-gray-800" id="totalStudents">Loading...</div>
                                </div>
                                <div class="col-auto">
                                    <i class="bi bi-people fa-2x text-gray-300"></i>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
                
                <div class="col-xl-3 col-md-6 mb-4">
                    <div class="card border-left-success shadow h-100 py-2">
                        <div class="card-body">
                            <div class="row no-gutters align-items-center">
                                <div class="col mr-2">
                                    <div class="text-xs font-weight-bold text-success text-uppercase mb-1" data-translate="dashboard.active_enrollments">Active Enrollments</div>
                                    <div class="h5 mb-0 font-weight-bold text-gray-800" id="activeEnrollments">Loading...</div>
                                </div>
                                <div class="col-auto">
                                    <i class="bi bi-person-check fa-2x text-gray-300"></i>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
                
                <div class="col-xl-3 col-md-6 mb-4">
                    <div class="card border-left-info shadow h-100 py-2">
                        <div class="card-body">
                            <div class="row no-gutters align-items-center">
                                <div class="col mr-2">
                                    <div class="text-xs font-weight-bold text-info text-uppercase mb-1" data-translate="dashboard.total_exams">Total Exams</div>
                                    <div class="h5 mb-0 font-weight-bold text-gray-800" id="totalExams">Loading...</div>
                                </div>
                                <div class="col-auto">
                                    <i class="bi bi-clipboard-check fa-2x text-gray-300"></i>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
                
                <div class="col-xl-3 col-md-6 mb-4">
                    <div class="card border-left-warning shadow h-100 py-2">
                        <div class="card-body">
                            <div class="row no-gutters align-items-center">
                                <div class="col mr-2">
                                    <div class="text-xs font-weight-bold text-warning text-uppercase mb-1" data-translate="dashboard.payment_balance">Payment Balance</div>
                                    <div class="h5 mb-0 font-weight-bold text-gray-800" id="paymentBalance">Loading...</div>
                                </div>
                                <div class="col-auto">
                                    <i class="bi bi-credit-card fa-2x text-gray-300"></i>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
            
            <div class="row">
                <div class="col-lg-6">
                    <div class="card shadow mb-4">
                        <div class="card-header py-3">
                            <h6 class="m-0 font-weight-bold text-primary" data-translate="dashboard.recent_activities">Recent Activities</h6>
                        </div>
                        <div class="card-body" id="recentActivity">
                            <div class="text-center py-3">
                                <div class="spinner-border spinner-border-sm text-primary" role="status"></div>
                                <span class="ms-2" data-translate="dashboard.loading">Loading...</span>
                            </div>
                        </div>
                    </div>
                </div>
                
                <div class="col-lg-6">
                    <div class="card shadow mb-4">
                        <div class="card-header py-3">
                            <h6 class="m-0 font-weight-bold text-primary">Grade Distribution</h6>
                        </div>
                        <div class="card-body" id="gradeDistribution">
                            <div class="text-center py-3">
                                <div class="spinner-border spinner-border-sm text-primary" role="status"></div>
                                <span class="ms-2">Loading distribution...</span>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
            
            <div class="row">
                <div class="col-lg-8">
                    <div class="card shadow mb-4">
                        <div class="card-header py-3">
                            <h6 class="m-0 font-weight-bold text-primary" data-translate="dashboard.payment_trends">Payment Trends (Last 6 Months)</h6>
                        </div>
                        <div class="card-body" id="paymentChart">
                            <div class="text-center py-3">
                                <div class="spinner-border spinner-border-sm text-primary" role="status"></div>
                                <span class="ms-2">Loading payment data...</span>
                            </div>
                        </div>
                    </div>
                </div>
                
                <div class="col-lg-4">
                    <div class="card shadow mb-4">
                        <div class="card-header py-3">
                            <h6 class="m-0 font-weight-bold text-primary" data-translate="dashboard.academic_performance">Academic Performance</h6>
                        </div>
                        <div class="card-body" id="resultsSummary">
                            <div class="text-center py-3">
                                <div class="spinner-border spinner-border-sm text-primary" role="status"></div>
                                <span class="ms-2">Loading results...</span>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
            
            <div class="row">
                <div class="col-12">
                    <div class="card shadow mb-4">
                        <div class="card-header py-3">
                            <h6 class="m-0 font-weight-bold text-primary" data-translate="dashboard.quick_actions">Quick Actions</h6>
                        </div>
                        <div class="card-body">
                            <div class="row">
                                <div class="col-md-3 mb-2">
                                    <button class="btn btn-primary w-100" onclick="app.loadPage('students')">
                                        <i class="bi bi-person-plus me-2"></i><span data-translate="dashboard.add_new_student">Add New Student</span>
                                    </button>
                                </div>
                                <div class="col-md-3 mb-2">
                                    <button class="btn btn-success w-100" onclick="app.loadPage('exams')">
                                        <i class="bi bi-file-earmark-text me-2"></i><span data-translate="dashboard.create_exam">Create Exam</span>
                                    </button>
                                </div>
                                <div class="col-md-3 mb-2">
                                    <button class="btn btn-info w-100" onclick="app.loadPage('payments')">
                                        <i class="bi bi-credit-card me-2"></i><span data-translate="payments.record_payment">Record Payment</span>
                                    </button>
                                </div>
                                <div class="col-md-3 mb-2">
                                    <button class="btn btn-warning w-100" onclick="app.loadPage('settings')">
                                        <i class="bi bi-gear me-2"></i><span data-translate="settings.title">System Settings</span>
                                    </button>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        `;
        
        document.getElementById('pageContent').innerHTML = content;
        
        // Apply translations to the new content
        if (typeof langManager !== 'undefined') {
            console.log('🌍 Applying dashboard translations...');
            
            // Immediate translation
            langManager.forceTranslation();
            
            // Additional translations with delays for dynamic content
            setTimeout(() => {
                langManager.forceTranslation();
                console.log('🌍 Second translation pass');
            }, 100);
            
            setTimeout(() => {
                langManager.forceTranslation();
                console.log('🌍 Final translation pass');
            }, 1000);
        }
        
        // Load real dashboard stats
        await this.loadDashboardStats();
    }
    
    async loadAcademicYears() {
        try {
            const response = await this.apiCall('GET', '/admin/academic-years');
            
            const content = `
                <div class="page-header">
                    <div class="d-flex justify-content-between align-items-center">
                        <div>
                            <h1 class="h2 mb-1"><i class="bi bi-calendar3"></i> <span data-translate="academic_years.title">Academic Years</span></h1>
                            <p class="text-muted mb-0" data-translate="academic_years.subtitle">Manage academic year periods and settings</p>
                        </div>
                        <button class="btn btn-primary" onclick="app.showAcademicYearModal()">
                            <i class="bi bi-plus-circle"></i> <span data-translate="academic_years.add_year">Add Academic Year</span>
                        </button>
                    </div>
                </div>
                
                <div class="filter-bar">
                    <div class="row align-items-end">
                        <div class="col-md-3">
                            <label class="form-label" data-translate="academic_years.search_by_name">Search by name</label>
                            <input type="text" class="form-control" id="nameFilter" data-translate-placeholder="academic_years.search_placeholder" placeholder="e.g., 2024-2025">
                        </div>
                        <div class="col-md-2">
                            <label class="form-label" data-translate="students.status">Status</label>
                            <select class="form-select" id="statusFilter">
                                <option value="" data-translate="academic_years.all_status">All Status</option>
                                <option value="open" data-translate="academic_years.open">Open</option>
                                <option value="closed" data-translate="academic_years.closed">Closed</option>
                            </select>
                        </div>
                        <div class="col-md-2">
                            <label class="form-label">&nbsp;</label>
                            <div class="d-grid">
                                <button class="btn btn-outline-primary" onclick="app.filterAcademicYears()">
                                    <i class="bi bi-search"></i> <span data-translate="academic_years.filter">Filter</span>
                                </button>
                            </div>
                        </div>
                        <div class="col-md-2">
                            <label class="form-label">&nbsp;</label>
                            <div class="d-grid">
                                <button class="btn btn-outline-secondary" onclick="app.clearFilters()">
                                    <i class="bi bi-arrow-clockwise"></i> <span data-translate="academic_years.clear">Clear</span>
                                </button>
                            </div>
                        </div>
                        <div class="col-md-3 text-end">
                            <label class="form-label">&nbsp;</label>
                            <div class="btn-group" role="group">
                                <button class="btn btn-outline-success btn-sm" onclick="app.exportAcademicYears('csv')">
                                    <i class="bi bi-file-earmark-csv"></i> CSV
                                </button>
                                <button class="btn btn-outline-success btn-sm" onclick="app.exportAcademicYears('excel')">
                                    <i class="bi bi-file-earmark-excel"></i> Excel
                                </button>
                                <button class="btn btn-outline-info btn-sm" onclick="app.printAcademicYears()">
                                    <i class="bi bi-printer"></i> Print
                                </button>
                            </div>
                        </div>
                    </div>
                </div>
                
                <div class="card">
                    <div class="card-body p-0">
                        <div class="table-responsive">
                            <table class="table table-hover mb-0">
                                <thead>
                                    <tr>
                                        <th class="sortable" data-sort="name">
                                            <i class="bi bi-arrow-down-up"></i> <span data-translate="table.year_name">Name</span>
                                        </th>
                                        <th class="sortable" data-sort="starts_on">
                                            <i class="bi bi-arrow-down-up"></i> <span data-translate="table.start_date">Start Date</span>
                                        </th>
                                        <th class="sortable" data-sort="ends_on">
                                            <i class="bi bi-arrow-down-up"></i> <span data-translate="table.end_date">End Date</span>
                                        </th>
                                        <th class="sortable" data-sort="status">
                                            <i class="bi bi-arrow-down-up"></i> <span data-translate="table.status">Status</span>
                                        </th>
                                        <th class="sortable" data-sort="created_at">
                                            <i class="bi bi-arrow-down-up"></i> <span data-translate="table.created">Created</span>
                                        </th>
                                        <th width="120" data-translate="table.actions">Actions</th>
                                    </tr>
                                </thead>
                                <tbody id="academicYearsTableBody">
                                    ${this.renderAcademicYearsTable(response.data)}
                                </tbody>
                            </table>
                        </div>
                    </div>
                </div>
                
                ${this.renderPagination(response.meta)}
                
                <!-- Academic Year Modal -->
                <div class="modal fade" id="academicYearModal" tabindex="-1">
                    <div class="modal-dialog">
                        <div class="modal-content">
                            <div class="modal-header">
                                <h5 class="modal-title">
                                    <i class="bi bi-calendar3"></i> <span id="modalTitle">Add Academic Year</span>
                                </h5>
                                <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal"></button>
                            </div>
                            <div class="modal-body">
                                <form id="academicYearForm">
                                    <input type="hidden" id="yearId">
                                    <div class="mb-3">
                                        <label for="yearName" class="form-label" data-translate="academic_years.year_name">Academic Year Name *</label>
                                        <input type="text" class="form-control" id="yearName" data-translate-placeholder="academic_years.name_placeholder" placeholder="e.g., 2024-2025" required>
                                        <div class="invalid-feedback"></div>
                                    </div>
                                    <div class="row">
                                        <div class="col-md-6">
                                            <div class="mb-3">
                                                <label for="startsOn" class="form-label" data-translate="academic_years.start_date">Start Date</label>
                                                <input type="date" class="form-control" id="startsOn">
                                            </div>
                                        </div>
                                        <div class="col-md-6">
                                            <div class="mb-3">
                                                <label for="endsOn" class="form-label" data-translate="academic_years.end_date">End Date</label>
                                                <input type="date" class="form-control" id="endsOn">
                                            </div>
                                        </div>
                                    </div>
                                    <div class="mb-3">
                                        <label for="yearStatus" class="form-label" data-translate="students.status">Status *</label>
                                        <select class="form-select" id="yearStatus" required>
                                            <option value="" data-translate="academic_years.select_status">Select status</option>
                                            <option value="open" data-translate="academic_years.open">Open</option>
                                            <option value="closed" data-translate="academic_years.closed">Closed</option>
                                        </select>
                                        <div class="form-text">
                                            <i class="bi bi-info-circle"></i> 
                                            Open years allow new data entry. Closed years become read-only.
                                        </div>
                                    </div>
                                </form>
                            </div>
                            <div class="modal-footer">
                                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">
                                    <i class="bi bi-x-circle"></i> <span data-translate="btn.cancel">Cancel</span>
                                </button>
                                <button type="button" class="btn btn-primary" onclick="app.saveAcademicYear()">
                                    <i class="bi bi-check-circle"></i> <span id="saveButtonText">Save Academic Year</span>
                                </button>
                            </div>
                        </div>
                    </div>
                </div>
            `;
            
            document.getElementById('pageContent').innerHTML = content;
            
            // Add event listeners for sorting
            document.querySelectorAll('.sortable').forEach(header => {
                header.style.cursor = 'pointer';
                header.addEventListener('click', () => {
                    const sortField = header.getAttribute('data-sort');
                    this.sortAcademicYears(sortField);
                });
            });
            
        } catch (error) {
            document.getElementById('pageContent').innerHTML = `
                <div class="alert alert-danger">
                    <i class="bi bi-exclamation-triangle"></i> Error loading academic years: ${error.message}
                </div>`;
        }
    }
    
    renderAcademicYearsTable(years) {
        if (!years || years.length === 0) {
            return `
                <tr>
                    <td colspan="6" class="text-center py-5">
                        <div class="text-muted">
                            <i class="bi bi-calendar3" style="font-size: 3rem; opacity: 0.3;"></i>
                            <p class="mt-3 mb-0">No academic years found</p>
                            <small>Click "Add Academic Year" to create your first entry</small>
                        </div>
                    </td>
                </tr>
            `;
        }
        
        return years.map(year => `
            <tr>
                <td>
                    <strong>${year.name}</strong>
                </td>
                <td>${year.starts_on ? new Date(year.starts_on).toLocaleDateString() : '<span class="text-muted">Not set</span>'}</td>
                <td>${year.ends_on ? new Date(year.ends_on).toLocaleDateString() : '<span class="text-muted">Not set</span>'}</td>
                <td>
                    <span class="badge ${year.status === 'open' ? 'bg-success' : 'bg-secondary'}">
                        <i class="bi bi-${year.status === 'open' ? 'unlock' : 'lock'}"></i> ${year.status.toUpperCase()}
                    </span>
                </td>
                <td>
                    <small class="text-muted">${new Date(year.created_at).toLocaleDateString()}</small>
                </td>
                <td>
                    <div class="btn-group" role="group">
                        <button class="btn btn-sm btn-outline-primary" onclick="app.editAcademicYear(${year.id})" title="Edit">
                            <i class="bi bi-pencil"></i>
                        </button>
                        <button class="btn btn-sm btn-outline-danger" onclick="app.deleteAcademicYear(${year.id}, '${year.name}')" title="Delete">
                            <i class="bi bi-trash"></i>
                        </button>
                    </div>
                </td>
            </tr>
        `).join('');
    }
    
    renderPagination(meta) {
        if (meta.total_pages <= 1) return '';
        
        const currentPage = meta.page;
        const totalPages = meta.total_pages;
        let pagination = '<nav class="mt-4"><ul class="pagination justify-content-center">';
        
        // Previous button
        if (currentPage > 1) {
            pagination += `<li class="page-item"><a class="page-link" href="#" onclick="app.loadAcademicYearsPage(${currentPage - 1})">Previous</a></li>`;
        }
        
        // Page numbers
        for (let i = Math.max(1, currentPage - 2); i <= Math.min(totalPages, currentPage + 2); i++) {
            pagination += `<li class="page-item ${i === currentPage ? 'active' : ''}">
                <a class="page-link" href="#" onclick="app.loadAcademicYearsPage(${i})">${i}</a>
            </li>`;
        }
        
        // Next button
        if (currentPage < totalPages) {
            pagination += `<li class="page-item"><a class="page-link" href="#" onclick="app.loadAcademicYearsPage(${currentPage + 1})">Next</a></li>`;
        }
        
        pagination += '</ul></nav>';
        return pagination;
    }
    
    showAcademicYearModal(yearData = null) {
        const modalElement = document.getElementById('academicYearModal');
        if (!modalElement) {
            this.showToast('error', 'Modal not found. Please refresh the page.');
            return;
        }
        
        // Remove any existing backdrops
        document.querySelectorAll('.modal-backdrop').forEach(backdrop => backdrop.remove());
        document.querySelectorAll('.custom-modal-backdrop').forEach(backdrop => backdrop.remove());
        
        // Reset form
        const form = document.getElementById('academicYearForm');
        if (form) form.reset();
        
        const yearIdField = document.getElementById('yearId');
        if (yearIdField) yearIdField.value = '';
        
        if (yearData) {
            // Edit mode
            const modalTitle = document.getElementById('modalTitle');
            const saveButtonText = document.getElementById('saveButtonText');
            const yearNameField = document.getElementById('yearName');
            const startsOnField = document.getElementById('startsOn');
            const endsOnField = document.getElementById('endsOn');
            const yearStatusField = document.getElementById('yearStatus');
            
            if (modalTitle) {
                modalTitle.textContent = (typeof langManager !== 'undefined') 
                    ? langManager.t('modal.edit_academic_year')
                    : 'Edit Academic Year';
            }
            if (saveButtonText) {
                saveButtonText.textContent = (typeof langManager !== 'undefined')
                    ? langManager.t('modal.update_academic_year')
                    : 'Update Academic Year';
            }
            if (yearIdField) yearIdField.value = yearData.id;
            if (yearNameField) yearNameField.value = yearData.name;
            if (startsOnField) startsOnField.value = yearData.starts_on || '';
            if (endsOnField) endsOnField.value = yearData.ends_on || '';
            if (yearStatusField) yearStatusField.value = yearData.status;
        } else {
            // Add mode
            const modalTitle = document.getElementById('modalTitle');
            const saveButtonText = document.getElementById('saveButtonText');
            
            if (modalTitle) {
                modalTitle.textContent = (typeof langManager !== 'undefined')
                    ? langManager.t('modal.add_academic_year')
                    : 'Add Academic Year';
            }
            if (saveButtonText) {
                saveButtonText.textContent = (typeof langManager !== 'undefined')
                    ? langManager.t('modal.save_academic_year')
                    : 'Save Academic Year';
            }
        }
        
        // Clear validation classes
        document.querySelectorAll('.is-invalid').forEach(el => el.classList.remove('is-invalid'));
        
        // Clean up any existing backdrops before showing
        document.querySelectorAll('.modal-backdrop').forEach(backdrop => backdrop.remove());
        document.querySelectorAll('.custom-modal-backdrop').forEach(backdrop => backdrop.remove());
        
        // Show modal using simple CSS classes instead of Bootstrap
        modalElement.style.display = 'flex';
        modalElement.classList.add('show');
        document.body.classList.add('modal-open');
        
        // Focus on first input
        setTimeout(() => {
            const firstInput = modalElement.querySelector('input[type="text"]');
            if (firstInput) firstInput.focus();
        }, 100);
        
        // Add close button handlers
        const closeButtons = modalElement.querySelectorAll('[data-bs-dismiss="modal"]');
        closeButtons.forEach(btn => {
            btn.onclick = (e) => {
                e.preventDefault();
                this.closeAcademicYearModal();
            };
        });
    }
    
    closeAcademicYearModal() {
        const modalElement = document.getElementById('academicYearModal');
        if (modalElement) {
            modalElement.style.display = 'none';
            modalElement.classList.remove('show');
        }
        
        // Clean up body classes and backdrops
        document.body.classList.remove('modal-open');
        document.querySelectorAll('.modal-backdrop').forEach(backdrop => backdrop.remove());
        document.querySelectorAll('.custom-modal-backdrop').forEach(backdrop => backdrop.remove());
    }
    
    async saveAcademicYear() {
        const yearIdField = document.getElementById('yearId');
        const yearNameField = document.getElementById('yearName');
        const startsOnField = document.getElementById('startsOn');
        const endsOnField = document.getElementById('endsOn');
        const yearStatusField = document.getElementById('yearStatus');
        
        if (!yearNameField || !yearStatusField) {
            this.showToast('error', 'Form fields not found. Please refresh the page.');
            return;
        }
        
        const yearId = yearIdField ? yearIdField.value : '';
        const isEdit = !!yearId;
        
        const data = {
            name: yearNameField.value.trim(),
            starts_on: startsOnField ? (startsOnField.value || null) : null,
            ends_on: endsOnField ? (endsOnField.value || null) : null,
            status: yearStatusField.value
        };
        
        // Clear previous validation
        document.querySelectorAll('.is-invalid').forEach(el => el.classList.remove('is-invalid'));
        
        try {
            const endpoint = isEdit ? `/admin/academic-years/${yearId}` : '/admin/academic-years';
            const method = isEdit ? 'PUT' : 'POST';
            
            const response = await this.apiCall(method, endpoint, data);
            
            if (response.success) {
                // Close modal manually
                this.closeAcademicYearModal();
                
                // Show success message
                this.showToast('success', `Academic year ${isEdit ? 'updated' : 'created'} successfully!`);
                
                // Reload the list
                this.loadAcademicYears();
            } else {
                this.showToast('error', response.message);
            }
            
        } catch (error) {
            if (error.message.includes('422')) {
                // Validation errors
                try {
                    const errorResponse = JSON.parse(error.message);
                    this.handleValidationErrors(errorResponse.errors);
                } catch {
                    this.showToast('error', 'Validation failed. Please check your input.');
                }
            } else {
                this.showToast('error', 'Failed to save academic year. Please try again.');
            }
        }
    }
    
    async editAcademicYear(id) {
        try {
            const response = await this.apiCall('GET', `/admin/academic-years/${id}`);
            if (response.success) {
                this.showAcademicYearModal(response.data);
            }
        } catch (error) {
            this.showToast('error', 'Failed to load academic year data');
        }
    }
    
    async deleteAcademicYear(id, name) {
        if (!confirm(`Are you sure you want to delete the academic year "${name}"?\n\nThis action cannot be undone and may affect related data.`)) {
            return;
        }
        
        try {
            const response = await this.apiCall('DELETE', `/admin/academic-years/${id}`);
            
            if (response.success) {
                this.showToast('success', 'Academic year deleted successfully!');
                this.loadAcademicYears();
            } else {
                this.showToast('error', response.message);
            }
        } catch (error) {
            this.showToast('error', 'Failed to delete academic year. It may have related data.');
        }
    }
    
    async filterAcademicYears() {
        const name = document.getElementById('nameFilter').value;
        const status = document.getElementById('statusFilter').value;
        
        const params = new URLSearchParams();
        if (name) params.append('name', name);
        if (status) params.append('status', status);
        
        try {
            const response = await this.apiCall('GET', `/admin/academic-years?${params.toString()}`);
            document.getElementById('academicYearsTableBody').innerHTML = this.renderAcademicYearsTable(response.data);
        } catch (error) {
            this.showToast('error', 'Failed to filter academic years');
        }
    }
    
    clearFilters() {
        document.getElementById('nameFilter').value = '';
        document.getElementById('statusFilter').value = '';
        this.loadAcademicYears();
    }
    
    async sortAcademicYears(field) {
        // Toggle sort direction
        this.currentSort = this.currentSort || {};
        const currentDir = this.currentSort[field] || 'asc';
        const newDir = currentDir === 'asc' ? 'desc' : 'asc';
        this.currentSort[field] = newDir;
        
        try {
            const response = await this.apiCall('GET', `/admin/academic-years?sort_by=${field}&sort_dir=${newDir}`);
            document.getElementById('academicYearsTableBody').innerHTML = this.renderAcademicYearsTable(response.data);
            
            // Update sort indicator
            document.querySelectorAll('.sortable i').forEach(icon => {
                icon.className = 'bi bi-arrow-down-up';
            });
            const sortIcon = document.querySelector(`[data-sort="${field}"] i`);
            if (sortIcon) {
                sortIcon.className = `bi bi-arrow-${newDir === 'asc' ? 'up' : 'down'}`;
            }
        } catch (error) {
            this.showToast('error', 'Failed to sort academic years');
        }
    }
    
    async loadAcademicYearsPage(page) {
        try {
            const response = await this.apiCall('GET', `/admin/academic-years?page=${page}`);
            document.getElementById('academicYearsTableBody').innerHTML = this.renderAcademicYearsTable(response.data);
            document.querySelector('.pagination').outerHTML = this.renderPagination(response.meta);
        } catch (error) {
            this.showToast('error', 'Failed to load page');
        }
    }
    
    async exportAcademicYears(format) {
        try {
            const response = await this.apiCall('GET', '/admin/academic-years?export=' + format);
            
            if (response.success && response.data) {
                if (format === 'csv') {
                    this.downloadFile(response.data, 'academic-years.csv', 'text/csv');
                } else if (format === 'excel') {
                    this.downloadFile(response.data, 'academic-years.xlsx', 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet');
                }
                
                this.showToast('success', `Academic years exported as ${format.toUpperCase()}`);
            } else {
                this.showToast('error', 'No data to export');
            }
        } catch (error) {
            this.showToast('error', 'Export failed. Please try again.');
        }
    }
    
    printAcademicYears() {
        const printContent = `
            <html>
            <head>
                <title>Academic Years Report</title>
                <style>
                    body { font-family: Arial, sans-serif; margin: 20px; }
                    h1 { color: #333; border-bottom: 2px solid #333; }
                    table { width: 100%; border-collapse: collapse; margin-top: 20px; }
                    th, td { border: 1px solid #ddd; padding: 8px; text-align: left; }
                    th { background-color: #f5f5f5; font-weight: bold; }
                    tr:nth-child(even) { background-color: #f9f9f9; }
                    .print-date { text-align: right; color: #666; margin-bottom: 20px; }
                </style>
            </head>
            <body>
                <div class="print-date">Generated on: ${new Date().toLocaleDateString()}</div>
                <h1>Academic Years Report</h1>
                <table>
                    <thead>
                        <tr>
                            <th>Name</th>
                            <th>Start Date</th>
                            <th>End Date</th>
                            <th>Status</th>
                            <th>Created</th>
                        </tr>
                    </thead>
                    <tbody>
                        ${document.getElementById('academicYearsTableBody').innerHTML.replace(/<button[^>]*>.*?<\/button>/gi, '')}
                    </tbody>
                </table>
            </body>
            </html>
        `;
        
        const printWindow = window.open('', '_blank');
        printWindow.document.write(printContent);
        printWindow.document.close();
        printWindow.print();
    }
    
    downloadFile(data, filename, mimeType) {
        const blob = new Blob([data], { type: mimeType });
        const url = window.URL.createObjectURL(blob);
        const link = document.createElement('a');
        link.href = url;
        link.download = filename;
        document.body.appendChild(link);
        link.click();
        document.body.removeChild(link);
        window.URL.revokeObjectURL(url);
    }
    
    handleValidationErrors(errors) {
        Object.keys(errors).forEach(field => {
            const input = document.getElementById(field === 'name' ? 'yearName' : field);
            if (input) {
                input.classList.add('is-invalid');
                const feedback = input.nextElementSibling;
                if (feedback && feedback.classList.contains('invalid-feedback')) {
                    feedback.textContent = errors[field][0];
                }
            }
        });
    }
    
    showToast(type, message) {
        // Create toast element
        const toastHtml = `
            <div class="toast align-items-center text-white bg-${type === 'success' ? 'success' : type === 'error' ? 'danger' : 'info'} border-0" role="alert" aria-live="assertive" aria-atomic="true">
                <div class="d-flex">
                    <div class="toast-body">
                        <i class="bi bi-${type === 'success' ? 'check-circle' : type === 'error' ? 'exclamation-triangle' : 'info-circle'}"></i>
                        ${message}
                    </div>
                    <button type="button" class="btn-close btn-close-white me-2 m-auto" data-bs-dismiss="toast"></button>
                </div>
            </div>
        `;
        
        // Add to toast container or create one
        let toastContainer = document.getElementById('toastContainer');
        if (!toastContainer) {
            toastContainer = document.createElement('div');
            toastContainer.id = 'toastContainer';
            toastContainer.className = 'toast-container position-fixed top-0 end-0 p-3';
            toastContainer.style.zIndex = '9999';
            document.body.appendChild(toastContainer);
        }
        
        // Add toast
        toastContainer.insertAdjacentHTML('beforeend', toastHtml);
        const toastElement = toastContainer.lastElementChild;
        const toast = new bootstrap.Toast(toastElement);
        toast.show();
        
        // Remove after hiding
        toastElement.addEventListener('hidden.bs.toast', () => {
            toastElement.remove();
        });
    }
    
    // Students Management
    async loadStudents() {
        try {
            const content = `
                <div class="page-header">
                    <div class="d-flex justify-content-between align-items-center">
                        <div>
                            <h1 class="h2 mb-1"><i class="bi bi-people"></i> <span data-translate="students.title">Students Management</span></h1>
                            <p class="text-muted mb-0" data-translate="students.subtitle">Manage student information and login codes</p>
                        </div>
                        <div class="btn-group">
                            <button class="btn btn-outline-danger" id="bulkDeleteBtn" onclick="app.bulkDeleteStudents()" style="display: none;">
                                <i class="bi bi-trash"></i> <span data-translate="students.delete_selected">Delete Selected</span> (<span id="selectedCount">0</span>)
                            </button>
                            <div class="btn-group">
                                <button class="btn btn-outline-primary" onclick="app.showStudentImportModal()">
                                    <i class="bi bi-upload"></i> <span data-translate="students.import_students">Import Students</span>
                                </button>
                                <button class="btn btn-outline-success" onclick="app.showStudentEnrollmentImportModal()">
                                    <i class="bi bi-file-earmark-spreadsheet"></i> <span data-translate="students.import_with_enrollment">Import with Enrollment</span>
                                </button>
                            </div>
                            <button class="btn btn-primary" onclick="app.showStudentModal()">
                                <i class="bi bi-plus-circle"></i> <span data-translate="students.add_student">Add Student</span>
                            </button>
                        </div>
                    </div>
                </div>
                
                <!-- Filters -->
                <div class="filter-bar">
                    <div class="row g-3">
                        <div class="col-md-3">
                            <label class="form-label" data-translate="students.search_name">Search Name</label>
                            <input type="text" class="form-control" id="studentNameFilter" data-translate-placeholder="students.search_by_name" placeholder="Search by name...">
                        </div>
                        <div class="col-md-2">
                            <label class="form-label" data-translate="students.national_id">National ID</label>
                            <input type="text" class="form-control" id="studentNationalIdFilter" data-translate-placeholder="students.search_by_id" placeholder="Search by ID...">
                        </div>
                        <div class="col-md-1">
                            <label class="form-label" data-translate="students.gender">Gender</label>
                            <select class="form-select" id="studentGenderFilter">
                                <option value="" data-translate="students.all">All</option>
                                <option value="M">M</option>
                                <option value="F">F</option>
                            </select>
                        </div>
                    </div>
                    <div class="row mb-3">
                        <div class="col-md-2">
                            <label class="form-label" data-translate="students.academic_year">Academic Year</label>
                            <select class="form-select" id="studentYearFilter">
                                <option value="" data-translate="students.all_years">All Years</option>
                            </select>
                        </div>
                        <div class="col-md-1">
                            <label class="form-label" data-translate="students.grade">Grade</label>
                            <select class="form-select" id="studentGradeFilter">
                                <option value="" data-translate="students.all">All</option>
                            </select>
                        </div>
                        <div class="col-md-1">
                            <label class="form-label" data-translate="students.class">Class</label>
                            <select class="form-select" id="studentClassFilter">
                                <option value="" data-translate="students.all">All</option>
                            </select>
                        </div>
                        <div class="col-md-1">
                            <label class="form-label" data-translate="students.status">Status</label>
                            <select class="form-select" id="studentStatusFilter">
                                <option value="">All</option>
                                <option value="1">Active</option>
                                <option value="0">Disabled</option>
                            </select>
                        </div>
                        <div class="col-md-2">
                            <label class="form-label" data-translate="students.enrollment_status">Enrollment Status</label>
                            <select class="form-select" id="studentEnrollmentStatusFilter">
                                <option value="" data-translate="students.all">All</option>
                                <option value="active" data-translate="students.active">Active</option>
                                <option value="inactive" data-translate="students.inactive">Inactive</option>
                                <option value="graduated" data-translate="students.graduated">Graduated</option>
                            </select>
                        </div>
                        <div class="col-md-1">
                            <label class="form-label" data-translate="students.per_page">Per Page</label>
                            <select class="form-select" id="studentsPerPage" onchange="app.changeStudentsPerPage()">
                                <option value="10">10</option>
                                <option value="25" selected>25</option>
                                <option value="50">50</option>
                                <option value="100">100</option>
                                <option value="200">200</option>
                                <option value="500">500</option>
                            </select>
                        </div>
                        <div class="col-md-2">
                            <label class="form-label">&nbsp;</label>
                            <div class="d-flex gap-2">
                                <button class="btn btn-primary" onclick="app.filterStudents()" title="Apply Filters">
                                    <i class="bi bi-search"></i> <span data-translate="students.filter">Filter</span>
                                </button>
                                <button class="btn btn-outline-secondary" onclick="app.clearStudentsFilters()" title="Clear All Filters">
                                    <i class="bi bi-x-circle"></i> <span data-translate="students.clear">Clear</span>
                                </button>
                            </div>
                        </div>
                    </div>
                </div>
                
                <!-- Students Table -->
                <div class="table-container">
                    <div class="table-responsive">
                        <table class="table table-hover">
                            <thead class="table-light">
                                <tr>
                                    <th style="width: 50px;">
                                        <input type="checkbox" class="form-check-input" id="selectAllStudents" onchange="app.toggleSelectAll()">
                                    </th>
                                    <th style="width: 70px;">
                                        <a href="#" onclick="app.sortStudents('s.id')" class="text-decoration-none text-dark">
                                            <span data-translate="students.id">ID</span> <i class="bi bi-arrow-down-up text-muted"></i>
                                        </a>
                                    </th>
                                    <th style="width: 110px;">
                                        <a href="#" onclick="app.sortStudents('login_code')" class="text-decoration-none text-dark">
                                            <span data-translate="students.login_code_header">Login Code</span> <i class="bi bi-arrow-down-up text-muted"></i>
                                        </a>
                                    </th>
                                    <th>
                                        <a href="#" onclick="app.sortStudents('s.full_name')" class="text-decoration-none text-dark">
                                            <span data-translate="students.full_name_header">Full Name</span> <i class="bi bi-arrow-down-up text-muted"></i>
                                        </a>
                                    </th>
                                    <th style="width: 70px;" data-translate="students.gender_header">Gender</th>
                                    <th style="width: 120px;">
                                        <a href="#" onclick="app.sortStudents('ay.name')" class="text-decoration-none text-dark">
                                            <span data-translate="students.year_header">Year</span> <i class="bi bi-arrow-down-up text-muted"></i>
                                        </a>
                                    </th>
                                    <th style="width: 100px;">
                                        <a href="#" onclick="app.sortStudents('g.title')" class="text-decoration-none text-dark">
                                            <span data-translate="students.grade_header">Grade</span> <i class="bi bi-arrow-down-up text-muted"></i>
                                        </a>
                                    </th>
                                    <th style="width: 100px;">
                                        <a href="#" onclick="app.sortStudents('c.name')" class="text-decoration-none text-dark">
                                            <span data-translate="students.class_header">Class</span> <i class="bi bi-arrow-down-up text-muted"></i>
                                        </a>
                                    </th>
                                    <th style="width: 80px;" data-translate="students.status_header">Status</th>
                                    <th style="width: 110px;" data-translate="students.actions">Actions</th>
                                </tr>
                            </thead>
                            <tbody id="studentsTableBody">
                                <tr>
                                    <td colspan="10" class="text-center py-4">
                                        <div class="spinner-border text-primary" role="status">
                                            <span class="visually-hidden" data-translate="dashboard.loading">Loading...</span>
                                        </div>
                                        <div class="mt-2" data-translate="dashboard.loading_students">Loading students...</div>
                                    </td>
                                </tr>
                            </tbody>
                        </table>
                    </div>
                </div>
                
                <!-- Pagination -->
                <div id="studentsPagination" class="d-flex justify-content-between align-items-center mt-3">
                    <div class="text-muted">
                        <span id="studentsInfo" data-translate="dashboard.loading_results">Loading...</span>
                    </div>
                    <nav>
                        <ul class="pagination pagination-sm mb-0" id="studentsPaginationNav">
                        </ul>
                    </nav>
                </div>
                
                <!-- Student Modal -->
                <div class="modal" id="studentModal">
                    <div class="modal-dialog modal-lg">
                        <div class="modal-content">
                            <div class="modal-header">
                                <h5 class="modal-title" id="studentModalTitle" data-translate="students.add_student">Add Student</h5>
                                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                            </div>
                            <div class="modal-body">
                                <form id="studentForm">
                                    <input type="hidden" id="studentId">
                                    
                                    <div class="row">
                                        <div class="col-md-8">
                                            <div class="mb-3">
                                                <label for="studentFullName" class="form-label" data-translate="students.full_name">Full Name *</label>
                                                <input type="text" class="form-control" id="studentFullName" required>
                                            </div>
                                        </div>
                                        <div class="col-md-4">
                                            <div class="mb-3">
                                                <label for="studentGender" class="form-label" data-translate="students.gender">Gender</label>
                                                <select class="form-select" id="studentGender">
                                                    <option value="" data-translate="students.select_gender">Select Gender</option>
                                                    <option value="M" data-translate="students.male">Male</option>
                                                    <option value="F" data-translate="students.female">Female</option>
                                                </select>
                                            </div>
                                        </div>
                                    </div>
                                    
                                    <div class="row">
                                        <div class="col-md-6">
                                            <div class="mb-3">
                                                <label for="studentNationalId" class="form-label" data-translate="students.national_id">National ID</label>
                                                <input type="text" class="form-control" id="studentNationalId" data-translate-placeholder="students.optional" placeholder="Optional">
                                            </div>
                                        </div>
                                        <div class="col-md-6">
                                            <div class="mb-3">
                                                <label for="studentBirthdate" class="form-label" data-translate="students.birthdate">Birth Date</label>
                                                <input type="date" class="form-control" id="studentBirthdate">
                                            </div>
                                        </div>
                                    </div>
                                    
                                    <div class="row">
                                        <div class="col-md-6">
                                            <div class="mb-3">
                                                <label for="studentLoginCode" class="form-label" data-translate="students.login_code">Login Code</label>
                                                <div class="input-group">
                                                    <input type="text" class="form-control" id="studentLoginCode" readonly data-translate-placeholder="students.auto_generated" placeholder="Auto-generated">
                                                    <button type="button" class="btn btn-outline-secondary" onclick="app.generateNewLoginCode()" title="Generate new code">
                                                        <i class="bi bi-arrow-clockwise"></i>
                                                    </button>
                                                </div>
                                                <div class="form-text" data-translate="students.login_code_help">8-digit unique code for student login</div>
                                            </div>
                                        </div>
                                        <div class="col-md-6">
                                            <div class="mb-3">
                                                <label for="studentEnabled" class="form-label" data-translate="students.status">Status</label>
                                                <select class="form-select" id="studentEnabled">
                                                    <option value="1" data-translate="students.active">Active</option>
                                                    <option value="0" data-translate="students.disabled">Disabled</option>
                                                </select>
                                            </div>
                                        </div>
                                    </div>
                                    
                                    <hr>
                                    <h6><i class="bi bi-person-plus"></i> <span data-translate="students.enrollment_info">Enrollment Information</span></h6>
                                    
                                    <div class="form-check mb-3">
                                        <input class="form-check-input" type="checkbox" id="enableEnrollment" onchange="app.toggleEnrollmentFields()">
                                        <label class="form-check-label" for="enableEnrollment">
                                            <strong data-translate="students.enroll_immediately">Enroll student immediately</strong>
                                        </label>
                                        <div class="form-text" data-translate="students.enroll_help">Check this to assign the student to a class right away</div>
                                    </div>
                                    
                                    <div id="enrollmentFields" class="d-none">
                                        <div class="row">
                                            <div class="col-md-4">
                                                <div class="mb-3">
                                                    <label for="enrollmentYear" class="form-label">Academic Year *</label>
                                                    <select class="form-select" id="enrollmentYear">
                                                        <option value="">Select Year</option>
                                                    </select>
                                                </div>
                                            </div>
                                            <div class="col-md-4">
                                                <div class="mb-3">
                                                    <label for="enrollmentGrade" class="form-label">Grade *</label>
                                                    <select class="form-select" id="enrollmentGrade">
                                                        <option value="">Select Grade</option>
                                                    </select>
                                                </div>
                                            </div>
                                            <div class="col-md-4">
                                                <div class="mb-3">
                                                    <label for="enrollmentClass" class="form-label">Class *</label>
                                                    <select class="form-select" id="enrollmentClass">
                                                        <option value="">Select Class</option>
                                                    </select>
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                    
                                    <div class="alert alert-info">
                                        <i class="bi bi-info-circle"></i>
                                        <strong>Note:</strong> The login code will be automatically generated and can be used by the student to access their portal.
                                    </div>
                                </form>
                            </div>
                            <div class="modal-footer">
                                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                                <button type="button" class="btn btn-primary" onclick="app.saveStudent()">
                                    <span id="studentSaveButtonText">Save Student</span>
                                </button>
                            </div>
                        </div>
                    </div>
                </div>
                
                <!-- Import Students Modal -->
                <div class="modal fade" id="studentImportModal" tabindex="-1">
                    <div class="modal-dialog modal-lg">
                        <div class="modal-content">
                            <div class="modal-header">
                                <h5 class="modal-title">
                                    <i class="bi bi-upload"></i> Import Students
                                </h5>
                                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                            </div>
                            <div class="modal-body">
                                <div class="row">
                                    <div class="col-md-6">
                                        <h6><i class="bi bi-info-circle"></i> Instructions</h6>
                                        <ul class="text-muted small">
                                            <li><strong>Required:</strong> full_name</li>
                                            <li><strong>Optional:</strong> national_id, gender (M/F), birthdate (YYYY-MM-DD)</li>
                                            <li>Login codes will be auto-generated</li>
                                            <li>Maximum file size: 5MB</li>
                                            <li>Supported format: CSV only</li>
                                        </ul>
                                        
                                        <div class="mb-3">
                                            <a href="#" onclick="app.downloadSampleCSV()" class="btn btn-outline-info btn-sm">
                                                <i class="bi bi-download"></i> Download Sample CSV
                                            </a>
                                        </div>
                                    </div>
                                    <div class="col-md-6">
                                        <h6><i class="bi bi-file-earmark-spreadsheet"></i> CSV Format Example</h6>
                                        <div class="bg-light p-2 rounded small font-monospace">
                                            full_name,national_id,gender,birthdate<br>
                                            John Smith,123456789,M,2010-05-15<br>
                                            Jane Doe,987654321,F,2011-03-20<br>
                                            Ahmed Ali,,M,2009-12-10
                                        </div>
                                    </div>
                                </div>
                                
                                <hr>
                                
                                <div class="row mb-3">
                                    <div class="col-md-12">
                                        <h6><i class="bi bi-person-plus"></i> Enrollment Settings</h6>
                                        <div class="form-check mb-3">
                                            <input class="form-check-input" type="checkbox" id="importEnableEnrollment" onchange="app.toggleImportEnrollmentFields()">
                                            <label class="form-check-label" for="importEnableEnrollment">
                                                <strong>Enroll all imported students</strong>
                                            </label>
                                            <div class="form-text">All students will be enrolled in the same class</div>
                                        </div>
                                        
                                        <div id="importEnrollmentFields" class="d-none">
                                            <div class="row">
                                                <div class="col-md-4">
                                                    <div class="mb-3">
                                                        <label for="importEnrollmentYear" class="form-label">Academic Year *</label>
                                                        <select class="form-select" id="importEnrollmentYear">
                                                            <option value="">Select Year</option>
                                                        </select>
                                                    </div>
                                                </div>
                                                <div class="col-md-4">
                                                    <div class="mb-3">
                                                        <label for="importEnrollmentGrade" class="form-label">Grade *</label>
                                                        <select class="form-select" id="importEnrollmentGrade">
                                                            <option value="">Select Grade</option>
                                                        </select>
                                                    </div>
                                                </div>
                                                <div class="col-md-4">
                                                    <div class="mb-3">
                                                        <label for="importEnrollmentClass" class="form-label">Class *</label>
                                                        <select class="form-select" id="importEnrollmentClass">
                                                            <option value="">Select Class</option>
                                                        </select>
                                                    </div>
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                                
                                <form id="studentImportForm" enctype="multipart/form-data">
                                    <div class="mb-3">
                                        <label for="importFile" class="form-label">Select CSV File</label>
                                        <input type="file" class="form-control" id="importFile" accept=".csv" required>
                                        <div class="form-text">Choose a CSV file containing student data</div>
                                    </div>
                                    
                                    <div id="importProgress" class="d-none">
                                        <div class="progress mb-3">
                                            <div class="progress-bar progress-bar-striped progress-bar-animated" style="width: 100%">
                                                Processing...
                                            </div>
                                        </div>
                                    </div>
                                    
                                    <div id="importResults" class="d-none">
                                        <div class="alert alert-info">
                                            <h6><i class="bi bi-info-circle"></i> Import Results</h6>
                                            <div id="importSummary"></div>
                                            <div id="importErrors" class="mt-2"></div>
                                        </div>
                                    </div>
                                </form>
                            </div>
                            <div class="modal-footer">
                                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                                <button type="button" class="btn btn-primary" onclick="app.importStudents()" id="importButton">
                                    <i class="bi bi-upload"></i> Import Students
                                </button>
                            </div>
                        </div>
                    </div>
                </div>
            `;
            
            document.getElementById('pageContent').innerHTML = content;
            
            // Load filter options
            await this.loadStudentsFilterOptions();
            
            // Load students data
            await this.loadStudentsData();
            
        } catch (error) {
            this.showToast('error', 'Failed to load students page');
        }
    }
    
    async loadStudentsFilterOptions() {
        try {
            // Load academic years
            const yearResponse = await this.apiCall('GET', '/admin/academic-years');
            const yearSelect = document.getElementById('studentYearFilter');
            if (yearSelect && yearResponse.data) {
                yearSelect.innerHTML = '<option value="">All Years</option>';
                yearResponse.data.forEach(year => {
                    yearSelect.innerHTML += `<option value="${year.id}">${year.name}</option>`;
                });
            }
            
            // Load grades
            const gradeResponse = await this.apiCall('GET', '/admin/grades');
            const gradeSelect = document.getElementById('studentGradeFilter');
            if (gradeSelect && gradeResponse.data) {
                gradeSelect.innerHTML = '<option value="">All</option>';
                gradeResponse.data.forEach(grade => {
                    gradeSelect.innerHTML += `<option value="${grade.id}">${grade.title}</option>`;
                });
            }
            
            // Load classes
            const classResponse = await this.apiCall('GET', '/admin/classes');
            const classSelect = document.getElementById('studentClassFilter');
            if (classSelect && classResponse.data) {
                classSelect.innerHTML = '<option value="">All</option>';
                classResponse.data.forEach(cls => {
                    classSelect.innerHTML += `<option value="${cls.id}">${cls.name}</option>`;
                });
            }
            
            // Note: Filters now work on submit button click only
            // No automatic filtering event listeners added
            
        } catch (error) {
            console.warn('Failed to load filter options:', error);
        }
    }
    
    async loadStudentsData(page = 1, perPage = 25, filters = {}) {
        try {
            // Build query parameters
            let queryParams = new URLSearchParams({
                page: page,
                per_page: perPage
            });
            
            // Add filters
            if (filters.name) queryParams.append('name', filters.name);
            if (filters.national_id) queryParams.append('national_id', filters.national_id);
            if (filters.gender) queryParams.append('gender', filters.gender);
            if (filters.enabled !== undefined) queryParams.append('enabled', filters.enabled);
            if (filters.year_id) queryParams.append('year_id', filters.year_id);
            if (filters.grade_id) queryParams.append('grade_id', filters.grade_id);
            if (filters.class_id) queryParams.append('class_id', filters.class_id);
            if (filters.enrollment_status) queryParams.append('enrollment_status', filters.enrollment_status);
            
            const response = await this.apiCall('GET', `/admin/students?${queryParams.toString()}`);
            document.getElementById('studentsTableBody').innerHTML = this.renderStudentsTable(response.data);
            this.updateStudentsPagination(response.meta);
        } catch (error) {
            document.getElementById('studentsTableBody').innerHTML = `
                <tr>
                    <td colspan="9" class="text-center py-4 text-danger">
                        <i class="bi bi-exclamation-triangle"></i> Failed to load students
                    </td>
                </tr>
            `;
        }
    }
    
    renderStudentsTable(students) {
        if (!students || students.length === 0) {
            return `
                <tr>
                    <td colspan="10" class="text-center py-4 text-muted">
                        <i class="bi bi-people"></i><br>
                        No students found
                    </td>
                </tr>
            `;
        }
        
        return students.map(student => `
            <tr>
                <td>
                    <input type="checkbox" class="form-check-input student-checkbox" value="${student.id}" onchange="app.updateSelectedCount()">
                </td>
                <td>${student.id}</td>
                <td>
                    <span class="badge bg-primary font-monospace">${student.login_code}</span>
                </td>
                <td>
                    <div class="fw-bold">${student.full_name}</div>
                    ${student.national_id ? `<small class="text-muted">ID: ${student.national_id}</small>` : ''}
                </td>
                <td>
                    ${student.gender === 'M' ? '<i class="bi bi-person text-primary"></i> M' : 
                      student.gender === 'F' ? '<i class="bi bi-person-dress text-pink"></i> F' : 
                      '<span class="text-muted">-</span>'}
                </td>
                <td>
                    ${student.year_name ? `<span class="badge bg-info">${student.year_name}</span>` : '<span class="text-muted">Not Enrolled</span>'}
                </td>
                <td>
                    ${student.grade_title ? `<span class="badge bg-secondary">${student.grade_title}</span>` : '<span class="text-muted">-</span>'}
                </td>
                <td>
                    ${student.class_name ? `<span class="badge bg-warning text-dark">${student.class_name}</span>` : '<span class="text-muted">-</span>'}
                </td>
                <td>
                    <span class="badge ${student.enabled ? 'bg-success' : 'bg-secondary'}">
                        ${student.enabled ? 'Active' : 'Disabled'}
                    </span>
                </td>
                <td>
                    <div class="btn-group" role="group">
                        <button class="btn btn-outline-primary btn-sm" onclick="app.editStudent(${student.id})" title="Edit">
                            <i class="bi bi-pencil"></i>
                        </button>
                        <button class="btn btn-outline-danger btn-sm" onclick="app.deleteStudent(${student.id}, '${student.full_name.replace(/'/g, '&#39;')}')" title="Delete">
                            <i class="bi bi-trash"></i>
                        </button>
                    </div>
                </td>
            </tr>
        `).join('');
    }
    
    updateStudentsPagination(meta) {
        const info = document.getElementById('studentsInfo');
        const nav = document.getElementById('studentsPaginationNav');
        
        if (meta) {
            const start = ((meta.page - 1) * meta.per_page) + 1;
            const end = Math.min(meta.page * meta.per_page, meta.total_rows);
            info.textContent = `Showing ${start}-${end} of ${meta.total_rows} students`;
            
            // Generate pagination
            let paginationHtml = '';
            if (meta.total_pages > 1) {
                // Previous button
                paginationHtml += `
                    <li class="page-item ${meta.page === 1 ? 'disabled' : ''}">
                        <a class="page-link" href="#" onclick="app.loadStudentsPage(${meta.page - 1})">Previous</a>
                    </li>
                `;
                
                // Page numbers
                for (let i = 1; i <= meta.total_pages; i++) {
                    if (i === meta.page || Math.abs(i - meta.page) <= 2 || i === 1 || i === meta.total_pages) {
                        paginationHtml += `
                            <li class="page-item ${i === meta.page ? 'active' : ''}">
                                <a class="page-link" href="#" onclick="app.loadStudentsPage(${i})">${i}</a>
                            </li>
                        `;
                    } else if (Math.abs(i - meta.page) === 3) {
                        paginationHtml += '<li class="page-item disabled"><span class="page-link">...</span></li>';
                    }
                }
                
                // Next button
                paginationHtml += `
                    <li class="page-item ${meta.page === meta.total_pages ? 'disabled' : ''}">
                        <a class="page-link" href="#" onclick="app.loadStudentsPage(${meta.page + 1})">Next</a>
                    </li>
                `;
            }
            nav.innerHTML = paginationHtml;
        }
    }
    
    async loadStudentsPage(page) {
        const perPage = document.getElementById('studentsPerPage')?.value || 25;
        const filters = this.getStudentsFilters();
        await this.loadStudentsData(page, perPage, filters);
    }
    
    changeStudentsPerPage() {
        // Reset to page 1 when changing page size
        this.loadStudentsPage(1);
    }
    
    getStudentsFilters() {
        return {
            name: document.getElementById('studentNameFilter')?.value || '',
            national_id: document.getElementById('studentNationalIdFilter')?.value || '',
            gender: document.getElementById('studentGenderFilter')?.value || '',
            enabled: document.getElementById('studentStatusFilter')?.value || undefined,
            year_id: document.getElementById('studentYearFilter')?.value || '',
            grade_id: document.getElementById('studentGradeFilter')?.value || '',
            class_id: document.getElementById('studentClassFilter')?.value || '',
            enrollment_status: document.getElementById('studentEnrollmentStatusFilter')?.value || ''
        };
    }
    
    async filterStudents() {
        // Reset to page 1 when applying filters
        await this.loadStudentsPage(1);
    }
    
    clearStudentsFilters() {
        document.getElementById('studentNameFilter').value = '';
        document.getElementById('studentNationalIdFilter').value = '';
        document.getElementById('studentGenderFilter').value = '';
        document.getElementById('studentStatusFilter').value = '';
        document.getElementById('studentYearFilter').value = '';
        document.getElementById('studentGradeFilter').value = '';
        document.getElementById('studentClassFilter').value = '';
        document.getElementById('studentEnrollmentStatusFilter').value = '';
        // Reload with cleared filters
        this.loadStudentsPage(1);
    }
    
    getCurrentStudentPage() {
        // Try to get current page from pagination display
        const activePageLink = document.querySelector('#studentsPaginationNav .page-item.active .page-link');
        return activePageLink ? parseInt(activePageLink.textContent) : 1;
    }
    
    // Student CRUD Functions
    showStudentModal(studentData = null) {
        const modalElement = document.getElementById('studentModal');
        if (!modalElement) {
            this.showToast('error', 'Modal not found. Please refresh the page.');
            return;
        }
        
        // Remove any existing backdrops
        document.querySelectorAll('.modal-backdrop').forEach(backdrop => backdrop.remove());
        
        // Reset form
        const form = document.getElementById('studentForm');
        if (form) form.reset();
        
        const studentIdField = document.getElementById('studentId');
        if (studentIdField) studentIdField.value = '';
        
        if (studentData) {
            // Edit mode
            document.getElementById('studentModalTitle').textContent = 'Edit Student';
            document.getElementById('studentSaveButtonText').textContent = 'Update Student';
            if (studentIdField) studentIdField.value = studentData.id;
            document.getElementById('studentFullName').value = studentData.full_name;
            document.getElementById('studentNationalId').value = studentData.national_id || '';
            document.getElementById('studentGender').value = studentData.gender || '';
            document.getElementById('studentBirthdate').value = studentData.birthdate || '';
            document.getElementById('studentLoginCode').value = studentData.login_code;
            document.getElementById('studentEnabled').value = studentData.enabled ? "1" : "0";
            
            // Handle enrollment information for edit mode
            const enableEnrollmentCheckbox = document.getElementById('enableEnrollment');
            const enrollmentFields = document.getElementById('enrollmentFields');
            
            if (studentData.year_id || studentData.grade_id || studentData.class_id) {
                // Student has enrollment data
                enableEnrollmentCheckbox.checked = true;
                enrollmentFields.classList.remove('d-none');
                
                // Load enrollment options and then set values
                this.loadEnrollmentOptions().then(() => {
                    if (studentData.year_id) {
                        document.getElementById('enrollmentYear').value = studentData.year_id;
                    }
                    if (studentData.grade_id) {
                        document.getElementById('enrollmentGrade').value = studentData.grade_id;
                    }
                    if (studentData.class_id) {
                        document.getElementById('enrollmentClass').value = studentData.class_id;
                    }
                });
            } else {
                // No enrollment data
                enableEnrollmentCheckbox.checked = false;
                enrollmentFields.classList.add('d-none');
            }
        } else {
            // Add mode
            document.getElementById('studentModalTitle').textContent = 'Add Student';
            document.getElementById('studentSaveButtonText').textContent = 'Save Student';
            document.getElementById('studentEnabled').value = "1";
            // Generate login code for new student
            this.generateNewLoginCode();
        }
        
        // Clear validation classes
        document.querySelectorAll('.is-invalid').forEach(el => el.classList.remove('is-invalid'));
        
        // Show modal
        modalElement.style.display = 'flex';
        modalElement.classList.add('show');
        document.body.classList.add('modal-open');
        
        // Add close button handlers
        const closeButtons = modalElement.querySelectorAll('[data-bs-dismiss="modal"]');
        closeButtons.forEach(btn => {
            btn.onclick = (e) => {
                e.preventDefault();
                this.closeStudentModal();
            };
        });
        
        // Focus on first input
        setTimeout(() => {
            const firstInput = modalElement.querySelector('input[type="text"]');
            if (firstInput) firstInput.focus();
        }, 100);
    }
    
    closeStudentModal() {
        const modalElement = document.getElementById('studentModal');
        if (modalElement) {
            modalElement.style.display = 'none';
            modalElement.classList.remove('show');
        }
        document.body.classList.remove('modal-open');
        document.querySelectorAll('.modal-backdrop').forEach(backdrop => backdrop.remove());
    }
    
    generateNewLoginCode() {
        // Generate 8-digit code
        const code = Math.floor(10000000 + Math.random() * 90000000).toString();
        document.getElementById('studentLoginCode').value = code;
        this.showToast('info', 'New login code generated');
    }
    
    async saveStudent() {
        const studentIdField = document.getElementById('studentId');
        const fullNameField = document.getElementById('studentFullName');
        const nationalIdField = document.getElementById('studentNationalId');
        const genderField = document.getElementById('studentGender');
        const birthdateField = document.getElementById('studentBirthdate');
        const loginCodeField = document.getElementById('studentLoginCode');
        const enabledField = document.getElementById('studentEnabled');
        const enableEnrollmentCheckbox = document.getElementById('enableEnrollment');
        
        if (!fullNameField || !enabledField) {
            this.showToast('error', 'Form fields not found. Please refresh the page.');
            return;
        }
        
        const studentId = studentIdField ? studentIdField.value : '';
        const isEdit = !!studentId;
        const enableEnrollment = enableEnrollmentCheckbox && enableEnrollmentCheckbox.checked;
        
        const data = {
            full_name: fullNameField.value.trim(),
            national_id: nationalIdField.value.trim() || null,
            gender: genderField.value || null,
            birthdate: birthdateField.value || null,
            enabled: parseInt(enabledField.value)
        };
        
        // Include login code for new students
        if (!isEdit && loginCodeField.value) {
            data.login_code = loginCodeField.value;
        }
        
        // Add enrollment data if enabled (for both new and edit)
        if (enableEnrollment) {
            const yearField = document.getElementById('enrollmentYear');
            const gradeField = document.getElementById('enrollmentGrade');
            const classField = document.getElementById('enrollmentClass');
            
            if (!yearField.value || !gradeField.value || !classField.value) {
                this.showToast('error', 'Please select Academic Year, Grade, and Class for enrollment.');
                return;
            }
            
            data.year_id = parseInt(yearField.value);
            data.grade_id = parseInt(gradeField.value);
            data.class_id = parseInt(classField.value);
        }
        
        // Clear previous validation
        document.querySelectorAll('.is-invalid').forEach(el => el.classList.remove('is-invalid'));
        
        try {
            let endpoint, method;
            
            if (isEdit) {
                // For edit mode, always use the standard update endpoint
                // The backend will handle enrollment updates based on provided data
                endpoint = `/admin/students/${studentId}`;
                method = 'PUT';
            } else if (enableEnrollment) {
                endpoint = '/admin/students/enroll';
                method = 'POST';
            } else {
                endpoint = '/admin/students';
                method = 'POST';
            }
            
            const response = await this.apiCall(method, endpoint, data);
            
            if (response.success) {
                this.closeStudentModal();
                let action;
                if (isEdit) {
                    action = enableEnrollment ? 'updated with enrollment changes' : 'updated';
                } else {
                    action = enableEnrollment ? 'created and enrolled' : 'created';
                }
                this.showToast('success', `Student ${action} successfully!`);
                // Refresh current page with current filters
                const currentPage = this.getCurrentStudentPage() || 1;
                this.loadStudentsPage(currentPage);
            } else {
                this.showToast('error', response.message);
            }
            
        } catch (error) {
            this.showToast('error', 'Failed to save student. Please try again.');
        }
    }
    
    async editStudent(id) {
        try {
            const response = await this.apiCall('GET', `/admin/students/${id}`);
            if (response.success) {
                this.showStudentModal(response.data);
            }
        } catch (error) {
            this.showToast('error', 'Failed to load student data');
        }
    }
    
    async deleteStudent(id, name) {
        if (!confirm(`Are you sure you want to delete student "${name}"?\n\nThis will permanently delete the student and ALL related data including:\n- Enrollments\n- Exam results (if any)\n- Payment records (if any)\n\nThis action cannot be undone!`)) {
            return;
        }
        
        try {
            const response = await this.apiCall('DELETE', `/admin/students/${id}`);
            
            if (response.success) {
                this.showToast('success', 'Student deleted successfully!');
                // Refresh current page with current filters
                const currentPage = this.getCurrentStudentPage() || 1;
                this.loadStudentsPage(currentPage);
            } else {
                this.showToast('error', response.message);
            }
        } catch (error) {
            this.showToast('error', 'Failed to delete student. They may have related data.');
        }
    }
    
    
    async sortStudents(field) {
        this.currentSort = this.currentSort || {};
        const currentDir = this.currentSort[field] || 'asc';
        const newDir = currentDir === 'asc' ? 'desc' : 'asc';
        this.currentSort[field] = newDir;
        
        try {
            const response = await this.apiCall('GET', `/admin/students?sort_by=${field}&sort_dir=${newDir}`);
            document.getElementById('studentsTableBody').innerHTML = this.renderStudentsTable(response.data);
            this.updateStudentsPagination(response.meta);
        } catch (error) {
            this.showToast('error', 'Failed to sort students');
        }
    }
    
    showStudentImportModal() {
        const modal = new bootstrap.Modal(document.getElementById('studentImportModal'));
        modal.show();
        
        // Reset form
        document.getElementById('studentImportForm').reset();
        document.getElementById('importProgress').classList.add('d-none');
        document.getElementById('importResults').classList.add('d-none');
        document.getElementById('importButton').disabled = false;
    }
    
    async importStudents() {
        const fileInput = document.getElementById('importFile');
        const file = fileInput.files[0];
        const enableEnrollmentCheckbox = document.getElementById('importEnableEnrollment');
        const enableEnrollment = enableEnrollmentCheckbox && enableEnrollmentCheckbox.checked;
        
        if (!file) {
            this.showToast('error', 'Please select a CSV file');
            return;
        }
        
        if (!file.name.toLowerCase().endsWith('.csv')) {
            this.showToast('error', 'Please select a valid CSV file');
            return;
        }
        
        // Validate enrollment fields if enabled
        if (enableEnrollment) {
            const yearField = document.getElementById('importEnrollmentYear');
            const gradeField = document.getElementById('importEnrollmentGrade');
            const classField = document.getElementById('importEnrollmentClass');
            
            if (!yearField.value || !gradeField.value || !classField.value) {
                this.showToast('error', 'Please select Academic Year, Grade, and Class for enrollment.');
                return;
            }
        }
        
        // Show progress
        document.getElementById('importProgress').classList.remove('d-none');
        document.getElementById('importResults').classList.add('d-none');
        document.getElementById('importButton').disabled = true;
        
        try {
            const formData = new FormData();
            formData.append('file', file);
            
            // Add enrollment data if enabled
            if (enableEnrollment) {
                formData.append('year_id', document.getElementById('importEnrollmentYear').value);
                formData.append('grade_id', document.getElementById('importEnrollmentGrade').value);
                formData.append('class_id', document.getElementById('importEnrollmentClass').value);
            }
            
            // Choose endpoint based on enrollment setting
            const endpoint = enableEnrollment ? '/student1/api/admin/students/import-enroll' : '/student1/api/admin/students/import';
            
            const response = await fetch(endpoint, {
                method: 'POST',
                headers: {
                    'Authorization': `Bearer ${this.token}`
                },
                body: formData
            });
            
            const result = await response.json();
            
            // Hide progress
            document.getElementById('importProgress').classList.add('d-none');
            document.getElementById('importButton').disabled = false;
            
            if (result.success) {
                this.showImportResults(result.data, enableEnrollment);
                this.showToast('success', result.message);
                
                // Refresh student list
                setTimeout(() => {
                    this.loadStudentsPage(1);
                }, 2000);
            } else {
                this.showToast('error', result.message);
            }
            
        } catch (error) {
            console.error('Import error:', error);
            document.getElementById('importProgress').classList.add('d-none');
            document.getElementById('importButton').disabled = false;
            this.showToast('error', 'Import failed. Please try again.');
        }
    }
    
    showImportResults(data, withEnrollment = false) {
        const resultsDiv = document.getElementById('importResults');
        const summaryDiv = document.getElementById('importSummary');
        const errorsDiv = document.getElementById('importErrors');
        
        const label = withEnrollment ? 'Enrolled' : 'Successful';
        const dataKey = withEnrollment ? (data.enrolled_students ? 'enrolled_students' : 'imported_students') : 'imported_students';
        
        // Summary
        summaryDiv.innerHTML = `
            <div class="row text-center">
                <div class="col-4">
                    <div class="text-primary">
                        <strong>${data.total_rows}</strong><br>
                        <small>Total Rows</small>
                    </div>
                </div>
                <div class="col-4">
                    <div class="text-success">
                        <strong>${data.successful}</strong><br>
                        <small>${label}</small>
                    </div>
                </div>
                <div class="col-4">
                    <div class="text-danger">
                        <strong>${data.failed}</strong><br>
                        <small>Failed</small>
                    </div>
                </div>
            </div>
        `;
        
        // Show successful imports/enrollments
        if (data[dataKey] && data[dataKey].length > 0) {
            const studentsHtml = `
                <div class="mt-3">
                    <h6>${withEnrollment ? 'Enrolled Students:' : 'Imported Students:'}</h6>
                    <div class="table-responsive">
                        <table class="table table-sm">
                            <thead>
                                <tr>
                                    <th>Name</th>
                                    <th>Login Code</th>
                                    ${withEnrollment ? '<th>Class</th>' : ''}
                                </tr>
                            </thead>
                            <tbody>
                                ${data[dataKey].slice(0, 10).map(student => `
                                    <tr>
                                        <td>${student.full_name}</td>
                                        <td><code>${student.login_code}</code></td>
                                        ${withEnrollment ? `<td>Grade ${student.grade_id || ''} - Class ${student.class_id || ''}</td>` : ''}
                                    </tr>
                                `).join('')}
                            </tbody>
                        </table>
                        ${data[dataKey].length > 10 ? `<p class="small text-muted">... and ${data[dataKey].length - 10} more students</p>` : ''}
                    </div>
                </div>
            `;
            summaryDiv.innerHTML += studentsHtml;
        }
        
        // Errors
        if (data.errors && data.errors.length > 0) {
            let errorsHtml = '<div class="mt-3"><h6>Errors:</h6><div class="table-responsive"><table class="table table-sm">';
            errorsHtml += '<thead><tr><th>Row</th><th>Data</th><th>Error</th></tr></thead><tbody>';
            
            data.errors.slice(0, 10).forEach(error => { // Show first 10 errors
                errorsHtml += `
                    <tr>
                        <td>${error.row}</td>
                        <td class="small">${JSON.stringify(error.data)}</td>
                        <td class="text-danger small">${error.error}</td>
                    </tr>
                `;
            });
            
            errorsHtml += '</tbody></table>';
            if (data.errors.length > 10) {
                errorsHtml += `<p class="small text-muted">... and ${data.errors.length - 10} more errors</p>`;
            }
            errorsHtml += '</div></div>';
            
            errorsDiv.innerHTML = errorsHtml;
        } else {
            errorsDiv.innerHTML = '';
        }
        
        // Show results
        resultsDiv.classList.remove('d-none');
    }
    
    showStudentEnrollmentImportModal() {
        const modalHtml = `
            <div class="modal fade" id="studentEnrollmentImportModal" tabindex="-1">
                <div class="modal-dialog modal-lg">
                    <div class="modal-content">
                        <div class="modal-header">
                            <h5 class="modal-title">
                                <i class="bi bi-file-earmark-spreadsheet"></i> Import Students with Enrollment
                            </h5>
                            <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                        </div>
                        <div class="modal-body">
                            <div class="row">
                                <div class="col-md-6">
                                    <h6><i class="bi bi-info-circle"></i> Instructions</h6>
                                    <ul class="text-muted small">
                                        <li><strong>Required:</strong> full_name, year_name, grade_code, class_code</li>
                                        <li><strong>Optional:</strong> national_id, gender (M/F), birthdate (YYYY-MM-DD)</li>
                                        <li>Each row specifies the student and their enrollment</li>
                                        <li>Login codes will be auto-generated</li>
                                        <li>Maximum file size: 5MB</li>
                                        <li>Supported format: CSV only</li>
                                    </ul>
                                    
                                    <div class="mb-3">
                                        <a href="${this.apiBase}/admin/students/enrollment-template" class="btn btn-outline-info btn-sm" target="_blank">
                                            <i class="bi bi-download"></i> Download Template CSV
                                        </a>
                                    </div>
                                </div>
                                <div class="col-md-6">
                                    <h6><i class="bi bi-file-earmark-spreadsheet"></i> CSV Format Example</h6>
                                    <div class="bg-light p-2 rounded small font-monospace">
                                        full_name,national_id,gender,birthdate,year_name,grade_code,class_code<br>
                                        Ahmed Ali,123456789,M,2010-05-15,2023-2024,KG1,A<br>
                                        Sara Hassan,234567890,F,2011-03-20,2023-2024,KG2,B
                                    </div>
                                </div>
                            </div>
                            
                            <hr>
                            
                            <form id="studentEnrollmentImportForm" enctype="multipart/form-data">
                                <div class="mb-3">
                                    <label for="enrollmentImportFile" class="form-label">CSV File *</label>
                                    <input type="file" class="form-control" id="enrollmentImportFile" accept=".csv" required>
                                </div>
                            </form>
                            
                            <div id="enrollmentImportProgress" class="d-none">
                                <div class="progress mb-3">
                                    <div class="progress-bar progress-bar-striped progress-bar-animated" style="width: 100%">
                                        Processing...
                                    </div>
                                </div>
                            </div>
                            
                            <div id="enrollmentImportResults" class="d-none">
                                <div class="alert alert-info">
                                    <h6><i class="bi bi-info-circle"></i> Import Results</h6>
                                    <div id="enrollmentImportSummary"></div>
                                </div>
                                <div id="enrollmentImportErrors" class="d-none">
                                    <h6 class="text-danger">Errors</h6>
                                    <div class="bg-danger bg-opacity-10 p-2 rounded small" id="enrollmentErrorList"></div>
                                </div>
                            </div>
                        </div>
                        <div class="modal-footer">
                            <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                            <button type="button" class="btn btn-success" onclick="app.importStudentsWithEnrollment()" id="enrollmentImportButton">
                                <i class="bi bi-upload"></i> Import Students with Enrollment
                            </button>
                        </div>
                    </div>
                </div>
            </div>
        `;
        
        // Remove existing modal if present
        const existingModal = document.getElementById('studentEnrollmentImportModal');
        if (existingModal) {
            existingModal.remove();
        }
        
        document.body.insertAdjacentHTML('beforeend', modalHtml);
        
        const modal = new bootstrap.Modal(document.getElementById('studentEnrollmentImportModal'));
        modal.show();
        
        // Reset form
        document.getElementById('studentEnrollmentImportForm').reset();
        document.getElementById('enrollmentImportProgress').classList.add('d-none');
        document.getElementById('enrollmentImportResults').classList.add('d-none');
        document.getElementById('enrollmentImportButton').disabled = false;
    }
    
    async importStudentsWithEnrollment() {
        const fileInput = document.getElementById('enrollmentImportFile');
        const file = fileInput.files[0];
        
        if (!file) {
            this.showToast('error', 'Please select a CSV file');
            return;
        }
        
        if (!file.name.toLowerCase().endsWith('.csv')) {
            this.showToast('error', 'Please select a valid CSV file');
            return;
        }
        
        const formData = new FormData();
        formData.append('file', file);
        
        try {
            document.getElementById('enrollmentImportProgress').classList.remove('d-none');
            document.getElementById('enrollmentImportResults').classList.add('d-none');
            document.getElementById('enrollmentImportButton').disabled = true;
            
            const response = await this.apiCall('POST', '/admin/students/import-individual-enroll', formData, false);
            
            if (response.success) {
                const results = response.data;
                
                // Show summary
                document.getElementById('enrollmentImportSummary').innerHTML = `
                    <strong>Total Rows:</strong> ${results.total_rows}<br>
                    <strong>Successful:</strong> <span class="text-success">${results.successful}</span><br>
                    <strong>Failed:</strong> <span class="text-danger">${results.failed}</span>
                `;
                
                // Show errors if any
                if (results.errors && results.errors.length > 0) {
                    let errorHtml = '<ul class="mb-0">';
                    results.errors.forEach(error => {
                        errorHtml += `<li>Row ${error.row}: ${error.error}</li>`;
                    });
                    errorHtml += '</ul>';
                    
                    document.getElementById('enrollmentErrorList').innerHTML = errorHtml;
                    document.getElementById('enrollmentImportErrors').classList.remove('d-none');
                }
                
                document.getElementById('enrollmentImportResults').classList.remove('d-none');
                
                // Reload students data to show new imports
                if (results.successful > 0) {
                    await this.loadStudentsData();
                    this.showToast('success', `${results.successful} students imported with enrollment successfully`);
                }
                
                if (results.failed === 0) {
                    // Close modal if everything was successful
                    setTimeout(() => {
                        const modal = bootstrap.Modal.getInstance(document.getElementById('studentEnrollmentImportModal'));
                        modal?.hide();
                    }, 2000);
                }
                
            } else {
                throw new Error(response.message || 'Import failed');
            }
            
        } catch (error) {
            console.error('Import error:', error);
            this.showToast('error', 'Import failed: ' + (error.message || 'Unknown error'));
        } finally {
            document.getElementById('enrollmentImportProgress').classList.add('d-none');
            document.getElementById('enrollmentImportButton').disabled = false;
        }
    }
    
    downloadSampleCSV() {
        const csvContent = `full_name,national_id,gender,birthdate
Ahmed Mohamed Ali,123456789012,M,2010-05-15
Fatima Hassan Ahmed,234567890123,F,2011-03-20
Omar Mahmoud Salem,345678901234,M,2009-12-10
Zahra Ali Mustafa,456789012345,F,2010-08-25
Khalid Saeed Omar,,M,2011-01-30
Nour Abdalla Hassan,567890123456,F,2010-11-05
Yusuf Ibrahim Ali,678901234567,M,2009-09-18
Mariam Farouk Said,,F,2011-06-12
Hassan Ahmed Omar,789012345678,M,2010-04-22
Aisha Mohamed Salem,890123456789,F,2009-07-08
Ali Hassan Mahmoud,901234567890,M,2011-02-14
Sara Omar Ahmed,012345678901,F,2010-10-30
Ibrahim Ali Hassan,123456789013,M,2009-11-25
Layla Mohamed Omar,234567890124,F,2011-05-18
Ammar Salem Ali,345678901235,M,2010-03-12
Rana Ahmed Hassan,,F,2009-08-07
Tariq Omar Salem,456789012346,M,2011-04-20
Dina Ali Ahmed,567890123457,F,2010-12-03
Samir Hassan Omar,678901234568,M,2009-06-15
Lina Mohamed Ali,,F,2011-01-08
Waleed Salem Ahmed,789012345679,M,2010-09-28
Hala Omar Hassan,890123456790,F,2009-12-22
Basim Ali Salem,901234567801,M,2011-03-16
Nada Hassan Ahmed,012345678902,F,2010-07-11
Majed Omar Ali,123456789014,M,2009-10-04
Reem Salem Hassan,234567890125,F,2011-08-19
Faisal Ahmed Omar,,M,2010-05-26
Yasmina Ali Hassan,345678901236,F,2009-11-13
Adel Hassan Salem,456789012347,M,2011-02-27
Noha Omar Ahmed,567890123458,F,2010-06-09`;
        
        const blob = new Blob([csvContent], { type: 'text/csv;charset=utf-8;' });
        const link = document.createElement('a');
        const url = URL.createObjectURL(blob);
        
        link.setAttribute('href', url);
        link.setAttribute('download', 'students_sample.csv');
        link.style.visibility = 'hidden';
        
        document.body.appendChild(link);
        link.click();
        document.body.removeChild(link);
        
        this.showToast('success', 'Sample CSV downloaded successfully!');
    }
    
    toggleSelectAll() {
        const selectAllCheckbox = document.getElementById('selectAllStudents');
        const studentCheckboxes = document.querySelectorAll('.student-checkbox');
        
        studentCheckboxes.forEach(checkbox => {
            checkbox.checked = selectAllCheckbox.checked;
        });
        
        this.updateSelectedCount();
    }
    
    updateSelectedCount() {
        const selectedCheckboxes = document.querySelectorAll('.student-checkbox:checked');
        const count = selectedCheckboxes.length;
        const totalCheckboxes = document.querySelectorAll('.student-checkbox');
        
        // Update count display
        document.getElementById('selectedCount').textContent = count;
        
        // Show/hide bulk delete button
        const bulkDeleteBtn = document.getElementById('bulkDeleteBtn');
        if (count > 0) {
            bulkDeleteBtn.style.display = 'inline-block';
        } else {
            bulkDeleteBtn.style.display = 'none';
        }
        
        // Update select all checkbox state
        const selectAllCheckbox = document.getElementById('selectAllStudents');
        if (count === 0) {
            selectAllCheckbox.indeterminate = false;
            selectAllCheckbox.checked = false;
        } else if (count === totalCheckboxes.length) {
            selectAllCheckbox.indeterminate = false;
            selectAllCheckbox.checked = true;
        } else {
            selectAllCheckbox.indeterminate = true;
            selectAllCheckbox.checked = false;
        }
    }
    
    async bulkDeleteStudents() {
        const selectedCheckboxes = document.querySelectorAll('.student-checkbox:checked');
        const studentIds = Array.from(selectedCheckboxes).map(cb => parseInt(cb.value));
        
        if (studentIds.length === 0) {
            this.showToast('warning', 'Please select students to delete');
            return;
        }
        
        const confirmed = confirm(`Are you sure you want to delete ${studentIds.length} selected student(s)?\n\nThis will permanently delete ALL selected students and their related data including:\n- Enrollments\n- Exam results (if any)\n- Payment records (if any)\n\nThis action cannot be undone!`);
        if (!confirmed) return;
        
        try {
            const response = await this.apiCall('DELETE', '/admin/students/bulk', {
                student_ids: studentIds
            });
            
            if (response.success) {
                this.showToast('success', response.message);
                
                // Refresh current page
                const currentPage = this.getCurrentStudentPage() || 1;
                this.loadStudentsPage(currentPage);
                
                // Reset selection
                document.getElementById('selectAllStudents').checked = false;
                this.updateSelectedCount();
            } else {
                this.showToast('error', response.message);
            }
            
        } catch (error) {
            console.error('Bulk delete error:', error);
            this.showToast('error', 'Failed to delete students. Please try again.');
        }
    }
    
    toggleEnrollmentFields() {
        const checkbox = document.getElementById('enableEnrollment');
        const fields = document.getElementById('enrollmentFields');
        
        if (checkbox.checked) {
            fields.classList.remove('d-none');
            this.loadEnrollmentOptions();
        } else {
            fields.classList.add('d-none');
        }
    }
    
    toggleImportEnrollmentFields() {
        const checkbox = document.getElementById('importEnableEnrollment');
        const fields = document.getElementById('importEnrollmentFields');
        
        if (checkbox.checked) {
            fields.classList.remove('d-none');
            this.loadImportEnrollmentOptions();
        } else {
            fields.classList.add('d-none');
        }
    }
    
    async loadEnrollmentOptions() {
        try {
            // Load academic years
            const yearResponse = await this.apiCall('GET', '/admin/academic-years');
            const yearSelect = document.getElementById('enrollmentYear');
            yearSelect.innerHTML = '<option value="">Select Year</option>';
            yearResponse.data.forEach(year => {
                yearSelect.innerHTML += `<option value="${year.id}">${year.name}</option>`;
            });
            
            // Load grades
            const gradeResponse = await this.apiCall('GET', '/admin/grades');
            const gradeSelect = document.getElementById('enrollmentGrade');
            gradeSelect.innerHTML = '<option value="">Select Grade</option>';
            gradeResponse.data.forEach(grade => {
                gradeSelect.innerHTML += `<option value="${grade.id}">${grade.title}</option>`;
            });
            
            // Load classes
            const classResponse = await this.apiCall('GET', '/admin/classes');
            const classSelect = document.getElementById('enrollmentClass');
            classSelect.innerHTML = '<option value="">Select Class</option>';
            classResponse.data.forEach(cls => {
                classSelect.innerHTML += `<option value="${cls.id}">${cls.name}</option>`;
            });
            
        } catch (error) {
            this.showToast('error', 'Failed to load enrollment options');
        }
    }
    
    async loadImportEnrollmentOptions() {
        try {
            // Load academic years
            const yearResponse = await this.apiCall('GET', '/admin/academic-years');
            const yearSelect = document.getElementById('importEnrollmentYear');
            yearSelect.innerHTML = '<option value="">Select Year</option>';
            yearResponse.data.forEach(year => {
                yearSelect.innerHTML += `<option value="${year.id}">${year.name}</option>`;
            });
            
            // Load grades
            const gradeResponse = await this.apiCall('GET', '/admin/grades');
            const gradeSelect = document.getElementById('importEnrollmentGrade');
            gradeSelect.innerHTML = '<option value="">Select Grade</option>';
            gradeResponse.data.forEach(grade => {
                gradeSelect.innerHTML += `<option value="${grade.id}">${grade.title}</option>`;
            });
            
            // Load classes
            const classResponse = await this.apiCall('GET', '/admin/classes');
            const classSelect = document.getElementById('importEnrollmentClass');
            classSelect.innerHTML = '<option value="">Select Class</option>';
            classResponse.data.forEach(cls => {
                classSelect.innerHTML += `<option value="${cls.id}">${cls.name}</option>`;
            });
            
        } catch (error) {
            this.showToast('error', 'Failed to load enrollment options');
        }
    }
    
    async loadEnrollments() {
        try {
            const content = `
                <div class="page-header">
                    <div class="d-flex justify-content-between align-items-center">
                        <div>
                            <h1 class="h2 mb-1"><i class="bi bi-person-plus"></i> <span data-translate="enrollments.title">Enrollments Management</span></h1>
                            <p class="text-muted mb-0" data-translate="enrollments.subtitle">Manage student enrollments and class assignments</p>
                        </div>
                        <div class="btn-group">
                            <button class="btn btn-outline-danger" id="bulkTransferBtn" onclick="app.showBulkTransferModal()" style="display: none;">
                                <i class="bi bi-arrow-right-circle"></i> <span data-translate="enrollments.transfer_selected">Transfer Selected</span> (<span id="selectedTransferCount">0</span>)
                            </button>
                            <button class="btn btn-outline-primary" onclick="app.showBulkEnrollModal()">
                                <i class="bi bi-people"></i> <span data-translate="enrollments.bulk_enroll">Bulk Enroll</span>
                            </button>
                            <button class="btn btn-primary" onclick="app.showEnrollmentModal()">
                                <i class="bi bi-plus-circle"></i> <span data-translate="enrollments.add_enrollment">Add Enrollment</span>
                            </button>
                        </div>
                    </div>
                </div>
                
                <!-- Filters -->
                <div class="card mb-4">
                    <div class="card-body">
                        <div class="row mb-3">
                            <div class="col-md-3">
                                <label class="form-label" data-translate="enrollments.student_name">Student Name</label>
                                <input type="text" class="form-control" id="enrollmentStudentFilter" data-translate-placeholder="enrollments.search_placeholder" placeholder="Search by student name...">
                            </div>
                            <div class="col-md-2">
                                <label class="form-label" data-translate="enrollments.academic_year">Academic Year</label>
                                <select class="form-select" id="enrollmentYearFilter">
                                    <option value="" data-translate="enrollments.all_years">All Years</option>
                                </select>
                            </div>
                            <div class="col-md-2">
                                <label class="form-label" data-translate="enrollments.grade">Grade</label>
                                <select class="form-select" id="enrollmentGradeFilter">
                                    <option value="" data-translate="enrollments.all_grades">All Grades</option>
                                </select>
                            </div>
                            <div class="col-md-2">
                                <label class="form-label" data-translate="enrollments.class">Class</label>
                                <select class="form-select" id="enrollmentClassFilter">
                                    <option value="" data-translate="enrollments.all_classes">All Classes</option>
                                </select>
                            </div>
                            <div class="col-md-1">
                                <label class="form-label" data-translate="enrollments.status">Status</label>
                                <select class="form-select" id="enrollmentStatusFilter">
                                    <option value="" data-translate="enrollments.all">All</option>
                                    <option value="active" data-translate="enrollments.active">Active</option>
                                    <option value="inactive" data-translate="enrollments.inactive">Inactive</option>
                                    <option value="graduated" data-translate="enrollments.graduated">Graduated</option>
                                    <option value="transferred" data-translate="enrollments.transferred">Transferred</option>
                                </select>
                            </div>
                            <div class="col-md-1">
                                <label class="form-label" data-translate="enrollments.per_page">Per Page</label>
                                <select class="form-select" id="enrollmentsPerPage" onchange="app.changeEnrollmentsPerPage()">
                                    <option value="10">10</option>
                                    <option value="25" selected>25</option>
                                    <option value="50">50</option>
                                    <option value="100">100</option>
                                </select>
                            </div>
                        </div>
                        <div class="row">
                            <div class="col-md-2">
                                <div class="d-flex gap-2">
                                    <button class="btn btn-primary" onclick="app.filterEnrollments()" title="Apply Filters">
                                        <i class="bi bi-search"></i> <span data-translate="enrollments.filter">Filter</span>
                                    </button>
                                    <button class="btn btn-outline-secondary" onclick="app.clearEnrollmentFilters()" title="Clear All Filters">
                                        <i class="bi bi-x-circle"></i> <span data-translate="enrollments.clear">Clear</span>
                                    </button>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
                
                <!-- Enrollments Table -->
                <div class="card">
                    <div class="card-header">
                        <h6 class="card-title mb-0"><i class="bi bi-table"></i> <span data-translate="enrollments.list_title">Enrollments List</span></h6>
                    </div>
                    <div class="card-body p-0">
                        <div class="table-container">
                            <div class="table-responsive">
                                <table class="table table-hover mb-0">
                                    <thead class="table-light">
                                        <tr>
                                            <th style="width: 50px;">
                                                <input type="checkbox" class="form-check-input" id="selectAllEnrollments" onchange="app.toggleSelectAllEnrollments()">
                                            </th>
                                            <th style="width: 70px;">
                                                <a href="#" onclick="app.sortEnrollments('e.id')" class="text-decoration-none text-dark">
                                                    <span data-translate="table.enrollment_id">ID</span> <i class="bi bi-arrow-down-up text-muted"></i>
                                                </a>
                                            </th>
                                            <th>
                                                <a href="#" onclick="app.sortEnrollments('s.full_name')" class="text-decoration-none text-dark">
                                                    <span data-translate="table.enrollment_student">Student</span> <i class="bi bi-arrow-down-up text-muted"></i>
                                                </a>
                                            </th>
                                            <th style="width: 120px;">
                                                <a href="#" onclick="app.sortEnrollments('ay.name')" class="text-decoration-none text-dark">
                                                    <span data-translate="table.enrollment_year">Year</span> <i class="bi bi-arrow-down-up text-muted"></i>
                                                </a>
                                            </th>
                                            <th style="width: 100px;">
                                                <a href="#" onclick="app.sortEnrollments('g.title')" class="text-decoration-none text-dark">
                                                    <span data-translate="table.enrollment_grade">Grade</span> <i class="bi bi-arrow-down-up text-muted"></i>
                                                </a>
                                            </th>
                                            <th style="width: 100px;">
                                                <a href="#" onclick="app.sortEnrollments('c.name')" class="text-decoration-none text-dark">
                                                    <span data-translate="table.enrollment_class">Class</span> <i class="bi bi-arrow-down-up text-muted"></i>
                                                </a>
                                            </th>
                                            <th style="width: 90px;">
                                                <a href="#" onclick="app.sortEnrollments('e.status')" class="text-decoration-none text-dark">
                                                    <span data-translate="table.enrollment_status">Status</span> <i class="bi bi-arrow-down-up text-muted"></i>
                                                </a>
                                            </th>
                                            <th style="width: 120px;">
                                                <a href="#" onclick="app.sortEnrollments('e.created_at')" class="text-decoration-none text-dark">
                                                    <span data-translate="table.enrollment_date">Enrolled Date</span> <i class="bi bi-arrow-down-up text-muted"></i>
                                                </a>
                                            </th>
                                            <th style="width: 120px;" data-translate="table.enrollment_actions">Actions</th>
                                        </tr>
                                    </thead>
                                    <tbody id="enrollmentsTableBody">
                                        <tr>
                                            <td colspan="9" class="text-center py-4">
                                                <div class="spinner-border text-primary" role="status">
                                                    <span class="visually-hidden" data-translate="dashboard.loading">Loading...</span>
                                                </div>
                                                <div class="mt-2" data-translate="enrollments.loading">Loading enrollments...</div>
                                            </td>
                                        </tr>
                                    </tbody>
                                </table>
                            </div>
                        </div>
                    </div>
                </div>
                
                <!-- Pagination -->
                <div id="enrollmentsPagination" class="d-flex justify-content-between align-items-center mt-3">
                    <div class="text-muted">
                        <span id="enrollmentsInfo">Loading...</span>
                    </div>
                    <nav>
                        <ul class="pagination pagination-sm mb-0" id="enrollmentsPaginationNav">
                        </ul>
                    </nav>
                </div>
            `;
            
            document.getElementById('pageContent').innerHTML = content;
            
            // Load filter options and enrollments data
            await this.loadEnrollmentFilterOptions();
            await this.loadEnrollmentsData();
            
        } catch (error) {
            this.showToast('error', 'Failed to load enrollments page');
        }
    }
    
    async loadTerms() {
        try {
            const response = await this.apiCall('GET', '/admin/terms');
            
            const content = `
                <div class="page-header">
                    <div class="d-flex justify-content-between align-items-center">
                        <div>
                            <h1 class="h2 mb-1"><i class="bi bi-calendar-event"></i> <span data-translate="terms.title">Terms</span></h1>
                            <p class="text-muted mb-0" data-translate="terms.subtitle">Manage academic terms and periods</p>
                        </div>
                        <button class="btn btn-primary" onclick="app.showTermModal()">
                            <i class="bi bi-plus-circle"></i> <span data-translate="terms.add_term">Add Term</span>
                        </button>
                    </div>
                </div>
                
                <div class="filter-bar">
                    <div class="row align-items-end">
                        <div class="col-md-3">
                            <label class="form-label" data-translate="terms.search_by_name">Search by name</label>
                            <input type="text" class="form-control" id="termNameFilter" data-translate-placeholder="terms.search_placeholder" placeholder="e.g., First Term">
                        </div>
                        <div class="col-md-2">
                            <label class="form-label" data-translate="terms.visibility">Visibility</label>
                            <select class="form-select" id="termVisibilityFilter">
                                <option value="" data-translate="terms.all">All</option>
                                <option value="1" data-translate="terms.visible_to_students">Visible to Students</option>
                                <option value="0" data-translate="terms.hidden_from_students">Hidden from Students</option>
                            </select>
                        </div>
                        <div class="col-md-2">
                            <label class="form-label">&nbsp;</label>
                            <div class="d-grid">
                                <button class="btn btn-outline-primary" onclick="app.filterTerms()">
                                    <i class="bi bi-search"></i> <span data-translate="terms.filter">Filter</span>
                                </button>
                            </div>
                        </div>
                        <div class="col-md-2">
                            <label class="form-label">&nbsp;</label>
                            <div class="d-grid">
                                <button class="btn btn-outline-secondary" onclick="app.clearTermsFilters()">
                                    <i class="bi bi-arrow-clockwise"></i> <span data-translate="terms.clear">Clear</span>
                                </button>
                            </div>
                        </div>
                        <div class="col-md-3 text-end">
                            <label class="form-label">&nbsp;</label>
                            <div class="btn-group" role="group">
                                <button class="btn btn-outline-success btn-sm" onclick="app.exportTerms('csv')">
                                    <i class="bi bi-file-earmark-csv"></i> CSV
                                </button>
                                <button class="btn btn-outline-info btn-sm" onclick="app.printTerms()">
                                    <i class="bi bi-printer"></i> Print
                                </button>
                            </div>
                        </div>
                    </div>
                </div>
                
                <div class="card">
                    <div class="card-body p-0">
                        <div class="table-responsive">
                            <table class="table table-hover mb-0">
                                <thead>
                                    <tr>
                                        <th class="sortable" data-sort="name">
                                            <i class="bi bi-arrow-down-up"></i> <span data-translate="table.term_name">Name</span>
                                        </th>
                                        <th class="sortable" data-sort="sort_order">
                                            <i class="bi bi-arrow-down-up"></i> <span data-translate="table.sort_order">Order</span>
                                        </th>
                                        <th class="sortable" data-sort="visible_to_students">
                                            <i class="bi bi-arrow-down-up"></i> <span data-translate="table.visibility">Visible to Students</span>
                                        </th>
                                        <th class="sortable" data-sort="publish_at">
                                            <i class="bi bi-arrow-down-up"></i> <span data-translate="table.publish_date">Publish Date</span>
                                        </th>
                                        <th class="sortable" data-sort="created_at">
                                            <i class="bi bi-arrow-down-up"></i> <span data-translate="table.created">Created</span>
                                        </th>
                                        <th width="120" data-translate="table.actions">Actions</th>
                                    </tr>
                                </thead>
                                <tbody id="termsTableBody">
                                    ${this.renderTermsTable(response.data)}
                                </tbody>
                            </table>
                        </div>
                    </div>
                </div>
                
                ${this.renderPagination(response.meta)}
                
                <!-- Term Modal -->
                <div class="modal fade" id="termModal" tabindex="-1">
                    <div class="modal-dialog">
                        <div class="modal-content">
                            <div class="modal-header">
                                <h5 class="modal-title">
                                    <i class="bi bi-calendar-event"></i> <span id="termModalTitle">Add Term</span>
                                </h5>
                                <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal"></button>
                            </div>
                            <div class="modal-body">
                                <form id="termForm">
                                    <input type="hidden" id="termId">
                                    <div class="mb-3">
                                        <label for="termName" class="form-label" data-translate="terms.term_name">Term Name *</label>
                                        <input type="text" class="form-control" id="termName" data-translate-placeholder="terms.name_placeholder" placeholder="e.g., First Term" required>
                                        <div class="invalid-feedback"></div>
                                    </div>
                                    <div class="row">
                                        <div class="col-md-6">
                                            <div class="mb-3">
                                                <label for="termSortOrder" class="form-label" data-translate="terms.sort_order">Sort Order *</label>
                                                <input type="number" class="form-control" id="termSortOrder" min="1" required>
                                                <div class="invalid-feedback"></div>
                                            </div>
                                        </div>
                                        <div class="col-md-6">
                                            <div class="mb-3">
                                                <label for="termVisibleToStudents" class="form-label" data-translate="terms.visible_to_students_label">Visible to Students *</label>
                                                <select class="form-select" id="termVisibleToStudents" required>
                                                    <option value="" data-translate="terms.select_visibility">Select visibility</option>
                                                    <option value="1" data-translate="terms.yes_visible">Yes - Visible to Students</option>
                                                    <option value="0" data-translate="terms.no_hidden">No - Hidden from Students</option>
                                                </select>
                                                <div class="invalid-feedback"></div>
                                            </div>
                                        </div>
                                    </div>
                                    <div class="mb-3">
                                        <label for="termPublishAt" class="form-label" data-translate="terms.publish_date">Publish Date & Time</label>
                                        <input type="datetime-local" class="form-control" id="termPublishAt">
                                        <div class="form-text">
                                            <i class="bi bi-info-circle"></i> 
                                            <span data-translate="terms.publish_help">Optional: When to make this term's results visible to students</span>
                                        </div>
                                    </div>
                                </form>
                            </div>
                            <div class="modal-footer">
                                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">
                                    <i class="bi bi-x-circle"></i> <span data-translate="btn.cancel">Cancel</span>
                                </button>
                                <button type="button" class="btn btn-primary" onclick="app.saveTerm()">
                                    <i class="bi bi-check-circle"></i> <span id="termSaveButtonText" data-translate="terms.save_term">Save Term</span>
                                </button>
                            </div>
                        </div>
                    </div>
                </div>
            `;
            
            document.getElementById('pageContent').innerHTML = content;
            
            // Add event listeners for sorting
            document.querySelectorAll('.sortable').forEach(header => {
                header.style.cursor = 'pointer';
                header.addEventListener('click', () => {
                    const sortField = header.getAttribute('data-sort');
                    this.sortTerms(sortField);
                });
            });
            
        } catch (error) {
            document.getElementById('pageContent').innerHTML = `
                <div class="alert alert-danger">
                    <i class="bi bi-exclamation-triangle"></i> <span data-translate="terms.error_loading">Error loading terms</span>: ${error.message}
                </div>`;
        }
    }
    
    renderTermsTable(terms) {
        if (!terms || terms.length === 0) {
            return `
                <tr>
                    <td colspan="6" class="text-center py-5">
                        <div class="text-muted">
                            <i class="bi bi-calendar-event" style="font-size: 3rem; opacity: 0.3;"></i>
                            <p class="mt-3 mb-0" data-translate="terms.no_terms_found">No terms found</p>
                            <small data-translate="terms.add_term">Click "Add Term" to create your first entry</small>
                        </div>
                    </td>
                </tr>
            `;
        }
        
        return terms.map(term => `
            <tr>
                <td><strong>${term.name}</strong></td>
                <td><span class="badge bg-info">${term.sort_order}</span></td>
                <td>
                    ${term.visible_to_students ? 
                        '<span class="badge bg-success"><i class="bi bi-eye"></i> Visible</span>' : 
                        '<span class="badge bg-secondary"><i class="bi bi-eye-slash"></i> Hidden</span>'
                    }
                </td>
                <td>${term.publish_at ? new Date(term.publish_at).toLocaleString() : '<span class="text-muted">Not set</span>'}</td>
                <td>${new Date(term.created_at).toLocaleDateString()}</td>
                <td>
                    <div class="btn-group" role="group">
                        <button class="btn btn-sm btn-outline-primary" onclick="app.editTerm(${term.id})" title="Edit">
                            <i class="bi bi-pencil"></i>
                        </button>
                        <button class="btn btn-sm btn-outline-danger" onclick="app.deleteTerm(${term.id}, '${term.name}')" title="Delete">
                            <i class="bi bi-trash"></i>
                        </button>
                    </div>
                </td>
            </tr>
        `).join('');
    }
    
    // Terms CRUD Functions
    showTermModal(termData = null) {
        const modalElement = document.getElementById('termModal');
        if (!modalElement) {
            this.showToast('error', 'Modal not found. Please refresh the page.');
            return;
        }
        
        // Remove any existing backdrops
        document.querySelectorAll('.modal-backdrop').forEach(backdrop => backdrop.remove());
        
        // Reset form
        const form = document.getElementById('termForm');
        if (form) form.reset();
        
        const termIdField = document.getElementById('termId');
        if (termIdField) termIdField.value = '';
        
        if (termData) {
            // Edit mode
            const modalTitle = document.getElementById('termModalTitle');
            const saveButtonText = document.getElementById('termSaveButtonText');
            if (modalTitle) {
                modalTitle.textContent = (typeof langManager !== 'undefined') 
                    ? langManager.t('terms.edit_term') 
                    : 'Edit Term';
            }
            if (saveButtonText) {
                saveButtonText.textContent = (typeof langManager !== 'undefined') 
                    ? langManager.t('terms.update_term') 
                    : 'Update Term';
            }
            if (termIdField) termIdField.value = termData.id;
            document.getElementById('termName').value = termData.name;
            document.getElementById('termSortOrder').value = termData.sort_order;
            document.getElementById('termVisibleToStudents').value = termData.visible_to_students;
            if (termData.publish_at) {
                // Convert to datetime-local format
                const date = new Date(termData.publish_at);
                const localDateTime = new Date(date.getTime() - date.getTimezoneOffset() * 60000).toISOString().slice(0, 16);
                document.getElementById('termPublishAt').value = localDateTime;
            }
        } else {
            // Add mode
            const modalTitle = document.getElementById('termModalTitle');
            const saveButtonText = document.getElementById('termSaveButtonText');
            if (modalTitle) {
                modalTitle.textContent = (typeof langManager !== 'undefined') 
                    ? langManager.t('terms.add_term') 
                    : 'Add Term';
            }
            if (saveButtonText) {
                saveButtonText.textContent = (typeof langManager !== 'undefined') 
                    ? langManager.t('terms.save_term') 
                    : 'Save Term';
            }
        }
        
        // Clear validation classes
        document.querySelectorAll('.is-invalid').forEach(el => el.classList.remove('is-invalid'));
        
        // Show modal
        modalElement.style.display = 'flex';
        modalElement.classList.add('show');
        document.body.classList.add('modal-open');
        
        // Add close button handlers
        const closeButtons = modalElement.querySelectorAll('[data-bs-dismiss="modal"]');
        closeButtons.forEach(btn => {
            btn.onclick = (e) => {
                e.preventDefault();
                this.closeTermModal();
            };
        });
        
        // Focus on first input
        setTimeout(() => {
            const firstInput = modalElement.querySelector('input[type="text"]');
            if (firstInput) firstInput.focus();
        }, 100);
    }
    
    closeTermModal() {
        const modalElement = document.getElementById('termModal');
        if (modalElement) {
            modalElement.style.display = 'none';
            modalElement.classList.remove('show');
        }
        document.body.classList.remove('modal-open');
        document.querySelectorAll('.modal-backdrop').forEach(backdrop => backdrop.remove());
    }
    
    async saveTerm() {
        const termIdField = document.getElementById('termId');
        const termNameField = document.getElementById('termName');
        const termSortOrderField = document.getElementById('termSortOrder');
        const termVisibleField = document.getElementById('termVisibleToStudents');
        const termPublishAtField = document.getElementById('termPublishAt');
        
        if (!termNameField || !termSortOrderField || !termVisibleField) {
            this.showToast('error', 'Form fields not found. Please refresh the page.');
            return;
        }
        
        const termId = termIdField ? termIdField.value : '';
        const isEdit = !!termId;
        
        const data = {
            name: termNameField.value.trim(),
            sort_order: parseInt(termSortOrderField.value),
            visible_to_students: parseInt(termVisibleField.value),
            publish_at: termPublishAtField.value || null
        };
        
        // Clear previous validation
        document.querySelectorAll('.is-invalid').forEach(el => el.classList.remove('is-invalid'));
        
        try {
            const endpoint = isEdit ? `/admin/terms/${termId}` : '/admin/terms';
            const method = isEdit ? 'PUT' : 'POST';
            
            const response = await this.apiCall(method, endpoint, data);
            
            if (response.success) {
                this.closeTermModal();
                this.showToast('success', `Term ${isEdit ? 'updated' : 'created'} successfully!`);
                this.loadTerms();
            } else {
                this.showToast('error', response.message);
            }
            
        } catch (error) {
            this.showToast('error', 'Failed to save term. Please try again.');
        }
    }
    
    async editTerm(id) {
        try {
            const response = await this.apiCall('GET', `/admin/terms/${id}`);
            if (response.success) {
                this.showTermModal(response.data);
            }
        } catch (error) {
            this.showToast('error', 'Failed to load term data');
        }
    }
    
    async deleteTerm(id, name) {
        if (!confirm(`Are you sure you want to delete the term "${name}"?\n\nThis action cannot be undone and may affect related data.`)) {
            return;
        }
        
        try {
            const response = await this.apiCall('DELETE', `/admin/terms/${id}`);
            
            if (response.success) {
                this.showToast('success', 'Term deleted successfully!');
                this.loadTerms();
            } else {
                this.showToast('error', response.message);
            }
        } catch (error) {
            this.showToast('error', 'Failed to delete term. It may have related data.');
        }
    }
    
    async filterTerms() {
        const name = document.getElementById('termNameFilter').value;
        const visible = document.getElementById('termVisibilityFilter').value;
        
        const params = new URLSearchParams();
        if (name) params.append('name', name);
        if (visible !== '') params.append('visible_to_students', visible);
        
        try {
            const response = await this.apiCall('GET', `/admin/terms?${params.toString()}`);
            document.getElementById('termsTableBody').innerHTML = this.renderTermsTable(response.data);
        } catch (error) {
            this.showToast('error', 'Failed to filter terms');
        }
    }
    
    clearTermsFilters() {
        document.getElementById('termNameFilter').value = '';
        document.getElementById('termVisibilityFilter').value = '';
        this.loadTerms();
    }
    
    async sortTerms(field) {
        this.currentSort = this.currentSort || {};
        const currentDir = this.currentSort[field] || 'asc';
        const newDir = currentDir === 'asc' ? 'desc' : 'asc';
        this.currentSort[field] = newDir;
        
        try {
            const response = await this.apiCall('GET', `/admin/terms?sort_by=${field}&sort_dir=${newDir}`);
            document.getElementById('termsTableBody').innerHTML = this.renderTermsTable(response.data);
        } catch (error) {
            this.showToast('error', 'Failed to sort terms');
        }
    }
    
    exportTerms(format) {
        this.showToast('info', `${format.toUpperCase()} export functionality coming soon!`);
    }
    
    printTerms() {
        this.showToast('info', 'Print functionality coming soon!');
    }
    
    // Grades Management
    async loadGrades() {
        try {
            const response = await this.apiCall('GET', '/admin/grades');
            
            const content = `
                <div class="page-header">
                    <div class="d-flex justify-content-between align-items-center">
                        <div>
                            <h1 class="h2 mb-1"><i class="bi bi-ladder"></i> <span data-translate="grades.title">Grades</span></h1>
                            <p class="text-muted mb-0" data-translate="grades.subtitle">Manage academic grade levels</p>
                        </div>
                        <button class="btn btn-primary" onclick="app.showGradeModal()">
                            <i class="bi bi-plus-circle"></i> <span data-translate="grades.add_grade">Add Grade</span>
                        </button>
                    </div>
                </div>
                
                <div class="filter-bar">
                    <div class="row align-items-end">
                        <div class="col-md-3">
                            <label class="form-label" data-translate="grades.search">Search</label>
                            <input type="text" class="form-control" id="gradeSearchFilter" data-translate-placeholder="grades.search_placeholder" placeholder="Search by code or title">
                        </div>
                        <div class="col-md-2">
                            <label class="form-label" data-translate="grades.status">Status</label>
                            <select class="form-select" id="gradeEnabledFilter">
                                <option value="" data-translate="grades.all">All</option>
                                <option value="1" data-translate="grades.enabled">Enabled</option>
                                <option value="0" data-translate="grades.disabled">Disabled</option>
                            </select>
                        </div>
                        <div class="col-md-2">
                            <label class="form-label">&nbsp;</label>
                            <div class="d-grid">
                                <button class="btn btn-outline-primary" onclick="app.filterGrades()">
                                    <i class="bi bi-search"></i> <span data-translate="grades.filter">Filter</span>
                                </button>
                            </div>
                        </div>
                        <div class="col-md-2">
                            <label class="form-label">&nbsp;</label>
                            <div class="d-grid">
                                <button class="btn btn-outline-secondary" onclick="app.clearGradesFilters()">
                                    <i class="bi bi-arrow-clockwise"></i> <span data-translate="grades.clear">Clear</span>
                                </button>
                            </div>
                        </div>
                        <div class="col-md-3 text-end">
                            <label class="form-label">&nbsp;</label>
                            <div class="btn-group" role="group">
                                <button class="btn btn-outline-success btn-sm" onclick="app.exportGrades('csv')">
                                    <i class="bi bi-file-earmark-csv"></i> CSV
                                </button>
                                <button class="btn btn-outline-info btn-sm" onclick="app.printGrades()">
                                    <i class="bi bi-printer"></i> Print
                                </button>
                            </div>
                        </div>
                    </div>
                </div>
                
                <div class="card">
                    <div class="card-body p-0">
                        <div class="table-responsive">
                            <table class="table table-hover mb-0">
                                <thead>
                                    <tr>
                                        <th class="sortable" data-sort="code">
                                            <i class="bi bi-arrow-down-up"></i> <span data-translate="table.grade_code">Code</span>
                                        </th>
                                        <th class="sortable" data-sort="title">
                                            <i class="bi bi-arrow-down-up"></i> <span data-translate="table.grade_title">Title</span>
                                        </th>
                                        <th class="sortable" data-sort="sort_order">
                                            <i class="bi bi-arrow-down-up"></i> <span data-translate="table.grade_order">Order</span>
                                        </th>
                                        <th class="sortable" data-sort="enabled">
                                            <i class="bi bi-arrow-down-up"></i> <span data-translate="table.grade_status">Status</span>
                                        </th>
                                        <th class="sortable" data-sort="created_at">
                                            <i class="bi bi-arrow-down-up"></i> <span data-translate="table.created">Created</span>
                                        </th>
                                        <th width="120" data-translate="table.actions">Actions</th>
                                    </tr>
                                </thead>
                                <tbody id="gradesTableBody">
                                    ${this.renderGradesTable(response.data)}
                                </tbody>
                            </table>
                        </div>
                    </div>
                </div>
                
                ${this.renderPagination(response.meta)}
                
                <!-- Grade Modal -->
                <div class="modal fade" id="gradeModal" tabindex="-1">
                    <div class="modal-dialog">
                        <div class="modal-content">
                            <div class="modal-header">
                                <h5 class="modal-title">
                                    <i class="bi bi-ladder"></i> <span id="gradeModalTitle">Add Grade</span>
                                </h5>
                                <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal"></button>
                            </div>
                            <div class="modal-body">
                                <form id="gradeForm">
                                    <input type="hidden" id="gradeId">
                                    <div class="row">
                                        <div class="col-md-6">
                                            <div class="mb-3">
                                                <label for="gradeCode" class="form-label" data-translate="grades.grade_code">Grade Code *</label>
                                                <input type="text" class="form-control" id="gradeCode" data-translate-placeholder="grades.code_placeholder" placeholder="e.g., G1, KG1" required>
                                                <div class="invalid-feedback"></div>
                                            </div>
                                        </div>
                                        <div class="col-md-6">
                                            <div class="mb-3">
                                                <label for="gradeSortOrder" class="form-label" data-translate="grades.sort_order">Sort Order *</label>
                                                <input type="number" class="form-control" id="gradeSortOrder" min="1" required>
                                                <div class="invalid-feedback"></div>
                                            </div>
                                        </div>
                                    </div>
                                    <div class="mb-3">
                                        <label for="gradeTitle" class="form-label" data-translate="grades.grade_title">Grade Title *</label>
                                        <input type="text" class="form-control" id="gradeTitle" data-translate-placeholder="grades.title_placeholder" placeholder="e.g., Grade 1, Kindergarten 1" required>
                                        <div class="invalid-feedback"></div>
                                    </div>
                                    <div class="mb-3">
                                        <label for="gradeEnabled" class="form-label" data-translate="grades.status_label">Status *</label>
                                        <select class="form-select" id="gradeEnabled" required>
                                            <option value="" data-translate="grades.select_status">Select status</option>
                                            <option value="1" data-translate="grades.enabled">Enabled</option>
                                            <option value="0" data-translate="grades.disabled">Disabled</option>
                                        </select>
                                        <div class="form-text">
                                            <i class="bi bi-info-circle"></i> 
                                            <span data-translate="grades.disabled_help">Disabled grades are not available for new enrollments</span>
                                        </div>
                                        <div class="invalid-feedback"></div>
                                    </div>
                                </form>
                            </div>
                            <div class="modal-footer">
                                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">
                                    <i class="bi bi-x-circle"></i> <span data-translate="btn.cancel">Cancel</span>
                                </button>
                                <button type="button" class="btn btn-primary" onclick="app.saveGrade()">
                                    <i class="bi bi-check-circle"></i> <span id="gradeSaveButtonText" data-translate="grades.save_grade">Save Grade</span>
                                </button>
                            </div>
                        </div>
                    </div>
                </div>
            `;
            
            document.getElementById('pageContent').innerHTML = content;
            
            // Add event listeners for sorting
            document.querySelectorAll('.sortable').forEach(header => {
                header.style.cursor = 'pointer';
                header.addEventListener('click', () => {
                    const sortField = header.getAttribute('data-sort');
                    this.sortGrades(sortField);
                });
            });
            
        } catch (error) {
            document.getElementById('pageContent').innerHTML = `
                <div class="alert alert-danger">
                    <i class="bi bi-exclamation-triangle"></i> <span data-translate="grades.error_loading">Error loading grades</span>: ${error.message}
                </div>`;
        }
    }
    
    renderGradesTable(grades) {
        if (!grades || grades.length === 0) {
            return `
                <tr>
                    <td colspan="6" class="text-center py-5">
                        <div class="text-muted">
                            <i class="bi bi-ladder" style="font-size: 3rem; opacity: 0.3;"></i>
                            <p class="mt-3 mb-0" data-translate="grades.no_grades_found">No grades found</p>
                            <small data-translate="grades.add_grade">Click "Add Grade" to create your first entry</small>
                        </div>
                    </td>
                </tr>
            `;
        }
        
        return grades.map(grade => `
            <tr>
                <td><strong>${grade.code}</strong></td>
                <td>${grade.title}</td>
                <td><span class="badge bg-info">${grade.sort_order}</span></td>
                <td>
                    ${grade.enabled ? 
                        `<span class="badge bg-success"><i class="bi bi-check-circle"></i> ${(typeof langManager !== 'undefined') ? langManager.t('grades.enabled') : 'Enabled'}</span>` : 
                        `<span class="badge bg-secondary"><i class="bi bi-x-circle"></i> ${(typeof langManager !== 'undefined') ? langManager.t('grades.disabled') : 'Disabled'}</span>`
                    }
                </td>
                <td>${new Date(grade.created_at).toLocaleDateString()}</td>
                <td>
                    <div class="btn-group" role="group">
                        <button class="btn btn-sm btn-outline-primary" onclick="app.editGrade(${grade.id})" title="Edit">
                            <i class="bi bi-pencil"></i>
                        </button>
                        <button class="btn btn-sm btn-outline-danger" onclick="app.deleteGrade(${grade.id}, '${grade.code}')" title="Delete">
                            <i class="bi bi-trash"></i>
                        </button>
                    </div>
                </td>
            </tr>
        `).join('');
    }
    
    // Grades CRUD Functions
    showGradeModal(gradeData = null) {
        const modalElement = document.getElementById('gradeModal');
        if (!modalElement) {
            this.showToast('error', 'Modal not found. Please refresh the page.');
            return;
        }
        
        // Remove any existing backdrops
        document.querySelectorAll('.modal-backdrop').forEach(backdrop => backdrop.remove());
        
        // Reset form
        const form = document.getElementById('gradeForm');
        if (form) form.reset();
        
        const gradeIdField = document.getElementById('gradeId');
        if (gradeIdField) gradeIdField.value = '';
        
        if (gradeData) {
            // Edit mode
            const modalTitle = document.getElementById('gradeModalTitle');
            const saveButtonText = document.getElementById('gradeSaveButtonText');
            if (modalTitle) {
                modalTitle.textContent = (typeof langManager !== 'undefined') 
                    ? langManager.t('grades.edit_grade') 
                    : 'Edit Grade';
            }
            if (saveButtonText) {
                saveButtonText.textContent = (typeof langManager !== 'undefined') 
                    ? langManager.t('grades.update_grade') 
                    : 'Update Grade';
            }
            if (gradeIdField) gradeIdField.value = gradeData.id;
            document.getElementById('gradeCode').value = gradeData.code;
            document.getElementById('gradeTitle').value = gradeData.title;
            document.getElementById('gradeSortOrder').value = gradeData.sort_order;
            document.getElementById('gradeEnabled').value = gradeData.enabled;
        } else {
            // Add mode
            const modalTitle = document.getElementById('gradeModalTitle');
            const saveButtonText = document.getElementById('gradeSaveButtonText');
            if (modalTitle) {
                modalTitle.textContent = (typeof langManager !== 'undefined') 
                    ? langManager.t('grades.add_grade') 
                    : 'Add Grade';
            }
            if (saveButtonText) {
                saveButtonText.textContent = (typeof langManager !== 'undefined') 
                    ? langManager.t('grades.save_grade') 
                    : 'Save Grade';
            }
        }
        
        // Clear validation classes
        document.querySelectorAll('.is-invalid').forEach(el => el.classList.remove('is-invalid'));
        
        // Show modal
        modalElement.style.display = 'flex';
        modalElement.classList.add('show');
        document.body.classList.add('modal-open');
        
        // Add close button handlers
        const closeButtons = modalElement.querySelectorAll('[data-bs-dismiss="modal"]');
        closeButtons.forEach(btn => {
            btn.onclick = (e) => {
                e.preventDefault();
                this.closeGradeModal();
            };
        });
        
        // Focus on first input
        setTimeout(() => {
            const firstInput = modalElement.querySelector('input[type="text"]');
            if (firstInput) firstInput.focus();
        }, 100);
    }
    
    closeGradeModal() {
        const modalElement = document.getElementById('gradeModal');
        if (modalElement) {
            modalElement.style.display = 'none';
            modalElement.classList.remove('show');
        }
        document.body.classList.remove('modal-open');
        document.querySelectorAll('.modal-backdrop').forEach(backdrop => backdrop.remove());
    }
    
    async saveGrade() {
        const gradeIdField = document.getElementById('gradeId');
        const gradeCodeField = document.getElementById('gradeCode');
        const gradeTitleField = document.getElementById('gradeTitle');
        const gradeSortOrderField = document.getElementById('gradeSortOrder');
        const gradeEnabledField = document.getElementById('gradeEnabled');
        
        if (!gradeCodeField || !gradeTitleField || !gradeSortOrderField || !gradeEnabledField) {
            this.showToast('error', 'Form fields not found. Please refresh the page.');
            return;
        }
        
        const gradeId = gradeIdField ? gradeIdField.value : '';
        const isEdit = !!gradeId;
        
        const data = {
            code: gradeCodeField.value.trim(),
            title: gradeTitleField.value.trim(),
            sort_order: parseInt(gradeSortOrderField.value),
            enabled: parseInt(gradeEnabledField.value)
        };
        
        // Clear previous validation
        document.querySelectorAll('.is-invalid').forEach(el => el.classList.remove('is-invalid'));
        
        try {
            const endpoint = isEdit ? `/admin/grades/${gradeId}` : '/admin/grades';
            const method = isEdit ? 'PUT' : 'POST';
            
            const response = await this.apiCall(method, endpoint, data);
            
            if (response.success) {
                this.closeGradeModal();
                this.showToast('success', `Grade ${isEdit ? 'updated' : 'created'} successfully!`);
                this.loadGrades();
            } else {
                this.showToast('error', response.message);
            }
            
        } catch (error) {
            this.showToast('error', 'Failed to save grade. Please try again.');
        }
    }
    
    async editGrade(id) {
        try {
            const response = await this.apiCall('GET', `/admin/grades/${id}`);
            if (response.success) {
                this.showGradeModal(response.data);
            }
        } catch (error) {
            this.showToast('error', 'Failed to load grade data');
        }
    }
    
    async deleteGrade(id, code) {
        if (!confirm(`Are you sure you want to delete the grade "${code}"?\n\nThis action cannot be undone and may affect related data.`)) {
            return;
        }
        
        try {
            const response = await this.apiCall('DELETE', `/admin/grades/${id}`);
            
            if (response.success) {
                this.showToast('success', 'Grade deleted successfully!');
                this.loadGrades();
            } else {
                this.showToast('error', response.message);
            }
        } catch (error) {
            this.showToast('error', 'Failed to delete grade. It may have related data.');
        }
    }
    
    async filterGrades() {
        const search = document.getElementById('gradeSearchFilter').value;
        const enabled = document.getElementById('gradeEnabledFilter').value;
        
        const params = new URLSearchParams();
        if (search) params.append('search', search);
        if (enabled !== '') params.append('enabled', enabled);
        
        try {
            const response = await this.apiCall('GET', `/admin/grades?${params.toString()}`);
            document.getElementById('gradesTableBody').innerHTML = this.renderGradesTable(response.data);
        } catch (error) {
            this.showToast('error', 'Failed to filter grades');
        }
    }
    
    clearGradesFilters() {
        document.getElementById('gradeSearchFilter').value = '';
        document.getElementById('gradeEnabledFilter').value = '';
        this.loadGrades();
    }
    
    async sortGrades(field) {
        this.currentSort = this.currentSort || {};
        const currentDir = this.currentSort[field] || 'asc';
        const newDir = currentDir === 'asc' ? 'desc' : 'asc';
        this.currentSort[field] = newDir;
        
        try {
            const response = await this.apiCall('GET', `/admin/grades?sort_by=${field}&sort_dir=${newDir}`);
            document.getElementById('gradesTableBody').innerHTML = this.renderGradesTable(response.data);
        } catch (error) {
            this.showToast('error', 'Failed to sort grades');
        }
    }
    
    exportGrades(format) {
        this.showToast('info', `${format.toUpperCase()} export functionality coming soon!`);
    }
    
    printGrades() {
        this.showToast('info', 'Print functionality coming soon!');
    }
    
    // Classes Management
    async loadClasses() {
        try {
            const response = await this.apiCall('GET', '/admin/classes');
            
            const content = `
                <div class="page-header">
                    <div class="d-flex justify-content-between align-items-center">
                        <div>
                            <h1 class="h2 mb-1"><i class="bi bi-people"></i> <span data-translate="classes.title">Classes</span></h1>
                            <p class="text-muted mb-0" data-translate="classes.subtitle">Manage class sections within grades</p>
                        </div>
                        <button class="btn btn-primary" onclick="app.showClassModal()">
                            <i class="bi bi-plus-circle"></i> <span data-translate="classes.add_class">Add Class</span>
                        </button>
                    </div>
                </div>
                
                <!-- Filters -->
                <div class="filter-bar">
                    <div class="row g-3">
                        <div class="col-md-4">
                            <label class="form-label" data-translate="classes.class_name">Class Name</label>
                            <input type="text" class="form-control" id="classNameFilter" data-translate-placeholder="classes.search_placeholder" placeholder="Search class name...">
                        </div>
                        <div class="col-md-4">
                            <label class="form-label" data-translate="classes.status">Status</label>
                            <select class="form-select" id="classStatusFilter">
                                <option value="" data-translate="classes.all_statuses">All Statuses</option>
                                <option value="1" data-translate="classes.active">Active</option>
                                <option value="0" data-translate="classes.disabled">Disabled</option>
                            </select>
                        </div>
                        <div class="col-md-4">
                            <label class="form-label">&nbsp;</label>
                            <div class="d-flex">
                                <button class="btn btn-primary" onclick="app.filterClasses()">
                                    <i class="bi bi-search"></i> <span data-translate="classes.search">Search</span>
                                </button>
                                <button class="btn btn-outline-secondary ms-2" onclick="app.clearClassesFilters()">
                                    <i class="bi bi-x-circle"></i> <span data-translate="classes.clear">Clear</span>
                                </button>
                            </div>
                        </div>
                    </div>
                    <div class="row mt-3">
                        <div class="col-12">
                            <button class="btn btn-primary" onclick="app.filterClasses()">
                                <i class="bi bi-search"></i> <span data-translate="classes.search">Search</span>
                            </button>
                            <button class="btn btn-outline-secondary ms-2" onclick="app.clearClassesFilters()">
                                <i class="bi bi-x-circle"></i> <span data-translate="classes.clear">Clear</span>
                            </button>
                            <div class="btn-group ms-3">
                                <button class="btn btn-outline-primary dropdown-toggle" data-bs-toggle="dropdown">
                                    <i class="bi bi-download"></i> <span data-translate="classes.export">Export</span>
                                </button>
                                <ul class="dropdown-menu">
                                    <li><a class="dropdown-item" href="#" onclick="app.exportClasses('csv')" data-translate="classes.export_csv">Export CSV</a></li>
                                    <li><a class="dropdown-item" href="#" onclick="app.exportClasses('excel')" data-translate="classes.export_excel">Export Excel</a></li>
                                </ul>
                            </div>
                            <button class="btn btn-outline-secondary ms-2" onclick="app.printClasses()">
                                <i class="bi bi-printer"></i> <span data-translate="classes.print">Print</span>
                            </button>
                        </div>
                    </div>
                </div>
                
                <!-- Classes Table -->
                <div class="card">
                    <div class="card-body">
                        <div class="table-responsive">
                            <table class="table table-hover">
                                <thead>
                                    <tr>
                                        <th><a href="#" onclick="app.sortClasses('name')" class="text-decoration-none"><span data-translate="table.class_name">Class Name</span> <i class="bi bi-arrow-down-up"></i></a></th>
                                        <th><a href="#" onclick="app.sortClasses('enabled')" class="text-decoration-none"><span data-translate="table.class_status">Status</span> <i class="bi bi-arrow-down-up"></i></a></th>
                                        <th><a href="#" onclick="app.sortClasses('created_at')" class="text-decoration-none"><span data-translate="table.class_created">Created</span> <i class="bi bi-arrow-down-up"></i></a></th>
                                        <th width="120" data-translate="table.actions">Actions</th>
                                    </tr>
                                </thead>
                                <tbody id="classesTableBody">
                                    <tr>
                                        <td colspan="4" class="text-center loading">
                                            <i class="bi bi-arrow-clockwise"></i> <span data-translate="classes.loading">Loading classes...</span>
                                        </td>
                                    </tr>
                                </tbody>
                            </table>
                        </div>
                    </div>
                </div>
                
                <!-- Class Modal -->
                <div class="modal" id="classModal">
                    <div class="modal-dialog">
                        <div class="modal-content">
                            <div class="modal-header">
                                <h5 class="modal-title" id="classModalTitle">Add Class</h5>
                                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                            </div>
                            <div class="modal-body">
                                <form id="classForm">
                                    <input type="hidden" id="classId">
                                    
                                    <div class="mb-3">
                                        <label for="className" class="form-label" data-translate="classes.class_name_label">Class Name *</label>
                                        <input type="text" class="form-control" id="className" required maxlength="50" data-translate-placeholder="classes.name_placeholder" placeholder="e.g., A, B, Red, Alpha, Green">
                                        <div class="form-text" data-translate="classes.name_help">Class identifier that can be reused across all grades and years</div>
                                    </div>
                                    
                                    <div class="mb-3">
                                        <label for="classEnabled" class="form-label" data-translate="classes.status_label">Status</label>
                                        <select class="form-select" id="classEnabled">
                                            <option value="1" data-translate="classes.active">Active</option>
                                            <option value="0" data-translate="classes.disabled">Disabled</option>
                                        </select>
                                    </div>
                                    
                                    <div class="alert alert-info">
                                        <i class="bi bi-info-circle"></i>
                                        <strong data-translate="classes.note">Note: Classes are now independent and reusable. When creating student enrollments, you'll select the academic year, grade, and class separately.</strong>
                                    </div>
                                </form>
                            </div>
                            <div class="modal-footer">
                                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal" data-translate="btn.cancel">Cancel</button>
                                <button type="button" class="btn btn-primary" onclick="app.saveClass()">
                                    <span id="classSaveButtonText" data-translate="classes.save_class">Save Class</span>
                                </button>
                            </div>
                        </div>
                    </div>
                </div>
            `;
            
            document.getElementById('pageContent').innerHTML = content;
            
            // Load classes data
            await this.loadClassesData();
            
        } catch (error) {
            document.getElementById('pageContent').innerHTML = `
                <div class="alert alert-danger">
                    <i class="bi bi-exclamation-triangle"></i> <span data-translate="classes.error_loading">Error loading classes</span>: ${error.message}
                </div>`;
        }
    }
    
    // No longer need dropdown loading for classes since they're independent
    
    async loadClassesData() {
        try {
            const response = await this.apiCall('GET', '/admin/classes');
            document.getElementById('classesTableBody').innerHTML = this.renderClassesTable(response.data);
        } catch (error) {
            document.getElementById('classesTableBody').innerHTML = `
                <tr>
                    <td colspan="4" class="text-center text-danger">
                        <i class="bi bi-exclamation-triangle"></i> <span data-translate="classes.failed_to_load">Failed to load classes</span>
                    </td>
                </tr>
            `;
        }
    }
    
    renderClassesTable(classes) {
        if (!classes || classes.length === 0) {
            return `
                <tr>
                    <td colspan="4" class="text-center text-muted">
                        <i class="bi bi-inbox"></i> <span data-translate="classes.no_classes_found">No classes found</span>
                        <br>
                        <small class="text-muted" data-translate="classes.add_classes_help">Add classes that can be reused across all grades and years</small>
                    </td>
                </tr>
            `;
        }
        
        return classes.map(classItem => {
            const safeName = classItem.name.replace(/'/g, '&#39;');
            return `
            <tr>
                <td><strong>${classItem.name}</strong></td>
                <td>
                    <span class="badge ${classItem.enabled ? 'bg-success' : 'bg-secondary'}">
                        ${classItem.enabled ? 
                            (typeof langManager !== 'undefined' ? langManager.t('classes.active') : 'Active') : 
                            (typeof langManager !== 'undefined' ? langManager.t('classes.disabled') : 'Disabled')
                        }
                    </span>
                </td>
                <td>${new Date(classItem.created_at).toLocaleDateString()}</td>
                <td>
                    <div class="btn-group btn-group-sm">
                        <button class="btn btn-outline-primary" onclick="app.editClass(${classItem.id})" title="Edit">
                            <i class="bi bi-pencil"></i>
                        </button>
                        <button class="btn btn-outline-danger" onclick="app.deleteClass(${classItem.id}, '${safeName}')" title="Delete">
                            <i class="bi bi-trash"></i>
                        </button>
                    </div>
                </td>
            </tr>
            `;
        }).join('');
    }
    
    // Classes CRUD Functions
    showClassModal(classData = null) {
        const modalElement = document.getElementById('classModal');
        if (!modalElement) {
            this.showToast('error', 'Modal not found. Please refresh the page.');
            return;
        }
        
        // Remove any existing backdrops
        document.querySelectorAll('.modal-backdrop').forEach(backdrop => backdrop.remove());
        
        // Reset form
        const form = document.getElementById('classForm');
        if (form) form.reset();
        
        const classIdField = document.getElementById('classId');
        if (classIdField) classIdField.value = '';
        
        if (classData) {
            // Edit mode
            const modalTitle = document.getElementById('classModalTitle');
            const saveButtonText = document.getElementById('classSaveButtonText');
            if (modalTitle) {
                modalTitle.textContent = (typeof langManager !== 'undefined') 
                    ? langManager.t('classes.edit_class') 
                    : 'Edit Class';
            }
            if (saveButtonText) {
                saveButtonText.textContent = (typeof langManager !== 'undefined') 
                    ? langManager.t('classes.update_class') 
                    : 'Update Class';
            }
            if (classIdField) classIdField.value = classData.id;
            document.getElementById('className').value = classData.name;
            document.getElementById('classEnabled').value = classData.enabled;
        } else {
            // Add mode
            const modalTitle = document.getElementById('classModalTitle');
            const saveButtonText = document.getElementById('classSaveButtonText');
            if (modalTitle) {
                modalTitle.textContent = (typeof langManager !== 'undefined') 
                    ? langManager.t('classes.add_class') 
                    : 'Add Class';
            }
            if (saveButtonText) {
                saveButtonText.textContent = (typeof langManager !== 'undefined') 
                    ? langManager.t('classes.save_class') 
                    : 'Save Class';
            }
        }
        
        // Clear validation classes
        document.querySelectorAll('.is-invalid').forEach(el => el.classList.remove('is-invalid'));
        
        // Show modal
        modalElement.style.display = 'flex';
        modalElement.classList.add('show');
        document.body.classList.add('modal-open');
        
        // Add close button handlers
        const closeButtons = modalElement.querySelectorAll('[data-bs-dismiss="modal"]');
        closeButtons.forEach(btn => {
            btn.onclick = (e) => {
                e.preventDefault();
                this.closeClassModal();
            };
        });
        
        // Focus on first input
        setTimeout(() => {
            const firstInput = modalElement.querySelector('select');
            if (firstInput) firstInput.focus();
        }, 100);
    }
    
    closeClassModal() {
        const modalElement = document.getElementById('classModal');
        if (modalElement) {
            modalElement.style.display = 'none';
            modalElement.classList.remove('show');
        }
        document.body.classList.remove('modal-open');
        document.querySelectorAll('.modal-backdrop').forEach(backdrop => backdrop.remove());
    }
    
    async saveClass() {
        const classIdField = document.getElementById('classId');
        const nameField = document.getElementById('className');
        const enabledField = document.getElementById('classEnabled');
        
        if (!nameField || !enabledField) {
            this.showToast('error', 'Form fields not found. Please refresh the page.');
            return;
        }
        
        const classId = classIdField ? classIdField.value : '';
        const isEdit = !!classId;
        
        const data = {
            name: nameField.value.trim(),
            enabled: parseInt(enabledField.value)
        };
        
        // Clear previous validation
        document.querySelectorAll('.is-invalid').forEach(el => el.classList.remove('is-invalid'));
        
        try {
            const endpoint = isEdit ? `/admin/classes/${classId}` : '/admin/classes';
            const method = isEdit ? 'PUT' : 'POST';
            
            const response = await this.apiCall(method, endpoint, data);
            
            if (response.success) {
                this.closeClassModal();
                this.showToast('success', `Class ${isEdit ? 'updated' : 'created'} successfully!`);
                this.loadClassesData();
            } else {
                this.showToast('error', response.message);
            }
            
        } catch (error) {
            this.showToast('error', 'Failed to save class. Please try again.');
        }
    }
    
    async editClass(id) {
        try {
            const response = await this.apiCall('GET', `/admin/classes/${id}`);
            if (response.success) {
                this.showClassModal(response.data);
            }
        } catch (error) {
            this.showToast('error', 'Failed to load class data');
        }
    }
    
    async deleteClass(id, name) {
        if (!confirm(`Are you sure you want to delete the class "${name}"?\n\nThis action cannot be undone and may affect related data.`)) {
            return;
        }
        
        try {
            const response = await this.apiCall('DELETE', `/admin/classes/${id}`);
            
            if (response.success) {
                this.showToast('success', 'Class deleted successfully!');
                this.loadClassesData();
            } else {
                this.showToast('error', response.message);
            }
        } catch (error) {
            this.showToast('error', 'Failed to delete class. It may have related data.');
        }
    }
    
    async filterClasses() {
        const name = document.getElementById('classNameFilter').value;
        const status = document.getElementById('classStatusFilter').value;
        
        const params = new URLSearchParams();
        if (name) params.append('name', name);
        if (status !== '') params.append('enabled', status);
        
        try {
            const response = await this.apiCall('GET', `/admin/classes?${params.toString()}`);
            document.getElementById('classesTableBody').innerHTML = this.renderClassesTable(response.data);
        } catch (error) {
            this.showToast('error', 'Failed to filter classes');
        }
    }
    
    clearClassesFilters() {
        document.getElementById('classNameFilter').value = '';
        document.getElementById('classStatusFilter').value = '';
        this.loadClassesData();
    }
    
    async sortClasses(field) {
        this.currentSort = this.currentSort || {};
        const currentDir = this.currentSort[field] || 'asc';
        const newDir = currentDir === 'asc' ? 'desc' : 'asc';
        this.currentSort[field] = newDir;
        
        try {
            const response = await this.apiCall('GET', `/admin/classes?sort_by=${field}&sort_dir=${newDir}`);
            document.getElementById('classesTableBody').innerHTML = this.renderClassesTable(response.data);
        } catch (error) {
            this.showToast('error', 'Failed to sort classes');
        }
    }
    
    exportClasses(format) {
        this.showToast('info', `${format.toUpperCase()} export functionality coming soon!`);
    }
    
    printClasses() {
        this.showToast('info', 'Print functionality coming soon!');
    }
    
    // Subjects Management
    async loadSubjects() {
        try {
            const response = await this.apiCall('GET', '/admin/subjects');
            
            const content = `
                <div class="page-header">
                    <div class="d-flex justify-content-between align-items-center">
                        <div>
                            <h1 class="h2 mb-1"><i class="bi bi-book"></i> <span data-translate="subjects.title">Subjects</span></h1>
                            <p class="text-muted mb-0" data-translate="subjects.subtitle">Manage academic subjects</p>
                        </div>
                        <button class="btn btn-primary" onclick="app.showSubjectModal()">
                            <i class="bi bi-plus-circle"></i> <span data-translate="subjects.add_subject">Add Subject</span>
                        </button>
                    </div>
                </div>
                
                <!-- Filters -->
                <div class="filter-bar">
                    <div class="row g-3">
                        <div class="col-md-4">
                            <label class="form-label" data-translate="subjects.subject_name">Subject Name</label>
                            <input type="text" class="form-control" id="subjectNameFilter" data-translate-placeholder="subjects.search_placeholder" placeholder="Search subject name...">
                        </div>
                        <div class="col-md-4">
                            <label class="form-label" data-translate="subjects.status">Status</label>
                            <select class="form-select" id="subjectStatusFilter">
                                <option value="" data-translate="subjects.all_statuses">All Statuses</option>
                                <option value="1" data-translate="subjects.active">Active</option>
                                <option value="0" data-translate="subjects.disabled">Disabled</option>
                            </select>
                        </div>
                        <div class="col-md-4">
                            <label class="form-label">&nbsp;</label>
                            <div class="d-flex">
                                <button class="btn btn-primary" onclick="app.filterSubjects()">
                                    <i class="bi bi-search"></i> <span data-translate="subjects.search">Search</span>
                                </button>
                                <button class="btn btn-outline-secondary ms-2" onclick="app.clearSubjectsFilters()">
                                    <i class="bi bi-x-circle"></i> <span data-translate="subjects.clear">Clear</span>
                                </button>
                            </div>
                        </div>
                    </div>
                    <div class="row mt-3">
                        <div class="col-12">
                            <div class="btn-group">
                                <button class="btn btn-outline-primary dropdown-toggle" data-bs-toggle="dropdown">
                                    <i class="bi bi-download"></i> <span data-translate="subjects.export">Export</span>
                                </button>
                                <ul class="dropdown-menu">
                                    <li><a class="dropdown-item" href="#" onclick="app.exportSubjects('csv')" data-translate="subjects.export_csv">Export CSV</a></li>
                                    <li><a class="dropdown-item" href="#" onclick="app.exportSubjects('excel')" data-translate="subjects.export_excel">Export Excel</a></li>
                                </ul>
                            </div>
                            <button class="btn btn-outline-secondary ms-2" onclick="app.printSubjects()">
                                <i class="bi bi-printer"></i> <span data-translate="subjects.print">Print</span>
                            </button>
                        </div>
                    </div>
                </div>
                
                <!-- Subjects Table -->
                <div class="card">
                    <div class="card-body">
                        <div class="table-responsive">
                            <table class="table table-hover">
                                <thead>
                                    <tr>
                                        <th><a href="#" onclick="app.sortSubjects('name')" class="text-decoration-none"><span data-translate="table.subject_name">Subject Name</span> <i class="bi bi-arrow-down-up"></i></a></th>
                                        <th><a href="#" onclick="app.sortSubjects('enabled')" class="text-decoration-none"><span data-translate="table.subject_status">Status</span> <i class="bi bi-arrow-down-up"></i></a></th>
                                        <th><a href="#" onclick="app.sortSubjects('created_at')" class="text-decoration-none"><span data-translate="table.subject_created">Created</span> <i class="bi bi-arrow-down-up"></i></a></th>
                                        <th width="120" data-translate="table.actions">Actions</th>
                                    </tr>
                                </thead>
                                <tbody id="subjectsTableBody">
                                    <tr>
                                        <td colspan="4" class="text-center loading">
                                            <i class="bi bi-arrow-clockwise"></i> <span data-translate="subjects.loading">Loading subjects...</span>
                                        </td>
                                    </tr>
                                </tbody>
                            </table>
                        </div>
                    </div>
                </div>
                
                <!-- Quick Add Common Subjects -->
                <div class="card">
                    <div class="card-header">
                        <h6 class="mb-0"><i class="bi bi-lightning"></i> <span data-translate="subjects.quick_add_title">Quick Add Common Subjects</span></h6>
                    </div>
                    <div class="card-body">
                        <div class="row g-2">
                            <div class="col-md-3">
                                <button class="btn btn-outline-primary btn-sm w-100" onclick="app.quickAddSubject('Mathematics')">
                                    <i class="bi bi-calculator"></i> <span data-translate="subjects.mathematics">Mathematics</span>
                                </button>
                            </div>
                            <div class="col-md-3">
                                <button class="btn btn-outline-primary btn-sm w-100" onclick="app.quickAddSubject('English Language')">
                                    <i class="bi bi-type"></i> <span data-translate="subjects.english_language">English Language</span>
                                </button>
                            </div>
                            <div class="col-md-3">
                                <button class="btn btn-outline-primary btn-sm w-100" onclick="app.quickAddSubject('Science')">
                                    <i class="bi bi-flask"></i> <span data-translate="subjects.science">Science</span>
                                </button>
                            </div>
                            <div class="col-md-3">
                                <button class="btn btn-outline-primary btn-sm w-100" onclick="app.quickAddSubject('Social Studies')">
                                    <i class="bi bi-globe"></i> <span data-translate="subjects.social_studies">Social Studies</span>
                                </button>
                            </div>
                            <div class="col-md-3">
                                <button class="btn btn-outline-primary btn-sm w-100" onclick="app.quickAddSubject('Art & Craft')">
                                    <i class="bi bi-palette"></i> <span data-translate="subjects.art_craft">Art & Craft</span>
                                </button>
                            </div>
                            <div class="col-md-3">
                                <button class="btn btn-outline-primary btn-sm w-100" onclick="app.quickAddSubject('Physical Education')">
                                    <i class="bi bi-activity"></i> <span data-translate="subjects.physical_education">Physical Education</span>
                                </button>
                            </div>
                            <div class="col-md-3">
                                <button class="btn btn-outline-primary btn-sm w-100" onclick="app.quickAddSubject('Computer Science')">
                                    <i class="bi bi-laptop"></i> <span data-translate="subjects.computer_science">Computer Science</span>
                                </button>
                            </div>
                            <div class="col-md-3">
                                <button class="btn btn-outline-primary btn-sm w-100" onclick="app.quickAddSubject('Music')">
                                    <i class="bi bi-music-note"></i> <span data-translate="subjects.music">Music</span>
                                </button>
                            </div>
                        </div>
                    </div>
                </div>
                
                <!-- Subject Modal -->
                <div class="modal" id="subjectModal">
                    <div class="modal-dialog">
                        <div class="modal-content">
                            <div class="modal-header">
                                <h5 class="modal-title" id="subjectModalTitle">Add Subject</h5>
                                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                            </div>
                            <div class="modal-body">
                                <form id="subjectForm">
                                    <input type="hidden" id="subjectId">
                                    
                                    <div class="mb-3">
                                        <label for="subjectName" class="form-label" data-translate="subjects.subject_name_label">Subject Name *</label>
                                        <input type="text" class="form-control" id="subjectName" required maxlength="100" data-translate-placeholder="subjects.name_placeholder" placeholder="e.g., Mathematics, English Language">
                                        <div class="form-text" data-translate="subjects.name_help">Enter the full subject name (2-100 characters)</div>
                                    </div>
                                    
                                    <div class="mb-3">
                                        <label for="subjectEnabled" class="form-label" data-translate="subjects.status_label">Status</label>
                                        <select class="form-select" id="subjectEnabled">
                                            <option value="1" data-translate="subjects.active">Active</option>
                                            <option value="0" data-translate="subjects.disabled">Disabled</option>
                                        </select>
                                    </div>
                                </form>
                            </div>
                            <div class="modal-footer">
                                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal" data-translate="btn.cancel">Cancel</button>
                                <button type="button" class="btn btn-primary" onclick="app.saveSubject()">
                                    <span id="subjectSaveButtonText" data-translate="subjects.save_subject">Save Subject</span>
                                </button>
                            </div>
                        </div>
                    </div>
                </div>
            `;
            
            document.getElementById('pageContent').innerHTML = content;
            
            // Load subjects data
            await this.loadSubjectsData();
            
        } catch (error) {
            document.getElementById('pageContent').innerHTML = `
                <div class="alert alert-danger">
                    <i class="bi bi-exclamation-triangle"></i> <span data-translate="subjects.error_loading">Error loading subjects</span>: ${error.message}
                </div>`;
        }
    }
    
    async loadSubjectsData() {
        try {
            const response = await this.apiCall('GET', '/admin/subjects');
            document.getElementById('subjectsTableBody').innerHTML = this.renderSubjectsTable(response.data);
        } catch (error) {
            document.getElementById('subjectsTableBody').innerHTML = `
                <tr>
                    <td colspan="4" class="text-center text-danger">
                        <i class="bi bi-exclamation-triangle"></i> <span data-translate="subjects.failed_to_load">Failed to load subjects</span>
                    </td>
                </tr>
            `;
        }
    }
    
    renderSubjectsTable(subjects) {
        if (!subjects || subjects.length === 0) {
            return `
                <tr>
                    <td colspan="4" class="text-center text-muted">
                        <i class="bi bi-inbox"></i> <span data-translate="subjects.no_subjects_found">No subjects found</span>
                    </td>
                </tr>
            `;
        }
        
        return subjects.map(subject => `
            <tr>
                <td><strong>${subject.name}</strong></td>
                <td>
                    <span class="badge ${subject.enabled ? 'bg-success' : 'bg-secondary'}">
                        ${subject.enabled ? 
                            (typeof langManager !== 'undefined' ? langManager.t('subjects.active') : 'Active') : 
                            (typeof langManager !== 'undefined' ? langManager.t('subjects.disabled') : 'Disabled')
                        }
                    </span>
                </td>
                <td>${new Date(subject.created_at).toLocaleDateString()}</td>
                <td>
                    <div class="btn-group btn-group-sm">
                        <button class="btn btn-outline-primary" onclick="app.editSubject(${subject.id})" title="Edit">
                            <i class="bi bi-pencil"></i>
                        </button>
                        <button class="btn btn-outline-danger" onclick="app.deleteSubject(${subject.id}, '${subject.name.replace(/'/g, '&#39;')}')" title="Delete">
                            <i class="bi bi-trash"></i>
                        </button>
                    </div>
                </td>
            </tr>
        `).join('');
    }
    
    // Subjects CRUD Functions
    showSubjectModal(subjectData = null) {
        const modalElement = document.getElementById('subjectModal');
        if (!modalElement) {
            this.showToast('error', 'Modal not found. Please refresh the page.');
            return;
        }
        
        // Remove any existing backdrops
        document.querySelectorAll('.modal-backdrop').forEach(backdrop => backdrop.remove());
        
        // Reset form
        const form = document.getElementById('subjectForm');
        if (form) form.reset();
        
        const subjectIdField = document.getElementById('subjectId');
        if (subjectIdField) subjectIdField.value = '';
        
        if (subjectData) {
            // Edit mode
            const modalTitle = document.getElementById('subjectModalTitle');
            const saveButtonText = document.getElementById('subjectSaveButtonText');
            if (modalTitle) {
                modalTitle.textContent = (typeof langManager !== 'undefined') 
                    ? langManager.t('subjects.edit_subject') 
                    : 'Edit Subject';
            }
            if (saveButtonText) {
                saveButtonText.textContent = (typeof langManager !== 'undefined') 
                    ? langManager.t('subjects.update_subject') 
                    : 'Update Subject';
            }
            if (subjectIdField) subjectIdField.value = subjectData.id;
            document.getElementById('subjectName').value = subjectData.name;
            document.getElementById('subjectEnabled').value = subjectData.enabled;
        } else {
            // Add mode
            const modalTitle = document.getElementById('subjectModalTitle');
            const saveButtonText = document.getElementById('subjectSaveButtonText');
            if (modalTitle) {
                modalTitle.textContent = (typeof langManager !== 'undefined') 
                    ? langManager.t('subjects.add_subject') 
                    : 'Add Subject';
            }
            if (saveButtonText) {
                saveButtonText.textContent = (typeof langManager !== 'undefined') 
                    ? langManager.t('subjects.save_subject') 
                    : 'Save Subject';
            }
        }
        
        // Clear validation classes
        document.querySelectorAll('.is-invalid').forEach(el => el.classList.remove('is-invalid'));
        
        // Show modal
        modalElement.style.display = 'flex';
        modalElement.classList.add('show');
        document.body.classList.add('modal-open');
        
        // Add close button handlers
        const closeButtons = modalElement.querySelectorAll('[data-bs-dismiss="modal"]');
        closeButtons.forEach(btn => {
            btn.onclick = (e) => {
                e.preventDefault();
                this.closeSubjectModal();
            };
        });
        
        // Focus on first input
        setTimeout(() => {
            const firstInput = modalElement.querySelector('input[type="text"]');
            if (firstInput) firstInput.focus();
        }, 100);
    }
    
    closeSubjectModal() {
        const modalElement = document.getElementById('subjectModal');
        if (modalElement) {
            modalElement.style.display = 'none';
            modalElement.classList.remove('show');
        }
        document.body.classList.remove('modal-open');
        document.querySelectorAll('.modal-backdrop').forEach(backdrop => backdrop.remove());
    }
    
    async saveSubject() {
        const subjectIdField = document.getElementById('subjectId');
        const nameField = document.getElementById('subjectName');
        const enabledField = document.getElementById('subjectEnabled');
        
        if (!nameField || !enabledField) {
            this.showToast('error', 'Form fields not found. Please refresh the page.');
            return;
        }
        
        const subjectId = subjectIdField ? subjectIdField.value : '';
        const isEdit = !!subjectId;
        
        const data = {
            name: nameField.value.trim(),
            enabled: parseInt(enabledField.value)
        };
        
        // Clear previous validation
        document.querySelectorAll('.is-invalid').forEach(el => el.classList.remove('is-invalid'));
        
        try {
            const endpoint = isEdit ? `/admin/subjects/${subjectId}` : '/admin/subjects';
            const method = isEdit ? 'PUT' : 'POST';
            
            const response = await this.apiCall(method, endpoint, data);
            
            if (response.success) {
                this.closeSubjectModal();
                this.showToast('success', `Subject ${isEdit ? 'updated' : 'created'} successfully!`);
                this.loadSubjectsData();
            } else {
                this.showToast('error', response.message);
            }
            
        } catch (error) {
            this.showToast('error', 'Failed to save subject. Please try again.');
        }
    }
    
    async editSubject(id) {
        try {
            const response = await this.apiCall('GET', `/admin/subjects/${id}`);
            if (response.success) {
                this.showSubjectModal(response.data);
            }
        } catch (error) {
            this.showToast('error', 'Failed to load subject data');
        }
    }
    
    async deleteSubject(id, name) {
        if (!confirm(`Are you sure you want to delete the subject "${name}"?\n\nThis action cannot be undone and may affect related data.`)) {
            return;
        }
        
        try {
            const response = await this.apiCall('DELETE', `/admin/subjects/${id}`);
            
            if (response.success) {
                this.showToast('success', 'Subject deleted successfully!');
                this.loadSubjectsData();
            } else {
                this.showToast('error', response.message);
            }
        } catch (error) {
            this.showToast('error', 'Failed to delete subject. It may have related data.');
        }
    }
    
    async quickAddSubject(name) {
        const data = {
            name: name,
            enabled: 1
        };
        
        try {
            const response = await this.apiCall('POST', '/admin/subjects', data);
            
            if (response.success) {
                this.showToast('success', `${name} added successfully!`);
                this.loadSubjectsData();
            } else {
                if (response.message.includes('duplicate') || response.message.includes('exists')) {
                    this.showToast('warning', `${name} already exists!`);
                } else {
                    this.showToast('error', response.message);
                }
            }
            
        } catch (error) {
            this.showToast('error', `Failed to add ${name}. Please try again.`);
        }
    }
    
    async filterSubjects() {
        const name = document.getElementById('subjectNameFilter').value;
        const status = document.getElementById('subjectStatusFilter').value;
        
        const params = new URLSearchParams();
        if (name) params.append('name', name);
        if (status !== '') params.append('enabled', status);
        
        try {
            const response = await this.apiCall('GET', `/admin/subjects?${params.toString()}`);
            document.getElementById('subjectsTableBody').innerHTML = this.renderSubjectsTable(response.data);
        } catch (error) {
            this.showToast('error', 'Failed to filter subjects');
        }
    }
    
    clearSubjectsFilters() {
        document.getElementById('subjectNameFilter').value = '';
        document.getElementById('subjectStatusFilter').value = '';
        this.loadSubjectsData();
    }
    
    async sortSubjects(field) {
        this.currentSort = this.currentSort || {};
        const currentDir = this.currentSort[field] || 'asc';
        const newDir = currentDir === 'asc' ? 'desc' : 'asc';
        this.currentSort[field] = newDir;
        
        try {
            const response = await this.apiCall('GET', `/admin/subjects?sort_by=${field}&sort_dir=${newDir}`);
            document.getElementById('subjectsTableBody').innerHTML = this.renderSubjectsTable(response.data);
        } catch (error) {
            this.showToast('error', 'Failed to sort subjects');
        }
    }
    
    exportSubjects(format) {
        this.showToast('info', `${format.toUpperCase()} export functionality coming soon!`);
    }
    
    printSubjects() {
        this.showToast('info', 'Print functionality coming soon!');
    }
    
    // Grade-Subject Rules Management
    async loadGradeSubjectRules() {
        try {
            const response = await this.apiCall('GET', '/admin/grade-subject-rules');
            
            const content = `
                <div class="page-header">
                    <div class="d-flex justify-content-between align-items-center">
                        <div>
                            <h1 class="h2 mb-1"><i class="bi bi-sliders"></i> Grade-Subject Rules</h1>
                            <p class="text-muted mb-0">Set min/max scores for each grade-subject combination</p>
                        </div>
                        <div class="btn-group">
                            <button class="btn btn-primary" onclick="app.showGradeSubjectRuleModal()">
                                <i class="bi bi-plus-circle"></i> Add Rule
                            </button>
                            <button class="btn btn-success" onclick="app.showMultiGradeRuleModal()">
                                <i class="bi bi-collection"></i> Add Multi-Grade Rules
                            </button>
                        </div>
                    </div>
                </div>
                
                <!-- Filters -->
                <div class="filter-bar">
                    <div class="row g-3">
                        <div class="col-md-3">
                            <label class="form-label">Grade</label>
                            <select class="form-select" id="ruleGradeFilter">
                                <option value="">All Grades</option>
                            </select>
                        </div>
                        <div class="col-md-3">
                            <label class="form-label">Subject</label>
                            <select class="form-select" id="ruleSubjectFilter">
                                <option value="">All Subjects</option>
                            </select>
                        </div>
                        <div class="col-md-3">
                            <label class="form-label">Search</label>
                            <input type="text" class="form-control" id="ruleSearchFilter" placeholder="Search grade or subject...">
                        </div>
                        <div class="col-md-3">
                            <label class="form-label">Status</label>
                            <select class="form-select" id="ruleStatusFilter">
                                <option value="">All Statuses</option>
                                <option value="1">Active</option>
                                <option value="0">Disabled</option>
                            </select>
                        </div>
                    </div>
                    <div class="row mt-3">
                        <div class="col-12">
                            <button class="btn btn-primary" onclick="app.filterGradeSubjectRules()">
                                <i class="bi bi-search"></i> Search
                            </button>
                            <button class="btn btn-outline-secondary ms-2" onclick="app.clearGradeSubjectRulesFilters()">
                                <i class="bi bi-x-circle"></i> Clear
                            </button>
                            <div class="btn-group ms-3">
                                <button class="btn btn-outline-primary dropdown-toggle" data-bs-toggle="dropdown">
                                    <i class="bi bi-download"></i> Export
                                </button>
                                <ul class="dropdown-menu">
                                    <li><a class="dropdown-item" href="#" onclick="app.exportGradeSubjectRules('csv')">Export CSV</a></li>
                                    <li><a class="dropdown-item" href="#" onclick="app.exportGradeSubjectRules('excel')">Export Excel</a></li>
                                </ul>
                            </div>
                            <button class="btn btn-outline-secondary ms-2" onclick="app.printGradeSubjectRules()">
                                <i class="bi bi-printer"></i> Print
                            </button>
                        </div>
                    </div>
                </div>
                
                <!-- Rules Table -->
                <div class="card">
                    <div class="card-body">
                        <div class="table-responsive">
                            <table class="table table-hover">
                                <thead>
                                    <tr>
                                        <th><a href="#" onclick="app.sortGradeSubjectRules('grade_title')" class="text-decoration-none">Grade <i class="bi bi-arrow-down-up"></i></a></th>
                                        <th><a href="#" onclick="app.sortGradeSubjectRules('subject_name')" class="text-decoration-none">Subject <i class="bi bi-arrow-down-up"></i></a></th>
                                        <th><a href="#" onclick="app.sortGradeSubjectRules('min_score')" class="text-decoration-none">Min Score <i class="bi bi-arrow-down-up"></i></a></th>
                                        <th><a href="#" onclick="app.sortGradeSubjectRules('max_score')" class="text-decoration-none">Max Score <i class="bi bi-arrow-down-up"></i></a></th>
                                        <th><a href="#" onclick="app.sortGradeSubjectRules('enabled')" class="text-decoration-none">Status <i class="bi bi-arrow-down-up"></i></a></th>
                                        <th><a href="#" onclick="app.sortGradeSubjectRules('created_at')" class="text-decoration-none">Created <i class="bi bi-arrow-down-up"></i></a></th>
                                        <th width="120" data-translate="table.actions">Actions</th>
                                    </tr>
                                </thead>
                                <tbody id="gradeSubjectRulesTableBody">
                                    <tr>
                                        <td colspan="7" class="text-center loading">
                                            <i class="bi bi-arrow-clockwise"></i> Loading rules...
                                        </td>
                                    </tr>
                                </tbody>
                            </table>
                        </div>
                    </div>
                </div>
                
                <!-- Grade Subject Rule Modal -->
                <div class="modal" id="gradeSubjectRuleModal">
                    <div class="modal-dialog">
                        <div class="modal-content">
                            <div class="modal-header">
                                <h5 class="modal-title" id="gradeSubjectRuleModalTitle">Add Grade-Subject Rule</h5>
                                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                            </div>
                            <div class="modal-body">
                                <form id="gradeSubjectRuleForm">
                                    <input type="hidden" id="gradeSubjectRuleId">
                                    
                                    <div class="mb-3">
                                        <label for="ruleGradeId" class="form-label">Grade *</label>
                                        <select class="form-select" id="ruleGradeId" required>
                                            <option value="">Select Grade</option>
                                        </select>
                                    </div>
                                    
                                    <div class="mb-3">
                                        <label for="ruleSubjectId" class="form-label">Subject *</label>
                                        <select class="form-select" id="ruleSubjectId" required>
                                            <option value="">Select Subject</option>
                                        </select>
                                    </div>
                                    
                                    <div class="row">
                                        <div class="col-md-6">
                                            <div class="mb-3">
                                                <label for="ruleMinScore" class="form-label">Minimum Score *</label>
                                                <input type="number" class="form-control" id="ruleMinScore" required min="0" max="999.99" step="0.01" placeholder="0.00">
                                                <div class="form-text">Lowest possible score for this grade-subject</div>
                                            </div>
                                        </div>
                                        <div class="col-md-6">
                                            <div class="mb-3">
                                                <label for="ruleMaxScore" class="form-label">Maximum Score *</label>
                                                <input type="number" class="form-control" id="ruleMaxScore" required min="0" max="999.99" step="0.01" placeholder="100.00">
                                                <div class="form-text">Highest possible score for this grade-subject</div>
                                            </div>
                                        </div>
                                    </div>
                                    
                                    <div class="mb-3">
                                        <label for="ruleEnabled" class="form-label">Status</label>
                                        <select class="form-select" id="ruleEnabled">
                                            <option value="1">Active</option>
                                            <option value="0">Disabled</option>
                                        </select>
                                    </div>
                                    
                                    <div class="alert alert-info">
                                        <i class="bi bi-info-circle"></i>
                                        <strong>Note:</strong> This rule will be used as the default min/max scores when creating exams for this grade-subject combination.
                                    </div>
                                </form>
                            </div>
                            <div class="modal-footer">
                                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                                <button type="button" class="btn btn-primary" onclick="app.saveGradeSubjectRule()">
                                    <span id="gradeSubjectRuleSaveButtonText">Save Rule</span>
                                </button>
                            </div>
                        </div>
                    </div>
                </div>
                
                <!-- Multi-Grade Rule Modal -->
                <div class="modal" id="multiGradeRuleModal">
                    <div class="modal-dialog modal-xl">
                        <div class="modal-content">
                            <div class="modal-header">
                                <h5 class="modal-title" id="multiGradeRuleModalTitle">Add Rules for Multiple Grades</h5>
                                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                            </div>
                            <div class="modal-body">
                                <form id="multiGradeRuleForm">
                                    <div class="row mb-3">
                                        <div class="col-md-4">
                                            <label for="multiRuleSubjectId" class="form-label">Subject *</label>
                                            <select class="form-select" id="multiRuleSubjectId" required onchange="app.loadAvailableGrades()">
                                                <option value="">Select Subject</option>
                                            </select>
                                        </div>
                                        <div class="col-md-3">
                                            <label for="defaultMinScore" class="form-label">Default Min Score</label>
                                            <input type="number" class="form-control" id="defaultMinScore" value="50" min="0" max="999.99" step="0.01">
                                        </div>
                                        <div class="col-md-3">
                                            <label for="defaultMaxScore" class="form-label">Default Max Score</label>
                                            <input type="number" class="form-control" id="defaultMaxScore" value="100" min="0" max="999.99" step="0.01">
                                        </div>
                                        <div class="col-md-2 d-flex align-items-end">
                                            <button type="button" class="btn btn-outline-info btn-sm w-100" onclick="app.applyDefaultScores()" title="Apply default scores to all grades">
                                                <i class="bi bi-arrow-down"></i> Apply
                                            </button>
                                        </div>
                                    </div>
                                    
                                    <div class="mb-3">
                                        <div id="gradeSelectionContainer">
                                            <div class="text-muted text-center py-4">
                                                <i class="bi bi-arrow-up"></i> Please select a subject first
                                            </div>
                                        </div>
                                    </div>
                                    
                                    <div class="alert alert-info py-2">
                                        <small>
                                            <i class="bi bi-info-circle"></i>
                                            <strong>Tip:</strong> Select a subject, set default scores, check grades you want, then click "Apply" to use defaults or customize individual scores.
                                        </small>
                                    </div>
                                </form>
                            </div>
                            <div class="modal-footer">
                                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                                <button type="button" class="btn btn-success" onclick="app.saveMultiGradeRules()" id="multiGradeSaveButton" disabled>
                                    <i class="bi bi-collection"></i> Create Selected Rules
                                </button>
                            </div>
                        </div>
                    </div>
                </div>
            `;
            
            document.getElementById('pageContent').innerHTML = content;
            
            // Load dropdown data and table
            await this.loadGradeSubjectRuleDropdowns();
            await this.loadGradeSubjectRulesData();
            
        } catch (error) {
            document.getElementById('pageContent').innerHTML = `
                <div class="alert alert-danger">
                    <i class="bi bi-exclamation-triangle"></i> Error loading grade-subject rules: ${error.message}
                </div>`;
        }
    }
    
    async loadGradeSubjectRuleDropdowns() {
        try {
            // Load grades
            const gradesResponse = await this.apiCall('GET', '/admin/grades');
            const gradeOptions = gradesResponse.data.map(grade => `<option value="${grade.id}">${grade.title}</option>`).join('');
            
            // Load subjects
            const subjectsResponse = await this.apiCall('GET', '/admin/subjects');
            const subjectOptions = subjectsResponse.data.map(subject => `<option value="${subject.id}">${subject.name}</option>`).join('');
            
            // Populate dropdowns
            const gradeSelects = document.querySelectorAll('#ruleGradeFilter, #ruleGradeId');
            gradeSelects.forEach(select => {
                if (select.id === 'ruleGradeFilter') {
                    select.innerHTML = '<option value="">All Grades</option>' + gradeOptions;
                } else {
                    select.innerHTML = '<option value="">Select Grade</option>' + gradeOptions;
                }
            });
            
            const subjectSelects = document.querySelectorAll('#ruleSubjectFilter, #ruleSubjectId');
            subjectSelects.forEach(select => {
                if (select.id === 'ruleSubjectFilter') {
                    select.innerHTML = '<option value="">All Subjects</option>' + subjectOptions;
                } else {
                    select.innerHTML = '<option value="">Select Subject</option>' + subjectOptions;
                }
            });
            
        } catch (error) {
            this.showToast('error', 'Failed to load dropdown data');
        }
    }
    
    async loadGradeSubjectRulesData() {
        try {
            const response = await this.apiCall('GET', '/admin/grade-subject-rules');
            document.getElementById('gradeSubjectRulesTableBody').innerHTML = this.renderGradeSubjectRulesTable(response.data);
        } catch (error) {
            document.getElementById('gradeSubjectRulesTableBody').innerHTML = `
                <tr>
                    <td colspan="7" class="text-center text-danger">
                        <i class="bi bi-exclamation-triangle"></i> Failed to load rules
                    </td>
                </tr>
            `;
        }
    }
    
    renderGradeSubjectRulesTable(rules) {
        if (!rules || rules.length === 0) {
            return `
                <tr>
                    <td colspan="7" class="text-center text-muted">
                        <i class="bi bi-inbox"></i> No grade-subject rules found
                        <br>
                        <small class="text-muted">Add rules to define min/max scores for grade-subject combinations</small>
                    </td>
                </tr>
            `;
        }
        
        return rules.map(rule => `
            <tr>
                <td><strong>${rule.grade_title}</strong> <small class="text-muted">(${rule.grade_code})</small></td>
                <td><strong>${rule.subject_name}</strong></td>
                <td><span class="badge bg-primary">${rule.min_score}</span></td>
                <td><span class="badge bg-success">${rule.max_score}</span></td>
                <td>
                    <span class="badge ${rule.enabled ? 'bg-success' : 'bg-secondary'}">
                        ${rule.enabled ? 'Active' : 'Disabled'}
                    </span>
                </td>
                <td>${new Date(rule.created_at).toLocaleDateString()}</td>
                <td>
                    <div class="btn-group btn-group-sm">
                        <button class="btn btn-outline-primary" onclick="app.editGradeSubjectRule(${rule.id})" title="Edit">
                            <i class="bi bi-pencil"></i>
                        </button>
                        <button class="btn btn-outline-danger" onclick="app.deleteGradeSubjectRule(${rule.id}, '${(rule.grade_title + ' - ' + rule.subject_name).replace(/'/g, '&#39;')}')\" title="Delete">
                            <i class="bi bi-trash"></i>
                        </button>
                    </div>
                </td>
            </tr>
        `).join('');
    }
    
    // Grade-Subject Rules CRUD Functions
    showGradeSubjectRuleModal(ruleData = null) {
        const modalElement = document.getElementById('gradeSubjectRuleModal');
        if (!modalElement) {
            this.showToast('error', 'Modal not found. Please refresh the page.');
            return;
        }
        
        // Remove any existing backdrops
        document.querySelectorAll('.modal-backdrop').forEach(backdrop => backdrop.remove());
        
        // Reset form
        const form = document.getElementById('gradeSubjectRuleForm');
        if (form) form.reset();
        
        const ruleIdField = document.getElementById('gradeSubjectRuleId');
        if (ruleIdField) ruleIdField.value = '';
        
        if (ruleData) {
            // Edit mode
            document.getElementById('gradeSubjectRuleModalTitle').textContent = 'Edit Grade-Subject Rule';
            document.getElementById('gradeSubjectRuleSaveButtonText').textContent = 'Update Rule';
            if (ruleIdField) ruleIdField.value = ruleData.id;
            document.getElementById('ruleGradeId').value = ruleData.grade_id;
            document.getElementById('ruleSubjectId').value = ruleData.subject_id;
            document.getElementById('ruleMinScore').value = parseFloat(ruleData.min_score);
            document.getElementById('ruleMaxScore').value = parseFloat(ruleData.max_score);
            // Convert boolean to string value for select dropdown
            document.getElementById('ruleEnabled').value = ruleData.enabled ? "1" : "0";
        } else {
            // Add mode
            document.getElementById('gradeSubjectRuleModalTitle').textContent = 'Add Grade-Subject Rule';
            document.getElementById('gradeSubjectRuleSaveButtonText').textContent = 'Save Rule';
        }
        
        // Clear validation classes
        document.querySelectorAll('.is-invalid').forEach(el => el.classList.remove('is-invalid'));
        
        // Show modal
        modalElement.style.display = 'flex';
        modalElement.classList.add('show');
        document.body.classList.add('modal-open');
        
        // Add close button handlers
        const closeButtons = modalElement.querySelectorAll('[data-bs-dismiss="modal"]');
        closeButtons.forEach(btn => {
            btn.onclick = (e) => {
                e.preventDefault();
                this.closeGradeSubjectRuleModal();
            };
        });
        
        // Focus on first input
        setTimeout(() => {
            const firstInput = modalElement.querySelector('select');
            if (firstInput) firstInput.focus();
        }, 100);
    }
    
    closeGradeSubjectRuleModal() {
        const modalElement = document.getElementById('gradeSubjectRuleModal');
        if (modalElement) {
            modalElement.style.display = 'none';
            modalElement.classList.remove('show');
        }
        document.body.classList.remove('modal-open');
        document.querySelectorAll('.modal-backdrop').forEach(backdrop => backdrop.remove());
    }
    
    async saveGradeSubjectRule() {
        const ruleIdField = document.getElementById('gradeSubjectRuleId');
        const gradeField = document.getElementById('ruleGradeId');
        const subjectField = document.getElementById('ruleSubjectId');
        const minScoreField = document.getElementById('ruleMinScore');
        const maxScoreField = document.getElementById('ruleMaxScore');
        const enabledField = document.getElementById('ruleEnabled');
        
        if (!gradeField || !subjectField || !minScoreField || !maxScoreField || !enabledField) {
            this.showToast('error', 'Form fields not found. Please refresh the page.');
            return;
        }
        
        const ruleId = ruleIdField ? ruleIdField.value : '';
        const isEdit = !!ruleId;
        
        const data = {
            grade_id: parseInt(gradeField.value),
            subject_id: parseInt(subjectField.value),
            min_score: parseFloat(minScoreField.value),
            max_score: parseFloat(maxScoreField.value),
            enabled: parseInt(enabledField.value)
        };
        
        // Clear previous validation
        document.querySelectorAll('.is-invalid').forEach(el => el.classList.remove('is-invalid'));
        
        try {
            const endpoint = isEdit ? `/admin/grade-subject-rules/${ruleId}` : '/admin/grade-subject-rules';
            const method = isEdit ? 'PUT' : 'POST';
            
            const response = await this.apiCall(method, endpoint, data);
            
            if (response.success) {
                this.closeGradeSubjectRuleModal();
                this.showToast('success', `Grade-subject rule ${isEdit ? 'updated' : 'created'} successfully!`);
                this.loadGradeSubjectRulesData();
            } else {
                this.showToast('error', response.message);
            }
            
        } catch (error) {
            this.showToast('error', 'Failed to save rule. Please try again.');
        }
    }
    
    async editGradeSubjectRule(id) {
        try {
            const response = await this.apiCall('GET', `/admin/grade-subject-rules/${id}`);
            if (response.success) {
                this.showGradeSubjectRuleModal(response.data);
            }
        } catch (error) {
            this.showToast('error', 'Failed to load rule data');
        }
    }
    
    async deleteGradeSubjectRule(id, name) {
        if (!confirm(`Are you sure you want to delete the rule for \"${name}\"?\\n\\nThis action cannot be undone and may affect exam creation.`)) {
            return;
        }
        
        try {
            const response = await this.apiCall('DELETE', `/admin/grade-subject-rules/${id}`);
            
            if (response.success) {
                this.showToast('success', 'Grade-subject rule deleted successfully!');
                this.loadGradeSubjectRulesData();
            } else {
                this.showToast('error', response.message);
            }
        } catch (error) {
            this.showToast('error', 'Failed to delete rule. It may have related data.');
        }
    }
    
    async filterGradeSubjectRules() {
        const grade = document.getElementById('ruleGradeFilter').value;
        const subject = document.getElementById('ruleSubjectFilter').value;
        const search = document.getElementById('ruleSearchFilter').value;
        const status = document.getElementById('ruleStatusFilter').value;
        
        const params = new URLSearchParams();
        if (grade) params.append('grade_id', grade);
        if (subject) params.append('subject_id', subject);
        if (search) params.append('search', search);
        if (status !== '') params.append('enabled', status);
        
        try {
            const response = await this.apiCall('GET', `/admin/grade-subject-rules?${params.toString()}`);
            document.getElementById('gradeSubjectRulesTableBody').innerHTML = this.renderGradeSubjectRulesTable(response.data);
        } catch (error) {
            this.showToast('error', 'Failed to filter rules');
        }
    }
    
    clearGradeSubjectRulesFilters() {
        document.getElementById('ruleGradeFilter').value = '';
        document.getElementById('ruleSubjectFilter').value = '';
        document.getElementById('ruleSearchFilter').value = '';
        document.getElementById('ruleStatusFilter').value = '';
        this.loadGradeSubjectRulesData();
    }
    
    async sortGradeSubjectRules(field) {
        this.currentSort = this.currentSort || {};
        const currentDir = this.currentSort[field] || 'asc';
        const newDir = currentDir === 'asc' ? 'desc' : 'asc';
        this.currentSort[field] = newDir;
        
        try {
            const response = await this.apiCall('GET', `/admin/grade-subject-rules?sort_by=${field}&sort_dir=${newDir}`);
            document.getElementById('gradeSubjectRulesTableBody').innerHTML = this.renderGradeSubjectRulesTable(response.data);
        } catch (error) {
            this.showToast('error', 'Failed to sort rules');
        }
    }
    
    exportGradeSubjectRules(format) {
        this.showToast('info', `${format.toUpperCase()} export functionality coming soon!`);
    }
    
    printGradeSubjectRules() {
        this.showToast('info', 'Print functionality coming soon!');
    }
    
    // Multi-Grade Rule Functions
    showMultiGradeRuleModal() {
        const modalElement = document.getElementById('multiGradeRuleModal');
        if (!modalElement) {
            this.showToast('error', 'Multi-grade modal not found. Please refresh the page.');
            return;
        }
        
        // Remove any existing backdrops
        document.querySelectorAll('.modal-backdrop').forEach(backdrop => backdrop.remove());
        
        // Reset form
        const form = document.getElementById('multiGradeRuleForm');
        if (form) form.reset();
        
        // Set default values
        document.getElementById('defaultMinScore').value = 50;
        document.getElementById('defaultMaxScore').value = 100;
        
        // Reset grade selection container
        document.getElementById('gradeSelectionContainer').innerHTML = `
            <div class="text-muted text-center py-3">
                <i class="bi bi-arrow-up"></i> Please select a subject first
            </div>
        `;
        
        // Disable save button
        document.getElementById('multiGradeSaveButton').disabled = true;
        
        // Load subjects dropdown
        this.loadMultiGradeSubjects();
        
        // Clear validation classes
        document.querySelectorAll('.is-invalid').forEach(el => el.classList.remove('is-invalid'));
        
        // Show modal
        modalElement.style.display = 'flex';
        modalElement.classList.add('show');
        document.body.classList.add('modal-open');
        
        // Add close button handlers
        const closeButtons = modalElement.querySelectorAll('[data-bs-dismiss="modal"]');
        closeButtons.forEach(btn => {
            btn.onclick = (e) => {
                e.preventDefault();
                this.closeMultiGradeRuleModal();
            };
        });
        
        // Focus on subject select
        setTimeout(() => {
            const firstInput = modalElement.querySelector('select');
            if (firstInput) firstInput.focus();
        }, 100);
    }
    
    closeMultiGradeRuleModal() {
        const modalElement = document.getElementById('multiGradeRuleModal');
        if (modalElement) {
            modalElement.style.display = 'none';
            modalElement.classList.remove('show');
        }
        document.body.classList.remove('modal-open');
        document.querySelectorAll('.modal-backdrop').forEach(backdrop => backdrop.remove());
    }
    
    async loadMultiGradeSubjects() {
        try {
            const response = await this.apiCall('GET', '/admin/subjects');
            const subjectSelect = document.getElementById('multiRuleSubjectId');
            
            if (subjectSelect && response.data) {
                subjectSelect.innerHTML = '<option value="">Select Subject</option>';
                response.data.forEach(subject => {
                    if (subject.enabled) {
                        subjectSelect.innerHTML += `<option value="${subject.id}">${subject.name}</option>`;
                    }
                });
            }
        } catch (error) {
            this.showToast('error', 'Failed to load subjects');
        }
    }
    
    async loadAvailableGrades() {
        const subjectId = document.getElementById('multiRuleSubjectId').value;
        const container = document.getElementById('gradeSelectionContainer');
        const saveButton = document.getElementById('multiGradeSaveButton');
        
        if (!subjectId) {
            container.innerHTML = `
                <div class="text-muted text-center py-3">
                    <i class="bi bi-arrow-up"></i> Please select a subject first
                </div>
            `;
            saveButton.disabled = true;
            return;
        }
        
        try {
            // Get all grades
            const gradesResponse = await this.apiCall('GET', '/admin/grades');
            
            // Get existing rules for this subject
            const rulesResponse = await this.apiCall('GET', `/admin/grade-subject-rules?subject_id=${subjectId}`);
            
            const existingGradeIds = rulesResponse.data.map(rule => rule.grade_id);
            const availableGrades = gradesResponse.data.filter(grade => 
                grade.enabled && !existingGradeIds.includes(grade.id)
            );
            
            if (availableGrades.length === 0) {
                container.innerHTML = `
                    <div class="text-warning text-center py-3">
                        <i class="bi bi-exclamation-triangle"></i> All grades already have rules for this subject
                    </div>
                `;
                saveButton.disabled = true;
                return;
            }
            
            // Get default scores
            const defaultMin = document.getElementById('defaultMinScore').value || 50;
            const defaultMax = document.getElementById('defaultMaxScore').value || 100;
            
            let html = `
                <div class="d-flex justify-content-between align-items-center mb-3">
                    <div>
                        <button type="button" class="btn btn-sm btn-outline-primary me-2" onclick="app.selectAllGrades()">
                            <i class="bi bi-check-all"></i> All
                        </button>
                        <button type="button" class="btn btn-sm btn-outline-secondary" onclick="app.deselectAllGrades()">
                            <i class="bi bi-x-square"></i> None
                        </button>
                    </div>
                    <small class="text-muted">${availableGrades.length} grades available</small>
                </div>
                
                <div class="table-responsive" style="max-height: 400px; overflow-y: auto;">
                    <table class="table table-sm table-hover">
                        <thead class="table-light sticky-top">
                            <tr>
                                <th style="width: 50px;">
                                    <input type="checkbox" class="form-check-input" id="selectAllGrades" onchange="app.toggleAllGrades(this)">
                                </th>
                                <th style="width: 80px;">Code</th>
                                <th>Grade Name</th>
                                <th style="width: 120px;">Min Score</th>
                                <th style="width: 120px;">Max Score</th>
                            </tr>
                        </thead>
                        <tbody>
            `;
            
            availableGrades.forEach(grade => {
                html += `
                    <tr data-grade-id="${grade.id}">
                        <td>
                            <input class="form-check-input grade-checkbox" type="checkbox" id="grade_${grade.id}" value="${grade.id}" onchange="app.updateSaveButton()">
                        </td>
                        <td>
                            <span class="badge bg-primary">${grade.code}</span>
                        </td>
                        <td>
                            <label class="form-check-label" for="grade_${grade.id}">${grade.title}</label>
                        </td>
                        <td>
                            <input type="number" class="form-control form-control-sm" id="min_${grade.id}" value="${defaultMin}" min="0" max="999.99" step="0.01">
                        </td>
                        <td>
                            <input type="number" class="form-control form-control-sm" id="max_${grade.id}" value="${defaultMax}" min="0" max="999.99" step="0.01">
                        </td>
                    </tr>
                `;
            });
            
            html += `
                        </tbody>
                    </table>
                </div>
            `;
            
            container.innerHTML = html;
            this.updateSaveButton();
            
        } catch (error) {
            container.innerHTML = `
                <div class="text-danger text-center py-3">
                    <i class="bi bi-exclamation-triangle"></i> Failed to load grades
                </div>
            `;
            saveButton.disabled = true;
        }
    }
    
    toggleAllGrades(masterCheckbox) {
        document.querySelectorAll('.grade-checkbox').forEach(checkbox => {
            checkbox.checked = masterCheckbox.checked;
        });
        this.updateSaveButton();
    }
    
    selectAllGrades() {
        const masterCheckbox = document.getElementById('selectAllGrades');
        if (masterCheckbox) masterCheckbox.checked = true;
        document.querySelectorAll('.grade-checkbox').forEach(checkbox => {
            checkbox.checked = true;
        });
        this.updateSaveButton();
    }
    
    deselectAllGrades() {
        const masterCheckbox = document.getElementById('selectAllGrades');
        if (masterCheckbox) masterCheckbox.checked = false;
        document.querySelectorAll('.grade-checkbox').forEach(checkbox => {
            checkbox.checked = false;
        });
        this.updateSaveButton();
    }
    
    applyDefaultScores() {
        const defaultMin = document.getElementById('defaultMinScore').value || 50;
        const defaultMax = document.getElementById('defaultMaxScore').value || 100;
        
        document.querySelectorAll('tr[data-grade-id]').forEach(row => {
            const gradeId = row.dataset.gradeId;
            const minInput = document.getElementById(`min_${gradeId}`);
            const maxInput = document.getElementById(`max_${gradeId}`);
            
            if (minInput) minInput.value = defaultMin;
            if (maxInput) maxInput.value = defaultMax;
        });
        
        this.showToast('info', 'Default scores applied to all grades');
    }
    
    updateSaveButton() {
        const checkedBoxes = document.querySelectorAll('.grade-checkbox:checked');
        const saveButton = document.getElementById('multiGradeSaveButton');
        const masterCheckbox = document.getElementById('selectAllGrades');
        
        // Update master checkbox state
        const totalCheckboxes = document.querySelectorAll('.grade-checkbox');
        if (masterCheckbox && totalCheckboxes.length > 0) {
            masterCheckbox.checked = checkedBoxes.length === totalCheckboxes.length;
            masterCheckbox.indeterminate = checkedBoxes.length > 0 && checkedBoxes.length < totalCheckboxes.length;
        }
        
        saveButton.disabled = checkedBoxes.length === 0;
        
        if (checkedBoxes.length > 0) {
            const ruleText = checkedBoxes.length === 1 ? 'Rule' : 'Rules';
            saveButton.innerHTML = `<i class="bi bi-collection"></i> Create ${checkedBoxes.length} ${ruleText}`;
        } else {
            saveButton.innerHTML = `<i class="bi bi-collection"></i> Create Selected Rules`;
        }
    }
    
    async saveMultiGradeRules() {
        const subjectId = document.getElementById('multiRuleSubjectId').value;
        const checkedBoxes = document.querySelectorAll('.grade-checkbox:checked');
        
        if (!subjectId || checkedBoxes.length === 0) {
            this.showToast('error', 'Please select a subject and at least one grade');
            return;
        }
        
        const rules = [];
        let hasErrors = false;
        
        checkedBoxes.forEach(checkbox => {
            const gradeId = checkbox.value;
            const minScore = parseFloat(document.getElementById(`min_${gradeId}`).value);
            const maxScore = parseFloat(document.getElementById(`max_${gradeId}`).value);
            
            if (isNaN(minScore) || isNaN(maxScore) || minScore > maxScore) {
                hasErrors = true;
                return;
            }
            
            rules.push({
                grade_id: parseInt(gradeId),
                subject_id: parseInt(subjectId),
                min_score: minScore,
                max_score: maxScore,
                enabled: 1
            });
        });
        
        if (hasErrors) {
            this.showToast('error', 'Please check that all min/max scores are valid numbers and min ≤ max');
            return;
        }
        
        try {
            let successCount = 0;
            let errorCount = 0;
            
            // Save each rule individually
            for (const rule of rules) {
                try {
                    const response = await this.apiCall('POST', '/admin/grade-subject-rules', rule);
                    if (response.success) {
                        successCount++;
                    } else {
                        errorCount++;
                    }
                } catch (error) {
                    errorCount++;
                }
            }
            
            if (successCount > 0) {
                this.closeMultiGradeRuleModal();
                const ruleText = successCount === 1 ? 'rule' : 'rules';
                let message = `Successfully created ${successCount} ${ruleText}!`;
                
                if (errorCount > 0) {
                    message += ` (${errorCount} failed)`;
                }
                
                this.showToast('success', message);
                this.loadGradeSubjectRulesData();
            } else {
                this.showToast('error', 'Failed to create any rules. They may already exist.');
            }
            
        } catch (error) {
            this.showToast('error', 'Failed to save rules. Please try again.');
        }
    }
    
    async apiCall(method, endpoint, data = null, isJson = true) {
        const config = {
            method,
            headers: {}
        };
        
        if (this.token) {
            config.headers.Authorization = `Bearer ${this.token}`;
        }
        
        if (data) {
            if (data instanceof FormData) {
                // Don't set Content-Type for FormData, let browser set it with boundary
                config.body = data;
            } else if (isJson) {
                config.headers['Content-Type'] = 'application/json';
                config.body = JSON.stringify(data);
            } else {
                config.body = data;
            }
        } else if (isJson) {
            config.headers['Content-Type'] = 'application/json';
        }
        
        const fullUrl = this.apiBase + endpoint;
        console.log('API Call:', method, fullUrl, 'Token:', this.token ? 'Present' : 'Missing');
        
        const response = await fetch(fullUrl, config);
        console.log('Response Status:', response.status, response.statusText);
        
        // Get response text first to debug
        const responseText = await response.text();
        console.log('Response Text:', responseText.substring(0, 200));
        
        if (!response.ok) {
            // Handle authentication errors
            if (response.status === 401) {
                this.logout();
                throw new Error('Authentication required');
            }
            
            // Try to parse JSON from response text
            try {
                const errorData = JSON.parse(responseText);
                throw new Error(errorData.message || `HTTP ${response.status}`);
            } catch (jsonError) {
                // If response is not JSON, throw with response content
                throw new Error(`HTTP ${response.status}: ${responseText.substring(0, 100)}`);
            }
        }
        
        // Parse successful response
        try {
            return JSON.parse(responseText);
        } catch (jsonError) {
            console.error('JSON Parse Error:', jsonError);
            console.error('Response Text:', responseText);
            throw new Error('Invalid JSON response: ' + responseText.substring(0, 100));
        }
    }
    
    showError(element, message) {
        element.textContent = message;
        element.style.display = 'block';
        setTimeout(() => {
            element.style.display = 'none';
        }, 5000);
    }
    
    // Enrollment Management Functions
    async loadEnrollmentsData(page = 1, perPage = 25, filters = {}) {
        try {
            // Build query parameters
            let queryParams = new URLSearchParams({
                page: page,
                per_page: perPage
            });
            
            // Add filters
            if (filters.student_name) queryParams.append('student_name', filters.student_name);
            if (filters.year_id) queryParams.append('year_id', filters.year_id);
            if (filters.grade_id) queryParams.append('grade_id', filters.grade_id);
            if (filters.class_id) queryParams.append('class_id', filters.class_id);
            if (filters.status) queryParams.append('status', filters.status);
            
            const response = await this.apiCall('GET', `/admin/enrollments?${queryParams.toString()}`);
            document.getElementById('enrollmentsTableBody').innerHTML = this.renderEnrollmentsTable(response.data);
            this.updateEnrollmentsPagination(response.meta);
        } catch (error) {
            document.getElementById('enrollmentsTableBody').innerHTML = `
                <tr>
                    <td colspan="9" class="text-center py-4 text-danger">
                        <i class="bi bi-exclamation-triangle"></i><br>
                        Failed to load enrollments: ${error.message}
                    </td>
                </tr>
            `;
        }
    }
    
    renderEnrollmentsTable(enrollments) {
        if (!enrollments || enrollments.length === 0) {
            return `
                <tr>
                    <td colspan="9" class="text-center py-4 text-muted">
                        <i class="bi bi-person-plus"></i><br>
                        No enrollments found
                    </td>
                </tr>
            `;
        }
        
        return enrollments.map(enrollment => `
            <tr>
                <td>
                    <input type="checkbox" class="form-check-input enrollment-checkbox" value="${enrollment.id}" onchange="app.updateSelectedTransferCount()">
                </td>
                <td>${enrollment.id}</td>
                <td>
                    <div class="fw-bold">${enrollment.student_name}</div>
                    <small class="text-muted">${enrollment.login_code}</small>
                </td>
                <td>
                    <span class="badge bg-info">${enrollment.year_name}</span>
                </td>
                <td>
                    <span class="badge bg-secondary">${enrollment.grade_title}</span>
                </td>
                <td>
                    <span class="badge bg-warning text-dark">${enrollment.class_name}</span>
                </td>
                <td>
                    <span class="badge ${
                        enrollment.status === 'active' ? 'bg-success' :
                        enrollment.status === 'inactive' ? 'bg-secondary' :
                        enrollment.status === 'graduated' ? 'bg-primary' :
                        enrollment.status === 'transferred' ? 'bg-warning' : 'bg-secondary'
                    }">
                        ${enrollment.status.charAt(0).toUpperCase() + enrollment.status.slice(1)}
                    </span>
                </td>
                <td>
                    <small class="text-muted">${new Date(enrollment.created_at).toLocaleDateString()}</small>
                </td>
                <td>
                    <div class="btn-group" role="group">
                        <button class="btn btn-outline-primary btn-sm" onclick="app.editEnrollment(${enrollment.id})" title="Edit">
                            <i class="bi bi-pencil"></i>
                        </button>
                        <button class="btn btn-outline-danger btn-sm" onclick="app.deleteEnrollment(${enrollment.id}, '${enrollment.student_name.replace(/'/g, '&#39;')}')" title="Delete">
                            <i class="bi bi-trash"></i>
                        </button>
                    </div>
                </td>
            </tr>
        `).join('');
    }
    
    updateEnrollmentsPagination(meta) {
        const info = document.getElementById('enrollmentsInfo');
        const nav = document.getElementById('enrollmentsPaginationNav');
        
        if (meta) {
            const start = ((meta.page - 1) * meta.per_page) + 1;
            const end = Math.min(meta.page * meta.per_page, meta.total_rows);
            info.textContent = `Showing ${start}-${end} of ${meta.total_rows} enrollments`;
            
            // Generate pagination
            let paginationHtml = '';
            if (meta.total_pages > 1) {
                // Previous button
                paginationHtml += `
                    <li class="page-item ${meta.page === 1 ? 'disabled' : ''}">
                        <a class="page-link" href="#" onclick="app.loadEnrollmentsPage(${meta.page - 1})">Previous</a>
                    </li>
                `;
                
                // Page numbers
                for (let i = 1; i <= meta.total_pages; i++) {
                    if (i === meta.page || Math.abs(i - meta.page) <= 2 || i === 1 || i === meta.total_pages) {
                        paginationHtml += `
                            <li class="page-item ${i === meta.page ? 'active' : ''}">
                                <a class="page-link" href="#" onclick="app.loadEnrollmentsPage(${i})">${i}</a>
                            </li>
                        `;
                    } else if (Math.abs(i - meta.page) === 3) {
                        paginationHtml += '<li class="page-item disabled"><span class="page-link">...</span></li>';
                    }
                }
                
                // Next button
                paginationHtml += `
                    <li class="page-item ${meta.page === meta.total_pages ? 'disabled' : ''}">
                        <a class="page-link" href="#" onclick="app.loadEnrollmentsPage(${meta.page + 1})">Next</a>
                    </li>
                `;
            }
            nav.innerHTML = paginationHtml;
        }
    }
    
    async loadEnrollmentsPage(page) {
        const perPage = document.getElementById('enrollmentsPerPage')?.value || 25;
        const filters = this.getEnrollmentFilters();
        await this.loadEnrollmentsData(page, perPage, filters);
    }
    
    getEnrollmentFilters() {
        return {
            student_name: document.getElementById('enrollmentStudentFilter')?.value || '',
            year_id: document.getElementById('enrollmentYearFilter')?.value || '',
            grade_id: document.getElementById('enrollmentGradeFilter')?.value || '',
            class_id: document.getElementById('enrollmentClassFilter')?.value || '',
            status: document.getElementById('enrollmentStatusFilter')?.value || ''
        };
    }
    
    async filterEnrollments() {
        await this.loadEnrollmentsPage(1);
    }
    
    clearEnrollmentFilters() {
        document.getElementById('enrollmentStudentFilter').value = '';
        document.getElementById('enrollmentYearFilter').value = '';
        document.getElementById('enrollmentGradeFilter').value = '';
        document.getElementById('enrollmentClassFilter').value = '';
        document.getElementById('enrollmentStatusFilter').value = '';
        this.loadEnrollmentsPage(1);
    }
    
    changeEnrollmentsPerPage() {
        this.loadEnrollmentsPage(1);
    }
    
    toggleSelectAllEnrollments() {
        const selectAllCheckbox = document.getElementById('selectAllEnrollments');
        const enrollmentCheckboxes = document.querySelectorAll('.enrollment-checkbox');
        enrollmentCheckboxes.forEach(checkbox => {
            checkbox.checked = selectAllCheckbox.checked;
        });
        this.updateSelectedTransferCount();
    }
    
    updateSelectedTransferCount() {
        const selectedCheckboxes = document.querySelectorAll('.enrollment-checkbox:checked');
        const count = selectedCheckboxes.length;
        document.getElementById('selectedTransferCount').textContent = count;
        const bulkTransferBtn = document.getElementById('bulkTransferBtn');
        if (count > 0) {
            bulkTransferBtn.style.display = 'inline-block';
        } else {
            bulkTransferBtn.style.display = 'none';
        }
        document.getElementById('selectAllEnrollments').checked = count > 0;
    }
    
    async showEnrollmentModal(enrollment = null) {
        const isEdit = !!enrollment;
        const modalTitle = isEdit ? 'Edit Enrollment' : 'Add New Enrollment';
        
        // Load required data
        await this.loadStudentOptions();
        await this.loadAcademicYearOptions();
        await this.loadGradeOptions();
        await this.loadClassOptions();
        
        const modal = document.createElement('div');
        modal.className = 'modal fade';
        modal.id = 'enrollmentModal';
        modal.innerHTML = `
            <div class="modal-dialog">
                <div class="modal-content">
                    <div class="modal-header">
                        <h5 class="modal-title">${modalTitle}</h5>
                        <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                    </div>
                    <div class="modal-body">
                        <form id="enrollmentForm">
                            <input type="hidden" id="enrollmentId" value="${enrollment ? enrollment.id : ''}">
                            
                            <div class="mb-3">
                                <label for="enrollmentStudentId" class="form-label">Student *</label>
                                <select class="form-select" id="enrollmentStudentId" required ${isEdit ? 'disabled' : ''}>
                                    <option value="">Select Student</option>
                                    ${this.renderStudentOptions(enrollment ? enrollment.student_id : null)}
                                </select>
                            </div>
                            
                            <div class="mb-3">
                                <label for="enrollmentYearId" class="form-label">Academic Year *</label>
                                <select class="form-select" id="enrollmentYearId" required>
                                    <option value="">Select Academic Year</option>
                                    ${this.renderYearOptions(enrollment ? enrollment.year_id : null)}
                                </select>
                            </div>
                            
                            <div class="mb-3">
                                <label for="enrollmentGradeId" class="form-label">Grade *</label>
                                <select class="form-select" id="enrollmentGradeId" required>
                                    <option value="">Select Grade</option>
                                    ${this.renderGradeOptions(enrollment ? enrollment.grade_id : null)}
                                </select>
                            </div>
                            
                            <div class="mb-3">
                                <label for="enrollmentClassId" class="form-label">Class *</label>
                                <select class="form-select" id="enrollmentClassId" required>
                                    <option value="">Select Class</option>
                                    ${this.renderClassOptions(enrollment ? enrollment.class_id : null)}
                                </select>
                            </div>
                            
                            <div class="mb-3">
                                <label for="enrollmentStatus" class="form-label">Status *</label>
                                <select class="form-select" id="enrollmentStatus" required>
                                    <option value="active" ${enrollment && enrollment.status === 'active' ? 'selected' : ''}>Active</option>
                                    <option value="inactive" ${enrollment && enrollment.status === 'inactive' ? 'selected' : ''}>Inactive</option>
                                    <option value="graduated" ${enrollment && enrollment.status === 'graduated' ? 'selected' : ''}>Graduated</option>
                                    <option value="transferred" ${enrollment && enrollment.status === 'transferred' ? 'selected' : ''}>Transferred</option>
                                </select>
                            </div>
                        </form>
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                        <button type="button" class="btn btn-primary" onclick="app.saveEnrollment()">
                            ${isEdit ? 'Update' : 'Create'} Enrollment
                        </button>
                    </div>
                </div>
            </div>
        `;
        
        document.body.appendChild(modal);
        const bsModal = new bootstrap.Modal(modal);
        bsModal.show();
        
        modal.addEventListener('hidden.bs.modal', () => {
            document.body.removeChild(modal);
        });
    }
    
    async showBulkEnrollModal() {
        // Load required data
        await this.loadEnrolledStudentsForBulk();
        await this.loadAcademicYearOptions();
        await this.loadGradeOptions();
        await this.loadClassOptions();
        
        const modal = document.createElement('div');
        modal.className = 'modal fade';
        modal.id = 'bulkEnrollModal';
        modal.innerHTML = `
            <div class="modal-dialog modal-lg">
                <div class="modal-content">
                    <div class="modal-header">
                        <h5 class="modal-title">Bulk Enroll Students</h5>
                        <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                    </div>
                    <div class="modal-body">
                        <form id="bulkEnrollForm">
                            <div class="row mb-3">
                                <div class="col-md-6">
                                    <label for="bulkEnrollYearId" class="form-label">Academic Year *</label>
                                    <select class="form-select" id="bulkEnrollYearId" required>
                                        <option value="">Select Academic Year</option>
                                        ${this.renderYearOptions()}
                                    </select>
                                </div>
                                <div class="col-md-6">
                                    <label for="bulkEnrollGradeId" class="form-label">Grade *</label>
                                    <select class="form-select" id="bulkEnrollGradeId" required>
                                        <option value="">Select Grade</option>
                                        ${this.renderGradeOptions()}
                                    </select>
                                </div>
                            </div>
                            
                            <div class="row mb-3">
                                <div class="col-md-6">
                                    <label for="bulkEnrollClassId" class="form-label">Class *</label>
                                    <select class="form-select" id="bulkEnrollClassId" required>
                                        <option value="">Select Class</option>
                                        ${this.renderClassOptions()}
                                    </select>
                                </div>
                                <div class="col-md-6">
                                    <label for="bulkEnrollStatus" class="form-label">Status *</label>
                                    <select class="form-select" id="bulkEnrollStatus" required>
                                        <option value="active">Active</option>
                                        <option value="inactive">Inactive</option>
                                        <option value="graduated">Graduated</option>
                                        <option value="transferred">Transferred</option>
                                    </select>
                                </div>
                            </div>
                            
                            <div class="mb-3">
                                <label class="form-label">Filter Students by Current Enrollment</label>
                                <div class="row mb-3">
                                    <div class="col-md-4">
                                        <label for="filterCurrentYear" class="form-label">Current Year</label>
                                        <select class="form-select" id="filterCurrentYear" onchange="app.filterBulkStudents()">
                                            <option value="">All Years</option>
                                            ${this.renderYearOptions()}
                                        </select>
                                    </div>
                                    <div class="col-md-4">
                                        <label for="filterCurrentGrade" class="form-label">Current Grade</label>
                                        <select class="form-select" id="filterCurrentGrade" onchange="app.filterBulkStudents()">
                                            <option value="">All Grades</option>
                                            ${this.renderGradeOptions()}
                                        </select>
                                    </div>
                                    <div class="col-md-4">
                                        <label for="filterCurrentClass" class="form-label">Current Class</label>
                                        <select class="form-select" id="filterCurrentClass" onchange="app.filterBulkStudents()">
                                            <option value="">All Classes</option>
                                            ${this.renderClassOptions()}
                                        </select>
                                    </div>
                                </div>
                            </div>
                            
                            <div class="mb-3">
                                <label class="form-label">Select Students *</label>
                                <div class="border rounded p-3" style="max-height: 300px; overflow-y: auto;">
                                    <div class="d-flex justify-content-between align-items-center mb-2">
                                        <div class="form-check">
                                            <input class="form-check-input" type="checkbox" id="selectAllBulkStudents" onchange="app.toggleAllBulkStudents()">
                                            <label class="form-check-label fw-bold" for="selectAllBulkStudents">
                                                Select All Visible Students
                                            </label>
                                        </div>
                                        <small class="text-muted" id="studentCount">Loading...</small>
                                    </div>
                                    <hr>
                                    <div id="bulkStudentsList">
                                        ${this.renderBulkStudentsList()}
                                    </div>
                                </div>
                            </div>
                        </form>
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                        <button type="button" class="btn btn-primary" onclick="app.saveBulkEnrollment()">
                            Enroll Selected Students
                        </button>
                    </div>
                </div>
            </div>
        `;
        
        document.body.appendChild(modal);
        const bsModal = new bootstrap.Modal(modal);
        bsModal.show();
        
        // Set initial student count
        setTimeout(() => {
            const studentCountDiv = document.getElementById('studentCount');
            const total = this.enrolledStudentsForBulk ? this.enrolledStudentsForBulk.length : 0;
            if (studentCountDiv) {
                studentCountDiv.textContent = `Showing ${total} of ${total} students`;
            }
        }, 100);
        
        modal.addEventListener('hidden.bs.modal', () => {
            document.body.removeChild(modal);
        });
    }
    
    async showBulkTransferModal() {
        // Load required data
        await this.loadAcademicYearOptions();
        await this.loadGradeOptions();
        await this.loadClassOptions();
        
        // Get selected enrollments
        const selectedCheckboxes = document.querySelectorAll('.enrollment-checkbox:checked');
        const selectedEnrollments = Array.from(selectedCheckboxes).map(cb => ({
            id: parseInt(cb.value),
            studentName: cb.getAttribute('data-student-name')
        }));
        
        if (selectedEnrollments.length === 0) {
            this.showToast('error', 'Please select enrollments to transfer.');
            return;
        }
        
        const modal = document.createElement('div');
        modal.className = 'modal fade';
        modal.id = 'bulkTransferModal';
        modal.innerHTML = `
            <div class="modal-dialog modal-lg">
                <div class="modal-content">
                    <div class="modal-header">
                        <h5 class="modal-title">Transfer Selected Enrollments</h5>
                        <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                    </div>
                    <div class="modal-body">
                        <div class="alert alert-info">
                            <i class="bi bi-info-circle"></i>
                            You are about to transfer ${selectedEnrollments.length} enrollment(s) to new academic year/grade/class.
                        </div>
                        
                        <form id="bulkTransferForm">
                            <div class="row mb-3">
                                <div class="col-md-4">
                                    <label for="transferYearId" class="form-label">Academic Year *</label>
                                    <select class="form-select" id="transferYearId" required>
                                        <option value="">Select Academic Year</option>
                                        ${this.renderYearOptions()}
                                    </select>
                                </div>
                                <div class="col-md-4">
                                    <label for="transferGradeId" class="form-label">Grade *</label>
                                    <select class="form-select" id="transferGradeId" required>
                                        <option value="">Select Grade</option>
                                        ${this.renderGradeOptions()}
                                    </select>
                                </div>
                                <div class="col-md-4">
                                    <label for="transferClassId" class="form-label">Class *</label>
                                    <select class="form-select" id="transferClassId" required>
                                        <option value="">Select Class</option>
                                        ${this.renderClassOptions()}
                                    </select>
                                </div>
                            </div>
                            
                            <div class="mb-3">
                                <label class="form-label">Selected Students:</label>
                                <div class="border rounded p-3" style="max-height: 200px; overflow-y: auto;">
                                    ${selectedEnrollments.map(enrollment => 
                                        `<div class="d-flex justify-content-between align-items-center py-1">
                                            <span>${enrollment.studentName}</span>
                                            <small class="text-muted">ID: ${enrollment.id}</small>
                                        </div>`
                                    ).join('')}
                                </div>
                            </div>
                        </form>
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                        <button type="button" class="btn btn-warning" onclick="app.saveBulkTransfer()">
                            Transfer Enrollments
                        </button>
                    </div>
                </div>
            </div>
        `;
        
        // Store selected enrollment IDs
        modal.setAttribute('data-enrollment-ids', JSON.stringify(selectedEnrollments.map(e => e.id)));
        
        document.body.appendChild(modal);
        const bsModal = new bootstrap.Modal(modal);
        bsModal.show();
        
        modal.addEventListener('hidden.bs.modal', () => {
            document.body.removeChild(modal);
        });
    }
    
    async editEnrollment(id) {
        try {
            const response = await this.apiCall('GET', `/admin/enrollments/${id}`);
            if (response.success) {
                this.showEnrollmentModal(response.data);
            } else {
                this.showToast('error', 'Failed to load enrollment data');
            }
        } catch (error) {
            this.showToast('error', 'Failed to load enrollment data');
        }
    }
    
    async deleteEnrollment(id, studentName) {
        if (!confirm(`Are you sure you want to delete the enrollment for "${studentName}"?\n\nThis action cannot be undone.`)) {
            return;
        }
        
        try {
            const response = await this.apiCall('DELETE', `/admin/enrollments/${id}`);
            
            if (response.success) {
                this.showToast('success', 'Enrollment deleted successfully!');
                // Refresh current page
                const currentPage = this.getCurrentEnrollmentPage() || 1;
                this.loadEnrollmentsPage(currentPage);
            } else {
                this.showToast('error', response.message);
            }
        } catch (error) {
            this.showToast('error', 'Failed to delete enrollment.');
        }
    }
    
    getCurrentEnrollmentPage() {
        const activePageLink = document.querySelector('#enrollmentsPaginationNav .page-item.active .page-link');
        return activePageLink ? parseInt(activePageLink.textContent) : 1;
    }
    
    async loadEnrollmentFilterOptions() {
        try {
            // Load academic years
            const yearResponse = await this.apiCall('GET', '/admin/academic-years');
            const yearSelect = document.getElementById('enrollmentYearFilter');
            if (yearSelect && yearResponse.data) {
                yearSelect.innerHTML = '<option value="">All Years</option>';
                yearResponse.data.forEach(year => {
                    yearSelect.innerHTML += `<option value="${year.id}">${year.name}</option>`;
                });
            }
            
            // Load grades
            const gradeResponse = await this.apiCall('GET', '/admin/grades');
            const gradeSelect = document.getElementById('enrollmentGradeFilter');
            if (gradeSelect && gradeResponse.data) {
                gradeSelect.innerHTML = '<option value="">All Grades</option>';
                gradeResponse.data.forEach(grade => {
                    gradeSelect.innerHTML += `<option value="${grade.id}">${grade.title}</option>`;
                });
            }
            
            // Load classes
            const classResponse = await this.apiCall('GET', '/admin/classes');
            const classSelect = document.getElementById('enrollmentClassFilter');
            if (classSelect && classResponse.data) {
                classSelect.innerHTML = '<option value="">All Classes</option>';
                classResponse.data.forEach(cls => {
                    classSelect.innerHTML += `<option value="${cls.id}">${cls.name}</option>`;
                });
            }
            
        } catch (error) {
            console.warn('Failed to load enrollment filter options:', error);
        }
    }
    
    async loadStudentOptions() {
        if (!this.studentOptions) {
            try {
                const response = await this.apiCall('GET', '/admin/students?per_page=1000');
                this.studentOptions = response.data || [];
            } catch (error) {
                this.studentOptions = [];
            }
        }
    }
    
    async loadAcademicYearOptions() {
        if (!this.academicYearOptions) {
            try {
                const response = await this.apiCall('GET', '/admin/academic-years');
                this.academicYearOptions = response.data || [];
            } catch (error) {
                this.academicYearOptions = [];
            }
        }
    }
    
    async loadGradeOptions() {
        if (!this.gradeOptions) {
            try {
                const response = await this.apiCall('GET', '/admin/grades');
                this.gradeOptions = response.data || [];
            } catch (error) {
                this.gradeOptions = [];
            }
        }
    }
    
    async loadClassOptions() {
        if (!this.classOptions) {
            try {
                const response = await this.apiCall('GET', '/admin/classes');
                this.classOptions = response.data || [];
            } catch (error) {
                this.classOptions = [];
            }
        }
    }
    
    renderStudentOptions(selectedId = null) {
        return this.studentOptions.map(student => 
            `<option value="${student.id}" ${student.id == selectedId ? 'selected' : ''}>
                ${student.full_name} (${student.login_code})
            </option>`
        ).join('');
    }
    
    renderYearOptions(selectedId = null) {
        return this.academicYearOptions.map(year => 
            `<option value="${year.id}" ${year.id == selectedId ? 'selected' : ''}>
                ${year.name}
            </option>`
        ).join('');
    }
    
    renderGradeOptions(selectedId = null) {
        return this.gradeOptions.map(grade => 
            `<option value="${grade.id}" ${grade.id == selectedId ? 'selected' : ''}>
                ${grade.title}
            </option>`
        ).join('');
    }
    
    renderClassOptions(selectedId = null) {
        return this.classOptions.map(cls => 
            `<option value="${cls.id}" ${cls.id == selectedId ? 'selected' : ''}>
                ${cls.name}
            </option>`
        ).join('');
    }
    
    async saveEnrollment() {
        const enrollmentIdField = document.getElementById('enrollmentId');
        const studentIdField = document.getElementById('enrollmentStudentId');
        const yearIdField = document.getElementById('enrollmentYearId');
        const gradeIdField = document.getElementById('enrollmentGradeId');
        const classIdField = document.getElementById('enrollmentClassId');
        const statusField = document.getElementById('enrollmentStatus');
        
        if (!studentIdField || !yearIdField || !gradeIdField || !classIdField || !statusField) {
            this.showToast('error', 'Form fields not found. Please refresh the page.');
            return;
        }
        
        const enrollmentId = enrollmentIdField ? enrollmentIdField.value : '';
        const isEdit = !!enrollmentId;
        
        const data = {
            student_id: parseInt(studentIdField.value),
            year_id: parseInt(yearIdField.value),
            grade_id: parseInt(gradeIdField.value),
            class_id: parseInt(classIdField.value),
            status: statusField.value
        };
        
        // Clear previous validation
        document.querySelectorAll('.is-invalid').forEach(el => el.classList.remove('is-invalid'));
        
        try {
            const endpoint = isEdit ? `/admin/enrollments/${enrollmentId}` : '/admin/enrollments';
            const method = isEdit ? 'PUT' : 'POST';
            
            const response = await this.apiCall(method, endpoint, data);
            
            if (response.success) {
                this.closeEnrollmentModal();
                this.showToast('success', `Enrollment ${isEdit ? 'updated' : 'created'} successfully!`);
                this.loadEnrollmentsPage(1);
            } else {
                this.showToast('error', response.message);
            }
            
        } catch (error) {
            this.showToast('error', 'Failed to save enrollment. Please try again.');
        }
    }
    
    closeEnrollmentModal() {
        const modal = document.getElementById('enrollmentModal');
        if (modal) {
            const bsModal = bootstrap.Modal.getInstance(modal);
            if (bsModal) {
                bsModal.hide();
            }
        }
    }
    
    renderBulkStudentsList() {
        if (!this.enrolledStudentsForBulk) {
            return '<div class="text-muted">Loading students...</div>';
        }

        const filteredStudents = this.getFilteredBulkStudents();
        
        if (filteredStudents.length === 0) {
            return '<div class="text-muted">No students found with the selected filters.</div>';
        }

        return filteredStudents.map(student => 
            `<div class="form-check">
                <input class="form-check-input student-checkbox" type="checkbox" value="${student.student_id}" id="student_${student.student_id}">
                <label class="form-check-label" for="student_${student.student_id}">
                    <strong>${student.student_name}</strong> (${student.login_code || student.student_id})
                    <br><small class="text-muted">
                        Current: ${student.year_name} - ${student.grade_title}${student.class_name ? ' - ' + student.class_name : ''}
                    </small>
                </label>
            </div>`
        ).join('');
    }
    
    toggleAllBulkStudents() {
        const selectAll = document.getElementById('selectAllBulkStudents');
        const checkboxes = document.querySelectorAll('.student-checkbox');
        
        checkboxes.forEach(checkbox => {
            checkbox.checked = selectAll.checked;
        });
    }
    
    async saveBulkEnrollment() {
        const yearIdField = document.getElementById('bulkEnrollYearId');
        const gradeIdField = document.getElementById('bulkEnrollGradeId');
        const classIdField = document.getElementById('bulkEnrollClassId');
        const statusField = document.getElementById('bulkEnrollStatus');
        
        if (!yearIdField || !gradeIdField || !classIdField || !statusField) {
            this.showToast('error', 'Form fields not found. Please refresh the page.');
            return;
        }
        
        // Get selected students
        const selectedStudents = Array.from(document.querySelectorAll('.student-checkbox:checked'))
            .map(checkbox => parseInt(checkbox.value));
            
        if (selectedStudents.length === 0) {
            this.showToast('error', 'Please select at least one student.');
            return;
        }
        
        const data = {
            student_ids: selectedStudents,
            year_id: parseInt(yearIdField.value),
            grade_id: parseInt(gradeIdField.value),
            class_id: parseInt(classIdField.value),
            status: statusField.value
        };
        
        try {
            const response = await this.apiCall('POST', '/admin/enrollments/bulk-enroll', data);
            
            if (response.success) {
                this.closeBulkEnrollModal();
                const results = response.data;
                this.showToast('success', 
                    `Bulk enrollment completed: ${results.successful} successful, ${results.failed} failed`
                );
                this.loadEnrollmentsPage(1);
            } else {
                this.showToast('error', response.message);
            }
            
        } catch (error) {
            this.showToast('error', 'Failed to enroll students. Please try again.');
        }
    }
    
    closeBulkEnrollModal() {
        const modal = document.getElementById('bulkEnrollModal');
        if (modal) {
            const bsModal = bootstrap.Modal.getInstance(modal);
            if (bsModal) {
                bsModal.hide();
            }
        }
    }
    
    async saveBulkTransfer() {
        const modal = document.getElementById('bulkTransferModal');
        const yearIdField = document.getElementById('transferYearId');
        const gradeIdField = document.getElementById('transferGradeId');
        const classIdField = document.getElementById('transferClassId');
        
        if (!yearIdField || !gradeIdField || !classIdField) {
            this.showToast('error', 'Form fields not found. Please refresh the page.');
            return;
        }
        
        const enrollmentIds = JSON.parse(modal.getAttribute('data-enrollment-ids'));
        
        const data = {
            enrollment_ids: enrollmentIds,
            year_id: parseInt(yearIdField.value),
            grade_id: parseInt(gradeIdField.value),
            class_id: parseInt(classIdField.value)
        };
        
        try {
            const response = await this.apiCall('POST', '/admin/enrollments/transfer', data);
            
            if (response.success) {
                this.closeBulkTransferModal();
                const results = response.data;
                this.showToast('success', 
                    `Transfer completed: ${results.successful} successful, ${results.failed} failed`
                );
                this.loadEnrollmentsPage(1);
            } else {
                this.showToast('error', response.message);
            }
            
        } catch (error) {
            this.showToast('error', 'Failed to transfer enrollments. Please try again.');
        }
    }
    
    closeBulkTransferModal() {
        const modal = document.getElementById('bulkTransferModal');
        if (modal) {
            const bsModal = bootstrap.Modal.getInstance(modal);
            if (bsModal) {
                bsModal.hide();
            }
        }
    }
    
    showImportEnrollmentsModal() {
        const modal = document.createElement('div');
        modal.className = 'modal fade';
        modal.id = 'importEnrollmentsModal';
        modal.innerHTML = `
            <div class="modal-dialog">
                <div class="modal-content">
                    <div class="modal-header">
                        <h5 class="modal-title">Import Enrollments</h5>
                        <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                    </div>
                    <div class="modal-body">
                        <div class="alert alert-info">
                            <h6><i class="bi bi-info-circle"></i> CSV Format Requirements:</h6>
                            <p class="mb-1">Required columns: <code>student_id</code>, <code>year_id</code>, <code>grade_id</code>, <code>class_id</code></p>
                            <p class="mb-1">Optional columns: <code>status</code> (default: active)</p>
                            <p class="mb-0">Valid status values: active, inactive, graduated, transferred</p>
                        </div>
                        
                        <form id="importEnrollmentsForm" enctype="multipart/form-data">
                            <div class="mb-3">
                                <label for="enrollmentImportFile" class="form-label">Select CSV File *</label>
                                <input type="file" class="form-control" id="enrollmentImportFile" accept=".csv" required>
                                <div class="form-text">Maximum file size: 5MB</div>
                            </div>
                            
                            <div class="mb-3">
                                <div class="form-check">
                                    <input class="form-check-input" type="checkbox" id="skipDuplicates" checked>
                                    <label class="form-check-label" for="skipDuplicates">
                                        Skip duplicate enrollments (same student + year)
                                    </label>
                                </div>
                            </div>
                        </form>
                        
                        <div class="mt-3">
                            <h6>Sample CSV Format:</h6>
                            <pre class="bg-light p-2 small">student_id,year_id,grade_id,class_id,status
1,1,2,3,active
2,1,2,3,active
3,1,3,4,inactive</pre>
                        </div>
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                        <button type="button" class="btn btn-primary" onclick="app.importEnrollments()">
                            <i class="bi bi-upload"></i> Import Enrollments
                        </button>
                    </div>
                </div>
            </div>
        `;
        
        document.body.appendChild(modal);
        const bsModal = new bootstrap.Modal(modal);
        bsModal.show();
        
        modal.addEventListener('hidden.bs.modal', () => {
            document.body.removeChild(modal);
        });
    }
    
    async importEnrollments() {
        const fileInput = document.getElementById('enrollmentImportFile');
        
        if (!fileInput || !fileInput.files[0]) {
            this.showToast('error', 'Please select a CSV file.');
            return;
        }
        
        const file = fileInput.files[0];
        const formData = new FormData();
        formData.append('file', file);
        
        try {
            const response = await fetch('/api/admin/enrollments/import', {
                method: 'POST',
                headers: {
                    'Authorization': `Bearer ${localStorage.getItem('auth_token')}`
                },
                body: formData
            });
            
            const result = await response.json();
            
            if (result.success) {
                this.closeImportEnrollmentsModal();
                const data = result.data;
                this.showToast('success', 
                    `Import completed: ${data.successful} successful, ${data.failed} failed from ${data.total_rows} total rows`
                );
                this.loadEnrollmentsPage(1);
            } else {
                this.showToast('error', result.message);
            }
            
        } catch (error) {
            this.showToast('error', 'Failed to import enrollments. Please try again.');
        }
    }
    
    closeImportEnrollmentsModal() {
        const modal = document.getElementById('importEnrollmentsModal');
        if (modal) {
            const bsModal = bootstrap.Modal.getInstance(modal);
            if (bsModal) {
                bsModal.hide();
            }
        }
    }
    
    // Exams Management
    async loadExams() {
        try {
            
            const content = `
                <div class="page-header">
                    <div class="d-flex justify-content-between align-items-center">
                        <div>
                            <h1 class="h2 mb-1"><i class="bi bi-file-earmark-text"></i> <span data-translate="exams.title">Exams</span></h1>
                            <p class="text-muted mb-0" data-translate="exams.subtitle">Manage exams and assessments</p>
                        </div>
                        <div class="btn-group">
                            <button class="btn btn-primary" onclick="app.showExamModal()">
                                <i class="bi bi-plus-circle"></i> <span data-translate="exams.add_exam">Add Exam</span>
                            </button>
                            <button class="btn btn-outline-primary" onclick="app.showBulkExamModal()">
                                <i class="bi bi-layers"></i> <span data-translate="exams.add_exam_group">Add Exam Group</span>
                            </button>
                        </div>
                    </div>
                </div>
                
                <div class="filter-bar">
                    <form onsubmit="app.filterExams(); return false;">
                        <div class="row align-items-end">
                            <div class="col-md-3">
                                <label class="form-label" data-translate="exams.search">Search</label>
                                <input type="text" class="form-control" id="examTitleFilter" data-translate-placeholder="exams.search_placeholder" placeholder="Search by title">
                            </div>
                            <div class="col-md-2">
                                <label class="form-label" data-translate="exams.year">Year</label>
                                <select class="form-select" id="examYearFilter">
                                    <option value="" data-translate="exams.all_years">All Years</option>
                                </select>
                            </div>
                            <div class="col-md-2">
                                <label class="form-label" data-translate="exams.term">Term</label>
                                <select class="form-select" id="examTermFilter">
                                    <option value="" data-translate="exams.all_terms">All Terms</option>
                                </select>
                            </div>
                            <div class="col-md-2">
                                <label class="form-label" data-translate="exams.grade">Grade</label>
                                <select class="form-select" id="examGradeFilter">
                                    <option value="" data-translate="exams.all_grades">All Grades</option>
                                </select>
                            </div>
                            <div class="col-md-1">
                                <label class="form-label">&nbsp;</label>
                                <div class="d-grid">
                                    <button type="submit" class="btn btn-outline-primary">
                                        <i class="bi bi-search"></i> <span data-translate="exams.filter">Filter</span>
                                    </button>
                                </div>
                            </div>
                            <div class="col-md-2">
                                <label class="form-label">&nbsp;</label>
                                <div class="d-grid">
                                    <button type="button" class="btn btn-outline-secondary" onclick="app.clearExamsFilters()">
                                        <i class="bi bi-arrow-clockwise"></i> <span data-translate="exams.clear">Clear</span>
                                    </button>
                                </div>
                            </div>
                        </div>
                    </form>
                </div>

                <div class="card">
                    <div class="card-header">
                        <div class="d-flex justify-content-between align-items-center">
                            <div class="d-flex align-items-center">
                                <input type="checkbox" class="form-check-input me-2" id="selectAllExams" onchange="app.toggleAllExams()">
                                <span data-translate="exams.select_all">Select All</span>
                            </div>
                            <div class="btn-group">
                                <button class="btn btn-outline-danger btn-sm" onclick="app.bulkDeleteExams()" style="display: none;" id="bulkDeleteExamsBtn">
                                    <i class="bi bi-trash"></i> <span data-translate="exams.delete_selected">Delete Selected</span>
                                </button>
                                <button class="btn btn-outline-success btn-sm" onclick="app.exportExams('csv')">
                                    <i class="bi bi-file-earmark-csv"></i> <span data-translate="exams.export">Export</span>
                                </button>
                            </div>
                        </div>
                    </div>
                    <div class="card-body p-0">
                        <div class="table-responsive">
                            <table class="table table-hover mb-0">
                                <thead>
                                    <tr>
                                        <th width="50"></th>
                                        <th data-translate="table.exam_title">Title</th>
                                        <th data-translate="table.exam_year">Year</th>
                                        <th data-translate="table.exam_term">Term</th>
                                        <th data-translate="table.exam_grade">Grade</th>
                                        <th data-translate="table.exam_subject">Subject</th>
                                        <th data-translate="table.exam_date">Exam Date</th>
                                        <th data-translate="table.exam_score_range">Score Range</th>
                                        <th data-translate="table.exam_status">Status</th>
                                        <th width="120" data-translate="table.exam_actions">Actions</th>
                                    </tr>
                                </thead>
                                <tbody id="examsTableBody">
                                    <tr><td colspan="10" class="text-center py-4 text-muted" data-translate="exams.loading">Loading exams...</td></tr>
                                </tbody>
                            </table>
                        </div>
                    </div>
                </div>
                
                <div id="examsPaginationNav">
                    <!-- Pagination will be rendered here -->
                </div>
            `;
            
            document.getElementById('pageContent').innerHTML = content;
            await this.loadExamsData();
            await this.loadExamFilterOptions();
            
        } catch (error) {
            console.error('Error loading exams:', error);
            this.showToast('error', (typeof langManager !== 'undefined' ? langManager.t('exams.failed_to_load') : 'Failed to load exams') + ': ' + error.message);
        }
    }
    
    async loadExamsData(page = 1, perPage = 25, filters = {}) {
        try {
            const params = new URLSearchParams({
                page: page,
                per_page: perPage,
                ...filters
            });
            
            const response = await this.apiCall('GET', `/admin/exams?${params.toString()}`);
            
            document.getElementById('examsTableBody').innerHTML = this.renderExamsTable(response.data);
            
            if (response.meta) {
                this.renderExamsPagination(response.meta);
            }
            
        } catch (error) {
            console.error('Error loading exams data:', error);
            this.showToast('error', (typeof langManager !== 'undefined' ? langManager.t('exams.failed_to_load') : 'Failed to load exams data') + ': ' + error.message);
        }
    }
    
    renderExamsTable(exams) {
        if (!exams || exams.length === 0) {
            return '<tr><td colspan="10" class="text-center py-4 text-muted" data-translate="exams.no_exams_found">No exams found</td></tr>';
        }
        
        return exams.map(exam => `
            <tr>
                <td>
                    <input type="checkbox" class="form-check-input exam-checkbox" value="${exam.id}" onchange="app.updateExamSelection()">
                </td>
                <td>
                    <div class="fw-bold">${exam.title}</div>
                </td>
                <td>
                    <span class="badge bg-info">${exam.year_name || 'N/A'}</span>
                </td>
                <td>
                    <span class="badge bg-secondary">${exam.term_name || 'N/A'}</span>
                </td>
                <td>
                    <span class="badge bg-warning text-dark">${exam.grade_title || 'N/A'}</span>
                </td>
                <td>
                    <span class="badge bg-primary">${exam.subject_name || 'N/A'}</span>
                </td>
                <td>
                    <small class="text-muted">${exam.exam_date ? new Date(exam.exam_date).toLocaleDateString() : 'N/A'}</small>
                </td>
                <td>
                    <small class="text-muted">${exam.min_score || '0'} - ${exam.max_score || '0'}</small>
                </td>
                <td>
                    <span class="badge ${exam.enabled == 1 ? 'bg-success' : 'bg-secondary'}">
                        ${exam.enabled == 1 ? (typeof langManager !== 'undefined' ? langManager.t('common.enabled') : 'Enabled') : (typeof langManager !== 'undefined' ? langManager.t('common.disabled') : 'Disabled')}
                    </span>
                </td>
                <td>
                    <div class="btn-group btn-group-sm">
                        <button class="btn btn-outline-primary btn-sm" onclick="app.editExam(${exam.id})" title="Edit">
                            <i class="bi bi-pencil"></i>
                        </button>
                        <button class="btn btn-outline-success btn-sm" onclick="app.viewExamResults(${exam.id})" title="View Results">
                            <i class="bi bi-list-check"></i>
                        </button>
                        <button class="btn btn-outline-danger btn-sm" onclick="app.deleteExam(${exam.id}, '${(exam.title || '').replace(/'/g, '&#39;')}')" title="Delete">
                            <i class="bi bi-trash"></i>
                        </button>
                    </div>
                </td>
            </tr>
        `).join('');
    }
    
    renderExamsPagination(meta) {
        const nav = document.getElementById('examsPaginationNav');
        if (!nav || meta.total_pages <= 1) {
            if (nav) nav.innerHTML = '';
            return;
        }
        
        let paginationHtml = '<nav><ul class="pagination justify-content-center">';
        
        if (meta.page > 1) {
            paginationHtml += `<li class="page-item">
                <a class="page-link" href="#" onclick="app.loadExamsPage(${meta.page - 1})">Previous</a>
            </li>`;
        }
        
        for (let i = Math.max(1, meta.page - 2); i <= Math.min(meta.total_pages, meta.page + 2); i++) {
            paginationHtml += `<li class="page-item ${i === meta.page ? 'active' : ''}">
                <a class="page-link" href="#" onclick="app.loadExamsPage(${i})">${i}</a>
            </li>`;
        }
        
        if (meta.page < meta.total_pages) {
            paginationHtml += `<li class="page-item">
                <a class="page-link" href="#" onclick="app.loadExamsPage(${meta.page + 1})">Next</a>
            </li>`;
        }
        
        paginationHtml += '</ul></nav>';
        nav.innerHTML = paginationHtml;
    }
    
    async loadExamsPage(page) {
        const perPage = document.getElementById('examsPerPage')?.value || 25;
        const filters = this.getExamFilters();
        await this.loadExamsData(page, perPage, filters);
    }
    
    getExamFilters() {
        const filters = {};
        const title = document.getElementById('examTitleFilter')?.value;
        const yearId = document.getElementById('examYearFilter')?.value;
        const termId = document.getElementById('examTermFilter')?.value;
        const gradeId = document.getElementById('examGradeFilter')?.value;
        
        if (title) filters.title = title;
        if (yearId) filters.year_id = yearId;
        if (termId) filters.term_id = termId;
        if (gradeId) filters.grade_id = gradeId;
        
        return filters;
    }
    
    async filterExams() {
        const filters = this.getExamFilters();
        await this.loadExamsPage(1);
    }
    
    clearExamsFilters() {
        document.getElementById('examTitleFilter').value = '';
        document.getElementById('examYearFilter').value = '';
        document.getElementById('examTermFilter').value = '';
        document.getElementById('examGradeFilter').value = '';
        this.loadExamsPage(1);
    }
    
    toggleAllExams() {
        const selectAll = document.getElementById('selectAllExams');
        const checkboxes = document.querySelectorAll('.exam-checkbox');
        
        checkboxes.forEach(checkbox => {
            checkbox.checked = selectAll.checked;
        });
        
        this.updateExamSelection();
    }
    
    updateExamSelection() {
        const checkboxes = document.querySelectorAll('.exam-checkbox:checked');
        const bulkDeleteBtn = document.getElementById('bulkDeleteExamsBtn');
        const selectAll = document.getElementById('selectAllExams');
        const totalCheckboxes = document.querySelectorAll('.exam-checkbox');
        
        if (bulkDeleteBtn) {
            bulkDeleteBtn.style.display = checkboxes.length > 0 ? 'inline-block' : 'none';
        }
        
        if (selectAll) {
            selectAll.checked = totalCheckboxes.length > 0 && checkboxes.length === totalCheckboxes.length;
        }
    }
    
    async loadExamFilterOptions() {
        try {
            // Load academic years
            const yearResponse = await this.apiCall('GET', '/admin/academic-years');
            const yearSelect = document.getElementById('examYearFilter');
            if (yearSelect && yearResponse.data) {
                yearSelect.innerHTML = '<option value="">All Years</option>';
                yearResponse.data.forEach(year => {
                    yearSelect.innerHTML += `<option value="${year.id}">${year.name}</option>`;
                });
            }
            
            // Load terms
            const termResponse = await this.apiCall('GET', '/admin/terms');
            const termSelect = document.getElementById('examTermFilter');
            if (termSelect && termResponse.data) {
                termSelect.innerHTML = '<option value="">All Terms</option>';
                termResponse.data.forEach(term => {
                    termSelect.innerHTML += `<option value="${term.id}">${term.name}</option>`;
                });
            }
            
            // Load grades
            const gradeResponse = await this.apiCall('GET', '/admin/grades');
            const gradeSelect = document.getElementById('examGradeFilter');
            if (gradeSelect && gradeResponse.data) {
                gradeSelect.innerHTML = '<option value="">All Grades</option>';
                gradeResponse.data.forEach(grade => {
                    gradeSelect.innerHTML += `<option value="${grade.id}">${grade.title}</option>`;
                });
            }
            
        } catch (error) {
            console.warn('Failed to load exam filter options:', error);
        }
    }
    
    async showExamModal(exam = null) {
        const isEdit = !!exam;
        const modalTitle = isEdit ? 'Edit Exam' : 'Add New Exam';
        
        // Load required data
        await this.loadAcademicYearOptions();
        await this.loadTermOptions();
        await this.loadGradeOptions();
        await this.loadSubjectOptions();
        
        const modal = document.createElement('div');
        modal.className = 'modal fade';
        modal.id = 'examModal';
        modal.innerHTML = `
            <div class="modal-dialog modal-lg">
                <div class="modal-content">
                    <div class="modal-header">
                        <h5 class="modal-title">${modalTitle}</h5>
                        <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                    </div>
                    <div class="modal-body">
                        <form id="examForm">
                            <input type="hidden" id="examId" value="${exam ? exam.id : ''}">
                            
                            <div class="row mb-3">
                                <div class="col-md-6">
                                    <label for="examTitle" class="form-label">Title *</label>
                                    <input type="text" class="form-control" id="examTitle" value="${exam ? exam.title : ''}" required>
                                </div>
                                <div class="col-md-6">
                                    <label for="examDate" class="form-label">Exam Date *</label>
                                    <input type="date" class="form-control" id="examDate" value="${exam ? exam.exam_date : ''}" required>
                                </div>
                            </div>
                            
                            <div class="row mb-3">
                                <div class="col-md-6">
                                    <label for="examYearId" class="form-label">Academic Year *</label>
                                    <select class="form-select" id="examYearId" required>
                                        <option value="">Select Academic Year</option>
                                        ${this.renderYearOptions(exam ? exam.year_id : null)}
                                    </select>
                                </div>
                                <div class="col-md-6">
                                    <label for="examTermId" class="form-label">Term *</label>
                                    <select class="form-select" id="examTermId" required>
                                        <option value="">Select Term</option>
                                        ${this.renderTermOptions(exam ? exam.term_id : null)}
                                    </select>
                                </div>
                            </div>
                            
                            <div class="row mb-3">
                                <div class="col-md-6">
                                    <label for="examGradeId" class="form-label">Grade *</label>
                                    <select class="form-select" id="examGradeId" required>
                                        <option value="">Select Grade</option>
                                        ${this.renderGradeOptions(exam ? exam.grade_id : null)}
                                    </select>
                                </div>
                                <div class="col-md-6">
                                    <label for="examSubjectId" class="form-label">Subject *</label>
                                    <select class="form-select" id="examSubjectId" required>
                                        <option value="">Select Subject</option>
                                        ${this.renderSubjectOptions(exam ? exam.subject_id : null)}
                                    </select>
                                </div>
                            </div>
                            
                            <div class="row mb-3">
                                <div class="col-md-4">
                                    <label for="examMinScore" class="form-label">Min Score *</label>
                                    <input type="number" class="form-control" id="examMinScore" value="${exam ? exam.min_score : ''}" step="0.01" min="0" required>
                                </div>
                                <div class="col-md-4">
                                    <label for="examMaxScore" class="form-label">Max Score *</label>
                                    <input type="number" class="form-control" id="examMaxScore" value="${exam ? exam.max_score : ''}" step="0.01" min="0" required>
                                </div>
                                <div class="col-md-4">
                                    <label for="examEnabled" class="form-label">Status</label>
                                    <select class="form-select" id="examEnabled">
                                        <option value="1" ${exam && exam.enabled == 1 ? 'selected' : ''}>Enabled</option>
                                        <option value="0" ${exam && exam.enabled == 0 ? 'selected' : ''}>Disabled</option>
                                    </select>
                                </div>
                            </div>
                        </form>
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                        <button type="button" class="btn btn-primary" onclick="app.saveExam()">
                            ${isEdit ? 'Update' : 'Create'} Exam
                        </button>
                    </div>
                </div>
            </div>
        `;
        
        document.body.appendChild(modal);
        const bsModal = new bootstrap.Modal(modal);
        bsModal.show();
        
        modal.addEventListener('hidden.bs.modal', () => {
            document.body.removeChild(modal);
        });
    }
    
    async loadTermOptions() {
        if (!this.termOptions) {
            try {
                const response = await this.apiCall('GET', '/admin/terms');
                this.termOptions = response.data || [];
            } catch (error) {
                this.termOptions = [];
            }
        }
    }
    
    async loadSubjectOptions() {
        if (!this.subjectOptions) {
            try {
                const response = await this.apiCall('GET', '/admin/subjects');
                this.subjectOptions = response.data || [];
            } catch (error) {
                this.subjectOptions = [];
            }
        }
    }
    
    renderTermOptions(selectedId = null) {
        return this.termOptions.map(term => 
            `<option value="${term.id}" ${term.id == selectedId ? 'selected' : ''}>
                ${term.name}
            </option>`
        ).join('');
    }
    
    renderSubjectOptions(selectedId = null) {
        return this.subjectOptions.map(subject => 
            `<option value="${subject.id}" ${subject.id == selectedId ? 'selected' : ''}>
                ${subject.name}
            </option>`
        ).join('');
    }
    
    async saveExam() {
        const examIdField = document.getElementById('examId');
        const titleField = document.getElementById('examTitle');
        const dateField = document.getElementById('examDate');
        const yearIdField = document.getElementById('examYearId');
        const termIdField = document.getElementById('examTermId');
        const gradeIdField = document.getElementById('examGradeId');
        const subjectIdField = document.getElementById('examSubjectId');
        const minScoreField = document.getElementById('examMinScore');
        const maxScoreField = document.getElementById('examMaxScore');
        const enabledField = document.getElementById('examEnabled');
        
        if (!titleField || !dateField || !yearIdField || !termIdField || !gradeIdField || !subjectIdField || !minScoreField || !maxScoreField) {
            this.showToast('error', 'Form fields not found. Please refresh the page.');
            return;
        }
        
        const examId = examIdField ? examIdField.value : '';
        const isEdit = !!examId;
        
        const data = {
            title: titleField.value.trim(),
            exam_date: dateField.value,
            year_id: parseInt(yearIdField.value),
            term_id: parseInt(termIdField.value),
            grade_id: parseInt(gradeIdField.value),
            subject_id: parseInt(subjectIdField.value),
            min_score: parseFloat(minScoreField.value),
            max_score: parseFloat(maxScoreField.value),
            enabled: parseInt(enabledField.value)
        };
        
        // Clear previous validation
        document.querySelectorAll('.is-invalid').forEach(el => el.classList.remove('is-invalid'));
        
        try {
            const endpoint = isEdit ? `/admin/exams/${examId}` : '/admin/exams';
            const method = isEdit ? 'PUT' : 'POST';
            
            const response = await this.apiCall(method, endpoint, data);
            
            if (response.success) {
                this.closeExamModal();
                this.showToast('success', `Exam ${isEdit ? 'updated' : 'created'} successfully!`);
                this.loadExamsPage(1);
            } else {
                this.showToast('error', response.message);
            }
            
        } catch (error) {
            this.showToast('error', 'Failed to save exam. Please try again.');
        }
    }
    
    closeExamModal() {
        const modal = document.getElementById('examModal');
        if (modal) {
            const bsModal = bootstrap.Modal.getInstance(modal);
            if (bsModal) {
                bsModal.hide();
            }
        }
    }
    
    async editExam(id) {
        try {
            const response = await this.apiCall('GET', `/admin/exams/${id}`);
            if (response.success) {
                this.showExamModal(response.data);
            } else {
                this.showToast('error', 'Failed to load exam data');
            }
        } catch (error) {
            this.showToast('error', 'Failed to load exam data');
        }
    }
    
    async deleteExam(id, title) {
        if (!confirm(`Are you sure you want to delete the exam "${title}"?\n\nThis action cannot be undone.`)) {
            return;
        }
        
        try {
            const response = await this.apiCall('DELETE', `/admin/exams/${id}`);
            
            if (response.success) {
                this.showToast('success', 'Exam deleted successfully!');
                const currentPage = this.getCurrentExamPage() || 1;
                this.loadExamsPage(currentPage);
            } else {
                this.showToast('error', response.message);
            }
        } catch (error) {
            this.showToast('error', 'Failed to delete exam.');
        }
    }
    
    getCurrentExamPage() {
        const activePageLink = document.querySelector('#examsPaginationNav .page-item.active .page-link');
        return activePageLink ? parseInt(activePageLink.textContent) : 1;
    }
    
    async bulkDeleteExams() {
        const selectedCheckboxes = document.querySelectorAll('.exam-checkbox:checked');
        const examIds = Array.from(selectedCheckboxes).map(cb => parseInt(cb.value));
        
        if (examIds.length === 0) {
            this.showToast('error', 'Please select exams to delete.');
            return;
        }
        
        if (!confirm(`Are you sure you want to delete ${examIds.length} exam(s)?\n\nThis action cannot be undone.`)) {
            return;
        }
        
        try {
            const response = await this.apiCall('DELETE', '/admin/exams/bulk', { exam_ids: examIds });
            
            if (response.success) {
                this.showToast('success', `${examIds.length} exam(s) deleted successfully!`);
                this.loadExamsPage(1);
            } else {
                this.showToast('error', response.message);
            }
        } catch (error) {
            this.showToast('error', 'Failed to delete exams.');
        }
    }
    
    viewExamResults(examId) {
        this.loadPage('results', `?exam_id=${examId}`);
    }
    
    exportExams(format) {
        this.showToast('info', `${format.toUpperCase()} export functionality coming soon!`);
    }
    
    async showBulkExamModal() {
        // Load required data
        await this.loadAcademicYearOptions();
        await this.loadTermOptions();
        await this.loadGradeOptions();
        
        const modal = document.createElement('div');
        modal.className = 'modal fade';
        modal.id = 'bulkExamModal';
        modal.innerHTML = `
            <div class="modal-dialog modal-lg">
                <div class="modal-content">
                    <div class="modal-header">
                        <h5 class="modal-title" data-translate="modal.create_exam_group">Create Exam Group</h5>
                        <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                    </div>
                    <div class="modal-body">
                        <div class="alert alert-info">
                            <i class="bi bi-info-circle"></i>
                            Create multiple exams for a grade with default scores from grade-subject rules.
                        </div>
                        
                        <form id="bulkExamForm">
                            <div class="row mb-3">
                                <div class="col-md-4">
                                    <label for="bulkExamYearId" class="form-label">Academic Year *</label>
                                    <select class="form-select" id="bulkExamYearId" required>
                                        <option value="">Select Academic Year</option>
                                        ${this.renderYearOptions()}
                                    </select>
                                </div>
                                <div class="col-md-4">
                                    <label for="bulkExamTermId" class="form-label">Term *</label>
                                    <select class="form-select" id="bulkExamTermId" required>
                                        <option value="">Select Term</option>
                                        ${this.renderTermOptions()}
                                    </select>
                                </div>
                                <div class="col-md-4">
                                    <label for="bulkExamGradeId" class="form-label">Grade *</label>
                                    <select class="form-select" id="bulkExamGradeId" required onchange="app.loadGradeSubjects()">
                                        <option value="">Select Grade</option>
                                        ${this.renderGradeOptions()}
                                    </select>
                                </div>
                            </div>
                            
                            <div class="row mb-3">
                                <div class="col-md-6">
                                    <label for="bulkExamDate" class="form-label">Exam Date *</label>
                                    <input type="date" class="form-control" id="bulkExamDate" required>
                                </div>
                                <div class="col-md-6">
                                    <label for="bulkExamEnabled" class="form-label">Status</label>
                                    <select class="form-select" id="bulkExamEnabled">
                                        <option value="1">Enabled</option>
                                        <option value="0">Disabled</option>
                                    </select>
                                </div>
                            </div>
                            
                            <div class="mb-3">
                                <label class="form-label">Select Subjects *</label>
                                <div class="border rounded p-3" style="max-height: 400px; overflow-y: auto;">
                                    <div class="form-check mb-2">
                                        <input class="form-check-input" type="checkbox" id="selectAllSubjects" onchange="app.toggleAllSubjects()">
                                        <label class="form-check-label fw-bold" for="selectAllSubjects">
                                            Select All Subjects
                                        </label>
                                    </div>
                                    <hr>
                                    <div id="subjectsList">
                                        <p class="text-muted">Please select a grade first...</p>
                                    </div>
                                </div>
                            </div>
                        </form>
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                        <button type="button" class="btn btn-primary" onclick="app.saveBulkExams()">
                            <span data-translate="modal.create_exam_group">Create Exam Group</span>
                        </button>
                    </div>
                </div>
            </div>
        `;
        
        document.body.appendChild(modal);
        const bsModal = new bootstrap.Modal(modal);
        bsModal.show();
        
        // Set default date to tomorrow
        const tomorrow = new Date();
        tomorrow.setDate(tomorrow.getDate() + 1);
        const dateField = document.getElementById('bulkExamDate');
        if (dateField) {
            dateField.value = tomorrow.toISOString().split('T')[0];
        }
        
        modal.addEventListener('hidden.bs.modal', () => {
            document.body.removeChild(modal);
        });
    }
    
    async loadGradeSubjects() {
        const gradeId = document.getElementById('bulkExamGradeId')?.value;
        const subjectsList = document.getElementById('subjectsList');
        
        if (!gradeId || !subjectsList) {
            return;
        }
        
        try {
            subjectsList.innerHTML = '<p class="text-muted">Loading subjects...</p>';
            
            const response = await this.apiCall('GET', `/admin/exams/grade-subjects/${gradeId}`);
            
            if (response.success && response.data) {
                if (response.data.length === 0) {
                    subjectsList.innerHTML = '<p class="text-muted">No subjects found for this grade.</p>';
                    return;
                }
                
                subjectsList.innerHTML = response.data.map(subject => `
                    <div class="card mb-2">
                        <div class="card-body py-2">
                            <div class="form-check">
                                <input class="form-check-input subject-checkbox" type="checkbox" 
                                       value="${subject.id}" id="subject_${subject.id}" 
                                       onchange="app.updateSubjectSelection()">
                                <label class="form-check-label fw-bold" for="subject_${subject.id}">
                                    ${subject.name}
                                </label>
                            </div>
                            <div class="row mt-2">
                                <div class="col-md-4">
                                    <label for="title_${subject.id}" class="form-label small">Exam Title</label>
                                    <input type="text" class="form-control form-control-sm" 
                                           id="title_${subject.id}" 
                                           value="${subject.name} Exam" 
                                           placeholder="Exam title">
                                </div>
                                <div class="col-md-4">
                                    <label for="min_${subject.id}" class="form-label small">Min Score</label>
                                    <input type="number" class="form-control form-control-sm" 
                                           id="min_${subject.id}" 
                                           value="${subject.min_score || 0}" 
                                           step="0.01" min="0">
                                </div>
                                <div class="col-md-4">
                                    <label for="max_${subject.id}" class="form-label small">Max Score</label>
                                    <input type="number" class="form-control form-control-sm" 
                                           id="max_${subject.id}" 
                                           value="${subject.max_score || 100}" 
                                           step="0.01" min="0">
                                </div>
                            </div>
                        </div>
                    </div>
                `).join('');
            } else {
                subjectsList.innerHTML = '<p class="text-muted text-danger">Failed to load subjects.</p>';
            }
            
        } catch (error) {
            subjectsList.innerHTML = '<p class="text-muted text-danger">Error loading subjects.</p>';
            console.error('Error loading grade subjects:', error);
        }
    }
    
    toggleAllSubjects() {
        const selectAll = document.getElementById('selectAllSubjects');
        const checkboxes = document.querySelectorAll('.subject-checkbox');
        
        checkboxes.forEach(checkbox => {
            checkbox.checked = selectAll.checked;
        });
        
        this.updateSubjectSelection();
    }
    
    updateSubjectSelection() {
        const checkboxes = document.querySelectorAll('.subject-checkbox:checked');
        const selectAll = document.getElementById('selectAllSubjects');
        const totalCheckboxes = document.querySelectorAll('.subject-checkbox');
        
        if (selectAll) {
            selectAll.checked = totalCheckboxes.length > 0 && checkboxes.length === totalCheckboxes.length;
        }
    }
    
    async saveBulkExams() {
        const yearIdField = document.getElementById('bulkExamYearId');
        const termIdField = document.getElementById('bulkExamTermId');
        const gradeIdField = document.getElementById('bulkExamGradeId');
        const dateField = document.getElementById('bulkExamDate');
        const enabledField = document.getElementById('bulkExamEnabled');
        
        if (!yearIdField || !termIdField || !gradeIdField || !dateField) {
            this.showToast('error', 'Form fields not found. Please refresh the page.');
            return;
        }
        
        // Get selected subjects with their data
        const selectedSubjects = Array.from(document.querySelectorAll('.subject-checkbox:checked'))
            .map(checkbox => {
                const subjectId = parseInt(checkbox.value);
                const titleField = document.getElementById(`title_${subjectId}`);
                const minField = document.getElementById(`min_${subjectId}`);
                const maxField = document.getElementById(`max_${subjectId}`);
                
                return {
                    subject_id: subjectId,
                    title: titleField ? titleField.value.trim() : '',
                    min_score: minField ? parseFloat(minField.value) : 0,
                    max_score: maxField ? parseFloat(maxField.value) : 100
                };
            });
            
        if (selectedSubjects.length === 0) {
            this.showToast('error', 'Please select at least one subject.');
            return;
        }
        
        const data = {
            year_id: parseInt(yearIdField.value),
            term_id: parseInt(termIdField.value),
            grade_id: parseInt(gradeIdField.value),
            exam_date: dateField.value,
            enabled: parseInt(enabledField.value),
            subjects: selectedSubjects
        };
        
        try {
            const response = await this.apiCall('POST', '/admin/exams/bulk-create', data);
            
            if (response.success) {
                this.closeBulkExamModal();
                const results = response.data;
                this.showToast('success', 
                    `Exam group created: ${results.successful} successful, ${results.failed} failed`
                );
                this.loadExamsPage(1);
            } else {
                this.showToast('error', response.message);
            }
            
        } catch (error) {
            this.showToast('error', 'Failed to create exam group. Please try again.');
        }
    }
    
    closeBulkExamModal() {
        const modal = document.getElementById('bulkExamModal');
        if (modal) {
            const bsModal = bootstrap.Modal.getInstance(modal);
            if (bsModal) {
                bsModal.hide();
            }
        }
    }
    
    // Results Management
    async loadResults() {
        try {
            const content = `
                <div class="page-header">
                    <div class="d-flex justify-content-between align-items-center">
                        <div>
                            <h1 class="h2 mb-1"><i class="bi bi-graph-up"></i> <span data-translate="results.title">Results Management</span></h1>
                            <p class="text-muted mb-0" data-translate="results.subtitle">Manage exam results and student performance</p>
                        </div>
                        <div class="btn-toolbar" role="toolbar">
                            <div class="btn-group me-2">
                                <button class="btn btn-success" onclick="app.showAddResultModal()">
                                    <i class="bi bi-plus-circle"></i> <span data-translate="results.add_result">Add Result</span>
                                </button>
                                <button class="btn btn-primary" onclick="app.showBulkResultsModal()">
                                    <i class="bi bi-plus-square"></i> <span data-translate="results.bulk_by_exam">Bulk by Exam</span>
                                </button>
                                <button class="btn btn-info" onclick="app.showStudentResultsModal()">
                                    <i class="bi bi-person-check"></i> <span data-translate="results.student_results">Student Results</span>
                                </button>
                            </div>
                            <div class="btn-group me-2">
                                <button class="btn btn-outline-secondary" onclick="app.showImportResultsModal()">
                                    <i class="bi bi-upload"></i> <span data-translate="results.import">Import</span>
                                </button>
                                <button class="btn btn-outline-secondary" onclick="app.exportResults()">
                                    <i class="bi bi-download"></i> <span data-translate="results.export">Export</span>
                                </button>
                            </div>
                            <div class="btn-group">
                                <button class="btn btn-danger" onclick="app.bulkDeleteResults()" disabled id="bulkDeleteBtn">
                                    <i class="bi bi-trash"></i> <span data-translate="results.delete_selected">Delete Selected</span>
                                </button>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Filters Section -->
                <div class="card mb-4">
                    <div class="card-body">
                        <h6 class="card-subtitle mb-3" data-translate="results.filters_search">Filters & Search</h6>
                        <div class="row g-3">
                            <div class="col-md-2">
                                <select class="form-select form-select-sm" id="yearFilter">
                                    <option value="" data-translate="results.all_years">All Years</option>
                                </select>
                            </div>
                            <div class="col-md-2">
                                <select class="form-select form-select-sm" id="termFilter">
                                    <option value="" data-translate="results.all_terms">All Terms</option>
                                </select>
                            </div>
                            <div class="col-md-2">
                                <select class="form-select form-select-sm" id="gradeFilter">
                                    <option value="" data-translate="results.all_grades">All Grades</option>
                                </select>
                            </div>
                            <div class="col-md-2">
                                <select class="form-select form-select-sm" id="subjectFilter">
                                    <option value="" data-translate="results.all_subjects">All Subjects</option>
                                </select>
                            </div>
                            <div class="col-md-2">
                                <select class="form-select form-select-sm" id="examFilter">
                                    <option value="" data-translate="results.all_exams">All Exams</option>
                                </select>
                            </div>
                            <div class="col-md-2">
                                <input type="text" class="form-control form-control-sm" id="searchResults" data-translate-placeholder="results.search_students" placeholder="Search students...">
                            </div>
                        </div>
                        <div class="row mt-2">
                            <div class="col-12">
                                <button class="btn btn-sm btn-primary" onclick="app.loadResultsData()">
                                    <i class="bi bi-search"></i> <span data-translate="results.apply_filters">Apply Filters</span>
                                </button>
                                <button class="btn btn-sm btn-outline-secondary" onclick="app.clearResultsFilters()">
                                    <i class="bi bi-x-circle"></i> <span data-translate="results.clear_filters">Clear</span>
                                </button>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Results Table -->
                <div class="card">
                    <div class="card-header d-flex justify-content-between align-items-center">
                        <div>
                            <h6 class="mb-0" data-translate="results.results_list">Results List</h6>
                            <small class="text-muted" id="resultsCount" data-translate="results.loading">Loading...</small>
                        </div>
                        <div class="form-check">
                            <input class="form-check-input" type="checkbox" id="selectAllResults" onchange="app.toggleSelectAllResults()">
                            <label class="form-check-label" for="selectAllResults" data-translate="results.select_all">
                                Select All
                            </label>
                        </div>
                    </div>
                    <div class="card-body p-0">
                        <div class="table-responsive">
                            <table class="table table-hover mb-0">
                                <thead class="table-light">
                                    <tr>
                                        <th width="40">
                                            <input type="checkbox" class="form-check-input" disabled>
                                        </th>
                                        <th class="sortable" data-sort="student_name" style="cursor: pointer;" data-translate="table.result_student">
                                            Student <i class="bi bi-arrow-down-up"></i>
                                        </th>
                                        <th class="sortable" data-sort="exam_title" style="cursor: pointer;" data-translate="table.result_exam">
                                            Exam <i class="bi bi-arrow-down-up"></i>
                                        </th>
                                        <th class="sortable" data-sort="score" style="cursor: pointer;" data-translate="table.result_score">
                                            Score <i class="bi bi-arrow-down-up"></i>
                                        </th>
                                        <th class="sortable" data-sort="percentage" style="cursor: pointer;" data-translate="table.result_percentage">
                                            Percentage <i class="bi bi-arrow-down-up"></i>
                                        </th>
                                        <th data-translate="table.result_pass_fail">Pass/Fail Status</th>
                                        <th data-translate="table.result_year_term">Year/Term</th>
                                        <th data-translate="table.result_subject">Subject</th>
                                        <th class="sortable" data-sort="created_at" style="cursor: pointer;" data-translate="table.result_date_added">
                                            Date Added <i class="bi bi-arrow-down-up"></i>
                                        </th>
                                        <th width="120" data-translate="table.result_actions">Actions</th>
                                    </tr>
                                </thead>
                                <tbody id="resultsTableBody">
                                    <tr>
                                        <td colspan="10" class="text-center py-4">
                                            <div class="spinner-border spinner-border-sm me-2" role="status"></div>
                                            <span data-translate="results.loading">Loading results...</span>
                                        </td>
                                    </tr>
                                </tbody>
                            </table>
                        </div>
                    </div>
                    <div class="card-footer">
                        <div class="d-flex justify-content-between align-items-center">
                            <div class="text-muted small">
                                <span data-translate="results.showing">Showing</span> <span id="resultsStart">0</span> <span data-translate="results.to">to</span> <span id="resultsEnd">0</span> <span data-translate="results.of">of</span> <span id="resultsTotal">0</span> <span data-translate="results.results">results</span>
                            </div>
                            <nav id="resultsPaginationNav">
                                <!-- Pagination will be rendered here -->
                            </nav>
                        </div>
                    </div>
                </div>
            `;
            
            document.getElementById('pageContent').innerHTML = content;
            
            // Load filter options and results data
            await this.loadResultsFilterOptions();
            await this.loadResultsData();
            
            // Setup event listeners
            this.setupResultsEventListeners();
            
        } catch (error) {
            console.error('Error loading results:', error);
            this.showToast('error', (typeof langManager !== 'undefined' ? langManager.t('results.failed_to_load') : 'Failed to load results') + ': ' + error.message);
        }
    }

    async loadResultsFilterOptions() {
        try {
            const [yearsRes, termsRes, gradesRes, subjectsRes, examsRes] = await Promise.all([
                this.apiCall('GET', '/admin/academic-years'),
                this.apiCall('GET', '/admin/terms'),
                this.apiCall('GET', '/admin/grades'),
                this.apiCall('GET', '/admin/subjects'),
                this.apiCall('GET', '/admin/exams')
            ]);

            // Populate filter dropdowns
            const yearSelect = document.getElementById('yearFilter');
            const termSelect = document.getElementById('termFilter');
            const gradeSelect = document.getElementById('gradeFilter');
            const subjectSelect = document.getElementById('subjectFilter');
            const examSelect = document.getElementById('examFilter');

            yearsRes.data.forEach(year => {
                yearSelect.innerHTML += `<option value="${year.id}">${year.name}</option>`;
            });

            termsRes.data.forEach(term => {
                termSelect.innerHTML += `<option value="${term.id}">${term.name}</option>`;
            });

            gradesRes.data.forEach(grade => {
                gradeSelect.innerHTML += `<option value="${grade.id}">${grade.title} (${grade.code})</option>`;
            });

            subjectsRes.data.forEach(subject => {
                subjectSelect.innerHTML += `<option value="${subject.id}">${subject.name}</option>`;
            });

            examsRes.data.forEach(exam => {
                examSelect.innerHTML += `<option value="${exam.id}">${exam.title} - ${exam.year_name} ${exam.term_name}</option>`;
            });

        } catch (error) {
            console.error('Error loading filter options:', error);
        }
    }

    async loadResultsData(page = 1, perPage = 25) {
        try {
            const filters = {
                year_id: document.getElementById('yearFilter')?.value || '',
                term_id: document.getElementById('termFilter')?.value || '',
                grade_id: document.getElementById('gradeFilter')?.value || '',
                subject_id: document.getElementById('subjectFilter')?.value || '',
                exam_id: document.getElementById('examFilter')?.value || '',
                q: document.getElementById('searchResults')?.value || ''
            };

            const params = new URLSearchParams({
                page: page.toString(),
                per_page: perPage.toString(),
                ...Object.fromEntries(Object.entries(filters).filter(([_, v]) => v))
            });

            const response = await this.apiCall('GET', `/admin/results?${params}`);
            
            document.getElementById('resultsTableBody').innerHTML = this.renderResultsTable(response.data);
            this.updateResultsPagination(response.meta);
            this.updateResultsCount(response.meta);

        } catch (error) {
            console.error('Error loading results data:', error);
            document.getElementById('resultsTableBody').innerHTML = `
                <tr>
                    <td colspan="10" class="text-center py-4 text-danger">
                        <i class="bi bi-exclamation-triangle"></i> <span data-translate="results.error_loading">Error loading results</span>: ${error.message}
                    </td>
                </tr>
            `;
        }
    }

    renderResultsTable(results) {
        if (!results || results.length === 0) {
            return `
                <tr>
                    <td colspan="10" class="text-center py-4">
                        <i class="bi bi-inbox text-muted" style="font-size: 2rem;"></i>
                        <div class="mt-2 text-muted" data-translate="results.no_results_found">No results found</div>
                    </td>
                </tr>
            `;
        }

        return results.map(result => `
            <tr>
                <td>
                    <input type="checkbox" class="form-check-input result-checkbox" value="${result.id}">
                </td>
                <td>
                    <div class="fw-semibold">${result.student_name || 'N/A'}</div>
                    <small class="text-muted">ID: ${result.student_id}</small>
                </td>
                <td>
                    <div>${result.exam_title || 'N/A'}</div>
                    <small class="text-muted">${result.exam_date ? new Date(result.exam_date).toLocaleDateString() : 'N/A'}</small>
                </td>
                <td>
                    <span class="badge bg-primary">${result.score}/${result.max_score || 'N/A'}</span>
                </td>
                <td>
                    <div class="fw-semibold">${result.percentage}%</div>
                </td>
                <td>
                    ${result.band_label ? 
                        `<span class="badge" style="background-color: ${result.band_color}; color: white;">${result.band_label}</span>` : 
                        '<span class="text-muted">No Band</span>'
                    }
                </td>
                <td>
                    <div>${result.year_name || 'N/A'}</div>
                    <small class="text-muted">${result.term_name || 'N/A'}</small>
                </td>
                <td>${result.subject_name || 'N/A'}</td>
                <td>
                    <small class="text-muted">
                        ${result.created_at ? new Date(result.created_at).toLocaleDateString() : 'N/A'}
                    </small>
                </td>
                <td>
                    <div class="btn-group btn-group-sm">
                        <button class="btn btn-outline-primary btn-sm" onclick="app.editResult(${result.id})" title="Edit">
                            <i class="bi bi-pencil"></i>
                        </button>
                        <button class="btn btn-outline-danger btn-sm" onclick="app.deleteResult(${result.id})" title="Delete">
                            <i class="bi bi-trash"></i>
                        </button>
                    </div>
                </td>
            </tr>
        `).join('');
    }

    updateResultsPagination(meta) {
        const nav = document.getElementById('resultsPaginationNav');
        if (!nav || !meta || meta.total_pages <= 1) {
            nav.innerHTML = '';
            return;
        }

        let pagination = `<ul class="pagination pagination-sm mb-0">`;
        
        // Previous button
        if (meta.page > 1) {
            pagination += `<li class="page-item"><a class="page-link" href="#" onclick="app.loadResultsData(${meta.page - 1})">&laquo;</a></li>`;
        }

        // Page numbers
        const startPage = Math.max(1, meta.page - 2);
        const endPage = Math.min(meta.total_pages, meta.page + 2);

        for (let i = startPage; i <= endPage; i++) {
            const active = i === meta.page ? 'active' : '';
            pagination += `<li class="page-item ${active}"><a class="page-link" href="#" onclick="app.loadResultsData(${i})">${i}</a></li>`;
        }

        // Next button
        if (meta.page < meta.total_pages) {
            pagination += `<li class="page-item"><a class="page-link" href="#" onclick="app.loadResultsData(${meta.page + 1})">&raquo;</a></li>`;
        }

        pagination += `</ul>`;
        nav.innerHTML = pagination;
    }

    updateResultsCount(meta) {
        const start = ((meta.page - 1) * meta.per_page) + 1;
        const end = Math.min(meta.page * meta.per_page, meta.total_rows);
        
        document.getElementById('resultsStart').textContent = meta.total_rows > 0 ? start : 0;
        document.getElementById('resultsEnd').textContent = end;
        document.getElementById('resultsTotal').textContent = meta.total_rows;
        document.getElementById('resultsCount').textContent = `${meta.total_rows} ${typeof langManager !== 'undefined' ? langManager.t('results.count') : 'results found'}`;
    }

    setupResultsEventListeners() {
        // Search input
        const searchInput = document.getElementById('searchResults');
        if (searchInput) {
            let searchTimeout;
            searchInput.addEventListener('input', () => {
                clearTimeout(searchTimeout);
                searchTimeout = setTimeout(() => this.loadResultsData(), 500);
            });
        }

        // Filter change listeners
        ['yearFilter', 'termFilter', 'gradeFilter', 'subjectFilter', 'examFilter'].forEach(filterId => {
            const filter = document.getElementById(filterId);
            if (filter) {
                filter.addEventListener('change', () => this.loadResultsData());
            }
        });

        // Checkbox change listeners
        document.addEventListener('change', (e) => {
            if (e.target.classList.contains('result-checkbox')) {
                this.updateBulkActions();
            }
        });

        // Sortable headers
        document.querySelectorAll('.sortable').forEach(header => {
            header.addEventListener('click', () => {
                const sortField = header.getAttribute('data-sort');
                this.sortResults(sortField);
            });
        });
    }

    toggleSelectAllResults() {
        const selectAll = document.getElementById('selectAllResults');
        const checkboxes = document.querySelectorAll('.result-checkbox');
        
        checkboxes.forEach(checkbox => {
            checkbox.checked = selectAll.checked;
        });
        
        this.updateBulkActions();
    }

    updateBulkActions() {
        const checkedBoxes = document.querySelectorAll('.result-checkbox:checked');
        const bulkDeleteBtn = document.getElementById('bulkDeleteBtn');
        
        if (bulkDeleteBtn) {
            bulkDeleteBtn.disabled = checkedBoxes.length === 0;
        }
        
        const selectAll = document.getElementById('selectAllResults');
        const allCheckboxes = document.querySelectorAll('.result-checkbox');
        
        if (selectAll) {
            selectAll.indeterminate = checkedBoxes.length > 0 && checkedBoxes.length < allCheckboxes.length;
            selectAll.checked = checkedBoxes.length === allCheckboxes.length && allCheckboxes.length > 0;
        }
    }

    clearResultsFilters() {
        document.getElementById('yearFilter').value = '';
        document.getElementById('termFilter').value = '';
        document.getElementById('gradeFilter').value = '';
        document.getElementById('subjectFilter').value = '';
        document.getElementById('examFilter').value = '';
        document.getElementById('searchResults').value = '';
        this.loadResultsData();
    }

    async showAddResultModal() {
        // Load exams first
        try {
            const examsRes = await this.apiCall('GET', '/admin/exams');

            const modal = this.createModal('Add Result', `
                <form id="addResultForm">
                    <div class="mb-3">
                        <label class="form-label">Exam *</label>
                        <select class="form-select" id="resultExamId" required onchange="app.loadStudentsForExam()">
                            <option value="">Select Exam</option>
                            ${examsRes.data.map(exam => 
                                `<option value="${exam.id}" data-min-score="${exam.min_score}" data-max-score="${exam.max_score}" data-grade-id="${exam.grade_id}" data-year-id="${exam.year_id}">${exam.title} - ${exam.year_name} ${exam.term_name} (${exam.grade_title})</option>`
                            ).join('')}
                        </select>
                    </div>
                    <div class="mb-3">
                        <label class="form-label">Student *</label>
                        <div class="position-relative">
                            <input type="text" class="form-control" id="studentSearch" placeholder="Type to search students..." disabled data-student-id="">
                            <div id="studentDropdown" class="position-absolute w-100 bg-white border border-top-0 rounded-bottom shadow-sm" style="max-height: 200px; overflow-y: auto; z-index: 9999; display: none; top: 100%;">
                                <!-- Student options will appear here -->
                            </div>
                        </div>
                        <div class="form-text" id="studentHelperText">Please select an exam first, then type to search students</div>
                    </div>
                    <div class="mb-3">
                        <label class="form-label">Score *</label>
                        <input type="number" class="form-control" id="resultScore" step="0.01" min="0" required disabled>
                        <div class="form-text" id="scoreHelperText">Select exam to see score range</div>
                    </div>
                </form>
            `, `
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                <button type="button" class="btn btn-primary" onclick="app.saveResult()">Save Result</button>
            `);

            modal.show();
        } catch (error) {
            this.showToast('error', 'Failed to load form data: ' + error.message);
        }
    }

    async loadStudentsForExam() {
        const examSelect = document.getElementById('resultExamId');
        const selectedOption = examSelect.options[examSelect.selectedIndex];
        const studentSearch = document.getElementById('studentSearch');
        const scoreInput = document.getElementById('resultScore');
        const scoreHelperText = document.getElementById('scoreHelperText');
        const studentHelperText = document.getElementById('studentHelperText');

        if (!examSelect.value) {
            // Disable student search when no exam selected
            studentSearch.disabled = true;
            studentSearch.value = '';
            scoreInput.disabled = true;
            scoreInput.value = '';
            scoreHelperText.textContent = 'Select exam to see score range';
            studentHelperText.textContent = 'Please select an exam first to load students';
            return;
        }

        try {
            const gradeId = selectedOption.getAttribute('data-grade-id');
            const yearId = selectedOption.getAttribute('data-year-id');
            const minScore = selectedOption.getAttribute('data-min-score');
            const maxScore = selectedOption.getAttribute('data-max-score');

            // Store exam context for AJAX searches
            this.currentExamContext = { gradeId, yearId };
            
            // Enable student search
            studentSearch.disabled = false;
            studentSearch.placeholder = 'Type to search students...';
            studentHelperText.innerHTML = 'Type at least 2 characters to search enrolled students';
            
            // Setup AJAX student search functionality
            this.setupAddResultStudentSearch();
            
            // Enable and setup score input
            scoreInput.disabled = false;
            scoreInput.max = maxScore;
            scoreHelperText.innerHTML = `Score range: <strong>0 - ${maxScore}</strong> (Min for pass: ${minScore})`;
            
        } catch (error) {
            console.error('Error setting up exam context:', error);
            this.showToast('error', 'Failed to setup student search for selected exam');
            studentSearch.disabled = true;
            this.currentExamContext = null;
        }
    }

    setupAddResultStudentSearch() {
        const studentSearch = document.getElementById('studentSearch');
        const studentDropdown = document.getElementById('studentDropdown');
        
        if (!studentSearch || !this.currentExamContext) return;

        let searchTimeout;

        // Clear previous listeners if any
        studentSearch.removeEventListener('input', this.handleAddResultStudentSearch);
        studentSearch.removeEventListener('focus', this.handleAddResultStudentFocus);
        studentSearch.removeEventListener('blur', this.handleAddResultStudentBlur);

        // Add new listeners
        this.handleAddResultStudentSearch = (e) => {
            const searchTerm = e.target.value.trim();
            
            // Clear previous timeout
            clearTimeout(searchTimeout);
            
            if (searchTerm.length < 2) {
                studentDropdown.style.display = 'none';
                return;
            }
            
            // Debounce search - wait 300ms after user stops typing
            searchTimeout = setTimeout(() => {
                this.searchExamStudentsAjax(searchTerm);
            }, 300);
        };

        this.handleAddResultStudentFocus = (e) => {
            const searchTerm = e.target.value.trim();
            if (searchTerm.length >= 2) {
                this.searchExamStudentsAjax(searchTerm);
            }
        };

        this.handleAddResultStudentBlur = (e) => {
            setTimeout(() => {
                studentDropdown.style.display = 'none';
            }, 200);
        };

        studentSearch.addEventListener('input', this.handleAddResultStudentSearch);
        studentSearch.addEventListener('focus', this.handleAddResultStudentFocus);
        studentSearch.addEventListener('blur', this.handleAddResultStudentBlur);
    }

    async searchExamStudentsAjax(searchTerm) {
        const studentDropdown = document.getElementById('studentDropdown');
        
        if (!this.currentExamContext) return;
        
        try {
            // Show loading state
            studentDropdown.innerHTML = '<div class="p-2 text-muted"><i class="bi bi-hourglass-split me-1"></i>Searching...</div>';
            studentDropdown.style.display = 'block';
            
            // Search students enrolled in this exam's grade and year with limit
            const { gradeId, yearId } = this.currentExamContext;
            const response = await this.apiCall('GET', `/admin/students?grade_id=${gradeId}&year_id=${yearId}&enrolled_only=true&q=${encodeURIComponent(searchTerm)}&per_page=10`);
            const students = response.data;

            if (!students || students.length === 0) {
                studentDropdown.innerHTML = '<div class="p-2 text-muted">No matching students found for this exam</div>';
            } else {
                studentDropdown.innerHTML = students.map(student => {
                    const escapedName = student.full_name.replace(/'/g, "\\'").replace(/"/g, '\\"');
                    return `
                        <div class="p-2 border-bottom student-option" style="cursor: pointer;" 
                             data-student-id="${student.id}"
                             data-student-name="${escapedName}"
                             onmouseover="this.style.backgroundColor='#f8f9fa'" 
                             onmouseout="this.style.backgroundColor='white'">
                            <div class="fw-semibold">${student.full_name}</div>
                            <small class="text-muted">ID: ${student.id} | Code: ${student.login_code || 'N/A'}</small>
                        </div>
                    `;
                }).join('');
                
                // Add "Show more" indicator if we hit the limit
                if (students.length === 10) {
                    studentDropdown.innerHTML += '<div class="p-2 text-center text-muted border-top"><small><i class="bi bi-info-circle me-1"></i>Showing first 10 matches. Type more to refine search.</small></div>';
                }
            }

            studentDropdown.style.display = 'block';
            
            // Add click event listeners
            studentDropdown.querySelectorAll('.student-option').forEach(option => {
                option.addEventListener('click', (e) => {
                    const studentId = option.getAttribute('data-student-id');
                    const studentName = option.getAttribute('data-student-name');
                    this.selectStudent(parseInt(studentId), studentName);
                });
            });
            
        } catch (error) {
            console.error('Error searching exam students:', error);
            studentDropdown.innerHTML = '<div class="p-2 text-danger">Error searching students</div>';
            studentDropdown.style.display = 'block';
        }
    }

    filterAndDisplayStudents(searchTerm) {
        const studentDropdown = document.getElementById('studentDropdown');
        const hiddenSelect = document.getElementById('resultStudentId');
        
        if (!this.examStudents) return;

        const filteredStudents = this.examStudents.filter(student => 
            student.full_name.toLowerCase().includes(searchTerm) ||
            student.id.toString().includes(searchTerm) ||
            (student.login_code && student.login_code.toLowerCase().includes(searchTerm))
        );

        if (filteredStudents.length === 0) {
            studentDropdown.innerHTML = '<div class="p-2 text-muted">No students found</div>';
        } else {
            studentDropdown.innerHTML = filteredStudents.map(student => {
                const escapedName = student.full_name.replace(/'/g, "\\'").replace(/"/g, '\\"');
                return `
                    <div class="p-2 border-bottom student-option" style="cursor: pointer;" 
                         data-student-id="${student.id}"
                         data-student-name="${escapedName}"
                         onmouseover="this.style.backgroundColor='#f8f9fa'" 
                         onmouseout="this.style.backgroundColor='white'">
                        <div class="fw-semibold">${student.full_name}</div>
                        <small class="text-muted">ID: ${student.id} | Code: ${student.login_code || 'N/A'}</small>
                    </div>
                `;
            }).join('');
        }

        studentDropdown.style.display = 'block';
        
        // Add click event listeners to student options
        studentDropdown.querySelectorAll('.student-option').forEach(option => {
            option.addEventListener('click', (e) => {
                const studentId = option.getAttribute('data-student-id');
                const studentName = option.getAttribute('data-student-name');
                this.selectStudent(parseInt(studentId), studentName);
            });
        });
    }

    selectStudent(studentId, studentName) {
        const studentSearch = document.getElementById('studentSearch');
        const studentDropdown = document.getElementById('studentDropdown');

        console.log('selectStudent called with:', { studentId, studentName });

        // Set values using data attribute
        studentSearch.value = studentName;
        studentSearch.setAttribute('data-student-id', studentId.toString());
        
        // Mark field as valid
        studentSearch.classList.remove('is-invalid');
        studentSearch.classList.add('is-valid');
        
        // Hide dropdown
        studentDropdown.style.display = 'none';
        
        // Debug log
        console.log('Student selected - after setting:', { 
            studentId, 
            studentName, 
            dataAttribute: studentSearch.getAttribute('data-student-id')
        });
    }

    async saveResult() {
        try {
            const examId = document.getElementById('resultExamId').value;
            const studentSearch = document.getElementById('studentSearch');
            const studentId = studentSearch.getAttribute('data-student-id');
            const score = document.getElementById('resultScore').value;
            const studentName = studentSearch.value;

            // Debug log
            console.log('Save result values:', { examId, studentId, score, studentName });

            if (!examId) {
                this.showToast('error', 'Please select an exam');
                return;
            }

            if (!studentId || studentId === '') {
                console.log('Student ID is empty from data attribute:', studentId);
                this.showToast('error', 'Please select a student from the dropdown');
                return;
            }

            if (!score || score === '') {
                this.showToast('error', 'Please enter a score');
                return;
            }

            // Validate score range (0 to max)
            const examSelect = document.getElementById('resultExamId');
            const selectedOption = examSelect.options[examSelect.selectedIndex];
            const maxScore = parseFloat(selectedOption.getAttribute('data-max-score'));

            if (parseFloat(score) < 0 || parseFloat(score) > maxScore) {
                this.showToast('error', `Score must be between 0 and ${maxScore}`);
                return;
            }

            const resultData = {
                exam_id: parseInt(examId),
                student_id: parseInt(studentId),
                score: parseFloat(score)
            };

            await this.apiCall('POST', '/admin/results', resultData);
            this.showToast('success', 'Result added successfully');
            this.closeModal();
            this.loadResultsData();

        } catch (error) {
            this.showToast('error', 'Failed to save result: ' + error.message);
        }
    }

    async editResult(id) {
        try {
            const result = await this.apiCall('GET', `/admin/results/${id}`);
            
            const modal = this.createModal('Edit Result', `
                <form id="editResultForm">
                    <div class="mb-3">
                        <label class="form-label">Student</label>
                        <input type="text" class="form-control" value="${result.data.student_name} (ID: ${result.data.student_id})" readonly>
                    </div>
                    <div class="mb-3">
                        <label class="form-label">Exam</label>
                        <input type="text" class="form-control" value="${result.data.exam_title}" readonly>
                    </div>
                    <div class="mb-3">
                        <label class="form-label">Score *</label>
                        <input type="number" class="form-control" id="editResultScore" step="0.01" min="0" value="${result.data.score}" required>
                        <div class="form-text">Range: ${result.data.min_score} - ${result.data.max_score}</div>
                    </div>
                    <div class="mb-3">
                        <label class="form-label">Current Percentage</label>
                        <input type="text" class="form-control" value="${result.data.percentage}%" readonly>
                    </div>
                </form>
            `, `
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                <button type="button" class="btn btn-primary" onclick="app.updateResult(${id})">Update Result</button>
            `);

            modal.show();
        } catch (error) {
            this.showToast('error', 'Failed to load result: ' + error.message);
        }
    }

    async updateResult(id) {
        try {
            const score = document.getElementById('editResultScore').value;

            if (!score) {
                this.showToast('error', 'Please enter a score');
                return;
            }

            const resultData = {
                score: parseFloat(score)
            };

            await this.apiCall('PUT', `/admin/results/${id}`, resultData);
            this.showToast('success', 'Result updated successfully');
            this.closeModal();
            this.loadResultsData();

        } catch (error) {
            this.showToast('error', 'Failed to update result: ' + error.message);
        }
    }

    async deleteResult(id) {
        if (!confirm('Are you sure you want to delete this result? This action cannot be undone.')) {
            return;
        }

        try {
            await this.apiCall('DELETE', `/admin/results/${id}`);
            this.showToast('success', 'Result deleted successfully');
            this.loadResultsData();
        } catch (error) {
            this.showToast('error', 'Failed to delete result: ' + error.message);
        }
    }

    async bulkDeleteResults() {
        const checkedBoxes = document.querySelectorAll('.result-checkbox:checked');
        if (checkedBoxes.length === 0) {
            this.showToast('warning', 'Please select results to delete');
            return;
        }

        if (!confirm(`Are you sure you want to delete ${checkedBoxes.length} result(s)? This action cannot be undone.`)) {
            return;
        }

        try {
            const resultIds = Array.from(checkedBoxes).map(cb => parseInt(cb.value));
            await this.apiCall('DELETE', '/admin/results/bulk', { result_ids: resultIds });
            this.showToast('success', `${checkedBoxes.length} result(s) deleted successfully`);
            this.loadResultsData();
        } catch (error) {
            this.showToast('error', 'Failed to delete results: ' + error.message);
        }
    }

    async showImportResultsModal() {
        const modal = this.createModal('Import Results', `
            <div class="mb-4">
                <h6><i class="bi bi-download"></i> Step 1: Download Template</h6>
                <div class="mb-3">
                    <label class="form-label">Select Exam for Template *</label>
                    <select class="form-select" id="templateExamId" onchange="app.toggleDownloadButton()">
                        <option value="">Select Exam</option>
                    </select>
                    <div class="form-text">Choose the exam to download a CSV template with enrolled students</div>
                </div>
                <button type="button" class="btn btn-outline-primary" id="downloadTemplateBtn" disabled onclick="app.downloadResultsTemplate()">
                    <i class="bi bi-download"></i> Download CSV Template
                </button>
                <div class="form-text mt-2">
                    Template includes: student_name, student_id, national_id, login_code, score (empty for you to fill)
                </div>
            </div>
            
            <hr>
            
            <div class="mb-3">
                <h6><i class="bi bi-upload"></i> Step 2: Upload Completed File</h6>
                <div class="mb-3">
                    <label class="form-label">Import File (CSV)</label>
                    <input type="file" class="form-control" id="importResultsFile" accept=".csv">
                    <div class="form-text">
                        Upload the CSV template you downloaded and filled with scores
                    </div>
                </div>
                <div class="mb-3">
                    <label class="form-label">Exam for Import *</label>
                    <select class="form-select" id="importExamId" required>
                        <option value="">Select Exam</option>
                    </select>
                    <div class="form-text">All results will be imported for this exam</div>
                </div>
            </div>
            
            <div class="alert alert-info">
                <h6><i class="bi bi-info-circle"></i> Import Guidelines:</h6>
                <ul class="mb-0">
                    <li>Required columns: <code>score</code> and at least one student identifier</li>
                    <li>Student identifiers: <code>student_id</code>, <code>login_code</code></li>
                    <li>Optional verification: <code>national_id</code></li>
                    <li>Scores must be between 0 and the exam's maximum score</li>
                    <li>Students must be enrolled in the appropriate grade/year</li>
                    <li>Leave score blank to skip that student</li>
                </ul>
            </div>
        `, `
            <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
            <button type="button" class="btn btn-primary" onclick="app.importResults()">Import Results</button>
        `);

        // Load exams for selection in both dropdowns
        try {
            const examsRes = await this.apiCall('GET', '/admin/exams');
            const templateExamSelect = document.getElementById('templateExamId');
            const importExamSelect = document.getElementById('importExamId');
            
            examsRes.data.forEach(exam => {
                const optionText = `${exam.title} - ${exam.year_name} ${exam.term_name} (${exam.grade_title})`;
                templateExamSelect.innerHTML += `<option value="${exam.id}">${optionText}</option>`;
                importExamSelect.innerHTML += `<option value="${exam.id}">${optionText}</option>`;
            });
        } catch (error) {
            console.error('Failed to load exams:', error);
        }

        modal.show();
    }

    toggleDownloadButton() {
        const examId = document.getElementById('templateExamId').value;
        const downloadBtn = document.getElementById('downloadTemplateBtn');
        downloadBtn.disabled = !examId;
    }

    async downloadResultsTemplate() {
        const examId = document.getElementById('templateExamId').value;
        
        if (!examId) {
            this.showToast('error', 'Please select an exam first');
            return;
        }

        try {
            // First, debug what data we have
            console.log('Debugging template data for exam:', examId);
            const debugResponse = await this.apiCall('GET', `/admin/results/template-debug/${examId}`);
            console.log('Debug data:', debugResponse.data);
            
            if (debugResponse.data.student_count === 0) {
                this.showToast('error', 'No enrolled students found for this exam. Please check enrollments.');
                return;
            }

            // Create a temporary link to download the CSV
            const response = await fetch(`/student1/api/download-template.php?exam_id=${examId}`, {
                method: 'GET',
                headers: {
                    'Authorization': `Bearer ${this.token}`
                }
            });

            if (!response.ok) {
                throw new Error(`HTTP ${response.status}: ${response.statusText}`);
            }

            // Get filename from header or create default
            const contentDisposition = response.headers.get('Content-Disposition');
            let filename = 'results_template.csv';
            if (contentDisposition) {
                const matches = contentDisposition.match(/filename="([^"]+)"/);
                if (matches) {
                    filename = matches[1];
                }
            }

            // Create blob and download
            const blob = await response.blob();
            
            // Debug: Check blob content
            const text = await blob.text();
            console.log('CSV content:', text.substring(0, 500) + (text.length > 500 ? '...' : ''));
            
            const url = window.URL.createObjectURL(blob);
            const link = document.createElement('a');
            link.href = url;
            link.download = filename;
            document.body.appendChild(link);
            link.click();
            document.body.removeChild(link);
            window.URL.revokeObjectURL(url);

            this.showToast('success', `Template downloaded successfully with ${debugResponse.data.student_count} students`);
        } catch (error) {
            this.showToast('error', 'Failed to download template: ' + error.message);
        }
    }

    async importResults() {
        const file = document.getElementById('importResultsFile').files[0];
        const examId = document.getElementById('importExamId').value;

        if (!file) {
            this.showToast('error', 'Please select a file to import');
            return;
        }

        if (!examId) {
            this.showToast('error', 'Please select an exam');
            return;
        }

        try {
            const formData = new FormData();
            formData.append('file', file);
            
            // Debug the file and formData
            console.log('File to upload:', file.name, file.size, file.type);
            console.log('FormData entries:');
            for (let pair of formData.entries()) {
                console.log(pair[0], pair[1]);
            }

            const response = await this.apiCall('POST', `/admin/results/import?exam_id=${examId}`, formData, false);
            
            this.showToast('success', `Import completed: ${response.data.successful} successful, ${response.data.failed} failed`);
            
            if (response.data.errors && response.data.errors.length > 0) {
                console.log('Import errors:', response.data.errors);
            }
            
            this.closeModal();
            this.loadResultsData();

        } catch (error) {
            this.showToast('error', 'Failed to import results: ' + error.message);
        }
    }

    async exportResults() {
        try {
            const filters = {
                year_id: document.getElementById('yearFilter')?.value || '',
                term_id: document.getElementById('termFilter')?.value || '',
                grade_id: document.getElementById('gradeFilter')?.value || '',
                subject_id: document.getElementById('subjectFilter')?.value || '',
                exam_id: document.getElementById('examFilter')?.value || '',
                q: document.getElementById('searchResults')?.value || ''
            };

            const params = new URLSearchParams({
                export: 'csv',
                ...Object.fromEntries(Object.entries(filters).filter(([_, v]) => v))
            });

            // Create a temporary link to download the file
            const link = document.createElement('a');
            link.href = `${this.apiBase}/admin/results?${params}`;
            link.download = `results_export_${new Date().toISOString().split('T')[0]}.csv`;
            link.click();

            this.showToast('success', 'Export initiated');

        } catch (error) {
            this.showToast('error', 'Failed to export results: ' + error.message);
        }
    }

    sortResults(field) {
        // This would implement sorting logic - for now just reload data
        this.loadResultsData();
    }

    // Modal utility methods for results management
    createModal(title, body, footer = '', size = '') {
        // Create a unique modal ID
        const modalId = 'dynamicModal_' + Date.now();
        const sizeClass = size ? `modal-${size}` : '';
        
        // Create modal HTML
        const modalHtml = `
            <div class="modal fade" id="${modalId}" tabindex="-1">
                <div class="modal-dialog ${sizeClass}">
                    <div class="modal-content">
                        <div class="modal-header">
                            <h5 class="modal-title">${title}</h5>
                            <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                        </div>
                        <div class="modal-body">
                            ${body}
                        </div>
                        ${footer ? `<div class="modal-footer">${footer}</div>` : ''}
                    </div>
                </div>
            </div>
        `;
        
        // Add modal to DOM
        document.body.insertAdjacentHTML('beforeend', modalHtml);
        const modalElement = document.getElementById(modalId);
        
        // Store current modal ID for cleanup
        this.currentModalId = modalId;
        
        // Add close handlers
        const closeButtons = modalElement.querySelectorAll('[data-bs-dismiss="modal"]');
        closeButtons.forEach(btn => {
            btn.onclick = (e) => {
                e.preventDefault();
                this.closeModal();
            };
        });
        
        return {
            show: () => {
                modalElement.style.display = 'flex';
                modalElement.classList.add('show');
                document.body.classList.add('modal-open');
            },
            hide: () => {
                this.closeModal();
            }
        };
    }

    closeModal() {
        if (this.currentModalId) {
            const modalElement = document.getElementById(this.currentModalId);
            if (modalElement) {
                modalElement.style.display = 'none';
                modalElement.classList.remove('show');
                
                // Remove modal from DOM after animation
                setTimeout(() => {
                    modalElement.remove();
                }, 150);
            }
            this.currentModalId = null;
        }
        
        document.body.classList.remove('modal-open');
        document.querySelectorAll('.modal-backdrop').forEach(backdrop => backdrop.remove());
    }

    async showBulkResultsModal() {
        try {
            // Load exams and classes for selection
            const [examsRes, classesRes] = await Promise.all([
                this.apiCall('GET', '/admin/exams'),
                this.apiCall('GET', '/admin/classes')
            ]);
            
            const modal = this.createModal('Bulk Results Entry', `
                <div class="row">
                    <div class="col-md-8">
                        <div class="mb-3">
                            <label class="form-label">Select Exam *</label>
                            <select class="form-select" id="bulkExamId" required onchange="app.loadExamStudents()">
                                <option value="">Select Exam</option>
                                ${examsRes.data.map(exam => 
                                    `<option value="${exam.id}">${exam.title} - ${exam.year_name} ${exam.term_name} (${exam.grade_title})</option>`
                                ).join('')}
                            </select>
                            <div class="form-text">All enrolled students for the selected exam will be shown below</div>
                        </div>
                    </div>
                    <div class="col-md-4">
                        <div class="mb-3">
                            <label class="form-label">Filter by Class</label>
                            <select class="form-select" id="bulkClassFilter" onchange="app.filterStudentsByClass()">
                                <option value="">All Classes</option>
                                ${classesRes.data.map(cls => 
                                    `<option value="${cls.id}">${cls.name} (${cls.grade_title})</option>`
                                ).join('')}
                            </select>
                            <div class="form-text">Show only students from selected class</div>
                        </div>
                    </div>
                </div>
                
                <div id="bulkResultsContainer" style="display: none;">
                    <div class="mb-3">
                        <div class="d-flex justify-content-between align-items-center">
                            <h6 class="mb-0">Student Results</h6>
                            <button type="button" class="btn btn-sm btn-outline-secondary" onclick="app.clearAllScores()">
                                Clear All Scores
                            </button>
                        </div>
                        <small class="text-muted">Enter scores for each student. Leave blank to skip.</small>
                    </div>
                    
                    <div class="table-responsive" style="max-height: 400px; overflow-y: auto;">
                        <table class="table table-sm">
                            <thead class="table-light sticky-top">
                                <tr>
                                    <th>Student</th>
                                    <th>Current Score</th>
                                    <th width="150">New Score</th>
                                </tr>
                            </thead>
                            <tbody id="bulkStudentsTableBody">
                                <!-- Students will be loaded here -->
                            </tbody>
                        </table>
                    </div>
                    
                    <div class="mt-3">
                        <div class="row">
                            <div class="col-md-6">
                                <small class="text-muted">
                                    Score Range: <span id="scoreRange">Select exam to see range</span>
                                </small>
                            </div>
                            <div class="col-md-6 text-end">
                                <small class="text-muted">
                                    Entries: <span id="entryCount">0</span> students
                                </small>
                            </div>
                        </div>
                    </div>
                </div>
            `, `
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                <button type="button" class="btn btn-primary" onclick="app.saveBulkResults()" disabled id="saveBulkBtn">Save Results</button>
            `, 'modal-lg');

            modal.show();
        } catch (error) {
            this.showToast('error', 'Failed to load bulk results form: ' + error.message);
        }
    }

    async loadExamStudents() {
        const examId = document.getElementById('bulkExamId').value;
        const classId = document.getElementById('bulkClassFilter')?.value || '';
        const container = document.getElementById('bulkResultsContainer');
        const saveBtn = document.getElementById('saveBulkBtn');
        
        if (!examId) {
            container.style.display = 'none';
            saveBtn.disabled = true;
            return;
        }

        try {
            // Get exam details and students with optional class filter
            let url = `/admin/results/exam/${examId}`;
            if (classId) {
                url += `?class_id=${classId}`;
            }
            const response = await this.apiCall('GET', url);
            const exam = response.data.exam;
            const studentResults = response.data.student_results;

            // Show container and update score range
            container.style.display = 'block';
            document.getElementById('scoreRange').innerHTML = `<strong>0 - ${exam.max_score}</strong> (Min for pass: ${exam.min_score})`;
            
            // Update entry count with class filter info
            const classFilterText = classId ? ' (filtered by class)' : '';
            document.getElementById('entryCount').textContent = `${studentResults.length} students${classFilterText}`;

            // Render students table
            const tbody = document.getElementById('bulkStudentsTableBody');
            tbody.innerHTML = studentResults.map(student => `
                <tr>
                    <td>
                        <div class="fw-semibold">${student.student_name}</div>
                        <small class="text-muted">ID: ${student.student_id} ${student.class_name ? '• ' + student.class_name : ''}</small>
                    </td>
                    <td>
                        ${student.status === 'completed' ? 
                            `<span class="badge bg-success">${student.score}</span>` : 
                            '<span class="text-muted">Not entered</span>'
                        }
                    </td>
                    <td>
                        <input type="number" 
                               class="form-control form-control-sm bulk-score-input" 
                               data-student-id="${student.student_id}"
                               value="${student.status === 'completed' ? student.score : ''}"
                               min="0" 
                               max="${exam.max_score}" 
                               step="0.01"
                               placeholder="0 - ${exam.max_score}">
                    </td>
                </tr>
            `).join('');

            // Enable save button
            saveBtn.disabled = false;

        } catch (error) {
            container.style.display = 'none';
            saveBtn.disabled = true;
            this.showToast('error', 'Failed to load exam students: ' + error.message);
        }
    }

    clearAllScores() {
        document.querySelectorAll('.bulk-score-input').forEach(input => {
            input.value = '';
        });
    }

    filterStudentsByClass() {
        // Reload students with class filter when class selection changes
        this.loadExamStudents();
    }

    async saveBulkResults() {
        const examId = document.getElementById('bulkExamId').value;
        const scoreInputs = document.querySelectorAll('.bulk-score-input');
        
        const results = [];
        scoreInputs.forEach(input => {
            const score = input.value;
            const studentId = input.getAttribute('data-student-id');
            
            if (score && score.trim() !== '') {
                results.push({
                    student_id: parseInt(studentId),
                    score: parseFloat(score)
                });
            }
        });

        if (results.length === 0) {
            this.showToast('warning', 'Please enter at least one score');
            return;
        }

        try {
            const response = await this.apiCall('POST', '/admin/results/bulk', {
                exam_id: parseInt(examId),
                results: results
            });

            this.showToast('success', `Bulk entry completed: ${response.data.successful} successful, ${response.data.failed} failed`);
            
            if (response.data.errors && response.data.errors.length > 0) {
                console.log('Bulk entry errors:', response.data.errors);
            }
            
            this.closeModal();
            this.loadResultsData();

        } catch (error) {
            this.showToast('error', 'Failed to save bulk results: ' + error.message);
        }
    }

    async showStudentResultsModal() {
        try {
            // Don't load all students at once - use AJAX search instead
            const modal = this.createModal('Student Results Entry', `
                <div class="mb-3">
                    <label class="form-label">Select Student *</label>
                    <div class="position-relative">
                        <input type="text" class="form-control" id="studentResultsSearch" placeholder="Type at least 2 characters to search students..." data-student-id="">
                        <div id="studentResultsDropdown" class="position-absolute w-100 bg-white border border-top-0 rounded-bottom shadow-sm" style="max-height: 200px; overflow-y: auto; z-index: 9999; display: none; top: 100%;">
                            <!-- Student options will appear here -->
                        </div>
                    </div>
                    <div class="form-text" id="studentResultsHelperText">Type at least 2 characters to search students, then select to view their available exams</div>
                </div>
                
                <div id="studentExamsContainer" style="display: none;">
                    <div class="mb-3">
                        <div class="d-flex justify-content-between align-items-center">
                            <h6 class="mb-0">Available Exams</h6>
                            <div>
                                <button type="button" class="btn btn-sm btn-outline-secondary" onclick="app.clearAllStudentScores()">
                                    Clear All
                                </button>
                                <button type="button" class="btn btn-sm btn-outline-primary" onclick="app.fillMissingOnly()">
                                    Fill Missing Only
                                </button>
                            </div>
                        </div>
                        <small class="text-muted">Enter scores for the exams you want to submit. Leave blank to skip.</small>
                    </div>
                    
                    <div class="table-responsive" style="max-height: 400px; overflow-y: auto;">
                        <table class="table table-sm">
                            <thead class="table-light sticky-top">
                                <tr>
                                    <th width="30%">Exam</th>
                                    <th width="15%">Subject</th>
                                    <th width="15%">Date</th>
                                    <th width="15%">Current Result</th>
                                    <th width="25%">New Score</th>
                                </tr>
                            </thead>
                            <tbody id="studentExamsTableBody">
                                <!-- Student exams will be loaded here -->
                            </tbody>
                        </table>
                    </div>
                    
                    <div class="mt-3">
                        <div class="row">
                            <div class="col-md-6">
                                <small class="text-muted">
                                    Student: <span id="selectedStudentInfo">None selected</span>
                                </small>
                            </div>
                            <div class="col-md-6 text-end">
                                <small class="text-muted">
                                    Available Exams: <span id="availableExamsCount">0</span>
                                </small>
                            </div>
                        </div>
                    </div>
                </div>
            `, `
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                <button type="button" class="btn btn-primary" onclick="app.saveStudentResults()" disabled id="saveStudentResultsBtn">Save Results</button>
            `, 'modal-lg');

            // Load percentage bands for band calculations
            await this.getPercentageBands();
            
            // Setup AJAX student search
            this.setupAjaxStudentResultsSearch();

            modal.show();
        } catch (error) {
            this.showToast('error', 'Failed to load student results form: ' + error.message);
        }
    }

    setupAjaxStudentResultsSearch() {
        const studentSearch = document.getElementById('studentResultsSearch');
        const studentDropdown = document.getElementById('studentResultsDropdown');
        
        if (!studentSearch) return;

        let searchTimeout;

        // Add event listeners
        studentSearch.addEventListener('input', (e) => {
            const searchTerm = e.target.value.trim();
            
            // Clear previous timeout
            clearTimeout(searchTimeout);
            
            if (searchTerm.length < 2) {
                // Hide dropdown for searches less than 2 characters
                studentDropdown.style.display = 'none';
                return;
            }
            
            // Debounce search - wait 300ms after user stops typing
            searchTimeout = setTimeout(() => {
                this.searchStudentsAjax(searchTerm);
            }, 300);
        });

        studentSearch.addEventListener('focus', (e) => {
            const searchTerm = e.target.value.trim();
            if (searchTerm.length >= 2) {
                this.searchStudentsAjax(searchTerm);
            }
        });

        studentSearch.addEventListener('blur', (e) => {
            setTimeout(() => {
                studentDropdown.style.display = 'none';
            }, 200);
        });
    }

    async searchStudentsAjax(searchTerm) {
        const studentDropdown = document.getElementById('studentResultsDropdown');
        
        try {
            // Show loading state
            studentDropdown.innerHTML = '<div class="p-2 text-muted"><i class="bi bi-hourglass-split me-1"></i>Searching...</div>';
            studentDropdown.style.display = 'block';
            
            // Search with limit of 10 students max
            const response = await this.apiCall('GET', `/admin/students?enrolled_only=true&q=${encodeURIComponent(searchTerm)}&per_page=10`);
            const students = response.data;

            if (!students || students.length === 0) {
                studentDropdown.innerHTML = '<div class="p-2 text-muted">No students found</div>';
            } else {
                studentDropdown.innerHTML = students.map(student => {
                    const escapedName = student.full_name.replace(/'/g, "\\'").replace(/"/g, '\\"');
                    return `
                        <div class="p-2 border-bottom student-option" style="cursor: pointer;" 
                             data-student-id="${student.id}"
                             data-student-name="${escapedName}"
                             data-grade-title="${student.grade_title || 'N/A'}"
                             data-year-name="${student.year_name || 'N/A'}"
                             onmouseover="this.style.backgroundColor='#f8f9fa'" 
                             onmouseout="this.style.backgroundColor='white'">
                            <div class="fw-semibold">${student.full_name}</div>
                            <small class="text-muted">ID: ${student.id} | ${student.grade_title || 'No Grade'} - ${student.year_name || 'No Year'}</small>
                        </div>
                    `;
                }).join('');
                
                // Add "Show more" indicator if we hit the limit
                if (students.length === 10) {
                    studentDropdown.innerHTML += '<div class="p-2 text-center text-muted border-top"><small><i class="bi bi-info-circle me-1"></i>Showing first 10 matches. Type more to refine search.</small></div>';
                }
            }

            studentDropdown.style.display = 'block';
            
            // Add click event listeners
            studentDropdown.querySelectorAll('.student-option').forEach(option => {
                option.addEventListener('click', (e) => {
                    const studentId = option.getAttribute('data-student-id');
                    const studentName = option.getAttribute('data-student-name');
                    const gradeTitle = option.getAttribute('data-grade-title');
                    const yearName = option.getAttribute('data-year-name');
                    this.selectStudentForResults(parseInt(studentId), studentName, gradeTitle, yearName);
                });
            });
            
        } catch (error) {
            console.error('Error searching students:', error);
            studentDropdown.innerHTML = '<div class="p-2 text-danger">Error searching students</div>';
            studentDropdown.style.display = 'block';
        }
    }

    async selectStudentForResults(studentId, studentName, gradeTitle, yearName) {
        const studentSearch = document.getElementById('studentResultsSearch');
        const studentDropdown = document.getElementById('studentResultsDropdown');
        const container = document.getElementById('studentExamsContainer');
        const saveBtn = document.getElementById('saveStudentResultsBtn');
        const helperText = document.getElementById('studentResultsHelperText');
        const selectedStudentInfo = document.getElementById('selectedStudentInfo');

        // Set values
        studentSearch.value = studentName;
        studentSearch.setAttribute('data-student-id', studentId.toString());
        studentDropdown.style.display = 'none';
        
        // Update info
        selectedStudentInfo.textContent = `${studentName} (${gradeTitle} - ${yearName})`;
        helperText.innerHTML = `<i class="bi bi-hourglass-split"></i> Loading available exams...`;

        try {
            // Load student's available exams
            const response = await this.apiCall('GET', `/admin/students/${studentId}/exams`);
            const exams = response.data;

            if (exams && exams.length > 0) {
                container.style.display = 'block';
                document.getElementById('availableExamsCount').textContent = exams.length;
                
                // Render exams table
                const tbody = document.getElementById('studentExamsTableBody');
                tbody.innerHTML = exams.map(exam => `
                    <tr>
                        <td>
                            <div class="fw-semibold">${exam.title}</div>
                            <small class="text-muted">${exam.term_name}</small>
                        </td>
                        <td>${exam.subject_name}</td>
                        <td>
                            <small>${exam.exam_date ? new Date(exam.exam_date).toLocaleDateString() : 'N/A'}</small>
                        </td>
                        <td>
                            ${exam.current_score !== null ? 
                                `<div>
                                    <span class="badge bg-primary">${exam.current_score}/${exam.max_score}</span>
                                    ${exam.current_percentage !== null ? 
                                        `<div class="mt-1">
                                            <span class="badge" style="background-color: ${exam.band_color || '#6c757d'}; color: white;">
                                                ${exam.band_label || Math.round(exam.current_percentage) + '%'}
                                            </span>
                                        </div>` : ''
                                    }
                                </div>` : 
                                '<span class="text-muted">Not entered</span>'
                            }
                        </td>
                        <td>
                            <input type="number" 
                                   class="form-control form-control-sm student-exam-score-input" 
                                   data-exam-id="${exam.id}"
                                   data-result-id="${exam.result_id || ''}"
                                   value="${exam.current_score || ''}"
                                   min="0" 
                                   max="${exam.max_score}" 
                                   step="0.01"
                                   placeholder="0 - ${exam.max_score}">
                        </td>
                    </tr>
                `).join('');

                saveBtn.disabled = false;
                helperText.textContent = `${exams.length} exams available for this student`;

            } else {
                container.style.display = 'none';
                saveBtn.disabled = true;
                helperText.textContent = 'No exams found for this student';
            }

        } catch (error) {
            console.error('Error loading student exams:', error);
            container.style.display = 'none';
            saveBtn.disabled = true;
            helperText.textContent = 'Error loading exams for this student';
            this.showToast('error', 'Failed to load student exams: ' + error.message);
        }
    }

    clearAllStudentScores() {
        document.querySelectorAll('.student-exam-score-input').forEach(input => {
            input.value = '';
        });
    }

    // Helper method to get percentage band info
    async getPercentageBands() {
        try {
            if (!this.percentageBands) {
                const response = await this.apiCall('GET', '/admin/percentage-bands');
                this.percentageBands = response.data.filter(band => band.enabled);
                this.percentageBands.sort((a, b) => a.start_percent - b.start_percent);
            }
            return this.percentageBands;
        } catch (error) {
            console.error('Error loading percentage bands:', error);
            return [];
        }
    }

    // Get band for a specific percentage
    getBandForPercentage(percentage) {
        if (!this.percentageBands) return null;
        
        return this.percentageBands.find(band => 
            percentage >= band.start_percent && percentage <= band.end_percent
        ) || null;
    }

    fillMissingOnly() {
        document.querySelectorAll('.student-exam-score-input').forEach(input => {
            const currentValue = input.value;
            const originalScore = input.getAttribute('data-original-score') || '';
            
            // Only clear inputs that already have scores (to focus on missing ones)
            if (currentValue && currentValue.trim() !== '') {
                input.value = '';
            }
        });
        
        this.showToast('info', 'Cleared existing scores - now only enter missing results');
    }

    async saveStudentResults() {
        const studentSearch = document.getElementById('studentResultsSearch');
        const studentId = studentSearch.getAttribute('data-student-id');
        const scoreInputs = document.querySelectorAll('.student-exam-score-input');
        
        if (!studentId) {
            this.showToast('error', 'Please select a student');
            return;
        }

        // Collect all score entries
        const results = [];
        const updates = [];
        
        scoreInputs.forEach(input => {
            const score = input.value;
            const examId = input.getAttribute('data-exam-id');
            const resultId = input.getAttribute('data-result-id');
            
            if (score && score.trim() !== '') {
                if (resultId) {
                    // Update existing result
                    updates.push({
                        result_id: parseInt(resultId),
                        score: parseFloat(score)
                    });
                } else {
                    // Create new result
                    results.push({
                        exam_id: parseInt(examId),
                        student_id: parseInt(studentId),
                        score: parseFloat(score)
                    });
                }
            }
        });

        if (results.length === 0 && updates.length === 0) {
            this.showToast('warning', 'Please enter at least one score');
            return;
        }

        try {
            let successCount = 0;
            let failCount = 0;
            const errors = [];

            // Process new results
            if (results.length > 0) {
                try {
                    const createResponse = await this.apiCall('POST', '/admin/results/bulk', {
                        exam_id: null, // Will be ignored since we provide individual exam_ids
                        results: results
                    });
                    successCount += createResponse.data.successful || 0;
                    failCount += createResponse.data.failed || 0;
                    if (createResponse.data.errors) {
                        errors.push(...createResponse.data.errors);
                    }
                } catch (error) {
                    failCount += results.length;
                    errors.push({ error: error.message, type: 'create' });
                }
            }

            // Process updates
            for (const update of updates) {
                try {
                    await this.apiCall('PUT', `/admin/results/${update.result_id}`, {
                        score: update.score
                    });
                    successCount++;
                } catch (error) {
                    failCount++;
                    errors.push({ result_id: update.result_id, error: error.message, type: 'update' });
                }
            }

            const totalOperations = results.length + updates.length;
            this.showToast('success', 
                `Student results saved: ${successCount} successful, ${failCount} failed out of ${totalOperations} operations`
            );
            
            if (errors.length > 0) {
                console.log('Student results errors:', errors);
            }
            
            this.closeModal();
            this.loadResultsData();

        } catch (error) {
            this.showToast('error', 'Failed to save student results: ' + error.message);
        }
    }

    // ===================
    // PAYMENTS SECTION
    // ===================

    async loadPayments() {
        const content = `
            <div class="page-header">
                <div class="d-flex justify-content-between align-items-center">
                    <div>
                        <h2><i class="bi bi-credit-card"></i> <span data-translate="payments.title">Payments</span></h2>
                        <p class="text-muted" data-translate="payments.subtitle">Manage student payments and fee tracking</p>
                    </div>
                </div>
            </div>

            <div class="card">
                <div class="card-body">
                    <div class="row">
                        <div class="col-md-12">
                            <h5 data-translate="payments.search_options">Search Options</h5>
                            <p class="text-muted small" data-translate="payments.search_help">Choose how you want to search for payments:</p>
                            
                            <div class="row mb-3">
                                <div class="col-md-6">
                                    <div class="card">
                                        <div class="card-body p-3">
                                            <h6 class="card-title"><i class="bi bi-person-search"></i> <span data-translate="payments.student_search_pay">Student Search & Pay</span></h6>
                                            <p class="small text-muted" data-translate="payments.student_search_desc">Find specific students to view their payment timeline and record new payments</p>
                                            <div class="input-group mb-2">
                                                <input type="text" id="studentSearchInput" class="form-control" 
                                                       data-translate-placeholder="payments.search_placeholder" placeholder="Search by name, student ID, or national ID...">
                                                <button class="btn btn-primary" onclick="app.searchStudentsForPayment()">
                                                    <i class="bi bi-search"></i> <span data-translate="payments.search_students">Search Students</span>
                                                </button>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                                <div class="col-md-6">
                                    <div class="card">
                                        <div class="card-body p-3">
                                            <h6 class="card-title"><i class="bi bi-clock-history"></i> <span data-translate="payments.payment_history_search">Payment History Search</span></h6>
                                            <p class="small text-muted" data-translate="payments.payment_history_desc">Search all payments across students using filters (Year, Grade, Class, Date, Amount)</p>
                                            <button class="btn btn-info w-100" onclick="app.searchPaymentHistory()">
                                                <i class="bi bi-search"></i> <span data-translate="payments.search_all_history">Search All Payment History</span>
                                            </button>
                                        </div>
                                    </div>
                                </div>
                            </div>
                            
                            <div class="row">
                                <div class="col-md-12">
                                    <button class="btn btn-outline-success" onclick="app.toggleAdvancedFilters()">
                                        <i class="bi bi-chevron-down"></i> <span data-translate="payments.show_advanced_filters">Show Advanced Filters</span>
                                    </button>
                                </div>
                            </div>
                            
                            <!-- Advanced Filters (Hidden by default) -->
                            <div id="advancedFilters" class="row" style="display: none;">
                                <div class="col-md-3">
                                    <label class="form-label" data-translate="payments.academic_year">Academic Year</label>
                                    <select class="form-select mb-2" id="paymentYearFilter" onchange="app.loadPaymentClasses()">
                                        <option value="" data-translate="payments.all_years">All Years</option>
                                    </select>
                                </div>
                                <div class="col-md-3">
                                    <label class="form-label" data-translate="payments.grade">Grade</label>
                                    <select class="form-select mb-2" id="paymentGradeFilter" onchange="app.loadPaymentClasses()">
                                        <option value="" data-translate="payments.all_grades">All Grades</option>
                                    </select>
                                </div>
                                <div class="col-md-3">
                                    <label class="form-label" data-translate="payments.class">Class</label>
                                    <select class="form-select mb-2" id="paymentClassFilter">
                                        <option value="" data-translate="payments.all_classes">All Classes</option>
                                    </select>
                                </div>
                                <div class="col-md-3">
                                    <label class="form-label" data-translate="payments.payment_method">Payment Method</label>
                                    <select class="form-select mb-2" id="paymentMethodFilter">
                                        <option value="" data-translate="payments.all_methods">All Methods</option>
                                        <option value="cash" data-translate="payments.cash">Cash</option>
                                        <option value="card" data-translate="payments.card">Card</option>
                                        <option value="bank" data-translate="payments.bank_transfer">Bank Transfer</option>
                                        <option value="wallet" data-translate="payments.digital_wallet">Digital Wallet</option>
                                    </select>
                                </div>
                                <div class="col-md-3">
                                    <label class="form-label" data-translate="payments.date_from">Date From</label>
                                    <input type="date" class="form-control mb-2" id="dateFromFilter">
                                </div>
                                <div class="col-md-3">
                                    <label class="form-label" data-translate="payments.date_to">Date To</label>
                                    <input type="date" class="form-control mb-2" id="dateToFilter">
                                </div>
                                <div class="col-md-3">
                                    <label class="form-label" data-translate="payments.min_amount">Min Amount</label>
                                    <input type="number" class="form-control mb-2" id="minAmountFilter" step="0.01" placeholder="0.00">
                                </div>
                                <div class="col-md-3">
                                    <label class="form-label" data-translate="payments.max_amount">Max Amount</label>
                                    <input type="number" class="form-control mb-2" id="maxAmountFilter" step="0.01" placeholder="1000.00">
                                </div>
                                <div class="col-md-6">
                                    <label class="form-label" data-translate="payments.quick_date_filters">Quick Date Filters</label>
                                    <div class="btn-group w-100" role="group">
                                        <button type="button" class="btn btn-outline-secondary btn-sm" onclick="app.setDateFilter('today')" data-translate="filter.today">Today</button>
                                        <button type="button" class="btn btn-outline-secondary btn-sm" onclick="app.setDateFilter('week')" data-translate="filter.this_week">This Week</button>
                                        <button type="button" class="btn btn-outline-secondary btn-sm" onclick="app.setDateFilter('month')" data-translate="filter.this_month">This Month</button>
                                        <button type="button" class="btn btn-outline-secondary btn-sm" onclick="app.setDateFilter('year')" data-translate="filter.this_year">This Year</button>
                                    </div>
                                </div>
                                <div class="col-md-6">
                                    <label class="form-label" data-translate="payments.show_only">Show Only</label>
                                    <div class="btn-group w-100" role="group">
                                        <button type="button" class="btn btn-outline-warning btn-sm" onclick="app.setSpecialFilter('last_payments')" data-translate="payments.last_payments">Last Payments</button>
                                        <button type="button" class="btn btn-outline-success btn-sm" onclick="app.setSpecialFilter('recent')" data-translate="payments.recent_30_days">Recent (30 days)</button>
                                        <button type="button" class="btn btn-outline-danger btn-sm" onclick="app.setSpecialFilter('old')" data-translate="payments.old_90_days">Old (90+ days)</button>
                                    </div>
                                </div>
                                <div class="col-md-12 mt-3">
                                    <div class="btn-group">
                                        <button class="btn btn-primary" onclick="app.searchPaymentHistory()">
                                            <i class="bi bi-search"></i> <span data-translate="payments.apply_filters">Apply Filters</span>
                                        </button>
                                        <button class="btn btn-outline-secondary" onclick="app.clearAllFilters()">
                                            <i class="bi bi-x-circle"></i> <span data-translate="payments.clear_all">Clear All</span>
                                        </button>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>

            <div id="studentSearchResults" class="mt-4" style="display: none;">
                <div class="card">
                    <div class="card-header">
                        <h5 data-translate="payments.search_results">Search Results</h5>
                    </div>
                    <div class="card-body">
                        <div id="studentsList" class="row">
                            <!-- Student results will be populated here -->
                        </div>
                    </div>
                </div>
            </div>

            <div id="studentTimeline" class="mt-4" style="display: none;">
                <!-- Student timeline will be populated here -->
            </div>
        `;

        document.getElementById('pageContent').innerHTML = content;
        await this.loadPaymentFilterOptions();
    }

    async loadPaymentFilterOptions() {
        try {
            const [yearsRes, gradesRes] = await Promise.all([
                this.apiCall('GET', '/admin/academic-years'),
                this.apiCall('GET', '/admin/grades')
            ]);

            const yearFilter = document.getElementById('paymentYearFilter');
            const gradeFilter = document.getElementById('paymentGradeFilter');

            // Populate year filter
            yearsRes.data.forEach(year => {
                yearFilter.innerHTML += `<option value="${year.id}">${year.name}</option>`;
            });

            // Populate grade filter
            gradesRes.data.forEach(grade => {
                gradeFilter.innerHTML += `<option value="${grade.id}">${grade.title}</option>`;
            });

            // Load classes for the default selection
            await this.loadPaymentClasses();

        } catch (error) {
            console.error('Failed to load payment filter options:', error);
        }
    }

    async loadPaymentClasses() {
        try {
            const yearId = document.getElementById('paymentYearFilter').value;
            const gradeId = document.getElementById('paymentGradeFilter').value;
            const classFilter = document.getElementById('paymentClassFilter');

            // Reset classes
            classFilter.innerHTML = `<option value="" data-translate="payments.all_classes">All Classes</option>`;

            // Only load classes if both year and grade are selected
            if (yearId && gradeId) {
                const response = await this.apiCall('GET', `/admin/classes?year_id=${yearId}&grade_id=${gradeId}`);
                
                response.data.forEach(classItem => {
                    classFilter.innerHTML += `<option value="${classItem.id}">${classItem.name || classItem.code}</option>`;
                });
            }

        } catch (error) {
            console.error('Failed to load classes:', error);
        }
    }

    async searchStudentsForPayment() {
        const searchTerm = document.getElementById('studentSearchInput').value.trim();
        const yearId = document.getElementById('paymentYearFilter').value;
        const gradeId = document.getElementById('paymentGradeFilter').value;
        const classId = document.getElementById('paymentClassFilter').value;

        if (searchTerm.length < 2 && !yearId && !gradeId && !classId) {
            this.showToast('error', 'Please enter at least 2 characters or select a filter');
            return;
        }

        try {
            const params = new URLSearchParams();
            if (searchTerm) params.append('q', searchTerm);
            if (yearId) params.append('year_id', yearId);
            if (gradeId) params.append('grade_id', gradeId);
            if (classId) params.append('class_id', classId);
            params.append('limit', '20');

            const response = await this.apiCall('GET', `/admin/payments/search?${params}`);
            this.displayStudentSearchResults(response.data);

        } catch (error) {
            this.showToast('error', (typeof langManager !== 'undefined' ? langManager.t('payments.failed_search') : 'Failed to search students') + ': ' + error.message);
        }
    }

    async searchPaymentHistory() {
        const searchTerm = document.getElementById('studentSearchInput').value.trim();
        const yearId = document.getElementById('paymentYearFilter').value;
        const gradeId = document.getElementById('paymentGradeFilter').value;
        const classId = document.getElementById('paymentClassFilter').value;
        const method = document.getElementById('paymentMethodFilter')?.value;
        const dateFrom = document.getElementById('dateFromFilter')?.value;
        const dateTo = document.getElementById('dateToFilter')?.value;
        const minAmount = document.getElementById('minAmountFilter')?.value;
        const maxAmount = document.getElementById('maxAmountFilter')?.value;

        try {
            const params = new URLSearchParams();
            if (searchTerm) params.append('q', searchTerm);
            if (yearId) params.append('year_id', yearId);
            if (gradeId) params.append('grade_id', gradeId);
            if (classId) params.append('class_id', classId);
            if (method) params.append('method', method);
            if (dateFrom) params.append('start_date', dateFrom);
            if (dateTo) params.append('end_date', dateTo);
            if (minAmount) params.append('min_amount', minAmount);
            if (maxAmount) params.append('max_amount', maxAmount);
            params.append('per_page', '100'); // Show more results for large searches

            const response = await this.apiCall('GET', `/admin/payments/history?${params}`);
            this.displayPaymentHistoryResults(response.data, response.meta);

        } catch (error) {
            this.showToast('error', (typeof langManager !== 'undefined' ? langManager.t('payments.failed_search') : 'Failed to search payment history') + ': ' + error.message);
        }
    }

    displayPaymentHistoryResults(payments, meta) {
        // Hide other sections
        document.getElementById('studentSearchResults').style.display = 'none';
        document.getElementById('studentTimeline').style.display = 'none';

        // Show payment history results
        const timelineDiv = document.getElementById('studentTimeline');
        
        const historyHtml = `
            <div class="card">
                <div class="card-header d-flex justify-content-between align-items-center">
                    <h5><i class="bi bi-clock-history"></i> <span data-translate="payments.payment_history_results">Payment History Search Results</span></h5>
                    <small class="text-muted">${meta.total_rows} <span data-translate="payments.payments_found">payments found</span></small>
                </div>
                <div class="card-body">
                    <div class="table-responsive">
                        <table class="table table-hover">
                            <thead>
                                <tr>
                                    <th>Student</th>
                                    <th>Academic Info</th>
                                    <th>Payment Date</th>
                                    <th>Amount</th>
                                    <th>Method</th>
                                    <th>Days Ago</th>
                                    <th>Notes</th>
                                    <th>Actions</th>
                                </tr>
                            </thead>
                            <tbody>
                                ${payments.map(payment => `
                                    <tr ${payment.is_last_payment ? 'class="table-warning"' : ''}>
                                        <td>
                                            <strong>${payment.student_name}</strong>
                                            <br><small class="text-muted">ID: ${payment.student_id}</small>
                                            ${payment.is_last_payment ? '<br><span class="badge bg-warning text-dark">Last Payment</span>' : ''}
                                        </td>
                                        <td>
                                            <small>
                                                ${payment.year_name}<br>
                                                ${payment.grade_title}${payment.class_name ? ' - ' + payment.class_name : ''}
                                            </small>
                                        </td>
                                        <td>${payment.paid_on}</td>
                                        <td>
                                            <strong>$${payment.amount.toFixed(2)}</strong>
                                        </td>
                                        <td>
                                            <span class="badge bg-secondary">${payment.method}</span>
                                        </td>
                                        <td>
                                            <span class="badge ${payment.days_ago <= 7 ? 'bg-success' : payment.days_ago <= 30 ? 'bg-warning' : 'bg-secondary'}">
                                                ${payment.days_ago} days
                                            </span>
                                        </td>
                                        <td>${payment.notes || '-'}</td>
                                        <td>
                                            <div class="btn-group">
                                                <button class="btn btn-sm btn-outline-primary" onclick="app.viewStudentTimeline(${payment.student_id})" title="View Timeline">
                                                    <i class="bi bi-person-lines-fill"></i>
                                                </button>
                                                <button class="btn btn-sm btn-outline-warning" onclick="app.editPayment(${payment.id})" title="Edit Payment">
                                                    <i class="bi bi-pencil"></i>
                                                </button>
                                            </div>
                                        </td>
                                    </tr>
                                `).join('')}
                            </tbody>
                        </table>
                    </div>
                    
                    ${meta.total_pages > 1 ? `
                        <nav aria-label="Payment history pagination">
                            <ul class="pagination justify-content-center">
                                ${meta.page > 1 ? `
                                    <li class="page-item">
                                        <a class="page-link" href="#" onclick="app.loadPaymentHistoryPage(${meta.page - 1})">Previous</a>
                                    </li>
                                ` : ''}
                                
                                ${Array.from({length: Math.min(5, meta.total_pages)}, (_, i) => {
                                    const pageNum = i + 1;
                                    return `
                                        <li class="page-item ${pageNum === meta.page ? 'active' : ''}">
                                            <a class="page-link" href="#" onclick="app.loadPaymentHistoryPage(${pageNum})">${pageNum}</a>
                                        </li>
                                    `;
                                }).join('')}
                                
                                ${meta.page < meta.total_pages ? `
                                    <li class="page-item">
                                        <a class="page-link" href="#" onclick="app.loadPaymentHistoryPage(${meta.page + 1})">Next</a>
                                    </li>
                                ` : ''}
                            </ul>
                        </nav>
                    ` : ''}
                </div>
            </div>
        `;

        timelineDiv.innerHTML = historyHtml;
        timelineDiv.style.display = 'block';
    }

    async loadPaymentHistoryPage(page) {
        const searchTerm = document.getElementById('studentSearchInput').value.trim();
        const yearId = document.getElementById('paymentYearFilter').value;
        const gradeId = document.getElementById('paymentGradeFilter').value;
        const classId = document.getElementById('paymentClassFilter').value;
        const method = document.getElementById('paymentMethodFilter')?.value;
        const dateFrom = document.getElementById('dateFromFilter')?.value;
        const dateTo = document.getElementById('dateToFilter')?.value;
        const minAmount = document.getElementById('minAmountFilter')?.value;
        const maxAmount = document.getElementById('maxAmountFilter')?.value;

        try {
            const params = new URLSearchParams();
            if (searchTerm) params.append('q', searchTerm);
            if (yearId) params.append('year_id', yearId);
            if (gradeId) params.append('grade_id', gradeId);
            if (classId) params.append('class_id', classId);
            if (method) params.append('method', method);
            if (dateFrom) params.append('start_date', dateFrom);
            if (dateTo) params.append('end_date', dateTo);
            if (minAmount) params.append('min_amount', minAmount);
            if (maxAmount) params.append('max_amount', maxAmount);
            params.append('page', page);
            params.append('per_page', '50');

            const response = await this.apiCall('GET', `/admin/payments/history?${params}`);
            this.displayPaymentHistoryResults(response.data, response.meta);

        } catch (error) {
            this.showToast('error', 'Failed to load payment history page: ' + error.message);
        }
    }

    displayStudentSearchResults(students) {
        const resultsDiv = document.getElementById('studentSearchResults');
        const studentsList = document.getElementById('studentsList');

        if (students.length === 0) {
            studentsList.innerHTML = '<div class="col-12"><div class="alert alert-info">No students found</div></div>';
        } else {
            studentsList.innerHTML = students.map(student => `
                <div class="col-md-6 col-lg-4 mb-3">
                    <div class="card h-100">
                        <div class="card-body">
                            <h6 class="card-title">${student.full_name}</h6>
                            <p class="card-text">
                                <small class="text-muted">
                                    ID: ${student.id}<br>
                                    ${student.national_id ? 'National ID: ' + student.national_id : ''}
                                </small>
                            </p>
                            <button class="btn btn-primary btn-sm" onclick="app.viewStudentTimeline(${student.id})">
                                <i class="bi bi-clock-history"></i> View Timeline
                            </button>
                        </div>
                    </div>
                </div>
            `).join('');
        }

        resultsDiv.style.display = 'block';
    }

    async viewStudentTimeline(studentId) {
        try {
            const response = await this.apiCall('GET', `/admin/students/${studentId}/timeline`);
            this.displayStudentTimeline(response.data);

        } catch (error) {
            this.showToast('error', 'Failed to load student timeline: ' + error.message);
        }
    }

    displayStudentTimeline(data) {
        const timelineDiv = document.getElementById('studentTimeline');
        const student = data.student;
        const timeline = data.timeline;

        const timelineHtml = `
            <div class="card">
                <div class="card-header">
                    <h5><i class="bi bi-person"></i> ${student.full_name} - Payment Timeline</h5>
                    <small class="text-muted">ID: ${student.id} ${student.national_id ? '| National ID: ' + student.national_id : ''}</small>
                </div>
                <div class="card-body">
                    <div class="table-responsive">
                        <table class="table">
                            <thead>
                                <tr>
                                    <th>Academic Year</th>
                                    <th>Grade & Class</th>
                                    <th>Status</th>
                                    <th>Configured Fee</th>
                                    <th>Total Paid</th>
                                    <th>Balance</th>
                                    <th>Actions</th>
                                </tr>
                            </thead>
                            <tbody>
                                ${timeline.map(year => `
                                    <tr>
                                        <td>
                                            ${year.year_name}
                                            <br><small class="text-muted">${year.year_status}</small>
                                        </td>
                                        <td>
                                            ${year.grade_title}
                                            ${year.class_name ? ' - ' + year.class_name : ''}
                                        </td>
                                        <td>
                                            <span class="badge ${year.status === 'active' ? 'bg-success' : 'bg-secondary'}">
                                                ${year.status}
                                            </span>
                                        </td>
                                        <td>$${year.configured_fee.toFixed(2)}</td>
                                        <td>$${year.total_paid.toFixed(2)}</td>
                                        <td>
                                            <span class="badge ${year.balance > 0 ? 'bg-warning' : year.balance < 0 ? 'bg-info' : 'bg-success'}">
                                                $${year.balance.toFixed(2)}
                                            </span>
                                        </td>
                                        <td>
                                            <div class="btn-group">
                                                <button class="btn btn-sm btn-outline-primary" onclick="app.viewPaymentHistory(${year.enrollment_id})">
                                                    <i class="bi bi-list"></i> History
                                                </button>
                                                ${year.can_pay ? `
                                                    <button class="btn btn-sm btn-success" onclick="app.showPaymentModal(${year.enrollment_id}, '${year.year_name}', '${student.full_name}')">
                                                        <i class="bi bi-plus"></i> Pay
                                                    </button>
                                                ` : ''}
                                            </div>
                                        </td>
                                    </tr>
                                `).join('')}
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>
        `;

        timelineDiv.innerHTML = timelineHtml;
        timelineDiv.style.display = 'block';

        // Hide search results to focus on timeline
        document.getElementById('studentSearchResults').style.display = 'none';
    }

    async viewPaymentHistory(enrollmentId) {
        try {
            const response = await this.apiCall('GET', `/admin/enrollments/${enrollmentId}/payments`);
            this.showPaymentHistoryModal(response.data);

        } catch (error) {
            this.showToast('error', 'Failed to load payment history: ' + error.message);
        }
    }

    showPaymentHistoryModal(payments) {
        const modal = document.createElement('div');
        modal.className = 'modal fade';
        modal.id = 'paymentHistoryModal';
        modal.innerHTML = `
            <div class="modal-dialog modal-lg">
                <div class="modal-content">
                    <div class="modal-header">
                        <h5 class="modal-title">Payment History</h5>
                        <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                    </div>
                    <div class="modal-body">
                        <div class="table-responsive">
                            <table class="table">
                                <thead>
                                    <tr>
                                        <th>Date</th>
                                        <th>Amount</th>
                                        <th>Method</th>
                                        <th>Notes</th>
                                        <th>Created By</th>
                                        <th>Actions</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    ${payments.map(payment => `
                                        <tr>
                                            <td>${payment.paid_on}</td>
                                            <td>$${parseFloat(payment.amount).toFixed(2)}</td>
                                            <td>
                                                <span class="badge bg-secondary">${payment.method}</span>
                                            </td>
                                            <td>${payment.notes || '-'}</td>
                                            <td>${payment.created_by_name || '-'}</td>
                                            <td>
                                                <div class="btn-group">
                                                    <button class="btn btn-sm btn-outline-warning" onclick="app.editPayment(${payment.id})">
                                                        <i class="bi bi-pencil"></i>
                                                    </button>
                                                    <button class="btn btn-sm btn-outline-danger" onclick="app.deletePayment(${payment.id}, '${payment.paid_on}', ${payment.amount})">
                                                        <i class="bi bi-trash"></i>
                                                    </button>
                                                </div>
                                            </td>
                                        </tr>
                                    `).join('')}
                                </tbody>
                            </table>
                        </div>
                    </div>
                </div>
            </div>
        `;

        document.body.appendChild(modal);
        const bsModal = new bootstrap.Modal(modal);
        bsModal.show();

        modal.addEventListener('hidden.bs.modal', () => {
            document.body.removeChild(modal);
        });
    }

    showPaymentModal(enrollmentId, yearName, studentName) {
        const today = new Date().toISOString().split('T')[0];
        
        const modal = document.createElement('div');
        modal.className = 'modal fade';
        modal.id = 'paymentModal';
        modal.innerHTML = `
            <div class="modal-dialog">
                <div class="modal-content">
                    <div class="modal-header">
                        <h5 class="modal-title">Record Payment</h5>
                        <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                    </div>
                    <div class="modal-body">
                        <form id="paymentForm">
                            <div class="mb-3">
                                <label class="form-label">Student & Year</label>
                                <input type="text" class="form-control" value="${studentName} - ${yearName}" readonly>
                            </div>
                            <div class="row">
                                <div class="col-md-6">
                                    <div class="mb-3">
                                        <label for="paymentAmount" class="form-label">Amount *</label>
                                        <div class="input-group">
                                            <span class="input-group-text">$</span>
                                            <input type="number" class="form-control" id="paymentAmount" step="0.01" min="0.01" required>
                                        </div>
                                    </div>
                                </div>
                                <div class="col-md-6">
                                    <div class="mb-3">
                                        <label for="paymentDate" class="form-label">Payment Date *</label>
                                        <input type="date" class="form-control" id="paymentDate" value="${today}" required>
                                    </div>
                                </div>
                            </div>
                            <div class="mb-3">
                                <label for="paymentMethod" class="form-label">Payment Method *</label>
                                <select class="form-select" id="paymentMethod" required>
                                    <option value="">Select method</option>
                                    <option value="cash">Cash</option>
                                    <option value="card">Card</option>
                                    <option value="bank">Bank Transfer</option>
                                    <option value="wallet">Digital Wallet</option>
                                </select>
                            </div>
                            <div class="mb-3">
                                <label for="paymentNotes" class="form-label">Notes</label>
                                <textarea class="form-control" id="paymentNotes" rows="3"></textarea>
                            </div>
                        </form>
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                        <button type="button" class="btn btn-primary" onclick="app.savePayment(${enrollmentId})">Record Payment</button>
                    </div>
                </div>
            </div>
        `;

        document.body.appendChild(modal);
        const bsModal = new bootstrap.Modal(modal);
        bsModal.show();

        modal.addEventListener('hidden.bs.modal', () => {
            document.body.removeChild(modal);
        });
    }

    async savePayment(enrollmentId) {
        const data = {
            amount: document.getElementById('paymentAmount').value,
            paid_on: document.getElementById('paymentDate').value,
            method: document.getElementById('paymentMethod').value,
            notes: document.getElementById('paymentNotes').value
        };

        if (!data.amount || !data.paid_on || !data.method) {
            this.showToast('error', 'Please fill in all required fields');
            return;
        }

        try {
            await this.apiCall('POST', `/admin/enrollments/${enrollmentId}/payments`, data);
            this.showToast('success', 'Payment recorded successfully');
            
            // Close modal
            const modal = document.getElementById('paymentModal');
            if (modal) {
                const bsModal = bootstrap.Modal.getInstance(modal);
                if (bsModal) bsModal.hide();
            }
            
            // Refresh the timeline if it's visible
            const timelineDiv = document.getElementById('studentTimeline');
            if (timelineDiv && timelineDiv.style.display !== 'none') {
                // Get student ID from the timeline and refresh
                const h5Element = timelineDiv.querySelector('h5');
                if (h5Element) {
                    const match = h5Element.textContent.match(/ID: (\d+)/);
                    if (match && match[1]) {
                        this.viewStudentTimeline(match[1]);
                    }
                }
            }

        } catch (error) {
            this.showToast('error', (typeof langManager !== 'undefined' ? langManager.t('payments.failed_record') : 'Failed to record payment') + ': ' + error.message);
        }
    }

    async editPayment(paymentId) {
        try {
            // Get payment details first
            const response = await this.apiCall('GET', `/admin/payments/${paymentId}`);
            const payment = response.data;

            const modal = document.createElement('div');
            modal.className = 'modal fade';
            modal.id = 'editPaymentModal';
            modal.innerHTML = `
                <div class="modal-dialog">
                    <div class="modal-content">
                        <div class="modal-header">
                            <h5 class="modal-title">Edit Payment</h5>
                            <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                        </div>
                        <div class="modal-body">
                            <form id="editPaymentForm">
                                <div class="row">
                                    <div class="col-md-6">
                                        <div class="mb-3">
                                            <label for="editPaymentAmount" class="form-label">Amount *</label>
                                            <div class="input-group">
                                                <span class="input-group-text">$</span>
                                                <input type="number" class="form-control" id="editPaymentAmount" step="0.01" min="0.01" value="${payment.amount}" required>
                                            </div>
                                        </div>
                                    </div>
                                    <div class="col-md-6">
                                        <div class="mb-3">
                                            <label for="editPaymentDate" class="form-label">Payment Date *</label>
                                            <input type="date" class="form-control" id="editPaymentDate" value="${payment.paid_on}" required>
                                        </div>
                                    </div>
                                </div>
                                <div class="mb-3">
                                    <label for="editPaymentMethod" class="form-label">Payment Method *</label>
                                    <select class="form-select" id="editPaymentMethod" required>
                                        <option value="cash" ${payment.method === 'cash' ? 'selected' : ''}>Cash</option>
                                        <option value="card" ${payment.method === 'card' ? 'selected' : ''}>Card</option>
                                        <option value="bank" ${payment.method === 'bank' ? 'selected' : ''}>Bank Transfer</option>
                                        <option value="wallet" ${payment.method === 'wallet' ? 'selected' : ''}>Digital Wallet</option>
                                    </select>
                                </div>
                                <div class="mb-3">
                                    <label for="editPaymentNotes" class="form-label">Notes</label>
                                    <textarea class="form-control" id="editPaymentNotes" rows="3">${payment.notes || ''}</textarea>
                                </div>
                            </form>
                        </div>
                        <div class="modal-footer">
                            <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                            <button type="button" class="btn btn-primary" onclick="app.updatePayment(${paymentId})">Update Payment</button>
                        </div>
                    </div>
                </div>
            `;

            document.body.appendChild(modal);
            const bsModal = new bootstrap.Modal(modal);
            bsModal.show();

            modal.addEventListener('hidden.bs.modal', () => {
                document.body.removeChild(modal);
            });

        } catch (error) {
            this.showToast('error', (typeof langManager !== 'undefined' ? langManager.t('payments.failed_to_load') : 'Failed to load payment details') + ': ' + error.message);
        }
    }

    async updatePayment(paymentId) {
        const data = {
            amount: document.getElementById('editPaymentAmount').value,
            paid_on: document.getElementById('editPaymentDate').value,
            method: document.getElementById('editPaymentMethod').value,
            notes: document.getElementById('editPaymentNotes').value
        };

        if (!data.amount || !data.paid_on || !data.method) {
            this.showToast('error', 'Please fill in all required fields');
            return;
        }

        try {
            await this.apiCall('PUT', `/admin/payments/${paymentId}`, data);
            this.showToast('success', 'Payment updated successfully');
            
            // Close modal
            const modal = document.getElementById('editPaymentModal');
            if (modal) {
                const bsModal = bootstrap.Modal.getInstance(modal);
                if (bsModal) bsModal.hide();
            }

        } catch (error) {
            this.showToast('error', (typeof langManager !== 'undefined' ? langManager.t('payments.failed_update') : 'Failed to update payment') + ': ' + error.message);
        }
    }

    async deletePayment(paymentId, paymentDate, amount) {
        if (!confirm(`Are you sure you want to delete the payment of $${amount} made on ${paymentDate}?`)) {
            return;
        }

        try {
            await this.apiCall('DELETE', `/admin/payments/${paymentId}`);
            this.showToast('success', 'Payment deleted successfully');
            
            // Close modal
            const modal = document.getElementById('paymentHistoryModal');
            if (modal) {
                const bsModal = bootstrap.Modal.getInstance(modal);
                if (bsModal) bsModal.hide();
            }

        } catch (error) {
            this.showToast('error', (typeof langManager !== 'undefined' ? langManager.t('payments.failed_delete') : 'Failed to delete payment') + ': ' + error.message);
        }
    }

    toggleAdvancedFilters() {
        const filtersDiv = document.getElementById('advancedFilters');
        const toggleBtn = document.querySelector('[onclick="app.toggleAdvancedFilters()"]');
        
        if (filtersDiv.style.display === 'none' || !filtersDiv.style.display) {
            filtersDiv.style.display = 'block';
            toggleBtn.innerHTML = '<i class="bi bi-chevron-up"></i> Hide Advanced Filters';
        } else {
            filtersDiv.style.display = 'none';
            toggleBtn.innerHTML = '<i class="bi bi-chevron-down"></i> Show Advanced Filters';
        }
    }

    setDateFilter(type) {
        const dateFromFilter = document.getElementById('dateFromFilter');
        const dateToFilter = document.getElementById('dateToFilter');
        const today = new Date();
        let fromDate, toDate;

        switch (type) {
            case 'today':
                fromDate = toDate = today.toISOString().split('T')[0];
                break;
            case 'week':
                fromDate = new Date(today.getTime() - 7 * 24 * 60 * 60 * 1000).toISOString().split('T')[0];
                toDate = today.toISOString().split('T')[0];
                break;
            case 'month':
                fromDate = new Date(today.getFullYear(), today.getMonth(), 1).toISOString().split('T')[0];
                toDate = today.toISOString().split('T')[0];
                break;
            case 'year':
                fromDate = new Date(today.getFullYear(), 0, 1).toISOString().split('T')[0];
                toDate = today.toISOString().split('T')[0];
                break;
        }

        if (dateFromFilter) dateFromFilter.value = fromDate;
        if (dateToFilter) dateToFilter.value = toDate;
        
        // Trigger search with new date filters
        this.searchPaymentHistory();
    }

    setSpecialFilter(type) {
        // Clear existing filters first
        this.clearAllFilters();
        
        const today = new Date();
        const dateFromFilter = document.getElementById('dateFromFilter');
        const dateToFilter = document.getElementById('dateToFilter');
        
        switch (type) {
            case 'recent':
                // Last 30 days
                const thirtyDaysAgo = new Date(today.getTime() - 30 * 24 * 60 * 60 * 1000);
                if (dateFromFilter) dateFromFilter.value = thirtyDaysAgo.toISOString().split('T')[0];
                if (dateToFilter) dateToFilter.value = today.toISOString().split('T')[0];
                break;
            case 'old':
                // More than 90 days ago
                const ninetyDaysAgo = new Date(today.getTime() - 90 * 24 * 60 * 60 * 1000);
                if (dateToFilter) dateToFilter.value = ninetyDaysAgo.toISOString().split('T')[0];
                break;
            case 'last_payments':
                // This will be handled by backend filtering
                // We'll add a special parameter for this
                break;
        }
        
        // Trigger search with special filter
        this.searchPaymentHistory();
    }

    clearAllFilters() {
        // Clear all filter inputs
        const filters = [
            'studentSearchInput',
            'paymentYearFilter', 
            'paymentGradeFilter',
            'paymentClassFilter',
            'paymentMethodFilter',
            'dateFromFilter',
            'dateToFilter',
            'minAmountFilter',
            'maxAmountFilter'
        ];
        
        filters.forEach(filterId => {
            const element = document.getElementById(filterId);
            if (element) {
                if (element.type === 'select-one') {
                    element.selectedIndex = 0;
                } else {
                    element.value = '';
                }
            }
        });
        
        // Trigger search to show all results
        this.searchPaymentHistory();
    }

    // ===================
    // SETTINGS SECTION  
    // ===================

    async loadSettings() {
        const content = `
            <div class="page-header">
                <div class="d-flex justify-content-between align-items-center">
                    <div>
                        <h2><i class="bi bi-gear"></i> <span data-translate="settings.title">System Settings</span></h2>
                        <p class="text-muted" data-translate="settings.subtitle">Configure system preferences and policies</p>
                    </div>
                    <button class="btn btn-primary" onclick="app.saveSettings()">
                        <i class="bi bi-check-circle"></i> <span data-translate="settings.save_all">Save All Settings</span>
                    </button>
                </div>
            </div>

            <div class="row">
                <!-- School Settings -->
                <div class="col-12 mb-4">
                    <div class="card">
                        <div class="card-header">
                            <h5><i class="bi bi-building"></i> <span data-translate="settings.school_settings">School Settings</span></h5>
                        </div>
                        <div class="card-body">
                            <div class="row">
                                <div class="col-md-6">
                                    <div class="mb-3">
                                        <label class="form-label" data-translate="settings.school_name">School Name</label>
                                        <input type="text" class="form-control" id="schoolName" data-translate-placeholder="settings.school_name_placeholder" placeholder="Enter school name">
                                        <small class="form-text text-muted" data-translate="settings.school_name_help">This will appear on student cards and portal</small>
                                    </div>
                                </div>
                                <div class="col-md-6">
                                    <div class="mb-3">
                                        <label class="form-label" data-translate="settings.portal_url">Student Portal URL</label>
                                        <input type="url" class="form-control" id="studentPortalUrl" data-translate-placeholder="settings.portal_url_placeholder" placeholder="http://localhost/student1/student/">
                                        <small class="form-text text-muted" data-translate="settings.portal_url_help">URL for QR codes on student cards</small>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
                
                <!-- Academic Visibility Settings -->
                <div class="col-lg-6 mb-4">
                    <div class="card">
                        <div class="card-header">
                            <h5><i class="bi bi-eye"></i> <span data-translate="settings.academic_visibility">Academic Visibility</span></h5>
                        </div>
                        <div class="card-body">
                            <div class="mb-3">
                                <label class="form-label" data-translate="settings.current_academic_year">Current Academic Year</label>
                                <select class="form-select" id="currentYearId">
                                    <option value="" data-translate="settings.select_year">Select Year</option>
                                </select>
                            </div>
                            
                            <div class="mb-3">
                                <div class="form-check">
                                    <input class="form-check-input" type="checkbox" id="firstTermVisible">
                                    <label class="form-check-label" for="firstTermVisible" data-translate="settings.first_term_visible">
                                        First Term Results Visible
                                    </label>
                                </div>
                            </div>
                            
                            <div class="mb-3">
                                <label class="form-label" data-translate="settings.first_term_publish_date">First Term Publish Date</label>
                                <input type="datetime-local" class="form-control" id="firstTermPublishAt">
                            </div>
                            
                            <div class="mb-3">
                                <div class="form-check">
                                    <input class="form-check-input" type="checkbox" id="secondTermVisible">
                                    <label class="form-check-label" for="secondTermVisible" data-translate="settings.second_term_visible">
                                        Second Term Results Visible
                                    </label>
                                </div>
                            </div>
                            
                            <div class="mb-3">
                                <label class="form-label" data-translate="settings.second_term_publish_date">Second Term Publish Date</label>
                                <input type="datetime-local" class="form-control" id="secondTermPublishAt">
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Display Options -->
                <div class="col-lg-6 mb-4">
                    <div class="card">
                        <div class="card-header">
                            <h5><i class="bi bi-palette"></i> <span data-translate="settings.display_options">Display Options</span></h5>
                        </div>
                        <div class="card-body">
                            <div class="mb-3">
                                <label class="form-label" data-translate="settings.display_mode">Display Mode</label>
                                <select class="form-select" id="displayMode">
                                    <option value="color_only" data-translate="settings.color_only">Color Only</option>
                                    <option value="color_score" data-translate="settings.color_score">Color + Score</option>
                                    <option value="color_score_percent" data-translate="settings.color_score_percent">Color + Score + Percent</option>
                                </select>
                            </div>
                            
                            <div class="mb-3">
                                <label class="form-label" data-translate="settings.percentage_rounding">Percentage Rounding (decimal places)</label>
                                <select class="form-select" id="percentRounding">
                                    <option value="0" data-translate="settings.rounding_0">0 (85%)</option>
                                    <option value="1" data-translate="settings.rounding_1">1 (84.5%)</option>
                                    <option value="2" data-translate="settings.rounding_2">2 (84.56%)</option>
                                </select>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Pass Rules -->
                <div class="col-lg-6 mb-4">
                    <div class="card">
                        <div class="card-header">
                            <h5><i class="bi bi-check-square"></i> <span data-translate="settings.pass_rules">Pass Rules</span></h5>
                        </div>
                        <div class="card-body">
                            <div class="mb-3">
                                <label class="form-label" data-translate="settings.per_subject_threshold">Per Subject Threshold Mode</label>
                                <select class="form-select" id="perSubjectThresholdMode">
                                    <option value="useGradeSubjectMin" data-translate="settings.use_grade_subject_min">Use Grade Subject Minimum</option>
                                    <option value="fixed" data-translate="settings.fixed_percentage">Fixed Percentage</option>
                                </select>
                            </div>
                            
                            <div class="mb-3">
                                <label class="form-label" data-translate="settings.subject_pass_min_percent">Subject Pass Min Percent</label>
                                <input type="number" class="form-control" id="subjectPassMinPercent" step="0.01" min="0" max="100">
                            </div>
                            
                            <div class="mb-3">
                                <label class="form-label" data-translate="settings.overall_pass_rule">Overall Pass Rule</label>
                                <select class="form-select" id="overallPassRule">
                                    <option value="none" data-translate="settings.none">None</option>
                                    <option value="avgAtLeast" data-translate="settings.avg_at_least">Average At Least</option>
                                </select>
                            </div>
                            
                            <div class="mb-3">
                                <label class="form-label" data-translate="settings.overall_pass_min_percent">Overall Pass Min Percent</label>
                                <input type="number" class="form-control" id="overallPassMinPercent" step="0.01" min="0" max="100">
                            </div>
                            
                            <div class="mb-3">
                                <label class="form-label" data-translate="settings.pass_condition">Pass Condition</label>
                                <select class="form-select" id="passCondition">
                                    <option value="allSubjectsOnly" data-translate="settings.all_subjects_only">All Subjects Only</option>
                                    <option value="overallOnly" data-translate="settings.overall_only">Overall Only</option>
                                    <option value="allSubjectsOrOverall" data-translate="settings.all_subjects_or_overall">All Subjects OR Overall</option>
                                    <option value="allSubjectsAndOverall" data-translate="settings.all_subjects_and_overall">All Subjects AND Overall</option>
                                </select>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Security Policy -->
                <div class="col-lg-6 mb-4">
                    <div class="card">
                        <div class="card-header">
                            <h5><i class="bi bi-shield-check"></i> <span data-translate="settings.security_policy">Security Policy</span></h5>
                        </div>
                        <div class="card-body">
                            <div class="mb-3">
                                <div class="form-check">
                                    <input class="form-check-input" type="checkbox" id="maskNationalIdInAdmin">
                                    <label class="form-check-label" for="maskNationalIdInAdmin" data-translate="settings.mask_national_id">
                                        Mask National ID in Admin Interface
                                    </label>
                                </div>
                            </div>
                            
                            <div class="mb-3">
                                <div class="form-check">
                                    <input class="form-check-input" type="checkbox" id="studentLoginRequiresUniqueAndNationalId">
                                    <label class="form-check-label" for="studentLoginRequiresUniqueAndNationalId" data-translate="settings.unique_national_id">
                                        Student Login Requires Unique National ID
                                    </label>
                                </div>
                            </div>
                            
                            <div class="mb-3">
                                <label class="form-label" data-translate="settings.session_timeout">Session Timeout (minutes)</label>
                                <input type="number" class="form-control" id="sessionTimeoutMinutes" min="5" max="1440" value="60">
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Grade Costs -->
                <div class="col-12 mb-4">
                    <div class="card">
                        <div class="card-header d-flex justify-content-between align-items-center">
                            <h5><i class="bi bi-currency-dollar"></i> <span data-translate="settings.grade_costs">Grade Costs</span></h5>
                            <button class="btn btn-outline-primary btn-sm" onclick="app.addGradeCost()">
                                <i class="bi bi-plus"></i> <span data-translate="settings.add_grade_cost">Add Grade Cost</span>
                            </button>
                        </div>
                        <div class="card-body">
                            <div id="gradeCostsList">
                                <!-- Grade costs will be populated here -->
                            </div>
                        </div>
                    </div>
                </div>
                
                <!-- Percentage Bands -->
                <div class="col-12 mb-4">
                    <div class="card">
                        <div class="card-header d-flex justify-content-between align-items-center">
                            <h5><i class="bi bi-palette"></i> <span data-translate="settings.percentage_bands">Grade Percentage Bands</span></h5>
                            <button class="btn btn-outline-primary btn-sm" onclick="app.addPercentageBand()">
                                <i class="bi bi-plus"></i> <span data-translate="settings.add_percentage_band">Add Percentage Band</span>
                            </button>
                        </div>
                        <div class="card-body">
                            <div class="alert alert-info">
                                <i class="bi bi-info-circle"></i> Configure the percentage ranges, labels, and colors for student grades. These will be used throughout the system for grade display.
                            </div>
                            <div id="percentageBandsList">
                                <!-- Percentage bands will be populated here -->
                            </div>
                        </div>
                    </div>
                </div>
                
                <!-- Grade Result Visibility -->
                <div class="col-12 mb-4">
                    <div class="card">
                        <div class="card-header d-flex justify-content-between align-items-center">
                            <h5><i class="bi bi-eye-slash"></i> <span data-translate="settings.grade_visibility">Grade Result Visibility</span></h5>
                            <button class="btn btn-outline-primary btn-sm" onclick="app.addGradeVisibility()">
                                <i class="bi bi-plus"></i> <span data-translate="settings.add_grade_visibility">Add Grade Visibility Rule</span>
                            </button>
                        </div>
                        <div class="card-body">
                            <div class="alert alert-warning">
                                <i class="bi bi-exclamation-triangle"></i> <span data-translate="settings.grade_visibility_help">Control which grades can see results in the student portal. Students also need to pay grade costs to view results.</span>
                            </div>
                            <div id="gradeVisibilityList">
                                <!-- Grade visibility rules will be populated here -->
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        `;

        document.getElementById('pageContent').innerHTML = content;
        await this.loadSettingsData();
    }

    async loadSettingsData() {
        try {
            const response = await this.apiCall('GET', '/admin/settings');
            const settings = response.data;
            
            // Populate Academic Visibility
            this.populateYearOptions(settings.academic_years, 'currentYearId');
            if (settings.academic_visibility) {
                const av = settings.academic_visibility;
                document.getElementById('currentYearId').value = av.current_year_id || '';
                document.getElementById('firstTermVisible').checked = av.first_term_visible;
                document.getElementById('firstTermPublishAt').value = av.first_term_publish_at ? av.first_term_publish_at.slice(0, 16) : '';
                document.getElementById('secondTermVisible').checked = av.second_term_visible;
                document.getElementById('secondTermPublishAt').value = av.second_term_publish_at ? av.second_term_publish_at.slice(0, 16) : '';
            }
            
            // Populate Display Options
            if (settings.display_options) {
                const do_opt = settings.display_options;
                document.getElementById('displayMode').value = do_opt.display_mode || 'color_only';
                document.getElementById('percentRounding').value = do_opt.percent_rounding || '0';
            }
            
            // Populate Pass Rules
            if (settings.pass_rules) {
                const pr = settings.pass_rules;
                document.getElementById('perSubjectThresholdMode').value = pr.per_subject_threshold_mode || 'useGradeSubjectMin';
                document.getElementById('subjectPassMinPercent').value = pr.subject_pass_min_percent || '';
                document.getElementById('overallPassRule').value = pr.overall_pass_rule || 'none';
                document.getElementById('overallPassMinPercent').value = pr.overall_pass_min_percent || '';
                document.getElementById('passCondition').value = pr.pass_condition || 'allSubjectsOnly';
            }
            
            // Populate Security Policy
            if (settings.security_policy) {
                const sp = settings.security_policy;
                document.getElementById('maskNationalIdInAdmin').checked = sp.mask_national_id_in_admin;
                document.getElementById('studentLoginRequiresUniqueAndNationalId').checked = sp.student_login_requires_unique_and_national_id;
                document.getElementById('sessionTimeoutMinutes').value = sp.session_timeout_minutes || 60;
            }
            
            // Populate School Settings
            if (settings.school_settings) {
                const ss = settings.school_settings;
                document.getElementById('schoolName').value = ss.school_name || '';
                document.getElementById('studentPortalUrl').value = ss.student_portal_url || '';
            }
            
            // Store for grade costs and percentage bands
            this.settingsData = settings;
            this.renderGradeCosts(settings.grade_costs || []);
            this.renderPercentageBands(settings.percentage_bands || []);
            this.renderGradeVisibility(settings.grade_result_visibility || []);
            
        } catch (error) {
            this.showToast('error', (typeof langManager !== 'undefined' ? langManager.t('settings.failed_to_load') : 'Failed to load settings') + ': ' + error.message);
        }
    }

    populateYearOptions(years, selectId) {
        const select = document.getElementById(selectId);
        if (select) {
            const currentOptions = select.innerHTML;
            const newOptions = years.map(year => `<option value="${year.id}">${year.name}</option>`).join('');
            select.innerHTML = currentOptions + newOptions;
        }
    }

    renderGradeCosts(gradeCosts) {
        const container = document.getElementById('gradeCostsList');
        
        if (gradeCosts.length === 0) {
            container.innerHTML = `<div class="text-muted" data-translate="settings.no_grade_costs">No grade costs configured. Click "Add Grade Cost" to get started.</div>`;
            return;
        }
        
        container.innerHTML = gradeCosts.map((cost, index) => `
            <div class="row align-items-center mb-2 grade-cost-row" data-index="${index}">
                <div class="col-md-4">
                    <select class="form-select grade-cost-year" data-index="${index}">
                        <option value="" data-translate="form.select_year">Select Year</option>
                        ${this.settingsData.academic_years.map(year => 
                            `<option value="${year.id}" ${cost.year_id == year.id ? 'selected' : ''}>${year.name}</option>`
                        ).join('')}
                    </select>
                </div>
                <div class="col-md-4">
                    <select class="form-select grade-cost-grade" data-index="${index}">
                        <option value="" data-translate="settings.select_grade">Select Grade</option>
                        ${this.settingsData.grades.map(grade => 
                            `<option value="${grade.id}" ${cost.grade_id == grade.id ? 'selected' : ''}>${grade.title}</option>`
                        ).join('')}
                    </select>
                </div>
                <div class="col-md-3">
                    <input type="number" class="form-control grade-cost-amount" data-index="${index}" 
                           step="0.01" min="0" data-translate-placeholder="settings.amount_placeholder" placeholder="Amount" value="${cost.fee_amount || ''}">
                </div>
                <div class="col-md-1">
                    <button class="btn btn-outline-danger btn-sm" onclick="app.removeGradeCost(${index})">
                        <i class="bi bi-trash"></i>
                    </button>
                </div>
            </div>
        `).join('');
    }

    addGradeCost() {
        const container = document.getElementById('gradeCostsList');
        const existingRows = container.querySelectorAll('.grade-cost-row');
        const index = existingRows.length;
        
        const newRow = document.createElement('div');
        newRow.className = 'row align-items-center mb-2 grade-cost-row';
        newRow.setAttribute('data-index', index);
        newRow.innerHTML = `
            <div class="col-md-4">
                <select class="form-select grade-cost-year" data-index="${index}">
                    <option value="" data-translate="form.select_year">Select Year</option>
                    ${this.settingsData.academic_years.map(year => 
                        `<option value="${year.id}">${year.name}</option>`
                    ).join('')}
                </select>
            </div>
            <div class="col-md-4">
                <select class="form-select grade-cost-grade" data-index="${index}">
                    <option value="" data-translate="settings.select_grade">Select Grade</option>
                    ${this.settingsData.grades.map(grade => 
                        `<option value="${grade.id}">${grade.title}</option>`
                    ).join('')}
                </select>
            </div>
            <div class="col-md-3">
                <input type="number" class="form-control grade-cost-amount" data-index="${index}" 
                       step="0.01" min="0" data-translate-placeholder="settings.amount_placeholder" placeholder="Amount">
            </div>
            <div class="col-md-1">
                <button class="btn btn-outline-danger btn-sm" onclick="app.removeGradeCost(${index})">
                    <i class="bi bi-trash"></i>
                </button>
            </div>
        `;
        
        if (container.querySelector('.text-muted')) {
            container.innerHTML = '';
        }
        container.appendChild(newRow);
    }

    removeGradeCost(index) {
        const row = document.querySelector(`.grade-cost-row[data-index="${index}"]`);
        if (row) {
            row.remove();
            
            // Check if no rows left
            const container = document.getElementById('gradeCostsList');
            if (container.children.length === 0) {
                container.innerHTML = `<div class="text-muted" data-translate="settings.no_grade_costs">No grade costs configured. Click "Add Grade Cost" to get started.</div>`;
            }
        }
    }

    async saveSettings() {
        try {
            const data = {
                school_settings: {
                    school_name: document.getElementById('schoolName').value,
                    student_portal_url: document.getElementById('studentPortalUrl').value
                },
                academic_visibility: {
                    current_year_id: document.getElementById('currentYearId').value,
                    first_term_visible: document.getElementById('firstTermVisible').checked,
                    first_term_publish_at: document.getElementById('firstTermPublishAt').value || null,
                    second_term_visible: document.getElementById('secondTermVisible').checked,
                    second_term_publish_at: document.getElementById('secondTermPublishAt').value || null
                },
                display_options: {
                    display_mode: document.getElementById('displayMode').value,
                    percent_rounding: document.getElementById('percentRounding').value
                },
                pass_rules: {
                    per_subject_threshold_mode: document.getElementById('perSubjectThresholdMode').value,
                    subject_pass_min_percent: document.getElementById('subjectPassMinPercent').value || null,
                    overall_pass_rule: document.getElementById('overallPassRule').value,
                    overall_pass_min_percent: document.getElementById('overallPassMinPercent').value || null,
                    pass_condition: document.getElementById('passCondition').value
                },
                security_policy: {
                    mask_national_id_in_admin: document.getElementById('maskNationalIdInAdmin').checked,
                    student_login_requires_unique_and_national_id: document.getElementById('studentLoginRequiresUniqueAndNationalId').checked,
                    session_timeout_minutes: document.getElementById('sessionTimeoutMinutes').value
                },
                grade_costs: this.collectGradeCosts(),
                percentage_bands: this.collectPercentageBands(),
                grade_result_visibility: this.collectGradeVisibility()
            };
            
            await this.apiCall('PUT', '/admin/settings', data);
            this.showToast('success', 'Settings updated successfully');
            
        } catch (error) {
            this.showToast('error', (typeof langManager !== 'undefined' ? langManager.t('settings.failed_to_save') : 'Failed to save settings') + ': ' + error.message);
        }
    }

    collectGradeCosts() {
        const costs = [];
        const rows = document.querySelectorAll('.grade-cost-row');
        
        console.log('Collecting grade costs, found rows:', rows.length);
        
        rows.forEach(row => {
            const index = row.getAttribute('data-index');
            const yearId = document.querySelector(`.grade-cost-year[data-index="${index}"]`).value;
            const gradeId = document.querySelector(`.grade-cost-grade[data-index="${index}"]`).value;
            const amount = document.querySelector(`.grade-cost-amount[data-index="${index}"]`).value;
            
            console.log(`Row ${index}: year=${yearId}, grade=${gradeId}, amount=${amount}`);
            
            if (yearId && gradeId && amount) {
                costs.push({
                    year_id: yearId,
                    grade_id: gradeId,
                    fee_amount: parseFloat(amount)
                });
            }
        });
        
        console.log('Final grade costs collection:', costs);
        return costs;
    }

    async loadEnrolledStudentsForBulk() {
        try {
            const response = await this.apiCall('GET', '/admin/enrollments?per_page=1000&status=active&sort_by=created_at&sort_dir=desc');
            
            // Remove duplicates by student_id, keeping the most recent enrollment
            const uniqueStudents = [];
            const seenStudentIds = new Set();
            
            for (const enrollment of response.data) {
                if (!seenStudentIds.has(enrollment.student_id)) {
                    seenStudentIds.add(enrollment.student_id);
                    uniqueStudents.push(enrollment);
                }
            }
            
            this.enrolledStudentsForBulk = uniqueStudents;
        } catch (error) {
            this.showToast('error', 'Failed to load enrolled students: ' + error.message);
            this.enrolledStudentsForBulk = [];
        }
    }

    getFilteredBulkStudents() {
        if (!this.enrolledStudentsForBulk) {
            return [];
        }

        const filterYear = document.getElementById('filterCurrentYear')?.value;
        const filterGrade = document.getElementById('filterCurrentGrade')?.value;
        const filterClass = document.getElementById('filterCurrentClass')?.value;

        return this.enrolledStudentsForBulk.filter(student => {
            if (filterYear && student.year_id != filterYear) return false;
            if (filterGrade && student.grade_id != filterGrade) return false;
            if (filterClass && student.class_id != filterClass) return false;
            return true;
        });
    }

    filterBulkStudents() {
        const studentsListDiv = document.getElementById('bulkStudentsList');
        const studentCountDiv = document.getElementById('studentCount');
        
        if (studentsListDiv) {
            studentsListDiv.innerHTML = this.renderBulkStudentsList();
            
            // Update count
            const filteredStudents = this.getFilteredBulkStudents();
            const total = this.enrolledStudentsForBulk ? this.enrolledStudentsForBulk.length : 0;
            studentCountDiv.textContent = `Showing ${filteredStudents.length} of ${total} students`;
            
            // Reset select all checkbox
            const selectAllCheckbox = document.getElementById('selectAllBulkStudents');
            if (selectAllCheckbox) {
                selectAllCheckbox.checked = false;
            }
        }
    }
    
    // ===== PERCENTAGE BANDS MANAGEMENT =====
    
    renderPercentageBands(percentageBands) {
        const container = document.getElementById('percentageBandsList');
        
        if (percentageBands.length === 0) {
            container.innerHTML = '<div class="text-muted">No percentage bands configured. Click "Add Percentage Band" to get started.</div>';
            return;
        }
        
        let html = '<div class="table-responsive"><table class="table table-sm">';
        html += '<thead><tr>';
        html += '<th>Range (%)</th>';
        html += '<th>Label</th>';
        html += '<th>Color</th>';
        html += '<th>Preview</th>';
        html += '<th>Enabled</th>';
        html += '<th>Actions</th>';
        html += '</tr></thead><tbody>';
        
        percentageBands.forEach((band, index) => {
            html += `
                <tr class="percentage-band-row" data-index="${index}">
                    <td>
                        <div class="row g-1">
                            <div class="col-5">
                                <input type="number" class="form-control form-control-sm percentage-band-start" 
                                       data-index="${index}" value="${band.start_percent}" 
                                       min="0" max="100" step="0.01" placeholder="From">
                            </div>
                            <div class="col-2 text-center pt-1">to</div>
                            <div class="col-5">
                                <input type="number" class="form-control form-control-sm percentage-band-end" 
                                       data-index="${index}" value="${band.end_percent}" 
                                       min="0" max="100" step="0.01" placeholder="To">
                            </div>
                        </div>
                    </td>
                    <td>
                        <input type="text" class="form-control form-control-sm percentage-band-label" 
                               data-index="${index}" value="${band.label}" placeholder="Grade Label" maxlength="20">
                    </td>
                    <td>
                        <input type="color" class="form-control form-control-color form-control-sm percentage-band-color" 
                               data-index="${index}" value="${band.color_hex}" title="Choose color">
                    </td>
                    <td>
                        <div class="grade-preview d-flex align-items-center">
                            <span class="badge me-2" style="background-color: ${band.color_hex}; color: white;">
                                ${band.label}
                            </span>
                            <small class="text-muted">${band.start_percent}%-${band.end_percent}%</small>
                        </div>
                    </td>
                    <td>
                        <div class="form-check">
                            <input class="form-check-input percentage-band-enabled" type="checkbox" 
                                   data-index="${index}" ${band.enabled ? 'checked' : ''}>
                        </div>
                    </td>
                    <td>
                        <button class="btn btn-outline-danger btn-sm" onclick="app.removePercentageBand(${index})" 
                                title="Remove Band">
                            <i class="bi bi-trash"></i>
                        </button>
                    </td>
                </tr>
            `;
        });
        
        html += '</tbody></table></div>';
        container.innerHTML = html;
        
        // Add event listeners for real-time preview updates
        container.querySelectorAll('.percentage-band-label, .percentage-band-color, .percentage-band-start, .percentage-band-end').forEach(input => {
            input.addEventListener('input', () => this.updateBandPreview(input));
        });
    }
    
    updateBandPreview(input) {
        const index = input.getAttribute('data-index');
        const row = document.querySelector(`.percentage-band-row[data-index="${index}"]`);
        
        if (row) {
            const label = row.querySelector('.percentage-band-label').value;
            const color = row.querySelector('.percentage-band-color').value;
            const startPercent = row.querySelector('.percentage-band-start').value;
            const endPercent = row.querySelector('.percentage-band-end').value;
            
            const preview = row.querySelector('.grade-preview');
            if (preview) {
                preview.innerHTML = `
                    <span class="badge me-2" style="background-color: ${color}; color: white;">
                        ${label}
                    </span>
                    <small class="text-muted">${startPercent}%-${endPercent}%</small>
                `;
            }
        }
    }
    
    addPercentageBand() {
        const container = document.getElementById('percentageBandsList');
        const currentBands = container.querySelectorAll('.percentage-band-row');
        const newIndex = currentBands.length;
        
        // Default values for new band
        const defaultBand = {
            start_percent: newIndex === 0 ? 90 : 70 - (newIndex * 20),
            end_percent: newIndex === 0 ? 100 : 89 - (newIndex * 20),
            label: newIndex === 0 ? 'Excellent' : newIndex === 1 ? 'Good' : newIndex === 2 ? 'Pass' : 'Grade ' + (newIndex + 1),
            color_hex: newIndex === 0 ? '#4CAF50' : newIndex === 1 ? '#2196F3' : newIndex === 2 ? '#FF9800' : '#9E9E9E',
            enabled: 1
        };
        
        // If this is the first band, replace the empty message
        if (newIndex === 0) {
            this.renderPercentageBands([defaultBand]);
        } else {
            // Add to existing bands
            const existingBands = this.collectPercentageBands();
            existingBands.push(defaultBand);
            this.renderPercentageBands(existingBands);
        }
    }
    
    removePercentageBand(index) {
        if (confirm('Are you sure you want to remove this percentage band?')) {
            const existingBands = this.collectPercentageBands();
            existingBands.splice(index, 1);
            this.renderPercentageBands(existingBands);
        }
    }
    
    collectPercentageBands() {
        const bands = [];
        const rows = document.querySelectorAll('.percentage-band-row');
        
        rows.forEach(row => {
            const index = row.getAttribute('data-index');
            const startPercent = parseFloat(document.querySelector(`.percentage-band-start[data-index="${index}"]`).value) || 0;
            const endPercent = parseFloat(document.querySelector(`.percentage-band-end[data-index="${index}"]`).value) || 0;
            const label = document.querySelector(`.percentage-band-label[data-index="${index}"]`).value.trim();
            const colorHex = document.querySelector(`.percentage-band-color[data-index="${index}"]`).value;
            const enabled = document.querySelector(`.percentage-band-enabled[data-index="${index}"]`).checked;
            
            if (label && startPercent >= 0 && endPercent >= 0 && endPercent >= startPercent) {
                bands.push({
                    start_percent: startPercent,
                    end_percent: endPercent,
                    label: label,
                    color_hex: colorHex,
                    enabled: enabled ? 1 : 0
                });
            }
        });
        
        // Sort by start_percent descending (highest to lowest)
        bands.sort((a, b) => b.start_percent - a.start_percent);
        
        return bands;
    }
    
    // ===== GRADE RESULT VISIBILITY =====
    
    renderGradeVisibility(gradeVisibility) {
        const container = document.getElementById('gradeVisibilityList');
        
        if (gradeVisibility.length === 0) {
            container.innerHTML = '<div class="text-muted">No grade visibility rules configured. Click "Add Grade Visibility Rule" to get started.</div>';
            return;
        }
        
        container.innerHTML = gradeVisibility.map((visibility, index) => `
            <div class="row align-items-center mb-3 grade-visibility-row border rounded p-3" data-index="${index}">
                <div class="col-md-3">
                    <label class="form-label">Academic Year</label>
                    <select class="form-select grade-visibility-year" data-index="${index}">
                        <option value="">Select Year</option>
                        ${this.settingsData.academic_years.map(year => 
                            `<option value="${year.id}" ${visibility.year_id == year.id ? 'selected' : ''}>${year.name}</option>`
                        ).join('')}
                    </select>
                </div>
                <div class="col-md-3">
                    <label class="form-label">Grade</label>
                    <select class="form-select grade-visibility-grade" data-index="${index}">
                        <option value="">Select Grade</option>
                        ${this.settingsData.grades.map(grade => 
                            `<option value="${grade.id}" ${visibility.grade_id == grade.id ? 'selected' : ''}>${grade.title}</option>`
                        ).join('')}
                    </select>
                </div>
                <div class="col-md-2">
                    <label class="form-label">Results Visible</label>
                    <div class="form-check form-switch">
                        <input class="form-check-input grade-visibility-enabled" type="checkbox" data-index="${index}" 
                               ${visibility.results_visible ? 'checked' : ''}>
                        <label class="form-check-label">
                            ${visibility.results_visible ? 'Enabled' : 'Disabled'}
                        </label>
                    </div>
                </div>
                <div class="col-md-3">
                    <label class="form-label">Hide Reason (Optional)</label>
                    <input type="text" class="form-control grade-visibility-reason" data-index="${index}" 
                           placeholder="Reason for hiding results" value="${visibility.hide_reason || ''}" maxlength="255">
                </div>
                <div class="col-md-1">
                    <label class="form-label">&nbsp;</label>
                    <button class="btn btn-outline-danger btn-sm d-block" onclick="app.removeGradeVisibility(${index})">
                        <i class="bi bi-trash"></i>
                    </button>
                </div>
            </div>
        `).join('');
        
        // Add event listeners for the switches to update labels
        container.querySelectorAll('.grade-visibility-enabled').forEach(checkbox => {
            checkbox.addEventListener('change', function() {
                const label = this.parentElement.querySelector('.form-check-label');
                label.textContent = this.checked ? 'Enabled' : 'Disabled';
            });
        });
    }
    
    addGradeVisibility() {
        const container = document.getElementById('gradeVisibilityList');
        const existingRows = container.querySelectorAll('.grade-visibility-row');
        const index = existingRows.length;
        
        const newRow = document.createElement('div');
        newRow.className = 'row align-items-center mb-3 grade-visibility-row border rounded p-3';
        newRow.setAttribute('data-index', index);
        newRow.innerHTML = `
            <div class="col-md-3">
                <label class="form-label">Academic Year</label>
                <select class="form-select grade-visibility-year" data-index="${index}">
                    <option value="">Select Year</option>
                    ${this.settingsData.academic_years.map(year => 
                        `<option value="${year.id}">${year.name}</option>`
                    ).join('')}
                </select>
            </div>
            <div class="col-md-3">
                <label class="form-label">Grade</label>
                <select class="form-select grade-visibility-grade" data-index="${index}">
                    <option value="">Select Grade</option>
                    ${this.settingsData.grades.map(grade => 
                        `<option value="${grade.id}">${grade.title}</option>`
                    ).join('')}
                </select>
            </div>
            <div class="col-md-2">
                <label class="form-label">Results Visible</label>
                <div class="form-check form-switch">
                    <input class="form-check-input grade-visibility-enabled" type="checkbox" data-index="${index}" checked>
                    <label class="form-check-label">
                        Enabled
                    </label>
                </div>
            </div>
            <div class="col-md-3">
                <label class="form-label">Hide Reason (Optional)</label>
                <input type="text" class="form-control grade-visibility-reason" data-index="${index}" 
                       placeholder="Reason for hiding results" maxlength="255">
            </div>
            <div class="col-md-1">
                <label class="form-label">&nbsp;</label>
                <button class="btn btn-outline-danger btn-sm d-block" onclick="app.removeGradeVisibility(${index})">
                    <i class="bi bi-trash"></i>
                </button>
            </div>
        `;
        
        // Clear placeholder text if this is the first row
        if (existingRows.length === 0) {
            container.innerHTML = '';
        }
        
        container.appendChild(newRow);
        
        // Add event listener for the new switch
        const checkbox = newRow.querySelector('.grade-visibility-enabled');
        checkbox.addEventListener('change', function() {
            const label = this.parentElement.querySelector('.form-check-label');
            label.textContent = this.checked ? 'Enabled' : 'Disabled';
        });
    }
    
    removeGradeVisibility(index) {
        if (confirm('Are you sure you want to remove this grade visibility rule?')) {
            const existingVisibility = this.collectGradeVisibility();
            existingVisibility.splice(index, 1);
            this.renderGradeVisibility(existingVisibility);
        }
    }
    
    collectGradeVisibility() {
        const visibility = [];
        const rows = document.querySelectorAll('.grade-visibility-row');
        
        rows.forEach(row => {
            const index = row.getAttribute('data-index');
            const yearId = document.querySelector(`.grade-visibility-year[data-index="${index}"]`).value;
            const gradeId = document.querySelector(`.grade-visibility-grade[data-index="${index}"]`).value;
            const resultsVisible = document.querySelector(`.grade-visibility-enabled[data-index="${index}"]`).checked;
            const hideReason = document.querySelector(`.grade-visibility-reason[data-index="${index}"]`).value.trim();
            
            if (yearId && gradeId) {
                visibility.push({
                    year_id: yearId,
                    grade_id: gradeId,
                    results_visible: resultsVisible ? 1 : 0,
                    hide_reason: hideReason || null
                });
            }
        });
        
        return visibility;
    }
    
    // ===== DASHBOARD STATS =====
    
    async loadDashboardStats() {
        try {
            console.log('Loading dashboard stats...');
            // Try authenticated endpoint first, fallback to test endpoint
            let response;
            try {
                response = await this.apiCall('GET', '/admin/dashboard/stats');
            } catch (authError) {
                console.log('Auth failed, using test endpoint:', authError.message);
                // Use test endpoint as fallback
                const testResponse = await fetch(`${this.apiBase}/public/dashboard/test-stats`);
                const testData = await testResponse.json();
                response = testData;
            }
            
            if (response.success && response.data) {
                const stats = response.data;
                console.log('Dashboard stats received:', stats);
                
                // Update main stats cards with debugging
                console.log('📊 Updating dashboard stats:', {
                    totalStudents: stats.total_students,
                    activeEnrollments: stats.active_enrollments,
                    totalExams: stats.total_exams,
                    paymentAmount: stats.total_payment_amount
                });
                
                const totalStudentsEl = document.getElementById('totalStudents');
                const activeEnrollmentsEl = document.getElementById('activeEnrollments');
                const totalExamsEl = document.getElementById('totalExams');
                const paymentBalanceEl = document.getElementById('paymentBalance');
                
                if (totalStudentsEl) {
                    totalStudentsEl.textContent = stats.total_students || '0';
                    console.log('✅ Total Students updated to:', totalStudentsEl.textContent);
                }
                if (activeEnrollmentsEl) {
                    activeEnrollmentsEl.textContent = stats.active_enrollments || '0';
                    console.log('✅ Active Enrollments updated to:', activeEnrollmentsEl.textContent);
                }
                if (totalExamsEl) {
                    totalExamsEl.textContent = stats.total_exams || '0';
                    console.log('✅ Total Exams updated to:', totalExamsEl.textContent);
                }
                if (paymentBalanceEl) {
                    paymentBalanceEl.textContent = `$${(stats.total_payment_amount || 0).toLocaleString()}`;
                    console.log('✅ Payment Balance updated to:', paymentBalanceEl.textContent);
                }
                
                // Render recent activity
                this.renderRecentActivity(stats.recent_activity || []);
                
                // Force translation update after data is loaded
                setTimeout(() => {
                    if (typeof langManager !== 'undefined') {
                        console.log('🌍 Translating dashboard after stats load...');
                        langManager.forceTranslation();
                    }
                }, 200);
                
                // Render grade distribution if exists
                if (this.renderGradeDistribution) {
                    this.renderGradeDistribution(stats.grade_distribution || []);
                }
                
                // Render payment stats if exists
                if (this.renderPaymentStats) {
                    this.renderPaymentStats(stats.payment_stats || []);
                }
                
                // Render results summary if exists
                if (this.renderResultsSummary) {
                    this.renderResultsSummary(stats.results_summary || []);
                }
                
            } else {
                // If API doesn't return success, show default values
                console.log('Dashboard stats API returned no data or failed');
                this.setDefaultDashboardStats();
            }
            
        } catch (error) {
            console.error('Failed to load dashboard stats:', error);
            // Show default values instead of keeping "Loading..."
            this.setDefaultDashboardStats();
            // Only show error toast in development or if user is admin
            if (window.location.hostname === 'localhost' || this.user?.role === 'super_admin') {
                this.showToast('warning', 'Using sample dashboard data');
            }
        }
    }
    
    // Set default stats when API fails
    setDefaultDashboardStats() {
        document.getElementById('totalStudents').textContent = '0';
        document.getElementById('activeEnrollments').textContent = '0';
        document.getElementById('totalExams').textContent = '0';
        document.getElementById('paymentBalance').textContent = '$0';
        
        // Show "No recent activity" message with translation
        const recentActivityContainer = document.getElementById('recentActivity');
        if (recentActivityContainer) {
            const noActivityMessage = (typeof langManager !== 'undefined') 
                ? langManager.t('dashboard.no_activity')
                : 'No recent activity';
            recentActivityContainer.innerHTML = `<div class="text-muted text-center py-3">${noActivityMessage}</div>`;
        }
    }
    
    renderRecentActivity(activities) {
        const container = document.getElementById('recentActivity');
        if (!container) return;
        
        if (activities.length === 0) {
            const noActivityMessage = (typeof langManager !== 'undefined') 
                ? langManager.t('dashboard.no_activity')
                : 'No recent activity';
            container.innerHTML = `<div class="text-muted text-center py-3">${noActivityMessage}</div>`;
            return;
        }
        
        let html = '<div class="list-group list-group-flush">';
        activities.forEach(activity => {
            const timeAgo = this.timeAgo(activity.time);
            
            // Translate activity description if it's "New student: ..."
            let description = activity.description;
            if (description && description.startsWith('New student:')) {
                const studentName = description.replace('New student: ', '');
                const prefix = (typeof langManager !== 'undefined') 
                    ? langManager.t('dashboard.new_student_prefix')
                    : 'New student:';
                description = `${prefix} ${studentName}`;
            }
            
            html += `
                <div class="list-group-item border-0 px-0">
                    <div class="d-flex align-items-center">
                        <div class="flex-shrink-0 me-3">
                            <i class="${activity.icon} text-primary"></i>
                        </div>
                        <div class="flex-grow-1">
                            <div class="fw-500">${description}</div>
                            <small class="text-muted">${timeAgo}</small>
                        </div>
                    </div>
                </div>
            `;
        });
        html += '</div>';
        
        container.innerHTML = html;
    }
    
    renderGradeDistribution(distribution) {
        const container = document.getElementById('gradeDistribution');
        if (!container) return;
        
        if (distribution.length === 0) {
            container.innerHTML = '<div class="text-muted text-center py-3">No enrollment data</div>';
            return;
        }
        
        let html = '';
        distribution.forEach(grade => {
            const percentage = distribution.reduce((sum, g) => sum + parseInt(g.student_count), 0);
            const gradePercentage = percentage > 0 ? Math.round((grade.student_count / percentage) * 100) : 0;
            
            html += `
                <div class="d-flex justify-content-between align-items-center mb-3">
                    <div>
                        <div class="fw-500">${grade.grade_name}</div>
                        <small class="text-muted">${grade.student_count} students</small>
                    </div>
                    <div class="text-end">
                        <div class="fw-600">${gradePercentage}%</div>
                        <div class="progress" style="width: 60px; height: 6px;">
                            <div class="progress-bar bg-primary" style="width: ${gradePercentage}%"></div>
                        </div>
                    </div>
                </div>
            `;
        });
        
        container.innerHTML = html;
    }
    
    renderPaymentStats(paymentStats) {
        const container = document.getElementById('paymentChart');
        if (!container) return;
        
        if (paymentStats.length === 0) {
            container.innerHTML = '<div class="text-muted text-center py-3">No payment data</div>';
            return;
        }
        
        // Simple bar chart representation
        let html = '<div class="payment-chart">';
        const maxAmount = Math.max(...paymentStats.map(p => parseFloat(p.total_amount)));
        
        paymentStats.slice(0, 6).forEach(stat => {
            const height = maxAmount > 0 ? (parseFloat(stat.total_amount) / maxAmount) * 100 : 0;
            const monthName = new Date(stat.month + '-01').toLocaleDateString('en', { month: 'short' });
            
            html += `
                <div class="chart-bar text-center me-2" style="display: inline-block; width: 40px;">
                    <div class="bar bg-primary" style="height: ${height}px; width: 100%; margin-bottom: 5px; min-height: 5px;"></div>
                    <small class="text-muted d-block">${monthName}</small>
                    <small class="fw-600 d-block">$${parseInt(stat.total_amount).toLocaleString()}</small>
                </div>
            `;
        });
        html += '</div>';
        
        container.innerHTML = html;
    }
    
    renderResultsSummary(resultsSummary) {
        const container = document.getElementById('resultsSummary');
        if (!container) return;
        
        if (resultsSummary.length === 0) {
            container.innerHTML = '<div class="text-muted text-center py-3">No results data</div>';
            return;
        }
        
        let html = '';
        resultsSummary.forEach(grade => {
            const passRate = grade.students_with_results > 0 ? 
                Math.round((grade.passing_students / grade.students_with_results) * 100) : 0;
            
            html += `
                <div class="d-flex justify-content-between align-items-center mb-3">
                    <div>
                        <div class="fw-500">${grade.grade_name}</div>
                        <small class="text-muted">${grade.students_with_results} students tested</small>
                    </div>
                    <div class="text-end">
                        <div class="fw-600 ${passRate >= 75 ? 'text-success' : passRate >= 50 ? 'text-warning' : 'text-danger'}">${passRate}%</div>
                        <small class="text-muted">${grade.passing_students}/${grade.students_with_results} passing</small>
                    </div>
                </div>
            `;
        });
        
        container.innerHTML = html;
    }
    
    timeAgo(dateString) {
        const now = new Date();
        const date = new Date(dateString);
        const diffInMinutes = Math.floor((now - date) / (1000 * 60));
        
        if (diffInMinutes < 1) return 'Just now';
        if (diffInMinutes < 60) return `${diffInMinutes}m ago`;
        
        const diffInHours = Math.floor(diffInMinutes / 60);
        if (diffInHours < 24) return `${diffInHours}h ago`;
        
        const diffInDays = Math.floor(diffInHours / 24);
        if (diffInDays < 7) return `${diffInDays}d ago`;
        
        return date.toLocaleDateString();
    }

    // User Management
    async loadUsers() {
        try {
            const content = `
                <div class="page-header">
                    <div class="d-flex justify-content-between align-items-center">
                        <div>
                            <h1 class="h2 mb-1"><i class="bi bi-people-fill"></i> <span data-translate="users.title">User Management</span></h1>
                            <p class="text-muted mb-0" data-translate="users.subtitle">Manage admin users and their permissions</p>
                        </div>
                        <button class="btn btn-primary" onclick="app.showUserModal()">
                            <i class="bi bi-plus-circle"></i> <span data-translate="users.add_user">Add User</span>
                        </button>
                    </div>
                </div>
                
                <div class="filter-bar">
                    <div class="row align-items-end">
                        <div class="col-md-3">
                            <label class="form-label" data-translate="users.search">Search</label>
                            <input type="text" class="form-control" id="userSearch" data-translate-placeholder="users.search_placeholder" placeholder="Username or full name..." onkeyup="app.filterUsers()">
                        </div>
                        <div class="col-md-2">
                            <label class="form-label" data-translate="users.role">Role</label>
                            <select class="form-select" id="userRoleFilter" onchange="app.filterUsers()">
                                <option value="" data-translate="users.all_roles">All Roles</option>
                                <option value="admin" data-translate="users.admin">Admin</option>
                                <option value="registrar" data-translate="users.registrar">Registrar</option>
                                <option value="teacher" data-translate="users.teacher">Teacher</option>
                                <option value="accountant" data-translate="users.accountant">Accountant</option>
                            </select>
                        </div>
                        <div class="col-md-2">
                            <label class="form-label" data-translate="users.status">Status</label>
                            <select class="form-select" id="userStatusFilter" onchange="app.filterUsers()">
                                <option value="" data-translate="users.all_statuses">All Status</option>
                                <option value="1" data-translate="users.enabled">Enabled</option>
                                <option value="0" data-translate="users.disabled">Disabled</option>
                            </select>
                        </div>
                        <div class="col-md-2">
                            <label class="form-label" data-translate="users.sort_by">Sort</label>
                            <select class="form-select" id="userSort" onchange="app.filterUsers()">
                                <option value="id" data-translate="table.user_id">ID</option>
                                <option value="username" data-translate="users.username">Username</option>
                                <option value="full_name" data-translate="users.full_name">Full Name</option>
                                <option value="role" data-translate="users.role">Role</option>
                                <option value="created_at" data-translate="users.created">Created Date</option>
                                <option value="last_login" data-translate="users.last_login">Last Login</option>
                            </select>
                        </div>
                        <div class="col-md-2">
                            <label class="form-label" data-translate="users.order">Order</label>
                            <select class="form-select" id="userSortDir" onchange="app.filterUsers()">
                                <option value="desc" data-translate="users.descending">Descending</option>
                                <option value="asc" data-translate="users.ascending">Ascending</option>
                            </select>
                        </div>
                        <div class="col-md-1">
                            <button class="btn btn-outline-secondary" onclick="app.clearUserFilters()">
                                <i class="bi bi-x-circle"></i>
                            </button>
                        </div>
                    </div>
                </div>
                
                <div class="card">
                    <div class="card-body">
                        <div id="usersTableContainer">
                            <div class="loading">
                                <i class="bi bi-hourglass-split"></i> <span data-translate="users.loading">Loading users...</span>
                            </div>
                        </div>
                    </div>
                </div>
            `;
            
            document.getElementById('pageContent').innerHTML = content;
            await this.loadUsersData();
            
        } catch (error) {
            console.error('Error loading users:', error);
            document.getElementById('pageContent').innerHTML = `<div class="alert alert-danger" data-translate="users.error_loading">Error loading users</div>`;
        }
    }
    
    async loadUsersData(page = 1, perPage = 25, filters = {}) {
        try {
            // Get current filter values
            const search = filters.search || document.getElementById('userSearch')?.value || '';
            const role = filters.role || document.getElementById('userRoleFilter')?.value || '';
            const enabled = filters.enabled || document.getElementById('userStatusFilter')?.value || '';
            const sortBy = filters.sortBy || document.getElementById('userSort')?.value || 'id';
            const sortDir = filters.sortDir || document.getElementById('userSortDir')?.value || 'desc';
            
            const params = new URLSearchParams({
                page: page.toString(),
                per_page: perPage.toString(),
                sort_by: sortBy,
                sort_dir: sortDir
            });
            
            if (search) params.append('q', search);
            if (role) params.append('role', role);
            if (enabled) params.append('enabled', enabled);
            
            const response = await this.apiCall('GET', `/admin/users?${params}`);
            
            if (response.success) {
                this.renderUsersTable(response.data, response.meta);
            } else {
                throw new Error(response.message);
            }
        } catch (error) {
            console.error('Error loading users data:', error);
            document.getElementById('usersTableContainer').innerHTML = `<div class="alert alert-danger" data-translate="users.error_loading_data">Error loading users data</div>`;
        }
    }
    
    renderUsersTable(users, meta) {
        const getRoleBadgeClass = (role) => {
            switch (role) {
                case 'admin': return 'bg-danger';
                case 'registrar': return 'bg-primary';
                case 'teacher': return 'bg-success';
                case 'accountant': return 'bg-warning text-dark';
                default: return 'bg-secondary';
            }
        };
        
        const formatDate = (dateString) => {
            if (!dateString) return typeof langManager !== 'undefined' ? langManager.t('users.never') : 'Never';
            return new Date(dateString).toLocaleDateString();
        };
        
        const tableHtml = `
            <div class="table-responsive">
                <table class="table table-hover">
                    <thead>
                        <tr>
                            <th data-translate="table.user_id">ID</th>
                            <th data-translate="table.user_username">Username</th>
                            <th data-translate="table.user_full_name">Full Name</th>
                            <th data-translate="table.user_role">Role</th>
                            <th data-translate="table.user_status">Status</th>
                            <th data-translate="table.user_last_login">Last Login</th>
                            <th data-translate="table.user_created">Created</th>
                            <th data-translate="table.user_actions">Actions</th>
                        </tr>
                    </thead>
                    <tbody>
                        ${users.map(user => `
                            <tr>
                                <td>${user.id}</td>
                                <td><strong>${user.username}</strong></td>
                                <td>${user.full_name}</td>
                                <td><span class="badge ${getRoleBadgeClass(user.role)}">${user.role.charAt(0).toUpperCase() + user.role.slice(1)}</span></td>
                                <td>
                                    ${user.enabled ? 
                                        `<span class="badge bg-success">${typeof langManager !== 'undefined' ? langManager.t('users.enabled') : 'Enabled'}</span>` : 
                                        `<span class="badge bg-danger">${typeof langManager !== 'undefined' ? langManager.t('users.disabled') : 'Disabled'}</span>`
                                    }
                                </td>
                                <td>${formatDate(user.last_login)}</td>
                                <td>${formatDate(user.created_at)}</td>
                                <td>
                                    <div class="btn-group btn-group-sm">
                                        <button class="btn btn-outline-primary" onclick="app.showUserModal(${user.id})" title="Edit">
                                            <i class="bi bi-pencil"></i>
                                        </button>
                                        <button class="btn btn-outline-${user.enabled ? 'warning' : 'success'}" 
                                                onclick="app.toggleUserStatus(${user.id}, ${!user.enabled})" 
                                                title="${user.enabled ? 'Disable' : 'Enable'}">
                                            <i class="bi bi-${user.enabled ? 'pause' : 'play'}"></i>
                                        </button>
                                        ${user.id != this.user?.id ? `
                                            <button class="btn btn-outline-danger" onclick="app.deleteUser(${user.id})" title="Delete">
                                                <i class="bi bi-trash"></i>
                                            </button>
                                        ` : ''}
                                    </div>
                                </td>
                            </tr>
                        `).join('')}
                    </tbody>
                </table>
            </div>
            
            ${this.renderPagination(meta, 'app.loadUsersPage')}
        `;
        
        document.getElementById('usersTableContainer').innerHTML = tableHtml;
    }
    
    async loadUsersPage(page) {
        const perPage = 25;
        const filters = this.getCurrentUserFilters();
        await this.loadUsersData(page, perPage, filters);
    }
    
    getCurrentUserFilters() {
        return {
            search: document.getElementById('userSearch')?.value || '',
            role: document.getElementById('userRoleFilter')?.value || '',
            enabled: document.getElementById('userStatusFilter')?.value || '',
            sortBy: document.getElementById('userSort')?.value || 'id',
            sortDir: document.getElementById('userSortDir')?.value || 'desc'
        };
    }
    
    filterUsers() {
        this.loadUsersPage(1);
    }
    
    clearUserFilters() {
        document.getElementById('userSearch').value = '';
        document.getElementById('userRoleFilter').value = '';
        document.getElementById('userStatusFilter').value = '';
        document.getElementById('userSort').value = 'id';
        document.getElementById('userSortDir').value = 'desc';
        this.loadUsersPage(1);
    }
    
    showUserModal(userId = null) {
        const isEdit = userId !== null;
        const modalHtml = `
            <div class="modal fade" id="userModal" tabindex="-1">
                <div class="modal-dialog">
                    <div class="modal-content">
                        <div class="modal-header">
                            <h5 class="modal-title">
                                <i class="bi bi-person-plus"></i> <span data-translate="${isEdit ? 'users.edit_user' : 'users.add_new_user'}">${isEdit ? 'Edit User' : 'Add New User'}</span>
                            </h5>
                            <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                        </div>
                        <div class="modal-body">
                            <form id="userForm" onsubmit="app.saveUser(event, ${userId})">
                                <div class="mb-3">
                                    <label for="userUsername" class="form-label" data-translate="users.username_label">Username *</label>
                                    <input type="text" class="form-control" id="userUsername" required 
                                           minlength="3" maxlength="50" data-translate-placeholder="users.username_placeholder" placeholder="Enter username">
                                    <div class="form-text">3-50 characters, letters, numbers, and underscores only</div>
                                </div>
                                
                                <div class="mb-3">
                                    <label for="userFullName" class="form-label" data-translate="users.full_name_label">Full Name *</label>
                                    <input type="text" class="form-control" id="userFullName" required 
                                           minlength="2" maxlength="200" data-translate-placeholder="users.full_name_placeholder" placeholder="Enter full name">
                                </div>
                                
                                <div class="mb-3">
                                    <label for="userRole" class="form-label" data-translate="users.role_label">Role *</label>
                                    <select class="form-select" id="userRole" required>
                                        <option value="" data-translate="users.select_role">Select Role</option>
                                        <option value="admin" data-translate="users.admin">Admin - Full system access</option>
                                        <option value="registrar" data-translate="users.registrar">Registrar - Student management</option>
                                        <option value="teacher" data-translate="users.teacher">Teacher - Academic management</option>
                                        <option value="accountant" data-translate="users.accountant">Accountant - Financial management</option>
                                    </select>
                                </div>
                                
                                <div class="mb-3">
                                    <label for="userPassword" class="form-label" data-translate="users.password_label">Password ${isEdit ? '' : '*'}</label>
                                    <input type="password" class="form-control" id="userPassword" 
                                           ${isEdit ? '' : 'required'} minlength="6" 
                                           data-translate-placeholder="users.password_placeholder" placeholder="${isEdit ? 'Leave blank to keep current password' : 'Enter password'}">
                                    <div class="form-text">Minimum 6 characters</div>
                                </div>
                                
                                <div class="mb-3">
                                    <div class="form-check">
                                        <input class="form-check-input" type="checkbox" id="userEnabled" checked>
                                        <label class="form-check-label" for="userEnabled" data-translate="users.enabled_label">
                                            Account Enabled
                                        </label>
                                    </div>
                                </div>
                                
                                <div class="d-flex justify-content-end gap-2">
                                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal" data-translate="btn.cancel">Cancel</button>
                                    <button type="submit" class="btn btn-primary">
                                        <i class="bi bi-check-circle"></i> <span data-translate="users.save_user">${isEdit ? 'Update User' : 'Create User'}</span>
                                    </button>
                                </div>
                            </form>
                        </div>
                    </div>
                </div>
            </div>
        `;
        
        // Remove existing modal
        const existingModal = document.getElementById('userModal');
        if (existingModal) {
            existingModal.remove();
        }
        
        document.body.insertAdjacentHTML('beforeend', modalHtml);
        
        if (isEdit) {
            this.loadUserForEdit(userId);
        }
        
        const modal = new bootstrap.Modal(document.getElementById('userModal'));
        modal.show();
    }
    
    async loadUserForEdit(userId) {
        try {
            const response = await this.apiCall('GET', `/admin/users/${userId}`);
            
            if (response.success) {
                const user = response.data;
                document.getElementById('userUsername').value = user.username;
                document.getElementById('userFullName').value = user.full_name;
                document.getElementById('userRole').value = user.role;
                document.getElementById('userEnabled').checked = user.enabled;
            } else {
                throw new Error(response.message);
            }
        } catch (error) {
            console.error('Error loading user:', error);
            this.showError(null, typeof langManager !== 'undefined' ? langManager.t('users.error_loading_user') : 'Error loading user data');
        }
    }
    
    async saveUser(event, userId = null) {
        event.preventDefault();
        
        const formData = {
            username: document.getElementById('userUsername').value,
            full_name: document.getElementById('userFullName').value,
            role: document.getElementById('userRole').value,
            enabled: document.getElementById('userEnabled').checked
        };
        
        const password = document.getElementById('userPassword').value;
        if (password) {
            formData.password = password;
        }
        
        try {
            const isEdit = userId !== null;
            const method = isEdit ? 'PUT' : 'POST';
            const url = isEdit ? `/admin/users/${userId}` : '/admin/users';
            
            const response = await this.apiCall(method, url, formData);
            
            if (response.success) {
                const modal = bootstrap.Modal.getInstance(document.getElementById('userModal'));
                modal.hide();
                
                this.showSuccess(typeof langManager !== 'undefined' ? langManager.t(isEdit ? 'users.user_updated' : 'users.user_created') : `User ${isEdit ? 'updated' : 'created'} successfully`);
                await this.loadUsersPage(1);
            } else {
                throw new Error(response.message);
            }
        } catch (error) {
            console.error('Error saving user:', error);
            this.showError(null, error.message || (typeof langManager !== 'undefined' ? langManager.t('users.error_saving') : 'Error saving user'));
        }
    }
    
    async toggleUserStatus(userId, newStatus) {
        try {
            const response = await this.apiCall('POST', `/admin/users/${userId}/toggle-status`, {});
            
            if (response.success) {
                this.showSuccess(response.message);
                await this.loadUsersPage(1);
            } else {
                throw new Error(response.message);
            }
        } catch (error) {
            console.error('Error toggling user status:', error);
            this.showError(null, error.message || (typeof langManager !== 'undefined' ? langManager.t('users.error_updating_status') : 'Error updating user status'));
        }
    }
    
    async deleteUser(userId) {
        if (!confirm(typeof langManager !== 'undefined' ? langManager.t('users.confirm_delete') : 'Are you sure you want to delete this user? This action cannot be undone.')) {
            return;
        }
        
        try {
            const response = await this.apiCall('DELETE', `/admin/users/${userId}`);
            
            if (response.success) {
                this.showSuccess(response.message);
                await this.loadUsersPage(1);
            } else {
                throw new Error(response.message);
            }
        } catch (error) {
            console.error('Error deleting user:', error);
            this.showError(null, error.message || (typeof langManager !== 'undefined' ? langManager.t('users.error_deleting') : 'Error deleting user'));
        }
    }
}

// Initialize app when DOM is loaded
document.addEventListener('DOMContentLoaded', () => {
    // Add delay to ensure all scripts are fully loaded
    setTimeout(() => {
        console.log('🚀 Initializing AdminApp...');
        window.app = new AdminApp();
        
        // Ensure app is globally accessible and ready
        setTimeout(() => {
            console.log('✅ AdminApp ready, window.app:', !!window.app);
            
            // Force re-setup navigation with working app
            if (window.langManager) {
                console.log('🔧 Re-setting up navigation with ready app');
                window.langManager.ensureEventListeners();
            }
        }, 500);
    }, 100);
});