// Language Manager
class LanguageManager {
    constructor() {
        this.currentLang = localStorage.getItem('admin_language') || 'en';
        this.translations = {
            'en': translations_en,
            'ar': translations_ar
        };
        this.rtlLanguages = ['ar'];
        
        this.init();
    }
    
    init() {
        this.applyLanguage();
        this.setupLanguageSwitcher();
        
        // Apply translations immediately for any existing content
        setTimeout(() => {
            this.translatePage();
            this.ensureEventListeners();
        }, 100);
        
        // Also apply translations when DOM changes
        this.observeContentChanges();
    }
    
    // Get translation for a key
    t(key) {
        const translation = this.translations[this.currentLang][key];
        return translation || key;
    }
    
    // Change language
    setLanguage(lang) {
        if (this.translations[lang]) {
            console.log('Switching language to:', lang);
            this.currentLang = lang;
            localStorage.setItem('admin_language', lang);
            
            this.applyLanguage();
            
            // Force immediate translation of all content
            this.forceTranslation();
            
            // Re-setup event listeners to ensure they still work
            this.ensureEventListeners();
            
            // Reload the current page content to apply translations
            if (window.app && window.app.currentPage) {
                setTimeout(() => {
                    this.forceTranslation(); // Apply translations again after page reload
                    window.app.navigateTo(window.app.currentPage);
                }, 300);
            }
            
            // Additional force translation after a short delay to catch any late-loading content
            setTimeout(() => {
                this.forceTranslation();
                this.ensureEventListeners();
            }, 1000);
        }
    }
    
    // Apply language settings (direction, fonts, etc.)
    applyLanguage() {
        const isRTL = this.rtlLanguages.includes(this.currentLang);
        const html = document.documentElement;
        
        // Set direction
        html.setAttribute('dir', isRTL ? 'rtl' : 'ltr');
        html.setAttribute('lang', this.currentLang);
        
        // Add/remove RTL class
        document.body.classList.toggle('rtl', isRTL);
        document.body.classList.toggle('arabic', this.currentLang === 'ar');
        
        // Update document direction for Bootstrap and other components
        if (isRTL) {
            // Load RTL Bootstrap if needed
            this.loadRTLStyles();
        } else {
            // Remove RTL styles when switching to LTR
            this.removeRTLStyles();
        }
    }
    
    // Load RTL styles for Arabic
    loadRTLStyles() {
        const existingRTL = document.getElementById('rtl-styles');
        if (!existingRTL) {
            const rtlStyles = document.createElement('link');
            rtlStyles.id = 'rtl-styles';
            rtlStyles.rel = 'stylesheet';
            rtlStyles.href = 'https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.rtl.min.css';
            document.head.appendChild(rtlStyles);
            
            // Add custom RTL styles
            const customRTL = document.createElement('style');
            customRTL.id = 'custom-rtl';
            customRTL.textContent = `
                /* Minimal RTL adjustments to avoid breaking functionality */
                html[dir="rtl"] .sidebar .nav-link {
                    text-align: right;
                    padding-right: 1rem;
                    padding-left: 0.5rem;
                }
                html[dir="rtl"] .sidebar .nav-link i {
                    margin-left: 0.8rem;
                    margin-right: 0;
                }
                html[dir="rtl"] .table th, 
                html[dir="rtl"] .table td {
                    text-align: right;
                }
                
                /* Arabic font styling */
                .arabic {
                    font-family: 'Segoe UI', 'Tahoma', 'Noto Sans Arabic', 'Arabic UI Text', sans-serif;
                }
                .arabic .sidebar .nav-link,
                .arabic h1, .arabic h2, .arabic h3, .arabic h4, .arabic h5, .arabic h6 {
                    font-family: 'Segoe UI', 'Tahoma', 'Noto Sans Arabic', 'Arabic UI Text', sans-serif;
                    font-weight: 600;
                }
                
                /* Language switcher positioning - ALWAYS in same position */
                .language-switcher {
                    position: fixed !important;
                    top: 20px !important;
                    right: 20px !important;
                    left: auto !important;
                    z-index: 9999 !important;
                    background: rgba(255, 255, 255, 0.95) !important;
                    backdrop-filter: blur(10px) !important;
                    border-radius: 25px !important;
                    padding: 10px 15px !important;
                    box-shadow: 0 4px 20px rgba(0, 0, 0, 0.15) !important;
                    border: 2px solid rgba(102, 126, 234, 0.3) !important;
                    min-width: 120px !important;
                }
                
                /* Keep language switcher in SAME position even in RTL */
                html[dir="rtl"] .language-switcher {
                    position: fixed !important;
                    top: 20px !important;
                    right: 20px !important;
                    left: auto !important;
                    z-index: 9999 !important;
                }
                
                .language-switcher select {
                    border: none !important;
                    background: transparent !important;
                    font-size: 14px !important;
                    font-weight: 500 !important;
                    color: #495057 !important;
                    cursor: pointer !important;
                    outline: none !important;
                    font-family: inherit !important;
                    pointer-events: auto !important;
                }
                
                /* Preserve click functionality for admin navigation */
                html[dir="rtl"] .nav-link,
                html[dir="rtl"] .btn,
                html[dir="rtl"] .sidebar .nav-link {
                    pointer-events: auto !important;
                    cursor: pointer !important;
                    position: relative !important;
                    z-index: 10 !important;
                }
            `;
            document.head.appendChild(customRTL);
        }
    }
    
    // Setup language switcher
    setupLanguageSwitcher() {
        // Remove existing switcher
        const existing = document.getElementById('language-switcher');
        if (existing) existing.remove();
        
        // Create language switcher
        const switcher = document.createElement('div');
        switcher.id = 'language-switcher';
        switcher.className = 'language-switcher';
        
        // Force consistent positioning with inline styles as backup
        switcher.style.cssText = `
            position: fixed !important;
            top: 20px !important;
            right: 20px !important;
            left: auto !important;
            z-index: 9999 !important;
            background: rgba(255, 255, 255, 0.95) !important;
            backdrop-filter: blur(10px) !important;
            border-radius: 25px !important;
            padding: 10px 15px !important;
            box-shadow: 0 4px 20px rgba(0, 0, 0, 0.15) !important;
            border: 2px solid rgba(102, 126, 234, 0.3) !important;
            min-width: 120px !important;
        `;
        
        switcher.innerHTML = `
            <select onchange="langManager.setLanguage(this.value)" style="border: none !important; background: transparent !important; font-size: 14px !important; font-weight: 500 !important; color: #495057 !important; cursor: pointer !important; font-family: inherit !important; outline: none !important; pointer-events: auto !important;">
                <option value="en" ${this.currentLang === 'en' ? 'selected' : ''}>🇺🇸 English</option>
                <option value="ar" ${this.currentLang === 'ar' ? 'selected' : ''}>🇸🇦 العربية</option>
            </select>
        `;
        
        document.body.appendChild(switcher);
    }
    
    // Translate all elements with data-translate attribute
    translatePage() {
        const elements = document.querySelectorAll('[data-translate]');
        elements.forEach(element => {
            const key = element.getAttribute('data-translate');
            element.textContent = this.t(key);
        });
        
        // Translate placeholders
        const placeholders = document.querySelectorAll('[data-translate-placeholder]');
        placeholders.forEach(element => {
            const key = element.getAttribute('data-translate-placeholder');
            element.setAttribute('placeholder', this.t(key));
        });
        
        // Translate titles
        const titles = document.querySelectorAll('[data-translate-title]');
        titles.forEach(element => {
            const key = element.getAttribute('data-translate-title');
            element.setAttribute('title', this.t(key));
        });
    }
    
    // Get current language
    getCurrentLanguage() {
        return this.currentLang;
    }
    
    // Check if current language is RTL
    isRTL() {
        return this.rtlLanguages.includes(this.currentLang);
    }
    
    // Ensure event listeners are working after language change
    ensureEventListeners() {
        // Check if already fixed to prevent infinite loops
        if (document.querySelector('[data-page][data-nav-fixed]')) {
            console.log('🔧 Navigation already fixed, skipping');
            return;
        }
        
        // ULTRA-AGGRESSIVE fix for navigation links
        const navLinks = document.querySelectorAll('[data-page]');
        console.log(`🔧 Fixing ${navLinks.length} navigation links`);
        
        navLinks.forEach((link, index) => {
            // Apply styles to ensure clickability
            link.style.cssText += `
                pointer-events: auto !important;
                cursor: pointer !important;
                position: relative !important;
                z-index: ${50 + index} !important;
                display: block !important;
                user-select: none !important;
                -webkit-user-select: none !important;
                -webkit-tap-highlight-color: transparent !important;
            `;
            
            // Remove ALL existing event listeners first
            const oldLink = link.cloneNode(true);
            link.parentNode.replaceChild(oldLink, link);
            
            // Add multiple event handlers for maximum compatibility
            ['click', 'mousedown', 'touchstart'].forEach(eventType => {
                oldLink.addEventListener(eventType, function(e) {
                    if (eventType === 'click') {
                        e.preventDefault();
                        e.stopPropagation();
                        e.stopImmediatePropagation();
                        
                        const page = this.getAttribute('data-page');
                        console.log(`🚀 Navigation click detected: ${page}`);
                        
                        if (page && window.app && typeof window.app.navigateTo === 'function') {
                            // Update active state immediately
                            document.querySelectorAll('.nav-link').forEach(l => l.classList.remove('active'));
                            this.classList.add('active');
                            
                            // Navigate
                            window.app.navigateTo(page);
                        }
                    }
                }, true); // Use capture phase
            });
            
            // Mark as fixed to prevent re-processing
            oldLink.setAttribute('data-nav-fixed', 'true');
            
            // Add visual feedback
            oldLink.addEventListener('mousedown', function() {
                this.style.opacity = '0.7';
            });
            
            oldLink.addEventListener('mouseup', function() {
                this.style.opacity = '1';
            });
            
            oldLink.addEventListener('mouseleave', function() {
                this.style.opacity = '1';
            });
        });
        
        // Fix buttons
        const buttons = document.querySelectorAll('.btn, button');
        buttons.forEach(button => {
            button.style.cssText += `
                pointer-events: auto !important;
                cursor: pointer !important;
                position: relative !important;
                z-index: 20 !important;
                user-select: none !important;
            `;
        });
        
        // Ensure language switcher positioning
        this.fixLanguageSwitcherPosition();
        
        console.log('Event listeners reinforced:', navLinks.length, 'nav links updated');
    }
    
    // Remove RTL styles when switching back to LTR
    removeRTLStyles() {
        const rtlStyles = document.getElementById('rtl-styles');
        const customRTL = document.getElementById('custom-rtl');
        
        if (rtlStyles) rtlStyles.remove();
        if (customRTL) customRTL.remove();
        
        // Reset any RTL classes
        document.body.classList.remove('rtl', 'arabic');
    }
    
    // Fix language switcher position
    fixLanguageSwitcherPosition() {
        const languageSwitcherContainer = document.querySelector('.language-switcher');
        const languageSwitcher = document.querySelector('.language-switcher select');
        
        if (languageSwitcherContainer) {
            languageSwitcherContainer.style.cssText = `
                position: fixed !important;
                top: 20px !important;
                right: 20px !important;
                left: auto !important;
                z-index: 9999 !important;
                background: rgba(255, 255, 255, 0.95) !important;
                backdrop-filter: blur(10px) !important;
                border-radius: 25px !important;
                padding: 10px 15px !important;
                box-shadow: 0 4px 20px rgba(0, 0, 0, 0.15) !important;
                border: 2px solid rgba(102, 126, 234, 0.3) !important;
                min-width: 120px !important;
            `;
        }
        
        if (languageSwitcher) {
            languageSwitcher.style.pointerEvents = 'auto';
            languageSwitcher.style.cursor = 'pointer';
            languageSwitcher.style.zIndex = '9999';
        }
    }
    
    // Observe content changes and apply translations
    observeContentChanges() {
        // Create observer to watch for new content
        const observer = new MutationObserver((mutations) => {
            let shouldTranslate = false;
            
            mutations.forEach(mutation => {
                if (mutation.type === 'childList' && mutation.addedNodes.length > 0) {
                    mutation.addedNodes.forEach(node => {
                        if (node.nodeType === Node.ELEMENT_NODE) {
                            // Check if the new content has translation attributes
                            if (node.querySelector && (
                                node.querySelector('[data-translate]') || 
                                node.hasAttribute && node.hasAttribute('data-translate')
                            )) {
                                shouldTranslate = true;
                            }
                        }
                    });
                }
            });
            
            if (shouldTranslate) {
                setTimeout(() => {
                    this.translatePage();
                    this.ensureEventListeners();
                }, 50);
            }
        });
        
        // Start observing
        observer.observe(document.body, {
            childList: true,
            subtree: true
        });
    }
    
    // Force immediate translation of all existing content
    forceTranslation() {
        // Translate everything immediately
        this.translatePage();
        
        // Also translate any content that might be hidden
        const hiddenElements = document.querySelectorAll('[style*="display: none"], [style*="display:none"]');
        hiddenElements.forEach(element => {
            const translatableElements = element.querySelectorAll('[data-translate]');
            translatableElements.forEach(el => {
                const key = el.getAttribute('data-translate');
                const params = el.hasAttribute('data-translate-params') 
                    ? JSON.parse(el.getAttribute('data-translate-params')) 
                    : {};
                el.textContent = this.t(key, params);
            });
        });
        
        console.log('Force translation applied for language:', this.currentLang);
    }
    
    // Special method to handle admin app initialization
    initializeForAdmin() {
        console.log('Initializing language manager for admin app');
        
        // Apply current language settings
        this.applyLanguage();
        
        // Force translation of all existing content
        this.forceTranslation();
        
        // Ensure all navigation is working
        this.ensureEventListeners();
        
        // Setup periodic checks to maintain functionality
        this.startMaintenanceTimer();
    }
    
    // Start maintenance timer to periodically check and fix issues
    startMaintenanceTimer() {
        // DISABLED: This was causing infinite loops that broke navigation
        return;
        
        setInterval(() => {
            // Check if language switcher is in correct position
            const switcher = document.querySelector('.language-switcher');
            if (switcher) {
                const styles = window.getComputedStyle(switcher);
                if (styles.left === '20px' && styles.right !== '20px') {
                    console.log('🔧 Auto-fixing language switcher position');
                    this.fixLanguageSwitcherPosition();
                }
            }
            
            // Check if navigation links are still clickable and fix them
            const navLinks = document.querySelectorAll('[data-page]');
            let problemLinks = 0;
            navLinks.forEach((link, index) => {
                const styles = window.getComputedStyle(link);
                if (styles.pointerEvents === 'none' || styles.cursor !== 'pointer') {
                    // Fix basic styles
                    link.style.pointerEvents = 'auto';
                    link.style.cursor = 'pointer';
                    link.style.position = 'relative';
                    link.style.zIndex = (20 + index).toString();
                    problemLinks++;
                    
                    // Ensure click handler exists
                    if (!link.hasAttribute('data-click-fixed')) {
                        link.addEventListener('click', function(e) {
                            e.preventDefault();
                            e.stopPropagation();
                            const page = this.getAttribute('data-page');
                            if (page && window.app && typeof window.app.navigateTo === 'function') {
                                console.log('🔧 Emergency navigation to:', page);
                                window.app.navigateTo(page);
                            }
                        }, true);
                        link.setAttribute('data-click-fixed', 'true');
                    }
                }
            });
            
            if (problemLinks > 0) {
                console.log(`🔧 Auto-fixed ${problemLinks} navigation links`);
            }
            
            // Also check and fix buttons
            const buttons = document.querySelectorAll('.btn, button');
            buttons.forEach(button => {
                if (button.style.pointerEvents === 'none') {
                    button.style.pointerEvents = 'auto';
                    button.style.cursor = 'pointer';
                }
            });
            
        }, 2000); // Check every 2 seconds for more responsive fixing
    }
}

// Initialize language manager
const langManager = new LanguageManager();

// Ensure translations are applied when page loads (especially after refresh)
document.addEventListener('DOMContentLoaded', function() {
    console.log('DOM loaded, applying translations for language:', langManager.getCurrentLanguage());
    
    // Apply initial translations and fix navigation immediately
    setTimeout(() => {
        langManager.forceTranslation();
        langManager.ensureEventListeners();
        
        // FIRST: Remove any invisible blocking elements
        console.log('🧹 Cleaning up invisible blocking elements...');
        
        // Remove modal backdrops that might be invisible but blocking
        document.querySelectorAll('.modal-backdrop').forEach(backdrop => {
            console.log('🗑️ Removing modal backdrop');
            backdrop.remove();
        });
        
        // Remove any elements with high z-index that might be blocking
        document.querySelectorAll('*').forEach(el => {
            const styles = window.getComputedStyle(el);
            const zIndex = parseInt(styles.zIndex);
            
            if (zIndex > 1000 && 
                el.offsetWidth > 200 && 
                el.offsetHeight > 200 && 
                !el.hasAttribute('data-page') &&
                !el.closest('.language-switcher')) {
                
                const rect = el.getBoundingClientRect();
                console.log(`🧹 Found potential blocker: ${el.tagName} (z-index: ${zIndex}, size: ${rect.width}x${rect.height})`);
                
                // Make it non-blocking
                el.style.pointerEvents = 'none';
            }
        });
        
        // Remove body modal classes
        document.body.classList.remove('modal-open');
        
        // NUCLEAR OPTION: Force all navigation links to work
        const navLinks = document.querySelectorAll('[data-page]');
        console.log(`🔧 NUCLEAR FIX: Found ${navLinks.length} navigation links`);
        
        navLinks.forEach((link, index) => {
            // Ultra-aggressive styling with MAXIMUM z-index
            link.style.cssText = `
                pointer-events: auto !important;
                cursor: pointer !important;
                position: relative !important;
                z-index: ${2147483647 - index} !important;
                display: block !important;
                user-select: none !important;
                -webkit-user-select: none !important;
                opacity: 1 !important;
                visibility: visible !important;
                background: rgba(0,0,0,0.01) !important;
                border: 1px solid transparent !important;
            `;
            
            // Remove ALL event listeners and clone
            const ultraLink = link.cloneNode(true);
            link.parentNode.replaceChild(ultraLink, link);
            
            // TRIPLE emergency click handler with app waiting
            ultraLink.onclick = function(e) {
                e.preventDefault();
                e.stopPropagation();
                e.stopImmediatePropagation();
                const page = this.getAttribute('data-page');
                console.log('🚨🚨🚨 ULTRA EMERGENCY CLICK:', page);
                
                // Wait for app to be ready
                const navigateWithWait = () => {
                    if (window.app && window.app.navigateTo) {
                        console.log('✅ App ready, navigating to:', page);
                        document.querySelectorAll('.nav-link').forEach(l => l.classList.remove('active'));
                        this.classList.add('active');
                        window.app.navigateTo(page);
                    } else {
                        console.log('⏳ App not ready, waiting... (app exists:', !!window.app, ')');
                        setTimeout(navigateWithWait, 100);
                    }
                };
                
                navigateWithWait();
                return false;
            };
            
            // Additional event listeners for extra safety
            ['click', 'mousedown'].forEach(eventType => {
                ultraLink.addEventListener(eventType, function(e) {
                    if (eventType === 'click') {
                        e.preventDefault();
                        e.stopPropagation();
                        const page = this.getAttribute('data-page');
                        console.log(`🚨 NUCLEAR ${eventType.toUpperCase()}:`, page);
                        
                        // Wait for app with timeout
                        const waitForApp = (attempts = 0) => {
                            if (window.app && window.app.navigateTo) {
                                window.app.navigateTo(page);
                            } else if (attempts < 50) { // Max 5 seconds
                                setTimeout(() => waitForApp(attempts + 1), 100);
                            } else {
                                console.error('❌ App never became ready after 5 seconds');
                            }
                        };
                        waitForApp();
                    }
                }, true);
            });
        });
    }, 200);
    
    // Apply translations again after a longer delay to catch dynamic content
    setTimeout(() => {
        langManager.forceTranslation();
        langManager.ensureEventListeners();
        console.log('Final translation sweep completed');
    }, 2000);
});

// Also apply when the page becomes visible (in case of tab switching)
document.addEventListener('visibilitychange', function() {
    if (!document.hidden) {
        setTimeout(() => {
            langManager.forceTranslation();
            langManager.ensureEventListeners();
        }, 200);
    }
});

// Make langManager globally accessible for debugging
window.langManager = langManager;