// Verify Admin Panel Language Fixes
const fs = require('fs');
const path = require('path');

console.log('🔧 Verifying Admin Panel Language Fixes...\n');

try {
    // Check if the language manager file exists and has the fixes
    const langManagerPath = path.join(__dirname, 'js/lang/lang-manager.js');
    
    if (!fs.existsSync(langManagerPath)) {
        console.log('❌ Language manager file not found');
        process.exit(1);
    }
    
    const langManagerContent = fs.readFileSync(langManagerPath, 'utf8');
    
    // Check for key fixes
    const fixes = [
        {
            name: 'ensureEventListeners method added',
            check: langManagerContent.includes('ensureEventListeners()'),
            pattern: 'ensureEventListeners()'
        },
        {
            name: 'Fixed language switcher positioning',
            check: langManagerContent.includes('position: fixed !important') && 
                   langManagerContent.includes('right: 20px !important'),
            pattern: 'position: fixed !important; top: 20px !important; right: 20px !important'
        },
        {
            name: 'RTL positioning override',
            check: langManagerContent.includes('html[dir="rtl"] .language-switcher') &&
                   langManagerContent.includes('right: 20px !important'),
            pattern: 'html[dir="rtl"] .language-switcher'
        },
        {
            name: 'Click event preservation',
            check: langManagerContent.includes('pointer-events: auto !important') &&
                   langManagerContent.includes('cursor: pointer !important'),
            pattern: 'pointer-events: auto !important'
        },
        {
            name: 'RTL cleanup method',
            check: langManagerContent.includes('removeRTLStyles()') &&
                   langManagerContent.includes('classList.remove(\'rtl\', \'arabic\')'),
            pattern: 'removeRTLStyles()'
        },
        {
            name: 'Navigation elements protection',
            check: langManagerContent.includes('[data-page], .nav-link, .sidebar .nav-link') &&
                   langManagerContent.includes('style.pointerEvents = \'auto\''),
            pattern: 'nav-link protection'
        },
        {
            name: 'Button elements protection',
            check: langManagerContent.includes('querySelectorAll(\'.btn, button\')') &&
                   langManagerContent.includes('style.cursor = \'pointer\''),
            pattern: 'button protection'
        },
        {
            name: 'Language switch with event listener reset',
            check: langManagerContent.includes('this.ensureEventListeners();') &&
                   langManagerContent.includes('setTimeout(() => {'),
            pattern: 'event listener reset on language switch'
        }
    ];
    
    console.log('🔍 Fix Verification Results:');
    let passedFixes = 0;
    
    fixes.forEach((fix, index) => {
        if (fix.check) {
            console.log(`✅ ${index + 1}. ${fix.name}`);
            passedFixes++;
        } else {
            console.log(`❌ ${index + 1}. ${fix.name}`);
            console.log(`   Missing: ${fix.pattern}`);
        }
    });
    
    const successRate = (passedFixes / fixes.length) * 100;
    console.log(`\n📊 Success Rate: ${passedFixes}/${fixes.length} (${successRate.toFixed(1)}%)\n`);
    
    if (successRate === 100) {
        console.log('🎉 All fixes have been successfully applied to the admin panel!\n');
    } else if (successRate >= 80) {
        console.log('⚠️  Most fixes applied, but some issues remain.\n');
    } else {
        console.log('❌ Significant issues found. Please review the fixes.\n');
    }
    
    // Check translation files
    const arPath = path.join(__dirname, 'js/lang/ar.js');
    const enPath = path.join(__dirname, 'js/lang/en.js');
    
    const arExists = fs.existsSync(arPath);
    const enExists = fs.existsSync(enPath);
    
    console.log('📁 Language Files:');
    console.log(`${arExists ? '✅' : '❌'} Arabic translations (ar.js)`);
    console.log(`${enExists ? '✅' : '❌'} English translations (en.js)`);
    
    if (arExists && enExists) {
        // Count translations
        const arContent = fs.readFileSync(arPath, 'utf8');
        const enContent = fs.readFileSync(enPath, 'utf8');
        
        const arMatches = arContent.match(/'[^']+'\s*:/g) || [];
        const enMatches = enContent.match(/'[^']+'\s*:/g) || [];
        
        console.log(`📝 Translation Keys: AR(${arMatches.length}) EN(${enMatches.length})`);
    }
    
    console.log('\n🧪 Testing Instructions:');
    console.log('1. Visit: http://localhost/student1/admin/test-admin-lang.html');
    console.log('2. Check language switcher is in top-right corner');
    console.log('3. Click navigation links - should work on first click');
    console.log('4. Switch to Arabic - switcher should stay in same position');
    console.log('5. Test buttons and links work in both languages');
    console.log('6. Switch back to English - RTL should be removed');
    
    console.log('\n🎯 Main Portal:');
    console.log('   http://localhost/student1/admin/');
    
    console.log('\n✨ Expected Results:');
    console.log('- Language switcher always floats top-right');
    console.log('- Navigation and buttons work on first click');
    console.log('- Smooth language switching without layout breaks');
    console.log('- Proper RTL layout in Arabic');
    console.log('- Clean LTR layout when switching back to English');
    
} catch (error) {
    console.error('❌ Error during verification:', error.message);
    process.exit(1);
}

console.log('\n🔧 Admin panel language fixes verification completed!');