<?php
// Direct CSV download script - bypasses the framework
error_reporting(E_ALL);
ini_set('display_errors', 1);

require_once __DIR__ . '/../src/Core/Database.php';
require_once __DIR__ . '/../src/Services/JWTService.php';

use App\Core\Database;
use App\Services\JWTService;

// Check for exam ID
if (!isset($_GET['exam_id'])) {
    http_response_code(400);
    header('Content-Type: application/json');
    echo json_encode(['error' => 'Exam ID is required']);
    exit;
}

$examId = $_GET['exam_id'];

// Check authorization
$jwtService = new JWTService();
$headers = getallheaders();
$authHeader = $headers['Authorization'] ?? '';

if (!$authHeader || strpos($authHeader, 'Bearer ') !== 0) {
    http_response_code(401);
    header('Content-Type: application/json');
    echo json_encode(['error' => 'Unauthorized']);
    exit;
}

$token = substr($authHeader, 7);
$payload = $jwtService->decode($token);

if (!$payload || !isset($payload['role']) || $payload['role'] !== 'admin') {
    http_response_code(401);
    header('Content-Type: application/json');
    echo json_encode(['error' => 'Unauthorized']);
    exit;
}

try {
    $db = Database::getConnection();
    
    // Get exam details
    $stmt = $db->prepare("
        SELECT e.*, ay.name as year_name, t.name as term_name, g.title as grade_title
        FROM exams e
        LEFT JOIN academic_years ay ON e.year_id = ay.id
        LEFT JOIN terms t ON e.term_id = t.id
        LEFT JOIN grades g ON e.grade_id = g.id
        WHERE e.id = ?
    ");
    $stmt->execute([$examId]);
    $exam = $stmt->fetch();
    
    if (!$exam) {
        http_response_code(404);
        header('Content-Type: application/json');
        echo json_encode(['error' => 'Exam not found']);
        exit;
    }
    
    // Get all enrolled students for this exam's grade/year with class info
    $sql = "SELECT s.id as student_id, s.full_name as student_name, s.login_code, 
                   s.national_id, c.name as class_name,
                   er.score as current_score
            FROM enrollments en
            LEFT JOIN students s ON en.student_id = s.id
            LEFT JOIN classes c ON en.class_id = c.id
            LEFT JOIN exam_results er ON er.student_id = s.id AND er.exam_id = ?
            WHERE en.year_id = ? AND en.grade_id = ? AND en.status = 'active' 
              AND s.enabled = 1 AND s.id IS NOT NULL
            ORDER BY c.name, s.full_name";
    
    $stmt = $db->prepare($sql);
    $stmt->execute([$examId, $exam['year_id'], $exam['grade_id']]);
    $students = $stmt->fetchAll();
    
    if (empty($students)) {
        http_response_code(404);
        header('Content-Type: application/json');
        echo json_encode(['error' => 'No enrolled students found for this exam']);
        exit;
    }
    
    // Generate CSV content
    $csvContent = "student_name,student_id,national_id,login_code,score\n";
    
    foreach ($students as $student) {
        $csvContent .= sprintf(
            "%s,%s,%s,%s,%s\n",
            '"' . str_replace('"', '""', $student['student_name'] ?? 'Unknown') . '"',
            $student['student_id'] ?? '',
            $student['national_id'] ?? '',
            $student['login_code'] ?? '',
            $student['current_score'] ?? ''
        );
    }
    
    // Clean output buffer
    if (ob_get_level()) {
        ob_end_clean();
    }
    
    // Set headers for CSV download
    $filename = sprintf(
        'results_template_%s_%s_%s_%s.csv',
        preg_replace('/[^a-zA-Z0-9_-]/', '_', $exam['title']),
        preg_replace('/[^a-zA-Z0-9_-]/', '_', $exam['year_name'] ?? 'unknown'),
        preg_replace('/[^a-zA-Z0-9_-]/', '_', $exam['term_name'] ?? 'unknown'),
        date('Y-m-d')
    );
    
    header('Content-Type: text/csv; charset=utf-8');
    header('Content-Disposition: attachment; filename="' . $filename . '"');
    header('Content-Length: ' . strlen($csvContent));
    header('Cache-Control: no-cache, must-revalidate');
    header('Pragma: no-cache');
    
    echo $csvContent;
    exit;
    
} catch (Exception $e) {
    http_response_code(500);
    header('Content-Type: application/json');
    echo json_encode(['error' => 'Failed to generate template: ' . $e->getMessage()]);
    exit;
}