<?php

error_reporting(E_ALL);
ini_set('display_errors', 1);

require_once __DIR__ . '/../src/Core/Database.php';
require_once __DIR__ . '/../src/Core/Router.php';
require_once __DIR__ . '/../src/Core/Request.php';
require_once __DIR__ . '/../src/Core/Response.php';
require_once __DIR__ . '/../src/Core/Controller.php';

// Auto-load controllers
spl_autoload_register(function ($class) {
    $prefix = 'App\\';
    $base_dir = __DIR__ . '/../src/';
    
    $len = strlen($prefix);
    if (strncmp($prefix, $class, $len) !== 0) {
        return;
    }
    
    $relative_class = substr($class, $len);
    $file = $base_dir . str_replace('\\', '/', $relative_class) . '.php';
    
    if (file_exists($file)) {
        require $file;
    }
});

use App\Core\Router;
use App\Core\Request;
use App\Core\Response;

// Set headers
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST, PUT, DELETE, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization');

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit();
}

try {
    $router = new Router();
    $request = new Request();
    
    // Define routes directly instead of loading from file
    // Authentication routes
    $router->post('/auth/admin/login', 'App\Controllers\AuthController@adminLogin');
    $router->post('/auth/student/login', 'App\Controllers\AuthController@studentLogin');
    $router->get('/auth/me', 'App\Controllers\AuthController@me', ['App\Middleware\AuthMiddleware']);
    $router->post('/auth/refresh', 'App\Controllers\AuthController@refresh', ['App\Middleware\AuthMiddleware']);
    
    // Academic Years
    $router->get('/admin/academic-years', 'App\Controllers\AcademicYearController@index', [
        'App\Middleware\AdminAuthMiddleware'
    ]);
    $router->post('/admin/academic-years', 'App\Controllers\AcademicYearController@store', [
        'App\Middleware\AdminAuthMiddleware'
    ]);
    $router->get('/admin/academic-years/{id}', 'App\Controllers\AcademicYearController@show', [
        'App\Middleware\AdminAuthMiddleware'
    ]);
    $router->put('/admin/academic-years/{id}', 'App\Controllers\AcademicYearController@update', [
        'App\Middleware\AdminAuthMiddleware'
    ]);
    $router->delete('/admin/academic-years/{id}', 'App\Controllers\AcademicYearController@destroy', [
        'App\Middleware\AdminAuthMiddleware'
    ]);
    
    // Student Portal
    $router->get('/student/profile', 'App\Controllers\StudentPortalController@profile', [
        'App\Middleware\StudentAuthMiddleware'
    ]);
    $router->get('/student/results', 'App\Controllers\StudentPortalController@results', [
        'App\Middleware\StudentAuthMiddleware'
    ]);
    
    // Dispatch request
    $route = $router->dispatch($request->getMethod(), $request->getUri());
    
    // Execute middlewares
    foreach ($route['middlewares'] as $middleware) {
        if (is_string($middleware)) {
            $middleware = new $middleware();
        }
        $middleware($request);
    }
    
    // Execute handler
    $handler = $route['handler'];
    $params = $route['params'];
    
    if (is_string($handler)) {
        [$controllerClass, $method] = explode('@', $handler);
        $controller = new $controllerClass($request);
        $response = $controller->$method(...array_values($params));
    } else {
        $response = $handler($request, $params);
    }
    
    if ($response instanceof Response) {
        $response->send();
    }
    
} catch (\App\Core\ValidationException $e) {
    (new Response())->error('Validation failed', 422, $e->getErrors())->send();
} catch (\Exception $e) {
    $statusCode = $e->getCode() ?: 500;
    $message = $e->getMessage() ?: 'Internal server error';
    
    if ($statusCode < 100 || $statusCode > 599) {
        $statusCode = 500;
    }
    
    (new Response())->error($message, $statusCode)->send();
}