<?php

error_reporting(E_ALL);
ini_set('display_errors', 1);

require_once __DIR__ . '/../src/Core/Database.php';
require_once __DIR__ . '/../src/Core/Router.php';
require_once __DIR__ . '/../src/Core/Request.php';
require_once __DIR__ . '/../src/Core/Response.php';
require_once __DIR__ . '/../src/Core/Controller.php';

// Auto-load controllers
spl_autoload_register(function ($class) {
    $prefix = 'App\\';
    $base_dir = __DIR__ . '/../src/';
    
    $len = strlen($prefix);
    if (strncmp($prefix, $class, $len) !== 0) {
        return;
    }
    
    $relative_class = substr($class, $len);
    $file = $base_dir . str_replace('\\', '/', $relative_class) . '.php';
    
    if (file_exists($file)) {
        require $file;
    }
});

use App\Core\Router;
use App\Core\Request;
use App\Core\Response;

// Set headers
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST, PUT, DELETE, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization');

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit();
}

try {
    $router = new Router();
    $request = new Request();
    
    // Define routes directly instead of loading from file
    // Authentication routes
    $router->post('/auth/admin/login', 'App\Controllers\AuthController@adminLogin');
    $router->post('/auth/student/login', 'App\Controllers\AuthController@studentLogin');
    $router->get('/auth/me', 'App\Controllers\AuthController@me');
    $router->post('/auth/refresh', 'App\Controllers\AuthController@refresh', ['App\Middleware\AuthMiddleware']);
    
    // Academic Years with auth middleware  
    $router->get('/admin/academic-years', 'App\Controllers\AcademicYearController@index', ['App\Middleware\AuthMiddleware']);
    $router->post('/admin/academic-years', 'App\Controllers\AcademicYearController@store', ['App\Middleware\AuthMiddleware']);
    $router->get('/admin/academic-years/{id}', 'App\Controllers\AcademicYearController@show', ['App\Middleware\AuthMiddleware']);
    $router->put('/admin/academic-years/{id}', 'App\Controllers\AcademicYearController@update', ['App\Middleware\AuthMiddleware']);
    $router->delete('/admin/academic-years/{id}', 'App\Controllers\AcademicYearController@destroy', ['App\Middleware\AuthMiddleware']);
    
    // Terms with auth middleware
    $router->get('/admin/terms', 'App\Controllers\TermController@index', ['App\Middleware\AuthMiddleware']);
    $router->post('/admin/terms', 'App\Controllers\TermController@store', ['App\Middleware\AuthMiddleware']);
    $router->get('/admin/terms/{id}', 'App\Controllers\TermController@show', ['App\Middleware\AuthMiddleware']);
    $router->put('/admin/terms/{id}', 'App\Controllers\TermController@update', ['App\Middleware\AuthMiddleware']);
    $router->delete('/admin/terms/{id}', 'App\Controllers\TermController@destroy', ['App\Middleware\AuthMiddleware']);

    // Percentage Bands with auth middleware
    $router->get('/admin/percentage-bands', 'App\Controllers\PercentageBandController@index', ['App\Middleware\AuthMiddleware']);
    
    // Grades with auth middleware
    $router->get('/admin/grades', 'App\Controllers\GradeController@index', ['App\Middleware\AuthMiddleware']);
    $router->post('/admin/grades', 'App\Controllers\GradeController@store', ['App\Middleware\AuthMiddleware']);
    $router->get('/admin/grades/{id}', 'App\Controllers\GradeController@show', ['App\Middleware\AuthMiddleware']);
    $router->put('/admin/grades/{id}', 'App\Controllers\GradeController@update', ['App\Middleware\AuthMiddleware']);
    $router->delete('/admin/grades/{id}', 'App\Controllers\GradeController@destroy', ['App\Middleware\AuthMiddleware']);
    
    // Classes with auth middleware
    $router->get('/admin/classes', 'App\Controllers\ClassController@index', ['App\Middleware\AuthMiddleware']);
    $router->post('/admin/classes', 'App\Controllers\ClassController@store', ['App\Middleware\AuthMiddleware']);
    $router->get('/admin/classes/{id}', 'App\Controllers\ClassController@show', ['App\Middleware\AuthMiddleware']);
    $router->put('/admin/classes/{id}', 'App\Controllers\ClassController@update', ['App\Middleware\AuthMiddleware']);
    $router->delete('/admin/classes/{id}', 'App\Controllers\ClassController@destroy', ['App\Middleware\AuthMiddleware']);
    
    // Subjects with auth middleware
    $router->get('/admin/subjects', 'App\Controllers\SubjectController@index', ['App\Middleware\AuthMiddleware']);
    $router->post('/admin/subjects', 'App\Controllers\SubjectController@store', ['App\Middleware\AuthMiddleware']);
    $router->get('/admin/subjects/{id}', 'App\Controllers\SubjectController@show', ['App\Middleware\AuthMiddleware']);
    $router->put('/admin/subjects/{id}', 'App\Controllers\SubjectController@update', ['App\Middleware\AuthMiddleware']);
    $router->delete('/admin/subjects/{id}', 'App\Controllers\SubjectController@destroy', ['App\Middleware\AuthMiddleware']);
    
    // Grade Subject Rules with auth middleware
    $router->get('/admin/grade-subject-rules', 'App\Controllers\GradeSubjectRuleController@index', ['App\Middleware\AuthMiddleware']);
    $router->post('/admin/grade-subject-rules', 'App\Controllers\GradeSubjectRuleController@store', ['App\Middleware\AuthMiddleware']);
    $router->get('/admin/grade-subject-rules/{id}', 'App\Controllers\GradeSubjectRuleController@show', ['App\Middleware\AuthMiddleware']);
    $router->put('/admin/grade-subject-rules/{id}', 'App\Controllers\GradeSubjectRuleController@update', ['App\Middleware\AuthMiddleware']);
    $router->delete('/admin/grade-subject-rules/{id}', 'App\Controllers\GradeSubjectRuleController@destroy', ['App\Middleware\AuthMiddleware']);
    
    // Students with auth middleware
    $router->get('/admin/students', 'App\Controllers\StudentController@index', ['App\Middleware\AuthMiddleware']);
    $router->post('/admin/students', 'App\Controllers\StudentController@store', ['App\Middleware\AuthMiddleware']);
    $router->post('/admin/students/import', 'App\Controllers\StudentController@import', ['App\Middleware\AuthMiddleware']);
    $router->post('/admin/students/enroll', 'App\Controllers\StudentController@storeWithEnrollment', ['App\Middleware\AuthMiddleware']);
    $router->post('/admin/students/import-enroll', 'App\Controllers\StudentController@importWithEnrollment', ['App\Middleware\AuthMiddleware']);
    $router->post('/admin/students/import-individual-enroll', 'App\Controllers\StudentController@importStudentsWithIndividualEnrollment', ['App\Middleware\AuthMiddleware']);
    $router->get('/admin/students/enrollment-template', 'App\Controllers\StudentController@downloadEnrollmentTemplate', ['App\Middleware\AuthMiddleware']);
    $router->delete('/admin/students/bulk', 'App\Controllers\StudentController@bulkDelete', ['App\Middleware\AuthMiddleware']);
    $router->get('/admin/students/{id}/exams', 'App\Controllers\StudentController@getStudentExams', ['App\Middleware\AuthMiddleware']);
    $router->get('/admin/students/{id}', 'App\Controllers\StudentController@show', ['App\Middleware\AuthMiddleware']);
    $router->put('/admin/students/{id}', 'App\Controllers\StudentController@update', ['App\Middleware\AuthMiddleware']);
    $router->delete('/admin/students/{id}', 'App\Controllers\StudentController@destroy', ['App\Middleware\AuthMiddleware']);
    
    // Enrollments with auth middleware
    $router->get('/admin/enrollments', 'App\Controllers\EnrollmentController@index', ['App\Middleware\AuthMiddleware']);
    $router->post('/admin/enrollments', 'App\Controllers\EnrollmentController@store', ['App\Middleware\AuthMiddleware']);
    $router->get('/admin/enrollments/{id}', 'App\Controllers\EnrollmentController@show', ['App\Middleware\AuthMiddleware']);
    $router->put('/admin/enrollments/{id}', 'App\Controllers\EnrollmentController@update', ['App\Middleware\AuthMiddleware']);
    $router->delete('/admin/enrollments/{id}', 'App\Controllers\EnrollmentController@destroy', ['App\Middleware\AuthMiddleware']);
    $router->post('/admin/enrollments/bulk-enroll', 'App\Controllers\EnrollmentController@bulkEnroll', ['App\Middleware\AuthMiddleware']);
    $router->post('/admin/enrollments/transfer', 'App\Controllers\EnrollmentController@transferStudents', ['App\Middleware\AuthMiddleware']);
    $router->post('/admin/enrollments/import', 'App\Controllers\EnrollmentController@import', ['App\Middleware\AuthMiddleware']);
    
    // Exams with auth middleware (specific routes before parameterized routes)
    $router->get('/admin/exams', 'App\Controllers\ExamController@index', ['App\Middleware\AuthMiddleware']);
    $router->post('/admin/exams', 'App\Controllers\ExamController@store', ['App\Middleware\AuthMiddleware']);
    $router->delete('/admin/exams/bulk', 'App\Controllers\ExamController@bulkDelete', ['App\Middleware\AuthMiddleware']);
    $router->post('/admin/exams/bulk-create', 'App\Controllers\ExamController@bulkCreate', ['App\Middleware\AuthMiddleware']);
    $router->get('/admin/exams/grade-subjects/{id}', 'App\Controllers\ExamController@getGradeSubjects', ['App\Middleware\AuthMiddleware']);
    $router->get('/admin/exams/{id}', 'App\Controllers\ExamController@show', ['App\Middleware\AuthMiddleware']);
    $router->put('/admin/exams/{id}', 'App\Controllers\ExamController@update', ['App\Middleware\AuthMiddleware']);
    $router->delete('/admin/exams/{id}', 'App\Controllers\ExamController@destroy', ['App\Middleware\AuthMiddleware']);
    
    // Results with auth middleware (specific routes before parameterized routes)
    $router->get('/admin/results', 'App\Controllers\ResultController@index', ['App\Middleware\AuthMiddleware']);
    $router->post('/admin/results', 'App\Controllers\ResultController@store', ['App\Middleware\AuthMiddleware']);
    $router->delete('/admin/results/bulk', 'App\Controllers\ResultController@bulkDelete', ['App\Middleware\AuthMiddleware']);
    $router->post('/admin/results/bulk', 'App\Controllers\ResultController@bulkStore', ['App\Middleware\AuthMiddleware']);
    $router->post('/admin/results/import', 'App\Controllers\ResultController@import', ['App\Middleware\AuthMiddleware']);
    $router->get('/admin/results/template/{examId}', 'App\Controllers\ResultController@downloadTemplate', ['App\Middleware\AuthMiddleware']);
    $router->get('/admin/results/template-debug/{examId}', 'App\Controllers\ResultController@debugTemplate', ['App\Middleware\AuthMiddleware']);
    $router->get('/admin/results/exam/{examId}', 'App\Controllers\ResultController@getByExam', ['App\Middleware\AuthMiddleware']);
    $router->get('/admin/results/{id}', 'App\Controllers\ResultController@show', ['App\Middleware\AuthMiddleware']);
    $router->put('/admin/results/{id}', 'App\Controllers\ResultController@update', ['App\Middleware\AuthMiddleware']);
    $router->delete('/admin/results/{id}', 'App\Controllers\ResultController@destroy', ['App\Middleware\AuthMiddleware']);
    
    // Payments
    $router->get('/admin/payments/search', 'App\Controllers\PaymentController@searchStudents', ['App\Middleware\AuthMiddleware']);
    $router->get('/admin/payments/history', 'App\Controllers\PaymentController@searchPaymentHistory', ['App\Middleware\AuthMiddleware']);
    $router->get('/admin/students/{studentId}/timeline', 'App\Controllers\PaymentController@studentTimeline', ['App\Middleware\AuthMiddleware']);
    $router->get('/admin/enrollments/{enrollmentId}/payments', 'App\Controllers\PaymentController@index', ['App\Middleware\AuthMiddleware']);
    $router->post('/admin/enrollments/{enrollmentId}/payments', 'App\Controllers\PaymentController@store', ['App\Middleware\AuthMiddleware']);
    $router->put('/admin/payments/{id}', 'App\Controllers\PaymentController@update', ['App\Middleware\AuthMiddleware']);
    $router->delete('/admin/payments/{id}', 'App\Controllers\PaymentController@destroy', ['App\Middleware\AuthMiddleware']);
    
    // Users (admin only)
    $router->get('/admin/users', 'App\Controllers\UserController@index', ['App\Middleware\AuthMiddleware']);
    $router->post('/admin/users', 'App\Controllers\UserController@store', ['App\Middleware\AuthMiddleware']);
    $router->get('/admin/users/{id}', 'App\Controllers\UserController@show', ['App\Middleware\AuthMiddleware']);
    $router->put('/admin/users/{id}', 'App\Controllers\UserController@update', ['App\Middleware\AuthMiddleware']);
    $router->delete('/admin/users/{id}', 'App\Controllers\UserController@destroy', ['App\Middleware\AuthMiddleware']);
    $router->post('/admin/users/{id}/toggle-status', 'App\Controllers\UserController@toggleStatus', ['App\Middleware\AuthMiddleware']);
    
    // Settings
    $router->get('/admin/settings', 'App\Controllers\SettingsController@index', ['App\Middleware\AuthMiddleware']);
    $router->put('/admin/settings', 'App\Controllers\SettingsController@update', ['App\Middleware\AuthMiddleware']);
    
    // School settings
    $router->get('/school/settings', 'App\Controllers\SchoolController@settings');
    
    // Public routes (for student cards and other non-authenticated access)
    $router->get('/public/grades', 'App\Controllers\PublicController@grades');
    $router->get('/public/subjects', 'App\Controllers\PublicController@subjects');
    $router->get('/public/classes', 'App\Controllers\PublicController@classes');
    $router->get('/public/academic-years', 'App\Controllers\PublicController@academicYears');
    
    // Dashboard
    $router->get('/admin/dashboard/stats', 'App\Controllers\DashboardController@stats', ['App\Middleware\AuthMiddleware']);
    
    // Public dashboard stats for testing
    $router->get('/public/dashboard/test-stats', 'App\Controllers\DashboardController@testStats');
    
    // Student Portal with auth middleware
    $router->get('/student/profile', 'App\Controllers\StudentPortalController@profile', ['App\Middleware\AuthMiddleware']);
    $router->get('/student/results', 'App\Controllers\StudentPortalController@results', ['App\Middleware\AuthMiddleware']);
    
    // Dispatch request
    $route = $router->dispatch($request->getMethod(), $request->getUri());
    
    // Execute middlewares
    foreach ($route['middlewares'] as $middleware) {
        if (is_string($middleware)) {
            $middleware = new $middleware();
        }
        $middleware($request);
    }
    
    // Execute handler
    $handler = $route['handler'];
    $params = $route['params'];
    
    if (is_string($handler)) {
        [$controllerClass, $method] = explode('@', $handler);
        $controller = new $controllerClass($request);
        $response = $controller->$method(...array_values($params));
    } else {
        $response = $handler($request, $params);
    }
    
    if ($response instanceof Response) {
        $response->send();
    }
    
} catch (\App\Core\ValidationException $e) {
    (new Response())->error('Validation failed', 422, $e->getErrors())->send();
} catch (\Exception $e) {
    $statusCode = (int)$e->getCode() ?: 500;
    $message = $e->getMessage() ?: 'Internal server error';
    
    // Ensure status code is valid HTTP status code
    if ($statusCode < 100 || $statusCode > 599) {
        $statusCode = 500;
    }
    
    (new Response())->error($message, $statusCode)->send();
}