<?php

namespace App\Controllers;

use App\Core\Controller;
use App\Core\Database;
use App\Core\Response;
use App\Services\AuthService;
use PDO;

class AcademicYearController extends Controller
{
    private PDO $db;
    private AuthService $authService;
    
    public function __construct($request)
    {
        parent::__construct($request);
        $this->db = Database::getConnection();
        $this->authService = new AuthService();
    }
    
    public function index(): Response
    {
        // Check if export is requested
        if ($exportFormat = $this->request->get('export')) {
            return $this->export($exportFormat);
        }
        
        $pagination = $this->getPaginationParams();
        $sort = $this->getSortParams(['id', 'name', 'starts_on', 'ends_on', 'status', 'created_at']);
        
        $whereConditions = ['1=1'];
        $params = [];
        
        // Filters
        if ($status = $this->request->get('status')) {
            $whereConditions[] = 'status = ?';
            $params[] = $status;
        }
        
        if ($name = $this->request->get('name')) {
            $whereConditions[] = 'name LIKE ?';
            $params[] = "%{$name}%";
        }
        
        $whereClause = implode(' AND ', $whereConditions);
        $orderClause = $sort['sortBy'] ? "ORDER BY {$sort['sortBy']} {$sort['sortDir']}" : 'ORDER BY starts_on DESC';
        
        // Get total count
        $countStmt = $this->db->prepare("SELECT COUNT(*) FROM academic_years WHERE {$whereClause}");
        $countStmt->execute($params);
        $totalRows = $countStmt->fetchColumn();
        
        // Get results
        $sql = "SELECT id, name, starts_on, ends_on, status, created_at 
                FROM academic_years 
                WHERE {$whereClause} 
                {$orderClause} 
                LIMIT ? OFFSET ?";
        
        $stmt = $this->db->prepare($sql);
        $stmt->execute([...$params, $pagination['perPage'], $pagination['offset']]);
        $years = $stmt->fetchAll();
        
        $meta = [
            'page' => $pagination['page'],
            'per_page' => $pagination['perPage'],
            'total_rows' => $totalRows,
            'total_pages' => ceil($totalRows / $pagination['perPage']),
            'sort_by' => $sort['sortBy'],
            'sort_dir' => $sort['sortDir']
        ];
        
        return $this->response->paginated($years, $meta);
    }
    
    private function export(string $format): Response
    {
        // Get all academic years for export
        $stmt = $this->db->prepare("SELECT name, starts_on, ends_on, status, created_at FROM academic_years ORDER BY starts_on DESC");
        $stmt->execute();
        $years = $stmt->fetchAll();
        
        if ($format === 'csv') {
            $csv = "Name,Start Date,End Date,Status,Created\n";
            foreach ($years as $year) {
                $csv .= sprintf('"%s","%s","%s","%s","%s"' . "\n",
                    $year['name'],
                    $year['starts_on'] ?: 'N/A',
                    $year['ends_on'] ?: 'N/A',
                    ucfirst($year['status']),
                    date('Y-m-d', strtotime($year['created_at']))
                );
            }
            
            return $this->response->success($csv, 'Export completed');
        }
        
        return $this->response->error('Export format not supported', 400);
    }
    
    public function show(string $id): Response
    {
        $stmt = $this->db->prepare("SELECT * FROM academic_years WHERE id = ?");
        $stmt->execute([$id]);
        $year = $stmt->fetch();
        
        if (!$year) {
            return $this->response->error('Academic year not found', 404);
        }
        
        return $this->response->success($year);
    }
    
    public function store(): Response
    {
        $data = $this->validate([
            'name' => 'required|min:4|max:50',
            'starts_on' => 'max:10',
            'ends_on' => 'max:10',
            'status' => 'required'
        ]);
        
        // Check for duplicate name
        $stmt = $this->db->prepare("SELECT id FROM academic_years WHERE name = ?");
        $stmt->execute([$data['name']]);
        if ($stmt->fetch()) {
            return $this->response->error('Academic year name already exists', 422);
        }
        
        try {
            $stmt = $this->db->prepare(
                "INSERT INTO academic_years (name, starts_on, ends_on, status) VALUES (?, ?, ?, ?)"
            );
            
            $stmt->execute([
                $data['name'],
                $data['starts_on'] ?? null,
                $data['ends_on'] ?? null,
                $data['status']
            ]);
            
            $yearId = $this->db->lastInsertId();
            
            // Get created year
            $stmt = $this->db->prepare("SELECT * FROM academic_years WHERE id = ?");
            $stmt->execute([$yearId]);
            $year = $stmt->fetch();
            
            // Log audit event
            $this->authService->logAuditEvent([
                'actor_id' => $this->request->user['user_id'],
                'actor_type' => 'admin',
                'role' => $this->request->user['role'],
                'action' => 'create',
                'entity' => 'academic_years',
                'entity_id' => $yearId,
                'after_data' => $year,
                'ip_address' => $this->request->getClientIp(),
                'user_agent' => $this->request->getUserAgent()
            ]);
            
            return $this->response->success($year, 'Academic year created successfully');
            
        } catch (\Exception $e) {
            return $this->response->error('Failed to create academic year: ' . $e->getMessage(), 500);
        }
    }
    
    public function update(string $id): Response
    {
        $data = $this->validate([
            'name' => 'required|min:4|max:50',
            'starts_on' => 'max:10',
            'ends_on' => 'max:10',
            'status' => 'required'
        ]);
        
        // Get existing year for audit
        $stmt = $this->db->prepare("SELECT * FROM academic_years WHERE id = ?");
        $stmt->execute([$id]);
        $oldYear = $stmt->fetch();
        
        if (!$oldYear) {
            return $this->response->error('Academic year not found', 404);
        }
        
        // Check for duplicate name (excluding current year)
        $stmt = $this->db->prepare("SELECT id FROM academic_years WHERE name = ? AND id != ?");
        $stmt->execute([$data['name'], $id]);
        if ($stmt->fetch()) {
            return $this->response->error('Academic year name already exists', 422);
        }
        
        try {
            $stmt = $this->db->prepare(
                "UPDATE academic_years SET name = ?, starts_on = ?, ends_on = ?, status = ? WHERE id = ?"
            );
            
            $stmt->execute([
                $data['name'],
                $data['starts_on'] ?? null,
                $data['ends_on'] ?? null,
                $data['status'],
                $id
            ]);
            
            // Get updated year
            $stmt = $this->db->prepare("SELECT * FROM academic_years WHERE id = ?");
            $stmt->execute([$id]);
            $newYear = $stmt->fetch();
            
            // Log audit event
            $this->authService->logAuditEvent([
                'actor_id' => $this->request->user['user_id'],
                'actor_type' => 'admin',
                'role' => $this->request->user['role'],
                'action' => 'update',
                'entity' => 'academic_years',
                'entity_id' => $id,
                'before_data' => $oldYear,
                'after_data' => $newYear,
                'ip_address' => $this->request->getClientIp(),
                'user_agent' => $this->request->getUserAgent()
            ]);
            
            return $this->response->success($newYear, 'Academic year updated successfully');
            
        } catch (\Exception $e) {
            return $this->response->error('Failed to update academic year: ' . $e->getMessage(), 500);
        }
    }
    
    public function destroy(string $id): Response
    {
        // Check for dependent classes
        $stmt = $this->db->prepare("SELECT COUNT(*) FROM classes WHERE year_id = ?");
        $stmt->execute([$id]);
        $classCount = $stmt->fetchColumn();
        
        if ($classCount > 0) {
            return $this->response->error('Cannot delete academic year with existing classes', 422);
        }
        
        // Get year for audit
        $stmt = $this->db->prepare("SELECT * FROM academic_years WHERE id = ?");
        $stmt->execute([$id]);
        $year = $stmt->fetch();
        
        if (!$year) {
            return $this->response->error('Academic year not found', 404);
        }
        
        try {
            $stmt = $this->db->prepare("DELETE FROM academic_years WHERE id = ?");
            $stmt->execute([$id]);
            
            // Log audit event
            $this->authService->logAuditEvent([
                'actor_id' => $this->request->user['user_id'],
                'actor_type' => 'admin',
                'role' => $this->request->user['role'],
                'action' => 'delete',
                'entity' => 'academic_years',
                'entity_id' => $id,
                'before_data' => $year,
                'ip_address' => $this->request->getClientIp(),
                'user_agent' => $this->request->getUserAgent()
            ]);
            
            return $this->response->success(null, 'Academic year deleted successfully');
            
        } catch (\Exception $e) {
            return $this->response->error('Failed to delete academic year: ' . $e->getMessage(), 500);
        }
    }
}