<?php

namespace App\Controllers;

use App\Core\Controller;
use App\Core\Response;
use App\Services\AuthService;
use App\Services\JWTService;

class AuthController extends Controller
{
    private AuthService $authService;
    private JWTService $jwtService;
    
    public function __construct($request)
    {
        parent::__construct($request);
        $this->authService = new AuthService();
        $this->jwtService = new JWTService();
    }
    
    public function adminLogin(): Response
    {
        $data = $this->validate([
            'username' => 'required|min:3',
            'password' => 'required|min:6'
        ]);
        
        $user = $this->authService->authenticateAdmin($data['username'], $data['password']);
        
        if (!$user) {
            // Log failed attempt
            $this->authService->logAuditEvent([
                'actor_type' => 'admin',
                'action' => 'login_failed',
                'entity' => 'admin_users',
                'ip_address' => $this->request->getClientIp(),
                'user_agent' => $this->request->getUserAgent()
            ]);
            
            return $this->response->error('Invalid credentials', 401);
        }
        
        // Generate JWT token
        $payload = [
            'type' => 'admin',
            'user_id' => $user['id'],
            'username' => $user['username'],
            'role' => $user['role'],
            'full_name' => $user['full_name']
        ];
        
        $token = $this->jwtService->encode($payload, 8 * 3600); // 8 hours
        
        // Log successful login
        $this->authService->logAuditEvent([
            'actor_id' => $user['id'],
            'actor_type' => 'admin',
            'role' => $user['role'],
            'action' => 'login_success',
            'entity' => 'admin_users',
            'ip_address' => $this->request->getClientIp(),
            'user_agent' => $this->request->getUserAgent()
        ]);
        
        return $this->response->success([
            'token' => $token,
            'user' => $user,
            'expires_in' => 8 * 3600
        ], 'Login successful');
    }
    
    public function studentLogin(): Response
    {
        $data = $this->validate([
            'login_code' => 'required|min:8|max:8',
            'national_id' => 'required'
        ]);
        
        $student = $this->authService->authenticateStudent($data['login_code'], $data['national_id']);
        
        if (!$student) {
            // Log failed attempt
            $this->authService->logAuditEvent([
                'actor_type' => 'student',
                'action' => 'login_failed',
                'entity' => 'students',
                'ip_address' => $this->request->getClientIp(),
                'user_agent' => $this->request->getUserAgent()
            ]);
            
            return $this->response->error('We couldn\'t find a student matching those credentials.', 401);
        }
        
        // Generate JWT token
        $payload = [
            'type' => 'student',
            'user_id' => $student['id'],
            'full_name' => $student['full_name'],
            'year_id' => $student['year_id'],
            'grade_id' => $student['grade_id'],
            'class_id' => $student['class_id']
        ];
        
        $token = $this->jwtService->encode($payload, 4 * 3600); // 4 hours
        
        // Log successful login
        $this->authService->logAuditEvent([
            'actor_id' => $student['id'],
            'actor_type' => 'student',
            'action' => 'login_success',
            'entity' => 'students',
            'ip_address' => $this->request->getClientIp(),
            'user_agent' => $this->request->getUserAgent()
        ]);
        
        return $this->response->success([
            'token' => $token,
            'student' => $student,
            'expires_in' => 4 * 3600
        ], 'Login successful');
    }
    
    public function me(): Response
    {
        $token = $this->request->getBearerToken();
        
        if (!$token) {
            return $this->response->error('Authentication required', 401);
        }
        
        $payload = $this->jwtService->decode($token);
        
        if (!$payload) {
            return $this->response->error('Invalid or expired token', 401);
        }
        
        return $this->response->success($payload);
    }
    
    public function refresh(): Response
    {
        if (!isset($this->request->user)) {
            return $this->response->error('Not authenticated', 401);
        }
        
        $user = $this->request->user;
        
        // Generate new token
        $expirationTime = $user['type'] === 'admin' ? 8 * 3600 : 4 * 3600;
        $newToken = $this->jwtService->encode($user, $expirationTime);
        
        return $this->response->success([
            'token' => $newToken,
            'expires_in' => $expirationTime
        ], 'Token refreshed');
    }
}