<?php

namespace App\Controllers;

use App\Core\Controller;
use App\Core\Database;
use App\Core\Response;
use Exception;

class ClassController extends Controller
{
    private $db;
    
    public function __construct($request = null)
    {
        parent::__construct($request);
        $this->db = Database::getConnection();
    }
    
    public function index(): Response
    {
        try {
            $page = max(1, intval($this->request->get('page', 1)));
            $perPage = min(200, max(10, intval($this->request->get('per_page', 50))));
            $offset = ($page - 1) * $perPage;
            
            // Build WHERE clause for filters
            $whereConditions = ['1=1'];
            $params = [];
            
            // Filter by class name
            if ($name = $this->request->get('name')) {
                $whereConditions[] = 'c.name LIKE ?';
                $params[] = '%' . $name . '%';
            }
            
            // Filter by enabled status
            if (($enabled = $this->request->get('enabled')) !== null && $enabled !== '') {
                $whereConditions[] = 'c.enabled = ?';
                $params[] = intval($enabled);
            }
            
            $whereClause = implode(' AND ', $whereConditions);
            
            // Handle sorting
            $sortBy = $this->request->get('sort_by', 'c.name');
            $sortDir = $this->request->get('sort_dir', 'asc');
            
            $allowedSortFields = ['c.name', 'c.enabled', 'c.created_at'];
            if (!in_array($sortBy, $allowedSortFields)) {
                $sortBy = 'c.name';
            }
            
            if (!in_array(strtolower($sortDir), ['asc', 'desc'])) {
                $sortDir = 'asc';
            }
            
            // Get total count
            $countSql = "
                SELECT COUNT(*) 
                FROM classes c
                WHERE $whereClause
            ";
            
            $countStmt = $this->db->prepare($countSql);
            $countStmt->execute($params);
            $totalCount = $countStmt->fetchColumn();
            
            // Get classes
            $sql = "
                SELECT 
                    c.*
                FROM classes c
                WHERE $whereClause
                ORDER BY $sortBy $sortDir
                LIMIT ? OFFSET ?
            ";
            
            $params[] = $perPage;
            $params[] = $offset;
            
            $stmt = $this->db->prepare($sql);
            $stmt->execute($params);
            $classes = $stmt->fetchAll(\PDO::FETCH_ASSOC);
            
            // Format data for display
            foreach ($classes as &$class) {
                $class['enabled'] = (bool)$class['enabled'];
            }
            
            return new Response([
                'success' => true,
                'data' => $classes,
                'meta' => [
                    'total' => $totalCount,
                    'page' => $page,
                    'per_page' => $perPage,
                    'total_pages' => ceil($totalCount / $perPage),
                    'sort_by' => $sortBy,
                    'sort_dir' => $sortDir
                ]
            ]);
            
        } catch (Exception $e) {
            error_log("Error in ClassController::index: " . $e->getMessage());
            return new Response(['success' => false, 'message' => 'Failed to fetch classes'], 500);
        }
    }
    
    public function show($id): Response
    {
        try {
            $sql = "SELECT * FROM classes WHERE id = ?";
            
            $stmt = $this->db->prepare($sql);
            $stmt->execute([$id]);
            $class = $stmt->fetch(\PDO::FETCH_ASSOC);
            
            if (!$class) {
                return new Response(['success' => false, 'message' => 'Class not found'], 404);
            }
            
            $class['enabled'] = (bool)$class['enabled'];
            
            return new Response(['success' => true, 'data' => $class]);
            
        } catch (Exception $e) {
            error_log("Error in ClassController::show: " . $e->getMessage());
            return new Response(['success' => false, 'message' => 'Failed to fetch class'], 500);
        }
    }
    
    public function store(): Response
    {
        try {
            $data = $this->request->all();
            
            // Validation
            $errors = $this->validateClassData($data);
            if (!empty($errors)) {
                return new Response(['success' => false, 'message' => 'Validation failed', 'errors' => $errors], 422);
            }
            
            $name = trim($data['name']);
            $enabled = isset($data['enabled']) ? intval($data['enabled']) : 1;
            
            // Check for duplicate name
            $checkSql = "SELECT id FROM classes WHERE name = ?";
            $checkStmt = $this->db->prepare($checkSql);
            $checkStmt->execute([$name]);
            if ($checkStmt->fetch()) {
                return new Response(['success' => false, 'message' => 'A class with this name already exists'], 422);
            }
            
            // Insert class
            $sql = "INSERT INTO classes (name, enabled) VALUES (?, ?)";
            $stmt = $this->db->prepare($sql);
            $stmt->execute([$name, $enabled]);
            
            $classId = $this->db->lastInsertId();
            
            // Get the created class
            $newClass = $this->show($classId);
            
            // Log the action
            $this->logAction('create', 'classes', $classId, null, $newClass->getData()['data']);
            
            return new Response(['success' => true, 'message' => 'Class created successfully', 'data' => $newClass->getData()['data']], 201);
            
        } catch (Exception $e) {
            error_log("Error in ClassController::store: " . $e->getMessage());
            return new Response(['success' => false, 'message' => 'Failed to create class'], 500);
        }
    }
    
    public function update($id): Response
    {
        try {
            $data = $this->request->all();
            
            // Check if class exists
            $existingClass = $this->show($id);
            if (!$existingClass->getData()['success']) {
                return $existingClass;
            }
            $oldClass = $existingClass->getData()['data'];
            
            // Validation
            $errors = $this->validateClassData($data);
            if (!empty($errors)) {
                return new Response(['success' => false, 'message' => 'Validation failed', 'errors' => $errors], 422);
            }
            
            $name = trim($data['name']);
            $enabled = isset($data['enabled']) ? intval($data['enabled']) : 1;
            
            // Check for duplicate name (excluding current class)
            $checkSql = "SELECT id FROM classes WHERE name = ? AND id != ?";
            $checkStmt = $this->db->prepare($checkSql);
            $checkStmt->execute([$name, $id]);
            if ($checkStmt->fetch()) {
                return new Response(['success' => false, 'message' => 'A class with this name already exists'], 422);
            }
            
            // Update class
            $sql = "UPDATE classes SET name = ?, enabled = ? WHERE id = ?";
            $stmt = $this->db->prepare($sql);
            $stmt->execute([$name, $enabled, $id]);
            
            // Get updated class
            $updatedClass = $this->show($id);
            
            // Log the action
            $this->logAction('update', 'classes', $id, $oldClass, $updatedClass->getData()['data']);
            
            return new Response(['success' => true, 'message' => 'Class updated successfully', 'data' => $updatedClass->getData()['data']]);
            
        } catch (Exception $e) {
            error_log("Error in ClassController::update: " . $e->getMessage());
            return new Response(['success' => false, 'message' => 'Failed to update class'], 500);
        }
    }
    
    public function destroy($id): Response
    {
        try {
            // Check if class exists
            $existingClass = $this->show($id);
            if (!$existingClass->getData()['success']) {
                return $existingClass;
            }
            $class = $existingClass->getData()['data'];
            
            // Check for dependencies - enrollments using this class
            $enrollmentCheckSql = "SELECT COUNT(*) FROM enrollments WHERE class_id = ?";
            $enrollmentCheckStmt = $this->db->prepare($enrollmentCheckSql);
            $enrollmentCheckStmt->execute([$id]);
            $enrollmentCount = $enrollmentCheckStmt->fetchColumn();
            
            if ($enrollmentCount > 0) {
                return new Response([
                    'success' => false, 
                    'message' => 'Cannot delete class. There are ' . $enrollmentCount . ' student enrollment(s) using this class.'
                ], 422);
            }
            
            // Delete the class
            $sql = "DELETE FROM classes WHERE id = ?";
            $stmt = $this->db->prepare($sql);
            $stmt->execute([$id]);
            
            // Log the action
            $this->logAction('delete', 'classes', $id, $class, null);
            
            return new Response(['success' => true, 'message' => 'Class deleted successfully']);
            
        } catch (Exception $e) {
            error_log("Error in ClassController::destroy: " . $e->getMessage());
            return new Response(['success' => false, 'message' => 'Failed to delete class'], 500);
        }
    }
    
    private function validateClassData($data): array
    {
        $errors = [];
        
        // Name validation
        if (empty($data['name'])) {
            $errors['name'] = 'Class name is required';
        } elseif (strlen(trim($data['name'])) < 1 || strlen(trim($data['name'])) > 50) {
            $errors['name'] = 'Class name must be between 1 and 50 characters';
        }
        
        // Enabled validation
        if (isset($data['enabled']) && !in_array($data['enabled'], [0, 1, '0', '1', true, false])) {
            $errors['enabled'] = 'Invalid enabled status';
        }
        
        return $errors;
    }
    
    private function logAction(string $action, string $entity, $entityId, $before, $after): void
    {
        try {
            $logData = [
                'actor' => $this->request->user['username'] ?? 'system',
                'role' => $this->request->user['role'] ?? 'admin',
                'timestamp' => date('Y-m-d H:i:s'),
                'action' => $action,
                'entity' => $entity,
                'entity_id' => $entityId,
                'before' => $before ? json_encode($before) : null,
                'after' => $after ? json_encode($after) : null
            ];
            
            error_log("Audit Log: " . json_encode($logData));
        } catch (Exception $e) {
            error_log("Failed to log action: " . $e->getMessage());
        }
    }
}