<?php

namespace App\Controllers;

use App\Core\Controller;
use App\Core\Database;
use App\Core\Response;
use PDO;

class DashboardController extends Controller
{
    private PDO $db;
    
    public function __construct($request)
    {
        parent::__construct($request);
        $this->db = Database::getConnection();
    }
    
    public function testStats(): Response
    {
        try {
            $stats = [];
            
            // Get basic counts without requiring academic year setup
            
            // Total Students (simple count)
            try {
                $stmt = $this->db->prepare("SELECT COUNT(*) as total FROM students WHERE enabled = 1");
                $stmt->execute();
                $stats['total_students'] = (int)$stmt->fetchColumn();
            } catch (\Exception $e) {
                $stats['total_students'] = 0;
                $stats['student_error'] = $e->getMessage();
            }
            
            // Total Enrollments (simple count)
            try {
                $stmt = $this->db->prepare("SELECT COUNT(*) as total FROM enrollments WHERE status = 'active'");
                $stmt->execute();
                $stats['active_enrollments'] = (int)$stmt->fetchColumn();
            } catch (\Exception $e) {
                $stats['active_enrollments'] = 0;
                $stats['enrollment_error'] = $e->getMessage();
            }
            
            // Total Exams (simple count)
            try {
                $stmt = $this->db->prepare("SELECT COUNT(*) as total FROM exams WHERE enabled = 1");
                $stmt->execute();
                $stats['total_exams'] = (int)$stmt->fetchColumn();
            } catch (\Exception $e) {
                $stats['total_exams'] = 0;
                $stats['exam_error'] = $e->getMessage();
            }
            
            // Total Payment Amount (simple sum)
            try {
                $stmt = $this->db->prepare("SELECT COALESCE(SUM(amount), 0) as total FROM payments");
                $stmt->execute();
                $stats['total_payment_amount'] = (float)$stmt->fetchColumn();
            } catch (\Exception $e) {
                $stats['total_payment_amount'] = 0;
                $stats['payment_error'] = $e->getMessage();
            }
            
            // Recent Activity (simplified)
            try {
                $stmt = $this->db->prepare("SELECT full_name, created_at FROM students WHERE enabled = 1 ORDER BY created_at DESC LIMIT 5");
                $stmt->execute();
                $recentStudents = $stmt->fetchAll();
                
                $activities = [];
                foreach ($recentStudents as $student) {
                    $activities[] = [
                        'type' => 'student',
                        'icon' => 'bi-person-plus',
                        'description' => "New student: {$student['full_name']}",
                        'time' => $student['created_at']
                    ];
                }
                $stats['recent_activity'] = $activities;
            } catch (\Exception $e) {
                $stats['recent_activity'] = [];
                $stats['activity_error'] = $e->getMessage();
            }
            
            $stats['test_mode'] = true;
            $stats['timestamp'] = date('Y-m-d H:i:s');
            
            return $this->response->success($stats);
            
        } catch (\Exception $e) {
            return $this->response->error('Failed to load test dashboard stats: ' . $e->getMessage(), 500);
        }
    }
    
    public function stats(): Response
    {
        try {
            $stats = [];
            
            // Get current academic year
            $yearStmt = $this->db->prepare("SELECT current_year_id FROM academic_visibility LIMIT 1");
            $yearStmt->execute();
            $currentYear = $yearStmt->fetch();
            $currentYearId = $currentYear['current_year_id'] ?? null;
            
            // Total Students
            $stmt = $this->db->prepare("SELECT COUNT(*) as total FROM students WHERE enabled = 1");
            $stmt->execute();
            $stats['total_students'] = $stmt->fetchColumn();
            
            // Active Enrollments (current year)
            if ($currentYearId) {
                $stmt = $this->db->prepare("
                    SELECT COUNT(*) as total 
                    FROM enrollments 
                    WHERE year_id = ? AND status = 'active'
                ");
                $stmt->execute([$currentYearId]);
                $stats['active_enrollments'] = $stmt->fetchColumn();
            } else {
                $stats['active_enrollments'] = 0;
            }
            
            // Total Exams (current year)
            if ($currentYearId) {
                $stmt = $this->db->prepare("
                    SELECT COUNT(*) as total 
                    FROM exams 
                    WHERE year_id = ? AND enabled = 1
                ");
                $stmt->execute([$currentYearId]);
                $stats['total_exams'] = $stmt->fetchColumn();
            } else {
                $stats['total_exams'] = 0;
            }
            
            // Total Payments (current year)
            if ($currentYearId) {
                $stmt = $this->db->prepare("
                    SELECT COUNT(*) as total 
                    FROM payments p
                    JOIN enrollments e ON p.enrollment_id = e.id
                    WHERE e.year_id = ?
                ");
                $stmt->execute([$currentYearId]);
                $stats['total_payments'] = $stmt->fetchColumn();
                
                // Total Payment Amount (current year)
                $stmt = $this->db->prepare("
                    SELECT COALESCE(SUM(p.amount), 0) as total 
                    FROM payments p
                    JOIN enrollments e ON p.enrollment_id = e.id
                    WHERE e.year_id = ?
                ");
                $stmt->execute([$currentYearId]);
                $stats['total_payment_amount'] = $stmt->fetchColumn();
            } else {
                $stats['total_payments'] = 0;
                $stats['total_payment_amount'] = 0;
            }
            
            // Recent Activity
            $stats['recent_activity'] = $this->getRecentActivity($currentYearId);
            
            // Grade Distribution
            $stats['grade_distribution'] = $this->getGradeDistribution($currentYearId);
            
            // Payment Statistics
            $stats['payment_stats'] = $this->getPaymentStats($currentYearId);
            
            // Exam Results Summary
            $stats['results_summary'] = $this->getResultsSummary($currentYearId);
            
            return $this->response->success($stats);
            
        } catch (\Exception $e) {
            return $this->response->error('Failed to load dashboard stats: ' . $e->getMessage(), 500);
        }
    }
    
    private function getRecentActivity($currentYearId): array
    {
        $activities = [];
        
        try {
            // Recent student registrations
            $stmt = $this->db->prepare("
                SELECT 'student_registered' as type, full_name as description, created_at
                FROM students 
                WHERE enabled = 1 
                ORDER BY created_at DESC 
                LIMIT 5
            ");
            $stmt->execute();
            $recentStudents = $stmt->fetchAll();
            
            foreach ($recentStudents as $student) {
                $activities[] = [
                    'type' => 'student',
                    'icon' => 'bi-person-plus',
                    'description' => "New student: {$student['description']}",
                    'time' => $student['created_at']
                ];
            }
            
            // Recent payments
            if ($currentYearId) {
                $stmt = $this->db->prepare("
                    SELECT p.amount, s.full_name, p.created_at
                    FROM payments p
                    JOIN enrollments e ON p.enrollment_id = e.id
                    JOIN students s ON e.student_id = s.id
                    WHERE e.year_id = ?
                    ORDER BY p.created_at DESC 
                    LIMIT 5
                ");
                $stmt->execute([$currentYearId]);
                $recentPayments = $stmt->fetchAll();
                
                foreach ($recentPayments as $payment) {
                    $activities[] = [
                        'type' => 'payment',
                        'icon' => 'bi-currency-dollar',
                        'description' => "Payment: {$payment['full_name']} - {$payment['amount']}",
                        'time' => $payment['created_at']
                    ];
                }
            }
            
            // Sort by time and limit to 10 most recent
            usort($activities, function($a, $b) {
                return strtotime($b['time']) - strtotime($a['time']);
            });
            
            return array_slice($activities, 0, 10);
            
        } catch (\Exception $e) {
            return [];
        }
    }
    
    private function getGradeDistribution($currentYearId): array
    {
        try {
            if (!$currentYearId) return [];
            
            $stmt = $this->db->prepare("
                SELECT g.title as grade_name, COUNT(e.id) as student_count
                FROM grades g
                LEFT JOIN enrollments e ON g.id = e.grade_id AND e.year_id = ? AND e.status = 'active'
                GROUP BY g.id, g.title
                ORDER BY g.title
            ");
            $stmt->execute([$currentYearId]);
            
            return $stmt->fetchAll();
            
        } catch (\Exception $e) {
            return [];
        }
    }
    
    private function getPaymentStats($currentYearId): array
    {
        try {
            if (!$currentYearId) return [];
            
            // Monthly payment totals for current year
            $stmt = $this->db->prepare("
                SELECT 
                    DATE_FORMAT(p.created_at, '%Y-%m') as month,
                    COUNT(*) as payment_count,
                    SUM(p.amount) as total_amount
                FROM payments p
                JOIN enrollments e ON p.enrollment_id = e.id
                WHERE e.year_id = ?
                GROUP BY DATE_FORMAT(p.created_at, '%Y-%m')
                ORDER BY month DESC
                LIMIT 12
            ");
            $stmt->execute([$currentYearId]);
            
            return $stmt->fetchAll();
            
        } catch (\Exception $e) {
            return [];
        }
    }
    
    private function getResultsSummary($currentYearId): array
    {
        try {
            if (!$currentYearId) return [];
            
            // Get pass/fail statistics by grade
            $stmt = $this->db->prepare("
                SELECT 
                    g.title as grade_name,
                    COUNT(DISTINCT er.student_id) as students_with_results,
                    COUNT(DISTINCT CASE WHEN er.score/e.max_score >= 0.5 THEN er.student_id END) as passing_students
                FROM exam_results er
                JOIN exams e ON er.exam_id = e.id
                JOIN enrollments en ON er.student_id = en.student_id AND en.year_id = e.year_id
                JOIN grades g ON en.grade_id = g.id
                WHERE e.year_id = ? AND e.enabled = 1
                GROUP BY g.id, g.title
                ORDER BY g.title
            ");
            $stmt->execute([$currentYearId]);
            
            return $stmt->fetchAll();
            
        } catch (\Exception $e) {
            return [];
        }
    }
}