<?php

namespace App\Controllers;

use App\Core\Controller;
use App\Core\Database;
use App\Core\Response;
use App\Services\AuthService;
use PDO;

class EnrollmentController extends Controller
{
    private PDO $db;
    private AuthService $authService;
    
    public function __construct($request)
    {
        parent::__construct($request);
        $this->db = Database::getConnection();
        $this->authService = new AuthService();
    }
    
    public function index(): Response
    {
        $pagination = $this->getPaginationParams();
        $sort = $this->getSortParams(['student_id', 'year_id', 'grade_id', 'class_id', 'status', 'created_at']);
        
        // Build base query
        $whereConditions = ['1=1'];
        $params = [];
        
        // Filters
        if ($yearId = $this->request->get('year_id')) {
            $whereConditions[] = 'e.year_id = ?';
            $params[] = $yearId;
        }
        
        if ($gradeId = $this->request->get('grade_id')) {
            $whereConditions[] = 'e.grade_id = ?';
            $params[] = $gradeId;
        }
        
        if ($classId = $this->request->get('class_id')) {
            $whereConditions[] = 'e.class_id = ?';
            $params[] = $classId;
        }
        
        if ($status = $this->request->get('status')) {
            $whereConditions[] = 'e.status = ?';
            $params[] = $status;
        }
        
        $whereClause = implode(' AND ', $whereConditions);
        $orderClause = $sort['sortBy'] ? "ORDER BY {$sort['sortBy']} {$sort['sortDir']}" : 'ORDER BY e.id DESC';
        
        // Get total count
        $countStmt = $this->db->prepare("SELECT COUNT(*) FROM enrollments e WHERE {$whereClause}");
        $countStmt->execute($params);
        $totalRows = $countStmt->fetchColumn();
        
        // Get paginated results with joins
        $sql = "SELECT e.*, 
                       s.full_name as student_name, s.login_code, s.national_id,
                       ay.name as year_name,
                       g.title as grade_title, g.code as grade_code,
                       c.name as class_name
                FROM enrollments e
                LEFT JOIN students s ON e.student_id = s.id
                LEFT JOIN academic_years ay ON e.year_id = ay.id
                LEFT JOIN grades g ON e.grade_id = g.id
                LEFT JOIN classes c ON e.class_id = c.id
                WHERE {$whereClause}
                {$orderClause}
                LIMIT ? OFFSET ?";
        
        $stmt = $this->db->prepare($sql);
        $stmt->execute([...$params, $pagination['perPage'], $pagination['offset']]);
        $enrollments = $stmt->fetchAll();
        
        $meta = [
            'page' => $pagination['page'],
            'per_page' => $pagination['perPage'],
            'total_rows' => $totalRows,
            'total_pages' => ceil($totalRows / $pagination['perPage']),
            'sort_by' => $sort['sortBy'],
            'sort_dir' => $sort['sortDir']
        ];
        
        return $this->response->paginated($enrollments, $meta);
    }
    
    public function store(): Response
    {
        $data = $this->validate([
            'student_id' => 'required|numeric',
            'year_id' => 'required|numeric',
            'grade_id' => 'required|numeric',
            'class_id' => 'required|numeric',
            'status' => 'required'
        ]);
        
        // Check if student exists
        $stmt = $this->db->prepare("SELECT id FROM students WHERE id = ?");
        $stmt->execute([$data['student_id']]);
        if (!$stmt->fetch()) {
            return $this->response->error('Student not found', 422);
        }
        
        // Check if enrollment already exists for this year
        $stmt = $this->db->prepare("SELECT id FROM enrollments WHERE student_id = ? AND year_id = ?");
        $stmt->execute([$data['student_id'], $data['year_id']]);
        if ($stmt->fetch()) {
            return $this->response->error('Student already enrolled in this academic year', 422);
        }
        
        try {
            $stmt = $this->db->prepare(
                "INSERT INTO enrollments (student_id, year_id, grade_id, class_id, status) 
                 VALUES (?, ?, ?, ?, ?)"
            );
            
            $stmt->execute([
                $data['student_id'],
                $data['year_id'],
                $data['grade_id'],
                $data['class_id'],
                $data['status']
            ]);
            
            $enrollmentId = $this->db->lastInsertId();
            
            // Get created enrollment with joins
            $stmt = $this->db->prepare(
                "SELECT e.*, 
                        s.full_name as student_name, s.login_code,
                        ay.name as year_name,
                        g.title as grade_title, g.code as grade_code,
                        c.name as class_name
                 FROM enrollments e
                 LEFT JOIN students s ON e.student_id = s.id
                 LEFT JOIN academic_years ay ON e.year_id = ay.id
                 LEFT JOIN grades g ON e.grade_id = g.id
                 LEFT JOIN classes c ON e.class_id = c.id
                 WHERE e.id = ?"
            );
            $stmt->execute([$enrollmentId]);
            $enrollment = $stmt->fetch();
            
            // Log audit event
            $this->authService->logAuditEvent([
                'actor_id' => $this->request->user['user_id'],
                'actor_type' => 'admin',
                'role' => $this->request->user['role'],
                'action' => 'create',
                'entity' => 'enrollments',
                'entity_id' => $enrollmentId,
                'after_data' => $enrollment,
                'ip_address' => $this->request->getClientIp(),
                'user_agent' => $this->request->getUserAgent()
            ]);
            
            return $this->response->success($enrollment, 'Enrollment created successfully');
            
        } catch (\Exception $e) {
            return $this->response->error('Failed to create enrollment: ' . $e->getMessage(), 500);
        }
    }
    
    public function show(string $id): Response
    {
        $stmt = $this->db->prepare(
            "SELECT e.*, 
                    s.full_name as student_name, s.login_code, s.national_id,
                    ay.name as year_name,
                    g.title as grade_title, g.code as grade_code,
                    c.name as class_name
             FROM enrollments e
             LEFT JOIN students s ON e.student_id = s.id
             LEFT JOIN academic_years ay ON e.year_id = ay.id
             LEFT JOIN grades g ON e.grade_id = g.id
             LEFT JOIN classes c ON e.class_id = c.id
             WHERE e.id = ?"
        );
        
        $stmt->execute([$id]);
        $enrollment = $stmt->fetch();
        
        if (!$enrollment) {
            return $this->response->error('Enrollment not found', 404);
        }
        
        return $this->response->success($enrollment);
    }
    
    public function update(string $id): Response
    {
        $data = $this->validate([
            'year_id' => 'required|numeric',
            'grade_id' => 'required|numeric',
            'class_id' => 'required|numeric',
            'status' => 'required'
        ]);
        
        // Get existing enrollment for audit
        $stmt = $this->db->prepare("SELECT * FROM enrollments WHERE id = ?");
        $stmt->execute([$id]);
        $oldEnrollment = $stmt->fetch();
        
        if (!$oldEnrollment) {
            return $this->response->error('Enrollment not found', 404);
        }
        
        // Check if enrollment already exists for this student/year combination (excluding current)
        $stmt = $this->db->prepare("SELECT id FROM enrollments WHERE student_id = ? AND year_id = ? AND id != ?");
        $stmt->execute([$oldEnrollment['student_id'], $data['year_id'], $id]);
        if ($stmt->fetch()) {
            return $this->response->error('Student already enrolled in this academic year', 422);
        }
        
        try {
            $stmt = $this->db->prepare(
                "UPDATE enrollments 
                 SET year_id = ?, grade_id = ?, class_id = ?, status = ? 
                 WHERE id = ?"
            );
            
            $stmt->execute([
                $data['year_id'],
                $data['grade_id'],
                $data['class_id'],
                $data['status'],
                $id
            ]);
            
            // Get updated enrollment with joins
            $stmt = $this->db->prepare(
                "SELECT e.*, 
                        s.full_name as student_name, s.login_code,
                        ay.name as year_name,
                        g.title as grade_title, g.code as grade_code,
                        c.name as class_name
                 FROM enrollments e
                 LEFT JOIN students s ON e.student_id = s.id
                 LEFT JOIN academic_years ay ON e.year_id = ay.id
                 LEFT JOIN grades g ON e.grade_id = g.id
                 LEFT JOIN classes c ON e.class_id = c.id
                 WHERE e.id = ?"
            );
            $stmt->execute([$id]);
            $newEnrollment = $stmt->fetch();
            
            // Log audit event
            $this->authService->logAuditEvent([
                'actor_id' => $this->request->user['user_id'],
                'actor_type' => 'admin',
                'role' => $this->request->user['role'],
                'action' => 'update',
                'entity' => 'enrollments',
                'entity_id' => $id,
                'before_data' => $oldEnrollment,
                'after_data' => $newEnrollment,
                'ip_address' => $this->request->getClientIp(),
                'user_agent' => $this->request->getUserAgent()
            ]);
            
            return $this->response->success($newEnrollment, 'Enrollment updated successfully');
            
        } catch (\Exception $e) {
            return $this->response->error('Failed to update enrollment: ' . $e->getMessage(), 500);
        }
    }
    
    public function destroy(string $id): Response
    {
        // Get enrollment for audit
        $stmt = $this->db->prepare(
            "SELECT e.*, 
                    s.full_name as student_name
             FROM enrollments e
             LEFT JOIN students s ON e.student_id = s.id
             WHERE e.id = ?"
        );
        $stmt->execute([$id]);
        $enrollment = $stmt->fetch();
        
        if (!$enrollment) {
            return $this->response->error('Enrollment not found', 404);
        }
        
        try {
            $stmt = $this->db->prepare("DELETE FROM enrollments WHERE id = ?");
            $stmt->execute([$id]);
            
            // Log audit event
            $this->authService->logAuditEvent([
                'actor_id' => $this->request->user['user_id'],
                'actor_type' => 'admin',
                'role' => $this->request->user['role'],
                'action' => 'delete',
                'entity' => 'enrollments',
                'entity_id' => $id,
                'before_data' => $enrollment,
                'ip_address' => $this->request->getClientIp(),
                'user_agent' => $this->request->getUserAgent()
            ]);
            
            return $this->response->success(null, 'Enrollment deleted successfully');
            
        } catch (\Exception $e) {
            return $this->response->error('Failed to delete enrollment: ' . $e->getMessage(), 500);
        }
    }
    
    public function bulkEnroll(): Response
    {
        $data = $this->request->all();
        
        if (!isset($data['student_ids']) || !is_array($data['student_ids']) || empty($data['student_ids'])) {
            return $this->response->error('No student IDs provided', 400);
        }
        
        $studentIds = array_map('intval', $data['student_ids']);
        $studentIds = array_filter($studentIds);
        
        if (empty($studentIds)) {
            return $this->response->error('Invalid student IDs provided', 400);
        }
        
        $enrollmentData = $this->validate([
            'year_id' => 'required|numeric',
            'grade_id' => 'required|numeric',
            'class_id' => 'required|numeric',
            'status' => 'required'
        ]);
        
        try {
            $this->db->beginTransaction();
            
            $results = [
                'total_students' => count($studentIds),
                'successful' => 0,
                'failed' => 0,
                'errors' => [],
                'enrolled_students' => []
            ];
            
            foreach ($studentIds as $studentId) {
                try {
                    // Check if student exists
                    $stmt = $this->db->prepare("SELECT full_name FROM students WHERE id = ?");
                    $stmt->execute([$studentId]);
                    $student = $stmt->fetch();
                    
                    if (!$student) {
                        throw new \Exception('Student not found');
                    }
                    
                    // Check if already enrolled for this year
                    $stmt = $this->db->prepare("SELECT id FROM enrollments WHERE student_id = ? AND year_id = ?");
                    $stmt->execute([$studentId, $enrollmentData['year_id']]);
                    if ($stmt->fetch()) {
                        throw new \Exception('Already enrolled in this academic year');
                    }
                    
                    // Create enrollment
                    $stmt = $this->db->prepare(
                        "INSERT INTO enrollments (student_id, year_id, grade_id, class_id, status) 
                         VALUES (?, ?, ?, ?, ?)"
                    );
                    
                    $stmt->execute([
                        $studentId,
                        $enrollmentData['year_id'],
                        $enrollmentData['grade_id'],
                        $enrollmentData['class_id'],
                        $enrollmentData['status']
                    ]);
                    
                    $enrollmentId = $this->db->lastInsertId();
                    $results['successful']++;
                    $results['enrolled_students'][] = [
                        'student_id' => $studentId,
                        'student_name' => $student['full_name'],
                        'enrollment_id' => $enrollmentId
                    ];
                    
                    // Log audit event
                    $this->authService->logAuditEvent([
                        'actor_id' => $this->request->user['user_id'],
                        'actor_type' => 'admin',
                        'role' => $this->request->user['role'],
                        'action' => 'bulk_enroll',
                        'entity' => 'enrollments',
                        'entity_id' => $enrollmentId,
                        'metadata' => ['student_id' => $studentId, 'student_name' => $student['full_name']],
                        'ip_address' => $this->request->getClientIp(),
                        'user_agent' => $this->request->getUserAgent()
                    ]);
                    
                } catch (\Exception $e) {
                    $results['failed']++;
                    $results['errors'][] = [
                        'student_id' => $studentId,
                        'student_name' => $student['full_name'] ?? 'Unknown',
                        'error' => $e->getMessage()
                    ];
                }
            }
            
            $this->db->commit();
            
            $message = sprintf(
                'Bulk enrollment completed: %d total, %d successful, %d failed',
                $results['total_students'],
                $results['successful'],
                $results['failed']
            );
            
            return $this->response->success($results, $message);
            
        } catch (\Exception $e) {
            $this->db->rollBack();
            return $this->response->error('Bulk enrollment failed: ' . $e->getMessage(), 500);
        }
    }
    
    public function transferStudents(): Response
    {
        $data = $this->request->all();
        
        if (!isset($data['enrollment_ids']) || !is_array($data['enrollment_ids']) || empty($data['enrollment_ids'])) {
            return $this->response->error('No enrollment IDs provided', 400);
        }
        
        $enrollmentIds = array_map('intval', $data['enrollment_ids']);
        $enrollmentIds = array_filter($enrollmentIds);
        
        if (empty($enrollmentIds)) {
            return $this->response->error('Invalid enrollment IDs provided', 400);
        }
        
        $transferData = $this->validate([
            'year_id' => 'required|numeric',
            'grade_id' => 'required|numeric',
            'class_id' => 'required|numeric'
        ]);
        
        try {
            $this->db->beginTransaction();
            
            $placeholders = str_repeat('?,', count($enrollmentIds) - 1) . '?';
            
            // Get enrollments to transfer
            $stmt = $this->db->prepare(
                "SELECT e.*, s.full_name as student_name 
                 FROM enrollments e 
                 LEFT JOIN students s ON e.student_id = s.id 
                 WHERE e.id IN ($placeholders)"
            );
            $stmt->execute($enrollmentIds);
            $enrollments = $stmt->fetchAll();
            
            if (empty($enrollments)) {
                $this->db->rollBack();
                return $this->response->error('No enrollments found with provided IDs', 404);
            }
            
            $results = [
                'total_enrollments' => count($enrollments),
                'successful' => 0,
                'failed' => 0,
                'errors' => [],
                'transferred_students' => []
            ];
            
            foreach ($enrollments as $enrollment) {
                try {
                    // Check for conflicts (same student, same year)
                    if ($enrollment['year_id'] == $transferData['year_id']) {
                        // Same year transfer - just update grade/class
                        $stmt = $this->db->prepare(
                            "UPDATE enrollments 
                             SET grade_id = ?, class_id = ? 
                             WHERE id = ?"
                        );
                        $stmt->execute([
                            $transferData['grade_id'],
                            $transferData['class_id'],
                            $enrollment['id']
                        ]);
                    } else {
                        // Different year - check for existing enrollment
                        $stmt = $this->db->prepare(
                            "SELECT id FROM enrollments 
                             WHERE student_id = ? AND year_id = ? AND id != ?"
                        );
                        $stmt->execute([
                            $enrollment['student_id'],
                            $transferData['year_id'],
                            $enrollment['id']
                        ]);
                        
                        if ($stmt->fetch()) {
                            throw new \Exception('Student already enrolled in target academic year');
                        }
                        
                        // Update to new year/grade/class
                        $stmt = $this->db->prepare(
                            "UPDATE enrollments 
                             SET year_id = ?, grade_id = ?, class_id = ? 
                             WHERE id = ?"
                        );
                        $stmt->execute([
                            $transferData['year_id'],
                            $transferData['grade_id'],
                            $transferData['class_id'],
                            $enrollment['id']
                        ]);
                    }
                    
                    $results['successful']++;
                    $results['transferred_students'][] = [
                        'enrollment_id' => $enrollment['id'],
                        'student_name' => $enrollment['student_name']
                    ];
                    
                    // Log audit event
                    $this->authService->logAuditEvent([
                        'actor_id' => $this->request->user['user_id'],
                        'actor_type' => 'admin',
                        'role' => $this->request->user['role'],
                        'action' => 'transfer',
                        'entity' => 'enrollments',
                        'entity_id' => $enrollment['id'],
                        'before_data' => $enrollment,
                        'metadata' => $transferData,
                        'ip_address' => $this->request->getClientIp(),
                        'user_agent' => $this->request->getUserAgent()
                    ]);
                    
                } catch (\Exception $e) {
                    $results['failed']++;
                    $results['errors'][] = [
                        'enrollment_id' => $enrollment['id'],
                        'student_name' => $enrollment['student_name'],
                        'error' => $e->getMessage()
                    ];
                }
            }
            
            $this->db->commit();
            
            $message = sprintf(
                'Transfer completed: %d total, %d successful, %d failed',
                $results['total_enrollments'],
                $results['successful'],
                $results['failed']
            );
            
            return $this->response->success($results, $message);
            
        } catch (\Exception $e) {
            $this->db->rollBack();
            return $this->response->error('Transfer failed: ' . $e->getMessage(), 500);
        }
    }
    
    public function import(): Response
    {
        if (!isset($_FILES['file']) || $_FILES['file']['error'] !== UPLOAD_ERR_OK) {
            return $this->response->error('No file uploaded or upload error', 400);
        }
        
        $file = $_FILES['file'];
        $allowedTypes = ['text/csv', 'application/csv', 'text/plain'];
        $maxSize = 5 * 1024 * 1024; // 5MB
        
        if ($file['size'] > $maxSize) {
            return $this->response->error('File size exceeds 5MB limit', 400);
        }
        
        if (!in_array($file['type'], $allowedTypes) && !str_ends_with($file['name'], '.csv')) {
            return $this->response->error('Only CSV files are allowed', 400);
        }
        
        try {
            $handle = fopen($file['tmp_name'], 'r');
            if (!$handle) {
                return $this->response->error('Failed to read uploaded file', 500);
            }
            
            $headers = fgetcsv($handle);
            if (!$headers) {
                fclose($handle);
                return $this->response->error('Invalid CSV file - no headers found', 400);
            }
            
            // Validate headers
            $requiredHeaders = ['student_id', 'year_id', 'grade_id', 'class_id'];
            $optionalHeaders = ['status'];
            $allValidHeaders = array_merge($requiredHeaders, $optionalHeaders);
            
            $invalidHeaders = array_diff($headers, $allValidHeaders);
            if (!empty($invalidHeaders)) {
                fclose($handle);
                return $this->response->error('Invalid headers: ' . implode(', ', $invalidHeaders), 400);
            }
            
            $missingRequired = array_diff($requiredHeaders, $headers);
            if (!empty($missingRequired)) {
                fclose($handle);
                return $this->response->error('Missing required headers: ' . implode(', ', $missingRequired), 400);
            }
            
            $results = [
                'total_rows' => 0,
                'successful' => 0,
                'failed' => 0,
                'errors' => [],
                'enrolled_students' => []
            ];
            
            $rowNumber = 1;
            
            while (($row = fgetcsv($handle)) !== false) {
                $rowNumber++;
                $results['total_rows']++;
                
                // Skip empty rows
                if (empty(array_filter($row))) {
                    continue;
                }
                
                $enrollmentData = array_combine($headers, $row);
                
                try {
                    $this->db->beginTransaction();
                    
                    // Validate required fields
                    if (empty(trim($enrollmentData['student_id'])) || 
                        empty(trim($enrollmentData['year_id'])) ||
                        empty(trim($enrollmentData['grade_id'])) ||
                        empty(trim($enrollmentData['class_id']))) {
                        throw new \Exception('All required fields must be provided');
                    }
                    
                    // Clean and validate data
                    $cleanData = [
                        'student_id' => (int)trim($enrollmentData['student_id']),
                        'year_id' => (int)trim($enrollmentData['year_id']),
                        'grade_id' => (int)trim($enrollmentData['grade_id']),
                        'class_id' => (int)trim($enrollmentData['class_id']),
                        'status' => isset($enrollmentData['status']) ? trim($enrollmentData['status']) : 'active'
                    ];
                    
                    // Validate status
                    if (!in_array($cleanData['status'], ['active', 'inactive', 'graduated', 'transferred'])) {
                        throw new \Exception('Status must be active, inactive, graduated, or transferred');
                    }
                    
                    // Check if student exists
                    $stmt = $this->db->prepare("SELECT full_name FROM students WHERE id = ?");
                    $stmt->execute([$cleanData['student_id']]);
                    $student = $stmt->fetch();
                    
                    if (!$student) {
                        throw new \Exception('Student ID ' . $cleanData['student_id'] . ' not found');
                    }
                    
                    // Check for duplicate enrollment
                    $stmt = $this->db->prepare("SELECT id FROM enrollments WHERE student_id = ? AND year_id = ?");
                    $stmt->execute([$cleanData['student_id'], $cleanData['year_id']]);
                    if ($stmt->fetch()) {
                        throw new \Exception('Student already enrolled in this academic year');
                    }
                    
                    // Validate academic year, grade, and class exist
                    $stmt = $this->db->prepare("SELECT id FROM academic_years WHERE id = ?");
                    $stmt->execute([$cleanData['year_id']]);
                    if (!$stmt->fetch()) {
                        throw new \Exception('Academic year ID ' . $cleanData['year_id'] . ' not found');
                    }
                    
                    $stmt = $this->db->prepare("SELECT id FROM grades WHERE id = ?");
                    $stmt->execute([$cleanData['grade_id']]);
                    if (!$stmt->fetch()) {
                        throw new \Exception('Grade ID ' . $cleanData['grade_id'] . ' not found');
                    }
                    
                    $stmt = $this->db->prepare("SELECT id FROM classes WHERE id = ?");
                    $stmt->execute([$cleanData['class_id']]);
                    if (!$stmt->fetch()) {
                        throw new \Exception('Class ID ' . $cleanData['class_id'] . ' not found');
                    }
                    
                    // Insert enrollment
                    $stmt = $this->db->prepare(
                        "INSERT INTO enrollments (student_id, year_id, grade_id, class_id, status) 
                         VALUES (?, ?, ?, ?, ?)"
                    );
                    
                    $stmt->execute([
                        $cleanData['student_id'],
                        $cleanData['year_id'],
                        $cleanData['grade_id'],
                        $cleanData['class_id'],
                        $cleanData['status']
                    ]);
                    
                    $enrollmentId = $this->db->lastInsertId();
                    
                    $this->db->commit();
                    
                    $results['successful']++;
                    $results['enrolled_students'][] = [
                        'enrollment_id' => $enrollmentId,
                        'student_name' => $student['full_name'],
                        'student_id' => $cleanData['student_id'],
                        'row' => $rowNumber
                    ];
                    
                    // Log audit event
                    $this->authService->logAuditEvent([
                        'actor_id' => $this->request->user['user_id'],
                        'actor_type' => 'admin',
                        'role' => $this->request->user['role'],
                        'action' => 'import',
                        'entity' => 'enrollments',
                        'entity_id' => $enrollmentId,
                        'after_data' => $cleanData,
                        'ip_address' => $this->request->getClientIp(),
                        'user_agent' => $this->request->getUserAgent()
                    ]);
                    
                } catch (\Exception $e) {
                    $this->db->rollBack();
                    $results['failed']++;
                    $results['errors'][] = [
                        'row' => $rowNumber,
                        'data' => $enrollmentData,
                        'error' => $e->getMessage()
                    ];
                }
            }
            
            fclose($handle);
            
            $message = sprintf(
                'Enrollment import completed: %d total, %d successful, %d failed',
                $results['total_rows'],
                $results['successful'],
                $results['failed']
            );
            
            return $this->response->success($results, $message);
            
        } catch (\Exception $e) {
            return $this->response->error('Import failed: ' . $e->getMessage(), 500);
        }
    }
}