<?php

namespace App\Controllers;

use App\Core\Controller;
use App\Core\Database;
use App\Core\Response;
use App\Services\AuthService;
use PDO;

class ExamController extends Controller
{
    private PDO $db;
    private AuthService $authService;
    
    public function __construct($request)
    {
        parent::__construct($request);
        $this->db = Database::getConnection();
        $this->authService = new AuthService();
    }
    
    public function index(): Response
    {
        $pagination = $this->getPaginationParams();
        $sort = $this->getSortParams(['title', 'exam_date', 'year_id', 'term_id', 'grade_id', 'subject_id', 'created_at']);
        
        $whereConditions = ['1=1'];
        $params = [];
        
        if ($yearId = $this->request->get('year_id')) {
            $whereConditions[] = 'e.year_id = ?';
            $params[] = $yearId;
        }
        
        if ($termId = $this->request->get('term_id')) {
            $whereConditions[] = 'e.term_id = ?';
            $params[] = $termId;
        }
        
        if ($gradeId = $this->request->get('grade_id')) {
            $whereConditions[] = 'e.grade_id = ?';
            $params[] = $gradeId;
        }
        
        if ($subjectId = $this->request->get('subject_id')) {
            $whereConditions[] = 'e.subject_id = ?';
            $params[] = $subjectId;
        }
        
        if ($title = $this->request->get('title')) {
            $whereConditions[] = 'e.title LIKE ?';
            $params[] = "%$title%";
        }
        
        if ($enabled = $this->request->get('enabled')) {
            $whereConditions[] = 'e.enabled = ?';
            $params[] = $enabled;
        }
        
        $whereClause = implode(' AND ', $whereConditions);
        $orderClause = $sort['sortBy'] ? "ORDER BY {$sort['sortBy']} {$sort['sortDir']}" : 'ORDER BY e.exam_date DESC';
        
        $countStmt = $this->db->prepare("SELECT COUNT(*) FROM exams e WHERE {$whereClause}");
        $countStmt->execute($params);
        $totalRows = $countStmt->fetchColumn();
        
        $sql = "SELECT e.*, 
                       ay.name as year_name,
                       t.name as term_name,
                       g.title as grade_title, g.code as grade_code,
                       s.name as subject_name
                FROM exams e
                LEFT JOIN academic_years ay ON e.year_id = ay.id
                LEFT JOIN terms t ON e.term_id = t.id
                LEFT JOIN grades g ON e.grade_id = g.id
                LEFT JOIN subjects s ON e.subject_id = s.id
                WHERE {$whereClause}
                {$orderClause}
                LIMIT ? OFFSET ?";
        
        $stmt = $this->db->prepare($sql);
        $stmt->execute([...$params, $pagination['perPage'], $pagination['offset']]);
        $exams = $stmt->fetchAll();
        
        $meta = [
            'page' => $pagination['page'],
            'per_page' => $pagination['perPage'],
            'total_rows' => $totalRows,
            'total_pages' => ceil($totalRows / $pagination['perPage']),
            'sort_by' => $sort['sortBy'],
            'sort_dir' => $sort['sortDir']
        ];
        
        return $this->response->paginated($exams, $meta);
    }
    
    public function store(): Response
    {
        $data = $this->validate([
            'year_id' => 'required|numeric',
            'term_id' => 'required|numeric',
            'grade_id' => 'required|numeric',
            'subject_id' => 'required|numeric',
            'title' => 'required|string|max:100',
            'exam_date' => 'required|date',
            'min_score' => 'required|numeric|min:0',
            'max_score' => 'required|numeric|min:0',
            'enabled' => 'boolean'
        ]);
        
        if ($data['min_score'] >= $data['max_score']) {
            return $this->response->error('Min score must be less than max score', 422);
        }
        
        $examDate = new \DateTime($data['exam_date']);
        $today = new \DateTime();
        
        if ($examDate < $today->setTime(0, 0, 0)) {
            return $this->response->error('Exam date cannot be in the past', 422);
        }
        
        $stmt = $this->db->prepare("SELECT id FROM exams WHERE year_id = ? AND term_id = ? AND grade_id = ? AND subject_id = ? AND title = ?");
        $stmt->execute([$data['year_id'], $data['term_id'], $data['grade_id'], $data['subject_id'], $data['title']]);
        if ($stmt->fetch()) {
            return $this->response->error('An exam with this title already exists for this year/term/grade/subject combination', 422);
        }
        
        try {
            $stmt = $this->db->prepare(
                "INSERT INTO exams (year_id, term_id, grade_id, subject_id, title, exam_date, min_score, max_score, enabled) 
                 VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)"
            );
            
            $stmt->execute([
                $data['year_id'],
                $data['term_id'],
                $data['grade_id'],
                $data['subject_id'],
                $data['title'],
                $data['exam_date'],
                $data['min_score'],
                $data['max_score'],
                $data['enabled'] ?? 1
            ]);
            
            $examId = $this->db->lastInsertId();
            
            $stmt = $this->db->prepare(
                "SELECT e.*, 
                        ay.name as year_name,
                        t.name as term_name,
                        g.title as grade_title, g.code as grade_code,
                        s.name as subject_name
                 FROM exams e
                 LEFT JOIN academic_years ay ON e.year_id = ay.id
                 LEFT JOIN terms t ON e.term_id = t.id
                 LEFT JOIN grades g ON e.grade_id = g.id
                 LEFT JOIN subjects s ON e.subject_id = s.id
                 WHERE e.id = ?"
            );
            $stmt->execute([$examId]);
            $exam = $stmt->fetch();
            
            $this->authService->logAuditEvent([
                'actor_id' => $this->request->user['user_id'],
                'actor_type' => 'admin',
                'role' => $this->request->user['role'],
                'action' => 'create',
                'entity' => 'exams',
                'entity_id' => $examId,
                'after_data' => $exam,
                'ip_address' => $this->request->getClientIp(),
                'user_agent' => $this->request->getUserAgent()
            ]);
            
            return $this->response->success($exam, 'Exam created successfully');
            
        } catch (\Exception $e) {
            return $this->response->error('Failed to create exam: ' . $e->getMessage(), 500);
        }
    }
    
    public function show(string $id): Response
    {
        $stmt = $this->db->prepare(
            "SELECT e.*, 
                    ay.name as year_name,
                    t.name as term_name,
                    g.title as grade_title, g.code as grade_code,
                    s.name as subject_name
             FROM exams e
             LEFT JOIN academic_years ay ON e.year_id = ay.id
             LEFT JOIN terms t ON e.term_id = t.id
             LEFT JOIN grades g ON e.grade_id = g.id
             LEFT JOIN subjects s ON e.subject_id = s.id
             WHERE e.id = ?"
        );
        
        $stmt->execute([$id]);
        $exam = $stmt->fetch();
        
        if (!$exam) {
            return $this->response->error('Exam not found', 404);
        }
        
        return $this->response->success($exam);
    }
    
    public function update(string $id): Response
    {
        $data = $this->validate([
            'year_id' => 'required|numeric',
            'term_id' => 'required|numeric',
            'grade_id' => 'required|numeric',
            'subject_id' => 'required|numeric',
            'title' => 'required|string|max:100',
            'exam_date' => 'required|date',
            'min_score' => 'required|numeric|min:0',
            'max_score' => 'required|numeric|min:0',
            'enabled' => 'boolean'
        ]);
        
        if ($data['min_score'] >= $data['max_score']) {
            return $this->response->error('Min score must be less than max score', 422);
        }
        
        $stmt = $this->db->prepare("SELECT * FROM exams WHERE id = ?");
        $stmt->execute([$id]);
        $oldExam = $stmt->fetch();
        
        if (!$oldExam) {
            return $this->response->error('Exam not found', 404);
        }
        
        $stmt = $this->db->prepare("SELECT id FROM exams WHERE year_id = ? AND term_id = ? AND grade_id = ? AND subject_id = ? AND title = ? AND id != ?");
        $stmt->execute([$data['year_id'], $data['term_id'], $data['grade_id'], $data['subject_id'], $data['title'], $id]);
        if ($stmt->fetch()) {
            return $this->response->error('An exam with this title already exists for this year/term/grade/subject combination', 422);
        }
        
        try {
            $stmt = $this->db->prepare(
                "UPDATE exams 
                 SET year_id = ?, term_id = ?, grade_id = ?, subject_id = ?, title = ?, exam_date = ?, min_score = ?, max_score = ?, enabled = ?
                 WHERE id = ?"
            );
            
            $stmt->execute([
                $data['year_id'],
                $data['term_id'],
                $data['grade_id'],
                $data['subject_id'],
                $data['title'],
                $data['exam_date'],
                $data['min_score'],
                $data['max_score'],
                $data['enabled'] ?? 1,
                $id
            ]);
            
            $stmt = $this->db->prepare(
                "SELECT e.*, 
                        ay.name as year_name,
                        t.name as term_name,
                        g.title as grade_title, g.code as grade_code,
                        s.name as subject_name
                 FROM exams e
                 LEFT JOIN academic_years ay ON e.year_id = ay.id
                 LEFT JOIN terms t ON e.term_id = t.id
                 LEFT JOIN grades g ON e.grade_id = g.id
                 LEFT JOIN subjects s ON e.subject_id = s.id
                 WHERE e.id = ?"
            );
            $stmt->execute([$id]);
            $newExam = $stmt->fetch();
            
            $this->authService->logAuditEvent([
                'actor_id' => $this->request->user['user_id'],
                'actor_type' => 'admin',
                'role' => $this->request->user['role'],
                'action' => 'update',
                'entity' => 'exams',
                'entity_id' => $id,
                'before_data' => $oldExam,
                'after_data' => $newExam,
                'ip_address' => $this->request->getClientIp(),
                'user_agent' => $this->request->getUserAgent()
            ]);
            
            return $this->response->success($newExam, 'Exam updated successfully');
            
        } catch (\Exception $e) {
            return $this->response->error('Failed to update exam: ' . $e->getMessage(), 500);
        }
    }
    
    public function destroy(string $id): Response
    {
        $stmt = $this->db->prepare(
            "SELECT e.*, 
                    ay.name as year_name, t.name as term_name,
                    g.title as grade_title, s.name as subject_name
             FROM exams e
             LEFT JOIN academic_years ay ON e.year_id = ay.id
             LEFT JOIN terms t ON e.term_id = t.id
             LEFT JOIN grades g ON e.grade_id = g.id
             LEFT JOIN subjects s ON e.subject_id = s.id
             WHERE e.id = ?"
        );
        $stmt->execute([$id]);
        $exam = $stmt->fetch();
        
        if (!$exam) {
            return $this->response->error('Exam not found', 404);
        }
        
        $stmt = $this->db->prepare("SELECT COUNT(*) FROM exam_results WHERE exam_id = ?");
        $stmt->execute([$id]);
        $resultCount = $stmt->fetchColumn();
        
        if ($resultCount > 0) {
            return $this->response->error('Cannot delete exam as it has associated results. Delete results first.', 422);
        }
        
        try {
            $stmt = $this->db->prepare("DELETE FROM exams WHERE id = ?");
            $stmt->execute([$id]);
            
            $this->authService->logAuditEvent([
                'actor_id' => $this->request->user['user_id'],
                'actor_type' => 'admin',
                'role' => $this->request->user['role'],
                'action' => 'delete',
                'entity' => 'exams',
                'entity_id' => $id,
                'before_data' => $exam,
                'ip_address' => $this->request->getClientIp(),
                'user_agent' => $this->request->getUserAgent()
            ]);
            
            return $this->response->success(null, 'Exam deleted successfully');
            
        } catch (\Exception $e) {
            return $this->response->error('Failed to delete exam: ' . $e->getMessage(), 500);
        }
    }
    
    public function bulkDelete(): Response
    {
        $data = $this->request->all();
        
        if (!isset($data['exam_ids']) || !is_array($data['exam_ids']) || empty($data['exam_ids'])) {
            return $this->response->error('No exam IDs provided', 400);
        }
        
        $examIds = array_map('intval', $data['exam_ids']);
        $examIds = array_filter($examIds);
        
        if (empty($examIds)) {
            return $this->response->error('Invalid exam IDs provided', 400);
        }
        
        try {
            $this->db->beginTransaction();
            
            $placeholders = str_repeat('?,', count($examIds) - 1) . '?';
            
            $stmt = $this->db->prepare("SELECT COUNT(*) FROM exam_results WHERE exam_id IN ($placeholders)");
            $stmt->execute($examIds);
            $resultCount = $stmt->fetchColumn();
            
            if ($resultCount > 0) {
                $this->db->rollBack();
                return $this->response->error('Cannot delete exams as some have associated results. Delete results first.', 422);
            }
            
            $stmt = $this->db->prepare("SELECT * FROM exams WHERE id IN ($placeholders)");
            $stmt->execute($examIds);
            $exams = $stmt->fetchAll();
            
            $stmt = $this->db->prepare("DELETE FROM exams WHERE id IN ($placeholders)");
            $stmt->execute($examIds);
            
            foreach ($exams as $exam) {
                $this->authService->logAuditEvent([
                    'actor_id' => $this->request->user['user_id'],
                    'actor_type' => 'admin',
                    'role' => $this->request->user['role'],
                    'action' => 'bulk_delete',
                    'entity' => 'exams',
                    'entity_id' => $exam['id'],
                    'before_data' => $exam,
                    'ip_address' => $this->request->getClientIp(),
                    'user_agent' => $this->request->getUserAgent()
                ]);
            }
            
            $this->db->commit();
            
            return $this->response->success([
                'deleted_count' => count($exams),
                'deleted_exams' => $exams
            ], count($exams) . ' exam(s) deleted successfully');
            
        } catch (\Exception $e) {
            $this->db->rollBack();
            return $this->response->error('Failed to delete exams: ' . $e->getMessage(), 500);
        }
    }
    
    public function getGradeSubjects(string $gradeId): Response
    {
        try {
            $stmt = $this->db->prepare("
                SELECT s.*, gsr.min_score, gsr.max_score
                FROM subjects s
                LEFT JOIN grade_subject_rules gsr ON s.id = gsr.subject_id AND gsr.grade_id = ?
                WHERE s.enabled = 1
                ORDER BY s.name
            ");
            $stmt->execute([$gradeId]);
            $subjects = $stmt->fetchAll();
            
            return $this->response->success($subjects);
            
        } catch (\Exception $e) {
            return $this->response->error('Failed to get grade subjects: ' . $e->getMessage(), 500);
        }
    }
    
    public function bulkCreate(): Response
    {
        $data = $this->validate([
            'year_id' => 'required|numeric',
            'term_id' => 'required|numeric',
            'grade_id' => 'required|numeric',
            'exam_date' => 'required|date',
            'subjects' => 'required|array',
            'enabled' => 'boolean'
        ]);
        
        $examDate = new \DateTime($data['exam_date']);
        $today = new \DateTime();
        
        if ($examDate < $today->setTime(0, 0, 0)) {
            return $this->response->error('Exam date cannot be in the past', 422);
        }
        
        if (empty($data['subjects'])) {
            return $this->response->error('At least one subject must be selected', 422);
        }
        
        try {
            $this->db->beginTransaction();
            
            $results = [
                'total_subjects' => count($data['subjects']),
                'successful' => 0,
                'failed' => 0,
                'errors' => [],
                'created_exams' => []
            ];
            
            foreach ($data['subjects'] as $subjectData) {
                try {
                    // Validate subject data
                    if (!isset($subjectData['subject_id']) || !isset($subjectData['title']) || 
                        !isset($subjectData['min_score']) || !isset($subjectData['max_score'])) {
                        throw new \Exception('Missing required subject data');
                    }
                    
                    if ($subjectData['min_score'] >= $subjectData['max_score']) {
                        throw new \Exception('Min score must be less than max score for subject ' . $subjectData['title']);
                    }
                    
                    // Check for existing exam
                    $stmt = $this->db->prepare("
                        SELECT id FROM exams 
                        WHERE year_id = ? AND term_id = ? AND grade_id = ? AND subject_id = ? AND title = ?
                    ");
                    $stmt->execute([
                        $data['year_id'], 
                        $data['term_id'], 
                        $data['grade_id'], 
                        $subjectData['subject_id'], 
                        $subjectData['title']
                    ]);
                    
                    if ($stmt->fetch()) {
                        throw new \Exception('Exam "' . $subjectData['title'] . '" already exists');
                    }
                    
                    // Create the exam
                    $stmt = $this->db->prepare("
                        INSERT INTO exams (year_id, term_id, grade_id, subject_id, title, exam_date, min_score, max_score, enabled) 
                        VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)
                    ");
                    
                    $stmt->execute([
                        $data['year_id'],
                        $data['term_id'],
                        $data['grade_id'],
                        $subjectData['subject_id'],
                        $subjectData['title'],
                        $data['exam_date'],
                        $subjectData['min_score'],
                        $subjectData['max_score'],
                        $data['enabled'] ?? 1
                    ]);
                    
                    $examId = $this->db->lastInsertId();
                    
                    // Get created exam with joins
                    $stmt = $this->db->prepare("
                        SELECT e.*, 
                               ay.name as year_name,
                               t.name as term_name,
                               g.title as grade_title, g.code as grade_code,
                               s.name as subject_name
                        FROM exams e
                        LEFT JOIN academic_years ay ON e.year_id = ay.id
                        LEFT JOIN terms t ON e.term_id = t.id
                        LEFT JOIN grades g ON e.grade_id = g.id
                        LEFT JOIN subjects s ON e.subject_id = s.id
                        WHERE e.id = ?
                    ");
                    $stmt->execute([$examId]);
                    $exam = $stmt->fetch();
                    
                    $results['successful']++;
                    $results['created_exams'][] = $exam;
                    
                    // Log audit event
                    $this->authService->logAuditEvent([
                        'actor_id' => $this->request->user['user_id'],
                        'actor_type' => 'admin',
                        'role' => $this->request->user['role'],
                        'action' => 'bulk_create',
                        'entity' => 'exams',
                        'entity_id' => $examId,
                        'after_data' => $exam,
                        'ip_address' => $this->request->getClientIp(),
                        'user_agent' => $this->request->getUserAgent()
                    ]);
                    
                } catch (\Exception $e) {
                    $results['failed']++;
                    $results['errors'][] = [
                        'subject_id' => $subjectData['subject_id'] ?? null,
                        'title' => $subjectData['title'] ?? 'Unknown',
                        'error' => $e->getMessage()
                    ];
                }
            }
            
            $this->db->commit();
            
            $message = sprintf(
                'Bulk exam creation completed: %d total, %d successful, %d failed',
                $results['total_subjects'],
                $results['successful'],
                $results['failed']
            );
            
            return $this->response->success($results, $message);
            
        } catch (\Exception $e) {
            $this->db->rollBack();
            return $this->response->error('Bulk exam creation failed: ' . $e->getMessage(), 500);
        }
    }
}