<?php

namespace App\Controllers;

use App\Core\Controller;
use App\Core\Database;
use App\Core\Response;
use App\Services\AuthService;
use PDO;

class GradeController extends Controller
{
    private PDO $db;
    private AuthService $authService;
    
    public function __construct($request)
    {
        parent::__construct($request);
        $this->db = Database::getConnection();
        $this->authService = new AuthService();
    }
    
    public function index(): Response
    {
        $pagination = $this->getPaginationParams();
        $sort = $this->getSortParams(['id', 'code', 'title', 'sort_order', 'enabled', 'created_at']);
        
        $whereConditions = ['1=1'];
        $params = [];
        
        // Filters
        if ($search = $this->request->get('search')) {
            $whereConditions[] = '(code LIKE ? OR title LIKE ?)';
            $params[] = "%{$search}%";
            $params[] = "%{$search}%";
        }
        
        if (isset($_GET['enabled']) && $_GET['enabled'] !== '') {
            $whereConditions[] = 'enabled = ?';
            $params[] = (int)$this->request->get('enabled');
        }
        
        $whereClause = implode(' AND ', $whereConditions);
        $orderClause = $sort['sortBy'] ? "ORDER BY {$sort['sortBy']} {$sort['sortDir']}" : 'ORDER BY sort_order ASC';
        
        // Get total count
        $countStmt = $this->db->prepare("SELECT COUNT(*) FROM grades WHERE {$whereClause}");
        $countStmt->execute($params);
        $totalRows = $countStmt->fetchColumn();
        
        // Get results
        $sql = "SELECT id, code, title, sort_order, enabled, created_at 
                FROM grades 
                WHERE {$whereClause} 
                {$orderClause} 
                LIMIT ? OFFSET ?";
        
        $stmt = $this->db->prepare($sql);
        $stmt->execute([...$params, $pagination['perPage'], $pagination['offset']]);
        $grades = $stmt->fetchAll();
        
        $meta = [
            'page' => $pagination['page'],
            'per_page' => $pagination['perPage'],
            'total_rows' => $totalRows,
            'total_pages' => ceil($totalRows / $pagination['perPage']),
            'sort_by' => $sort['sortBy'],
            'sort_dir' => $sort['sortDir']
        ];
        
        return $this->response->paginated($grades, $meta);
    }
    
    public function show(string $id): Response
    {
        $stmt = $this->db->prepare("SELECT * FROM grades WHERE id = ?");
        $stmt->execute([$id]);
        $grade = $stmt->fetch();
        
        if (!$grade) {
            return $this->response->error('Grade not found', 404);
        }
        
        return $this->response->success($grade);
    }
    
    public function store(): Response
    {
        $data = $this->validate([
            'code' => 'required|min:1|max:20',
            'title' => 'required|min:2|max:100',
            'sort_order' => 'required|numeric',
            'enabled' => 'required'
        ]);
        
        // Check for duplicate code
        $stmt = $this->db->prepare("SELECT id FROM grades WHERE code = ?");
        $stmt->execute([$data['code']]);
        if ($stmt->fetch()) {
            return $this->response->error('Grade code already exists', 422);
        }
        
        // Check for duplicate sort order
        $stmt = $this->db->prepare("SELECT id FROM grades WHERE sort_order = ?");
        $stmt->execute([$data['sort_order']]);
        if ($stmt->fetch()) {
            return $this->response->error('Sort order already exists', 422);
        }
        
        try {
            $stmt = $this->db->prepare(
                "INSERT INTO grades (code, title, sort_order, enabled) VALUES (?, ?, ?, ?)"
            );
            
            $stmt->execute([
                $data['code'],
                $data['title'],
                $data['sort_order'],
                (int)$data['enabled']
            ]);
            
            $gradeId = $this->db->lastInsertId();
            
            // Get created grade
            $stmt = $this->db->prepare("SELECT * FROM grades WHERE id = ?");
            $stmt->execute([$gradeId]);
            $grade = $stmt->fetch();
            
            // Log audit event
            if (isset($this->request->user)) {
                $this->authService->logAuditEvent([
                    'actor_id' => $this->request->user['user_id'],
                    'actor_type' => 'admin',
                    'role' => $this->request->user['role'],
                    'action' => 'create',
                    'entity' => 'grades',
                    'entity_id' => $gradeId,
                    'after_data' => $grade,
                    'ip_address' => $this->request->getClientIp(),
                    'user_agent' => $this->request->getUserAgent()
                ]);
            }
            
            return $this->response->success($grade, 'Grade created successfully');
            
        } catch (\Exception $e) {
            return $this->response->error('Failed to create grade: ' . $e->getMessage(), 500);
        }
    }
    
    public function update(string $id): Response
    {
        $data = $this->validate([
            'code' => 'required|min:1|max:20',
            'title' => 'required|min:2|max:100',
            'sort_order' => 'required|numeric',
            'enabled' => 'required'
        ]);
        
        // Get existing grade for audit
        $stmt = $this->db->prepare("SELECT * FROM grades WHERE id = ?");
        $stmt->execute([$id]);
        $oldGrade = $stmt->fetch();
        
        if (!$oldGrade) {
            return $this->response->error('Grade not found', 404);
        }
        
        // Check for duplicate code (excluding current grade)
        $stmt = $this->db->prepare("SELECT id FROM grades WHERE code = ? AND id != ?");
        $stmt->execute([$data['code'], $id]);
        if ($stmt->fetch()) {
            return $this->response->error('Grade code already exists', 422);
        }
        
        // Check for duplicate sort order (excluding current grade)
        $stmt = $this->db->prepare("SELECT id FROM grades WHERE sort_order = ? AND id != ?");
        $stmt->execute([$data['sort_order'], $id]);
        if ($stmt->fetch()) {
            return $this->response->error('Sort order already exists', 422);
        }
        
        try {
            $stmt = $this->db->prepare(
                "UPDATE grades SET code = ?, title = ?, sort_order = ?, enabled = ? WHERE id = ?"
            );
            
            $stmt->execute([
                $data['code'],
                $data['title'],
                $data['sort_order'],
                (int)$data['enabled'],
                $id
            ]);
            
            // Get updated grade
            $stmt = $this->db->prepare("SELECT * FROM grades WHERE id = ?");
            $stmt->execute([$id]);
            $newGrade = $stmt->fetch();
            
            // Log audit event
            if (isset($this->request->user)) {
                $this->authService->logAuditEvent([
                    'actor_id' => $this->request->user['user_id'],
                    'actor_type' => 'admin',
                    'role' => $this->request->user['role'],
                    'action' => 'update',
                    'entity' => 'grades',
                    'entity_id' => $id,
                    'before_data' => $oldGrade,
                    'after_data' => $newGrade,
                    'ip_address' => $this->request->getClientIp(),
                    'user_agent' => $this->request->getUserAgent()
                ]);
            }
            
            return $this->response->success($newGrade, 'Grade updated successfully');
            
        } catch (\Exception $e) {
            return $this->response->error('Failed to update grade: ' . $e->getMessage(), 500);
        }
    }
    
    public function destroy(string $id): Response
    {
        // Check for dependent classes
        $stmt = $this->db->prepare("SELECT COUNT(*) FROM classes WHERE grade_id = ?");
        $stmt->execute([$id]);
        $classCount = $stmt->fetchColumn();
        
        if ($classCount > 0) {
            return $this->response->error('Cannot delete grade with existing classes', 422);
        }
        
        // Check for dependent enrollments
        $stmt = $this->db->prepare("SELECT COUNT(*) FROM enrollments WHERE grade_id = ?");
        $stmt->execute([$id]);
        $enrollmentCount = $stmt->fetchColumn();
        
        if ($enrollmentCount > 0) {
            return $this->response->error('Cannot delete grade with existing student enrollments', 422);
        }
        
        // Get grade for audit
        $stmt = $this->db->prepare("SELECT * FROM grades WHERE id = ?");
        $stmt->execute([$id]);
        $grade = $stmt->fetch();
        
        if (!$grade) {
            return $this->response->error('Grade not found', 404);
        }
        
        try {
            $stmt = $this->db->prepare("DELETE FROM grades WHERE id = ?");
            $stmt->execute([$id]);
            
            // Log audit event
            if (isset($this->request->user)) {
                $this->authService->logAuditEvent([
                    'actor_id' => $this->request->user['user_id'],
                    'actor_type' => 'admin',
                    'role' => $this->request->user['role'],
                    'action' => 'delete',
                    'entity' => 'grades',
                    'entity_id' => $id,
                    'before_data' => $grade,
                    'ip_address' => $this->request->getClientIp(),
                    'user_agent' => $this->request->getUserAgent()
                ]);
            }
            
            return $this->response->success(null, 'Grade deleted successfully');
            
        } catch (\Exception $e) {
            return $this->response->error('Failed to delete grade: ' . $e->getMessage(), 500);
        }
    }
}