<?php

namespace App\Controllers;

use App\Core\Controller;
use App\Core\Database;
use App\Core\Response;
use Exception;

class GradeSubjectRuleController extends Controller
{
    private $db;
    
    public function __construct($request = null)
    {
        parent::__construct($request);
        $this->db = Database::getConnection();
    }
    
    public function index(): Response
    {
        try {
            $page = max(1, intval($this->request->get('page', 1)));
            $perPage = min(200, max(10, intval($this->request->get('per_page', 50))));
            $offset = ($page - 1) * $perPage;
            
            // Build WHERE clause for filters
            $whereConditions = ['1=1'];
            $params = [];
            
            // Filter by grade
            if ($gradeId = $this->request->get('grade_id')) {
                $whereConditions[] = 'gsr.grade_id = ?';
                $params[] = $gradeId;
            }
            
            // Filter by subject
            if ($subjectId = $this->request->get('subject_id')) {
                $whereConditions[] = 'gsr.subject_id = ?';
                $params[] = $subjectId;
            }
            
            // Filter by enabled status
            if (($enabled = $this->request->get('enabled')) !== null && $enabled !== '') {
                $whereConditions[] = 'gsr.enabled = ?';
                $params[] = intval($enabled);
            }
            
            // Search functionality
            if ($search = $this->request->get('search')) {
                $whereConditions[] = '(g.title LIKE ? OR g.code LIKE ? OR s.name LIKE ?)';
                $searchTerm = '%' . $search . '%';
                $params[] = $searchTerm;
                $params[] = $searchTerm;
                $params[] = $searchTerm;
            }
            
            $whereClause = implode(' AND ', $whereConditions);
            
            // Handle sorting
            $sortBy = $this->request->get('sort_by', 'g.sort_order');
            $sortDir = $this->request->get('sort_dir', 'asc');
            
            $allowedSortFields = ['g.sort_order', 'g.title', 'g.code', 's.name', 'gsr.min_score', 'gsr.max_score', 'gsr.enabled', 'gsr.created_at'];
            if (!in_array($sortBy, $allowedSortFields)) {
                $sortBy = 'g.sort_order';
            }
            
            if (!in_array(strtolower($sortDir), ['asc', 'desc'])) {
                $sortDir = 'asc';
            }
            
            // Get total count
            $countSql = "
                SELECT COUNT(*) 
                FROM grade_subject_rules gsr
                INNER JOIN grades g ON gsr.grade_id = g.id
                INNER JOIN subjects s ON gsr.subject_id = s.id
                WHERE $whereClause
            ";
            
            $countStmt = $this->db->prepare($countSql);
            $countStmt->execute($params);
            $totalCount = $countStmt->fetchColumn();
            
            // Get rules with grade and subject details
            $sql = "
                SELECT 
                    gsr.*,
                    g.code as grade_code,
                    g.title as grade_title,
                    g.sort_order as grade_sort_order,
                    s.name as subject_name
                FROM grade_subject_rules gsr
                INNER JOIN grades g ON gsr.grade_id = g.id
                INNER JOIN subjects s ON gsr.subject_id = s.id
                WHERE $whereClause
                ORDER BY $sortBy $sortDir
                LIMIT ? OFFSET ?
            ";
            
            $params[] = $perPage;
            $params[] = $offset;
            
            $stmt = $this->db->prepare($sql);
            $stmt->execute($params);
            $rules = $stmt->fetchAll(\PDO::FETCH_ASSOC);
            
            // Format numbers for display
            foreach ($rules as &$rule) {
                $rule['min_score'] = number_format($rule['min_score'], 2);
                $rule['max_score'] = number_format($rule['max_score'], 2);
                $rule['enabled'] = (bool)$rule['enabled'];
            }
            
            return new Response([
                'success' => true,
                'data' => $rules,
                'meta' => [
                    'total' => $totalCount,
                    'page' => $page,
                    'per_page' => $perPage,
                    'total_pages' => ceil($totalCount / $perPage),
                    'sort_by' => $sortBy,
                    'sort_dir' => $sortDir
                ]
            ]);
            
        } catch (Exception $e) {
            error_log("Error in GradeSubjectRuleController::index: " . $e->getMessage());
            return new Response(['success' => false, 'message' => 'Failed to fetch grade subject rules'], 500);
        }
    }
    
    public function show($id): Response
    {
        try {
            $sql = "
                SELECT 
                    gsr.*,
                    g.code as grade_code,
                    g.title as grade_title,
                    s.name as subject_name
                FROM grade_subject_rules gsr
                INNER JOIN grades g ON gsr.grade_id = g.id
                INNER JOIN subjects s ON gsr.subject_id = s.id
                WHERE gsr.id = ?
            ";
            
            $stmt = $this->db->prepare($sql);
            $stmt->execute([$id]);
            $rule = $stmt->fetch(\PDO::FETCH_ASSOC);
            
            if (!$rule) {
                return new Response(['success' => false, 'message' => 'Grade subject rule not found'], 404);
            }
            
            // Format numbers
            $rule['min_score'] = number_format($rule['min_score'], 2);
            $rule['max_score'] = number_format($rule['max_score'], 2);
            $rule['enabled'] = (bool)$rule['enabled'];
            
            return new Response(['success' => true, 'data' => $rule]);
            
        } catch (Exception $e) {
            error_log("Error in GradeSubjectRuleController::show: " . $e->getMessage());
            return new Response(['success' => false, 'message' => 'Failed to fetch grade subject rule'], 500);
        }
    }
    
    public function store(): Response
    {
        try {
            $data = $this->request->all();
            
            // Validation
            $errors = $this->validateRuleData($data);
            if (!empty($errors)) {
                return new Response(['success' => false, 'message' => 'Validation failed', 'errors' => $errors], 422);
            }
            
            $gradeId = intval($data['grade_id']);
            $subjectId = intval($data['subject_id']);
            $minScore = floatval($data['min_score']);
            $maxScore = floatval($data['max_score']);
            $enabled = isset($data['enabled']) ? intval($data['enabled']) : 1;
            
            // Check for duplicate grade-subject combination
            $checkSql = "SELECT id FROM grade_subject_rules WHERE grade_id = ? AND subject_id = ?";
            $checkStmt = $this->db->prepare($checkSql);
            $checkStmt->execute([$gradeId, $subjectId]);
            if ($checkStmt->fetch()) {
                return new Response(['success' => false, 'message' => 'A rule already exists for this grade-subject combination'], 422);
            }
            
            // Verify grade and subject exist
            if (!$this->verifyGradeExists($gradeId)) {
                return new Response(['success' => false, 'message' => 'Invalid grade selected'], 422);
            }
            
            if (!$this->verifySubjectExists($subjectId)) {
                return new Response(['success' => false, 'message' => 'Invalid subject selected'], 422);
            }
            
            // Insert rule
            $sql = "INSERT INTO grade_subject_rules (grade_id, subject_id, min_score, max_score, enabled) VALUES (?, ?, ?, ?, ?)";
            $stmt = $this->db->prepare($sql);
            $stmt->execute([$gradeId, $subjectId, $minScore, $maxScore, $enabled]);
            
            $ruleId = $this->db->lastInsertId();
            
            // Get the created rule with details
            $newRule = $this->show($ruleId);
            
            // Log the action
            $this->logAction('create', 'grade_subject_rules', $ruleId, null, $newRule->getData()['data']);
            
            return new Response(['success' => true, 'message' => 'Grade subject rule created successfully', 'data' => $newRule->getData()['data']], 201);
            
        } catch (Exception $e) {
            error_log("Error in GradeSubjectRuleController::store: " . $e->getMessage());
            return new Response(['success' => false, 'message' => 'Failed to create grade subject rule'], 500);
        }
    }
    
    public function update($id): Response
    {
        try {
            $data = $this->request->all();
            
            // Check if rule exists
            $existingRule = $this->show($id);
            if (!$existingRule->getData()['success']) {
                return $existingRule;
            }
            $oldRule = $existingRule->getData()['data'];
            
            // Validation
            $errors = $this->validateRuleData($data);
            if (!empty($errors)) {
                return new Response(['success' => false, 'message' => 'Validation failed', 'errors' => $errors], 422);
            }
            
            $gradeId = intval($data['grade_id']);
            $subjectId = intval($data['subject_id']);
            $minScore = floatval($data['min_score']);
            $maxScore = floatval($data['max_score']);
            $enabled = isset($data['enabled']) ? intval($data['enabled']) : 1;
            
            // Check for duplicate grade-subject combination (excluding current rule)
            $checkSql = "SELECT id FROM grade_subject_rules WHERE grade_id = ? AND subject_id = ? AND id != ?";
            $checkStmt = $this->db->prepare($checkSql);
            $checkStmt->execute([$gradeId, $subjectId, $id]);
            if ($checkStmt->fetch()) {
                return new Response(['success' => false, 'message' => 'A rule already exists for this grade-subject combination'], 422);
            }
            
            // Verify grade and subject exist
            if (!$this->verifyGradeExists($gradeId)) {
                return new Response(['success' => false, 'message' => 'Invalid grade selected'], 422);
            }
            
            if (!$this->verifySubjectExists($subjectId)) {
                return new Response(['success' => false, 'message' => 'Invalid subject selected'], 422);
            }
            
            // Update rule
            $sql = "UPDATE grade_subject_rules SET grade_id = ?, subject_id = ?, min_score = ?, max_score = ?, enabled = ? WHERE id = ?";
            $stmt = $this->db->prepare($sql);
            $stmt->execute([$gradeId, $subjectId, $minScore, $maxScore, $enabled, $id]);
            
            // Get updated rule
            $updatedRule = $this->show($id);
            
            // Log the action
            $this->logAction('update', 'grade_subject_rules', $id, $oldRule, $updatedRule->getData()['data']);
            
            return new Response(['success' => true, 'message' => 'Grade subject rule updated successfully', 'data' => $updatedRule->getData()['data']]);
            
        } catch (Exception $e) {
            error_log("Error in GradeSubjectRuleController::update: " . $e->getMessage());
            return new Response(['success' => false, 'message' => 'Failed to update grade subject rule'], 500);
        }
    }
    
    public function destroy($id): Response
    {
        try {
            // Check if rule exists
            $existingRule = $this->show($id);
            if (!$existingRule->getData()['success']) {
                return $existingRule;
            }
            $rule = $existingRule->getData()['data'];
            
            // Check for dependencies - exams using this grade-subject combination
            $examCheckSql = "SELECT COUNT(*) FROM exams WHERE grade_id = ? AND subject_id = ?";
            $examCheckStmt = $this->db->prepare($examCheckSql);
            $examCheckStmt->execute([$rule['grade_id'], $rule['subject_id']]);
            $examCount = $examCheckStmt->fetchColumn();
            
            if ($examCount > 0) {
                return new Response([
                    'success' => false, 
                    'message' => 'Cannot delete rule. There are ' . $examCount . ' exam(s) using this grade-subject combination.'
                ], 422);
            }
            
            // Delete the rule
            $sql = "DELETE FROM grade_subject_rules WHERE id = ?";
            $stmt = $this->db->prepare($sql);
            $stmt->execute([$id]);
            
            // Log the action
            $this->logAction('delete', 'grade_subject_rules', $id, $rule, null);
            
            return new Response(['success' => true, 'message' => 'Grade subject rule deleted successfully']);
            
        } catch (Exception $e) {
            error_log("Error in GradeSubjectRuleController::destroy: " . $e->getMessage());
            return new Response(['success' => false, 'message' => 'Failed to delete grade subject rule'], 500);
        }
    }
    
    private function validateRuleData($data): array
    {
        $errors = [];
        
        // Grade ID validation
        if (empty($data['grade_id'])) {
            $errors['grade_id'] = 'Grade is required';
        } elseif (!is_numeric($data['grade_id']) || intval($data['grade_id']) <= 0) {
            $errors['grade_id'] = 'Invalid grade selected';
        }
        
        // Subject ID validation
        if (empty($data['subject_id'])) {
            $errors['subject_id'] = 'Subject is required';
        } elseif (!is_numeric($data['subject_id']) || intval($data['subject_id']) <= 0) {
            $errors['subject_id'] = 'Invalid subject selected';
        }
        
        // Min score validation
        if (!isset($data['min_score']) || $data['min_score'] === '') {
            $errors['min_score'] = 'Minimum score is required';
        } elseif (!is_numeric($data['min_score'])) {
            $errors['min_score'] = 'Minimum score must be a number';
        } elseif (floatval($data['min_score']) < 0) {
            $errors['min_score'] = 'Minimum score cannot be negative';
        } elseif (floatval($data['min_score']) > 999.99) {
            $errors['min_score'] = 'Minimum score cannot exceed 999.99';
        }
        
        // Max score validation
        if (!isset($data['max_score']) || $data['max_score'] === '') {
            $errors['max_score'] = 'Maximum score is required';
        } elseif (!is_numeric($data['max_score'])) {
            $errors['max_score'] = 'Maximum score must be a number';
        } elseif (floatval($data['max_score']) < 0) {
            $errors['max_score'] = 'Maximum score cannot be negative';
        } elseif (floatval($data['max_score']) > 999.99) {
            $errors['max_score'] = 'Maximum score cannot exceed 999.99';
        }
        
        // Cross-validation: min <= max
        if (empty($errors['min_score']) && empty($errors['max_score'])) {
            $minScore = floatval($data['min_score']);
            $maxScore = floatval($data['max_score']);
            
            if ($minScore > $maxScore) {
                $errors['max_score'] = 'Maximum score must be greater than or equal to minimum score';
            }
        }
        
        // Enabled validation
        if (isset($data['enabled']) && !in_array($data['enabled'], [0, 1, '0', '1', true, false])) {
            $errors['enabled'] = 'Invalid enabled status';
        }
        
        return $errors;
    }
    
    private function verifyGradeExists($gradeId): bool
    {
        $sql = "SELECT id FROM grades WHERE id = ? AND enabled = 1";
        $stmt = $this->db->prepare($sql);
        $stmt->execute([$gradeId]);
        return (bool)$stmt->fetch();
    }
    
    private function verifySubjectExists($subjectId): bool
    {
        $sql = "SELECT id FROM subjects WHERE id = ? AND enabled = 1";
        $stmt = $this->db->prepare($sql);
        $stmt->execute([$subjectId]);
        return (bool)$stmt->fetch();
    }
    
    private function logAction(string $action, string $entity, $entityId, $before, $after): void
    {
        try {
            $logData = [
                'actor' => $this->request->user['username'] ?? 'system',
                'role' => $this->request->user['role'] ?? 'admin',
                'timestamp' => date('Y-m-d H:i:s'),
                'action' => $action,
                'entity' => $entity,
                'entity_id' => $entityId,
                'before' => $before ? json_encode($before) : null,
                'after' => $after ? json_encode($after) : null
            ];
            
            error_log("Audit Log: " . json_encode($logData));
        } catch (Exception $e) {
            error_log("Failed to log action: " . $e->getMessage());
        }
    }
}