<?php

namespace App\Controllers;

use App\Core\Controller;
use App\Core\Database;
use App\Core\Response;
use App\Services\AuthService;
use PDO;

class PaymentController extends Controller
{
    private PDO $db;
    private AuthService $authService;
    
    public function __construct($request)
    {
        parent::__construct($request);
        $this->db = Database::getConnection();
        $this->authService = new AuthService();
    }
    
    /**
     * Get payment history for a specific enrollment
     */
    public function index(string $enrollmentId): Response
    {
        $pagination = $this->getPaginationParams();
        $sort = $this->getSortParams(['paid_on', 'amount', 'method', 'created_at']);
        
        $whereConditions = ['p.enrollment_id = ?'];
        $params = [$enrollmentId];
        
        // Filter by date range
        if ($startDate = $this->request->get('start_date')) {
            $whereConditions[] = 'p.paid_on >= ?';
            $params[] = $startDate;
        }
        
        if ($endDate = $this->request->get('end_date')) {
            $whereConditions[] = 'p.paid_on <= ?';
            $params[] = $endDate;
        }
        
        // Filter by method
        if ($method = $this->request->get('method')) {
            $whereConditions[] = 'p.method = ?';
            $params[] = $method;
        }
        
        // Filter by amount range
        if ($minAmount = $this->request->get('min_amount')) {
            $whereConditions[] = 'p.amount >= ?';
            $params[] = $minAmount;
        }
        
        if ($maxAmount = $this->request->get('max_amount')) {
            $whereConditions[] = 'p.amount <= ?';
            $params[] = $maxAmount;
        }
        
        $whereClause = implode(' AND ', $whereConditions);
        $orderClause = $sort['sortBy'] ? "ORDER BY {$sort['sortBy']} {$sort['sortDir']}" : 'ORDER BY p.paid_on DESC';
        
        // Get total count
        $countStmt = $this->db->prepare("SELECT COUNT(*) FROM payments p WHERE {$whereClause}");
        $countStmt->execute($params);
        $totalRows = $countStmt->fetchColumn();
        
        // Get payments with creator info
        $sql = "SELECT p.*, 
                       au.full_name as created_by_name,
                       e.student_id, e.year_id, e.grade_id, e.class_id,
                       s.full_name as student_name,
                       ay.name as year_name,
                       g.title as grade_title,
                       c.name as class_name
                FROM payments p
                LEFT JOIN admin_users au ON p.created_by = au.id
                LEFT JOIN enrollments e ON p.enrollment_id = e.id
                LEFT JOIN students s ON e.student_id = s.id
                LEFT JOIN academic_years ay ON e.year_id = ay.id
                LEFT JOIN grades g ON e.grade_id = g.id
                LEFT JOIN classes c ON e.class_id = c.id
                WHERE {$whereClause}
                {$orderClause}
                LIMIT ? OFFSET ?";
        
        $stmt = $this->db->prepare($sql);
        $stmt->execute([...$params, $pagination['perPage'], $pagination['offset']]);
        $payments = $stmt->fetchAll();
        
        $meta = [
            'page' => $pagination['page'],
            'per_page' => $pagination['perPage'],
            'total_rows' => $totalRows,
            'total_pages' => ceil($totalRows / $pagination['perPage']),
            'sort_by' => $sort['sortBy'],
            'sort_dir' => $sort['sortDir']
        ];
        
        return $this->response->paginated($payments, $meta);
    }
    
    /**
     * Create a new payment
     */
    public function store(string $enrollmentId): Response
    {
        $data = $this->validate([
            'amount' => 'required|numeric|min:0.01',
            'paid_on' => 'required|date',
            'method' => 'required|in:cash,card,bank,wallet',
            'notes' => 'string|max:1000'
        ]);
        
        // Verify enrollment exists and get details
        $stmt = $this->db->prepare("
            SELECT e.*, ay.status as year_status, s.full_name as student_name
            FROM enrollments e
            LEFT JOIN academic_years ay ON e.year_id = ay.id
            LEFT JOIN students s ON e.student_id = s.id
            WHERE e.id = ? AND e.status = 'active'
        ");
        $stmt->execute([$enrollmentId]);
        $enrollment = $stmt->fetch();
        
        if (!$enrollment) {
            return $this->response->error('Enrollment not found or not active', 404);
        }
        
        // Check if academic year is open
        if ($enrollment['year_status'] === 'closed') {
            return $this->response->error('Cannot add payments to a closed academic year', 422);
        }
        
        // Validate payment date
        $paymentDate = new \DateTime($data['paid_on']);
        $today = new \DateTime();
        
        if ($paymentDate > $today) {
            return $this->response->error('Payment date cannot be in the future', 422);
        }
        
        try {
            $stmt = $this->db->prepare("
                INSERT INTO payments (enrollment_id, amount, paid_on, method, notes, created_by) 
                VALUES (?, ?, ?, ?, ?, ?)
            ");
            
            $stmt->execute([
                $enrollmentId,
                $data['amount'],
                $data['paid_on'],
                $data['method'],
                $data['notes'] ?? null,
                $this->request->user['user_id']
            ]);
            
            $paymentId = $this->db->lastInsertId();
            
            // Get the created payment with all details
            $stmt = $this->db->prepare("
                SELECT p.*, 
                       au.full_name as created_by_name,
                       e.student_id, e.year_id, e.grade_id, e.class_id,
                       s.full_name as student_name,
                       ay.name as year_name,
                       g.title as grade_title,
                       c.name as class_name
                FROM payments p
                LEFT JOIN admin_users au ON p.created_by = au.id
                LEFT JOIN enrollments e ON p.enrollment_id = e.id
                LEFT JOIN students s ON e.student_id = s.id
                LEFT JOIN academic_years ay ON e.year_id = ay.id
                LEFT JOIN grades g ON e.grade_id = g.id
                LEFT JOIN classes c ON e.class_id = c.id
                WHERE p.id = ?
            ");
            $stmt->execute([$paymentId]);
            $payment = $stmt->fetch();
            
            $this->authService->logAuditEvent([
                'actor_id' => $this->request->user['user_id'],
                'actor_type' => 'admin',
                'role' => $this->request->user['role'],
                'action' => 'create',
                'entity' => 'payments',
                'entity_id' => $paymentId,
                'after_data' => $payment,
                'ip_address' => $this->request->getClientIp(),
                'user_agent' => $this->request->getUserAgent()
            ]);
            
            return $this->response->success($payment, 'Payment recorded successfully');
            
        } catch (\Exception $e) {
            return $this->response->error('Failed to create payment: ' . $e->getMessage(), 500);
        }
    }
    
    /**
     * Update a payment
     */
    public function update(string $id): Response
    {
        $data = $this->validate([
            'amount' => 'required|numeric|min:0.01',
            'paid_on' => 'required|date',
            'method' => 'required|in:cash,card,bank,wallet',
            'notes' => 'string|max:1000'
        ]);
        
        // Get existing payment
        $stmt = $this->db->prepare("
            SELECT p.*, e.year_id, ay.status as year_status
            FROM payments p
            LEFT JOIN enrollments e ON p.enrollment_id = e.id
            LEFT JOIN academic_years ay ON e.year_id = ay.id
            WHERE p.id = ?
        ");
        $stmt->execute([$id]);
        $oldPayment = $stmt->fetch();
        
        if (!$oldPayment) {
            return $this->response->error('Payment not found', 404);
        }
        
        // Check if academic year is open
        if ($oldPayment['year_status'] === 'closed') {
            return $this->response->error('Cannot modify payments in a closed academic year', 422);
        }
        
        // Validate payment date
        $paymentDate = new \DateTime($data['paid_on']);
        $today = new \DateTime();
        
        if ($paymentDate > $today) {
            return $this->response->error('Payment date cannot be in the future', 422);
        }
        
        try {
            $stmt = $this->db->prepare("
                UPDATE payments 
                SET amount = ?, paid_on = ?, method = ?, notes = ?
                WHERE id = ?
            ");
            
            $stmt->execute([
                $data['amount'],
                $data['paid_on'],
                $data['method'],
                $data['notes'] ?? null,
                $id
            ]);
            
            // Get updated payment with all details
            $stmt = $this->db->prepare("
                SELECT p.*, 
                       au.full_name as created_by_name,
                       e.student_id, e.year_id, e.grade_id, e.class_id,
                       s.full_name as student_name,
                       ay.name as year_name,
                       g.title as grade_title,
                       c.name as class_name
                FROM payments p
                LEFT JOIN admin_users au ON p.created_by = au.id
                LEFT JOIN enrollments e ON p.enrollment_id = e.id
                LEFT JOIN students s ON e.student_id = s.id
                LEFT JOIN academic_years ay ON e.year_id = ay.id
                LEFT JOIN grades g ON e.grade_id = g.id
                LEFT JOIN classes c ON e.class_id = c.id
                WHERE p.id = ?
            ");
            $stmt->execute([$id]);
            $newPayment = $stmt->fetch();
            
            $this->authService->logAuditEvent([
                'actor_id' => $this->request->user['user_id'],
                'actor_type' => 'admin',
                'role' => $this->request->user['role'],
                'action' => 'update',
                'entity' => 'payments',
                'entity_id' => $id,
                'before_data' => $oldPayment,
                'after_data' => $newPayment,
                'ip_address' => $this->request->getClientIp(),
                'user_agent' => $this->request->getUserAgent()
            ]);
            
            return $this->response->success($newPayment, 'Payment updated successfully');
            
        } catch (\Exception $e) {
            return $this->response->error('Failed to update payment: ' . $e->getMessage(), 500);
        }
    }
    
    /**
     * Delete a payment
     */
    public function destroy(string $id): Response
    {
        $stmt = $this->db->prepare("
            SELECT p.*, e.year_id, ay.status as year_status,
                   s.full_name as student_name, ay.name as year_name
            FROM payments p
            LEFT JOIN enrollments e ON p.enrollment_id = e.id
            LEFT JOIN academic_years ay ON e.year_id = ay.id
            LEFT JOIN students s ON e.student_id = s.id
            WHERE p.id = ?
        ");
        $stmt->execute([$id]);
        $payment = $stmt->fetch();
        
        if (!$payment) {
            return $this->response->error('Payment not found', 404);
        }
        
        // Check if academic year is open
        if ($payment['year_status'] === 'closed') {
            return $this->response->error('Cannot delete payments from a closed academic year', 422);
        }
        
        try {
            $stmt = $this->db->prepare("DELETE FROM payments WHERE id = ?");
            $stmt->execute([$id]);
            
            $this->authService->logAuditEvent([
                'actor_id' => $this->request->user['user_id'],
                'actor_type' => 'admin',
                'role' => $this->request->user['role'],
                'action' => 'delete',
                'entity' => 'payments',
                'entity_id' => $id,
                'before_data' => $payment,
                'ip_address' => $this->request->getClientIp(),
                'user_agent' => $this->request->getUserAgent()
            ]);
            
            return $this->response->success(null, 'Payment deleted successfully');
            
        } catch (\Exception $e) {
            return $this->response->error('Failed to delete payment: ' . $e->getMessage(), 500);
        }
    }
    
    /**
     * Search students for payment management
     */
    public function searchStudents(): Response
    {
        $search = $this->request->get('q', '');
        $yearId = $this->request->get('year_id');
        $gradeId = $this->request->get('grade_id');
        $classId = $this->request->get('class_id');
        $limit = min((int)$this->request->get('limit', 50), 100);
        
        if (strlen($search) < 2 && !$yearId && !$gradeId && !$classId) {
            return $this->response->success([]);
        }
        
        $whereConditions = ['s.enabled = 1'];
        $params = [];
        
        // Search conditions
        if (strlen($search) >= 2) {
            $whereConditions[] = '(s.full_name LIKE ? OR s.id = ? OR s.national_id LIKE ?)';
            $searchTerm = "%$search%";
            $params = array_merge($params, [$searchTerm, $search, $searchTerm]);
        }
        
        // Filter conditions
        if ($yearId) {
            $whereConditions[] = 'e.year_id = ?';
            $params[] = $yearId;
        }
        
        if ($gradeId) {
            $whereConditions[] = 'e.grade_id = ?';
            $params[] = $gradeId;
        }
        
        if ($classId) {
            $whereConditions[] = 'e.class_id = ?';
            $params[] = $classId;
        }
        
        $whereClause = implode(' AND ', $whereConditions);
        
        $sql = "SELECT DISTINCT s.id, s.full_name, s.national_id,
                       COUNT(e.id) as enrollment_count
                FROM students s
                LEFT JOIN enrollments e ON s.id = e.student_id AND e.status = 'active'
                WHERE {$whereClause}
                GROUP BY s.id, s.full_name, s.national_id
                HAVING enrollment_count > 0
                ORDER BY s.full_name
                LIMIT ?";
        
        $stmt = $this->db->prepare($sql);
        $stmt->execute([...$params, $limit]);
        $students = $stmt->fetchAll();
        
        return $this->response->success($students);
    }
    
    /**
     * Get student timeline showing all years with payment info
     */
    public function studentTimeline(string $studentId): Response
    {
        // Verify student exists
        $stmt = $this->db->prepare("SELECT * FROM students WHERE id = ? AND enabled = 1");
        $stmt->execute([$studentId]);
        $student = $stmt->fetch();
        
        if (!$student) {
            return $this->response->error('Student not found', 404);
        }
        
        // Get all enrollments for this student with payment summary
        $sql = "SELECT e.id as enrollment_id, e.year_id, e.grade_id, e.class_id, e.status,
                       ay.name as year_name, ay.status as year_status,
                       g.title as grade_title, g.code as grade_code,
                       c.name as class_name,
                       gc.fee_amount as configured_fee,
                       COALESCE(SUM(p.amount), 0) as total_paid,
                       COUNT(p.id) as payment_count,
                       MAX(p.paid_on) as last_payment_date
                FROM enrollments e
                LEFT JOIN academic_years ay ON e.year_id = ay.id
                LEFT JOIN grades g ON e.grade_id = g.id
                LEFT JOIN classes c ON e.class_id = c.id
                LEFT JOIN grade_costs gc ON e.year_id = gc.year_id AND e.grade_id = gc.grade_id AND gc.enabled = 1
                LEFT JOIN payments p ON e.id = p.enrollment_id
                WHERE e.student_id = ?
                GROUP BY e.id, e.year_id, e.grade_id, e.class_id, e.status,
                         ay.name, ay.status, g.title, g.code, c.name, gc.fee_amount
                ORDER BY ay.name DESC";
        
        $stmt = $this->db->prepare($sql);
        $stmt->execute([$studentId]);
        $timeline = $stmt->fetchAll();
        
        // Calculate balance for each year
        foreach ($timeline as &$year) {
            $year['configured_fee'] = $year['configured_fee'] ? (float)$year['configured_fee'] : 0;
            $year['total_paid'] = (float)$year['total_paid'];
            $year['balance'] = $year['configured_fee'] - $year['total_paid'];
            $year['can_pay'] = $year['year_status'] === 'open' && $year['status'] === 'active';
        }
        
        return $this->response->success([
            'student' => $student,
            'timeline' => $timeline
        ]);
    }
    
    /**
     * Search payment history across all students
     */
    public function searchPaymentHistory(): Response
    {
        $pagination = $this->getPaginationParams();
        $sort = $this->getSortParams(['paid_on', 'amount', 'student_name', 'created_at']);
        
        $whereConditions = ['1 = 1'];
        $params = [];
        
        // Search by student name or ID
        if ($search = $this->request->get('q')) {
            $whereConditions[] = '(s.full_name LIKE ? OR s.id = ? OR s.national_id LIKE ?)';
            $searchTerm = "%$search%";
            $params = array_merge($params, [$searchTerm, $search, $searchTerm]);
        }
        
        // Filter by year
        if ($yearId = $this->request->get('year_id')) {
            $whereConditions[] = 'e.year_id = ?';
            $params[] = $yearId;
        }
        
        // Filter by grade
        if ($gradeId = $this->request->get('grade_id')) {
            $whereConditions[] = 'e.grade_id = ?';
            $params[] = $gradeId;
        }
        
        // Filter by class
        if ($classId = $this->request->get('class_id')) {
            $whereConditions[] = 'e.class_id = ?';
            $params[] = $classId;
        }
        
        // Filter by payment method
        if ($method = $this->request->get('method')) {
            $whereConditions[] = 'p.method = ?';
            $params[] = $method;
        }
        
        // Filter by date range
        if ($startDate = $this->request->get('start_date')) {
            $whereConditions[] = 'p.paid_on >= ?';
            $params[] = $startDate;
        }
        
        if ($endDate = $this->request->get('end_date')) {
            $whereConditions[] = 'p.paid_on <= ?';
            $params[] = $endDate;
        }
        
        // Filter by amount range
        if ($minAmount = $this->request->get('min_amount')) {
            $whereConditions[] = 'p.amount >= ?';
            $params[] = $minAmount;
        }
        
        if ($maxAmount = $this->request->get('max_amount')) {
            $whereConditions[] = 'p.amount <= ?';
            $params[] = $maxAmount;
        }
        
        $whereClause = implode(' AND ', $whereConditions);
        $orderClause = $sort['sortBy'] ? "ORDER BY {$sort['sortBy']} {$sort['sortDir']}" : 'ORDER BY p.paid_on DESC';
        
        // Get total count
        $countStmt = $this->db->prepare("
            SELECT COUNT(*)
            FROM payments p
            LEFT JOIN enrollments e ON p.enrollment_id = e.id
            LEFT JOIN students s ON e.student_id = s.id
            WHERE {$whereClause}
        ");
        $countStmt->execute($params);
        $totalRows = $countStmt->fetchColumn();
        
        // Get payment history with student and enrollment details
        $sql = "SELECT p.*, 
                       s.id as student_id, s.full_name as student_name, s.national_id,
                       e.year_id, e.grade_id, e.class_id,
                       ay.name as year_name,
                       g.title as grade_title,
                       c.name as class_name,
                       au.full_name as created_by_name,
                       p.paid_on,
                       DATEDIFF(CURRENT_DATE, p.paid_on) as days_ago
                FROM payments p
                LEFT JOIN enrollments e ON p.enrollment_id = e.id
                LEFT JOIN students s ON e.student_id = s.id
                LEFT JOIN academic_years ay ON e.year_id = ay.id
                LEFT JOIN grades g ON e.grade_id = g.id
                LEFT JOIN classes c ON e.class_id = c.id
                LEFT JOIN admin_users au ON p.created_by = au.id
                WHERE {$whereClause}
                {$orderClause}
                LIMIT ? OFFSET ?";
        
        $stmt = $this->db->prepare($sql);
        $stmt->execute([...$params, $pagination['perPage'], $pagination['offset']]);
        $payments = $stmt->fetchAll();
        
        // Add last payment indicator
        foreach ($payments as &$payment) {
            $payment['amount'] = (float)$payment['amount'];
            $payment['days_ago'] = (int)$payment['days_ago'];
            
            // Check if this is the student's most recent payment
            $lastPaymentStmt = $this->db->prepare("
                SELECT MAX(paid_on) as last_payment_date
                FROM payments p2
                LEFT JOIN enrollments e2 ON p2.enrollment_id = e2.id
                WHERE e2.student_id = ?
            ");
            $lastPaymentStmt->execute([$payment['student_id']]);
            $lastPayment = $lastPaymentStmt->fetch();
            
            $payment['is_last_payment'] = ($payment['paid_on'] === $lastPayment['last_payment_date']);
        }
        
        $meta = [
            'page' => $pagination['page'],
            'per_page' => $pagination['perPage'],
            'total_rows' => $totalRows,
            'total_pages' => ceil($totalRows / $pagination['perPage']),
            'sort_by' => $sort['sortBy'],
            'sort_dir' => $sort['sortDir']
        ];
        
        return $this->response->paginated($payments, $meta);
    }
}