<?php

namespace App\Controllers;

use App\Core\Controller;
use App\Core\Database;
use App\Core\Response;
use App\Services\AuthService;
use PDO;

class ResultController extends Controller
{
    private PDO $db;
    private AuthService $authService;
    
    public function __construct($request)
    {
        parent::__construct($request);
        $this->db = Database::getConnection();
        $this->authService = new AuthService();
    }
    
    public function index(): Response
    {
        $pagination = $this->getPaginationParams();
        $sort = $this->getSortParams(['score', 'percentage', 'created_at', 'student_name', 'exam_title']);
        
        $whereConditions = ['1=1'];
        $params = [];
        
        if ($examId = $this->request->get('exam_id')) {
            $whereConditions[] = 'er.exam_id = ?';
            $params[] = $examId;
        }
        
        if ($studentId = $this->request->get('student_id')) {
            $whereConditions[] = 'er.student_id = ?';
            $params[] = $studentId;
        }
        
        if ($yearId = $this->request->get('year_id')) {
            $whereConditions[] = 'e.year_id = ?';
            $params[] = $yearId;
        }
        
        if ($termId = $this->request->get('term_id')) {
            $whereConditions[] = 'e.term_id = ?';
            $params[] = $termId;
        }
        
        if ($gradeId = $this->request->get('grade_id')) {
            $whereConditions[] = 'e.grade_id = ?';
            $params[] = $gradeId;
        }
        
        if ($subjectId = $this->request->get('subject_id')) {
            $whereConditions[] = 'e.subject_id = ?';
            $params[] = $subjectId;
        }
        
        if ($search = $this->request->get('q')) {
            $whereConditions[] = '(s.full_name LIKE ? OR s.id LIKE ? OR e.title LIKE ?)';
            $searchTerm = "%$search%";
            $params = array_merge($params, [$searchTerm, $searchTerm, $searchTerm]);
        }
        
        $whereClause = implode(' AND ', $whereConditions);
        $orderClause = $sort['sortBy'] ? "ORDER BY {$sort['sortBy']} {$sort['sortDir']}" : 'ORDER BY er.created_at DESC';
        
        $countStmt = $this->db->prepare("
            SELECT COUNT(*) 
            FROM exam_results er
            LEFT JOIN exams e ON er.exam_id = e.id
            LEFT JOIN students s ON er.student_id = s.id
            WHERE {$whereClause}
        ");
        $countStmt->execute($params);
        $totalRows = $countStmt->fetchColumn();
        
        $sql = "SELECT er.id, er.exam_id, er.student_id, er.score, er.percentage,
                       er.created_at, er.updated_at,
                       e.title as exam_title, e.exam_date, e.min_score, e.max_score,
                       s.id as student_id, s.full_name as student_name, s.login_code,
                       ay.name as year_name,
                       t.name as term_name,
                       g.title as grade_title, g.code as grade_code,
                       sub.name as subject_name,
                       -- Calculate band label and color based on percentage
                       pb.label as band_label,
                       pb.color_hex as band_color
                FROM exam_results er
                LEFT JOIN exams e ON er.exam_id = e.id
                LEFT JOIN students s ON er.student_id = s.id
                LEFT JOIN academic_years ay ON e.year_id = ay.id
                LEFT JOIN terms t ON e.term_id = t.id
                LEFT JOIN grades g ON e.grade_id = g.id
                LEFT JOIN subjects sub ON e.subject_id = sub.id
                LEFT JOIN percentage_bands pb ON (er.percentage >= pb.start_percent AND er.percentage <= pb.end_percent AND pb.enabled = 1)
                WHERE {$whereClause}
                {$orderClause}
                LIMIT ? OFFSET ?";
        
        $stmt = $this->db->prepare($sql);
        $stmt->execute([...$params, $pagination['perPage'], $pagination['offset']]);
        $results = $stmt->fetchAll();
        
        $meta = [
            'page' => $pagination['page'],
            'per_page' => $pagination['perPage'],
            'total_rows' => $totalRows,
            'total_pages' => ceil($totalRows / $pagination['perPage']),
            'sort_by' => $sort['sortBy'],
            'sort_dir' => $sort['sortDir']
        ];
        
        return $this->response->paginated($results, $meta);
    }
    
    public function store(): Response
    {
        $data = $this->validate([
            'exam_id' => 'required|numeric',
            'student_id' => 'required|numeric',
            'score' => 'required|numeric|min:0'
        ]);
        
        // Get exam details for validation
        $stmt = $this->db->prepare("SELECT * FROM exams WHERE id = ?");
        $stmt->execute([$data['exam_id']]);
        $exam = $stmt->fetch();
        
        if (!$exam) {
            return $this->response->error('Exam not found', 422);
        }
        
        if (!$exam['enabled']) {
            return $this->response->error('Exam is disabled', 422);
        }
        
        // Validate score range (0 to max_score, min_score is just for pass/fail logic)
        if ($data['score'] < 0 || $data['score'] > $exam['max_score']) {
            return $this->response->error("Score must be between 0 and {$exam['max_score']}", 422);
        }
        
        // Verify student exists and is enrolled in this exam's year/grade
        $stmt = $this->db->prepare("
            SELECT s.*, en.id as enrollment_id
            FROM students s
            LEFT JOIN enrollments en ON s.id = en.student_id 
                AND en.year_id = ? AND en.grade_id = ? AND en.status = 'active'
            WHERE s.id = ? AND s.enabled = 1
        ");
        $stmt->execute([$exam['year_id'], $exam['grade_id'], $data['student_id']]);
        $student = $stmt->fetch();
        
        if (!$student || !$student['enrollment_id']) {
            return $this->response->error('Student not found or not enrolled in the appropriate grade/year for this exam', 422);
        }
        
        // Check for existing result
        $stmt = $this->db->prepare("SELECT id FROM exam_results WHERE exam_id = ? AND student_id = ?");
        $stmt->execute([$data['exam_id'], $data['student_id']]);
        if ($stmt->fetch()) {
            return $this->response->error('Result already exists for this exam and student', 422);
        }
        
        // Calculate percentage and band
        $percentage = ($data['score'] / $exam['max_score']) * 100;
        
        try {
            $stmt = $this->db->prepare("
                INSERT INTO exam_results (exam_id, student_id, score, percentage) 
                VALUES (?, ?, ?, ?)
            ");
            
            $stmt->execute([
                $data['exam_id'],
                $data['student_id'],
                $data['score'],
                round($percentage, 2)
            ]);
            
            $resultId = $this->db->lastInsertId();
            
            // Get the created result with all joins
            $stmt = $this->db->prepare("
                SELECT er.id, er.exam_id, er.student_id, er.score, er.percentage,
                       er.created_at, er.updated_at,
                       e.title as exam_title, e.exam_date, e.min_score, e.max_score,
                       s.full_name as student_name, s.login_code,
                       ay.name as year_name,
                       t.name as term_name,
                       g.title as grade_title, g.code as grade_code,
                       sub.name as subject_name,
                       pb.label as band_label,
                       pb.color_hex as band_color
                FROM exam_results er
                LEFT JOIN exams e ON er.exam_id = e.id
                LEFT JOIN students s ON er.student_id = s.id
                LEFT JOIN academic_years ay ON e.year_id = ay.id
                LEFT JOIN terms t ON e.term_id = t.id
                LEFT JOIN grades g ON e.grade_id = g.id
                LEFT JOIN subjects sub ON e.subject_id = sub.id
                LEFT JOIN percentage_bands pb ON (er.percentage >= pb.start_percent AND er.percentage <= pb.end_percent AND pb.enabled = 1)
                WHERE er.id = ?
            ");
            $stmt->execute([$resultId]);
            $result = $stmt->fetch();
            
            $this->authService->logAuditEvent([
                'actor_id' => $this->request->user['user_id'],
                'actor_type' => 'admin',
                'role' => $this->request->user['role'],
                'action' => 'create',
                'entity' => 'exam_results',
                'entity_id' => $resultId,
                'after_data' => $result,
                'ip_address' => $this->request->getClientIp(),
                'user_agent' => $this->request->getUserAgent()
            ]);
            
            return $this->response->success($result, 'Result created successfully');
            
        } catch (\Exception $e) {
            return $this->response->error('Failed to create result: ' . $e->getMessage(), 500);
        }
    }
    
    public function show(string $id): Response
    {
        $stmt = $this->db->prepare("
            SELECT er.id, er.exam_id, er.student_id, er.score, er.percentage,
                   er.created_at, er.updated_at,
                   e.title as exam_title, e.exam_date, e.min_score, e.max_score,
                   s.full_name as student_name, s.login_code, s.national_id,
                   ay.name as year_name,
                   t.name as term_name,
                   g.title as grade_title, g.code as grade_code,
                   sub.name as subject_name,
                   pb.label as band_label,
                   pb.color_hex as band_color
            FROM exam_results er
            LEFT JOIN exams e ON er.exam_id = e.id
            LEFT JOIN students s ON er.student_id = s.id
            LEFT JOIN academic_years ay ON e.year_id = ay.id
            LEFT JOIN terms t ON e.term_id = t.id
            LEFT JOIN grades g ON e.grade_id = g.id
            LEFT JOIN subjects sub ON e.subject_id = sub.id
            LEFT JOIN percentage_bands pb ON (er.percentage >= pb.start_percent AND er.percentage <= pb.end_percent AND pb.enabled = 1)
            WHERE er.id = ?
        ");
        
        $stmt->execute([$id]);
        $result = $stmt->fetch();
        
        if (!$result) {
            return $this->response->error('Result not found', 404);
        }
        
        return $this->response->success($result);
    }
    
    public function update(string $id): Response
    {
        $data = $this->validate([
            'score' => 'required|numeric|min:0'
        ]);
        
        $stmt = $this->db->prepare("SELECT * FROM exam_results WHERE id = ?");
        $stmt->execute([$id]);
        $oldResult = $stmt->fetch();
        
        if (!$oldResult) {
            return $this->response->error('Result not found', 404);
        }
        
        // Get exam details for validation
        $stmt = $this->db->prepare("SELECT * FROM exams WHERE id = ?");
        $stmt->execute([$oldResult['exam_id']]);
        $exam = $stmt->fetch();
        
        // Validate score range (0 to max_score, min_score is just for pass/fail logic)
        if ($data['score'] < 0 || $data['score'] > $exam['max_score']) {
            return $this->response->error("Score must be between 0 and {$exam['max_score']}", 422);
        }
        
        // Calculate new percentage
        $percentage = ($data['score'] / $exam['max_score']) * 100;
        
        try {
            $stmt = $this->db->prepare("
                UPDATE exam_results 
                SET score = ?, percentage = ?
                WHERE id = ?
            ");
            
            $stmt->execute([
                $data['score'],
                round($percentage, 2),
                $id
            ]);
            
            // Get updated result with all joins
            $stmt = $this->db->prepare("
                SELECT er.id, er.exam_id, er.student_id, er.score, er.percentage,
                       er.created_at, er.updated_at,
                       e.title as exam_title, e.exam_date, e.min_score, e.max_score,
                       s.full_name as student_name, s.login_code,
                       ay.name as year_name,
                       t.name as term_name,
                       g.title as grade_title, g.code as grade_code,
                       sub.name as subject_name,
                       pb.label as band_label,
                       pb.color_hex as band_color
                FROM exam_results er
                LEFT JOIN exams e ON er.exam_id = e.id
                LEFT JOIN students s ON er.student_id = s.id
                LEFT JOIN academic_years ay ON e.year_id = ay.id
                LEFT JOIN terms t ON e.term_id = t.id
                LEFT JOIN grades g ON e.grade_id = g.id
                LEFT JOIN subjects sub ON e.subject_id = sub.id
                LEFT JOIN percentage_bands pb ON (er.percentage >= pb.start_percent AND er.percentage <= pb.end_percent AND pb.enabled = 1)
                WHERE er.id = ?
            ");
            $stmt->execute([$id]);
            $newResult = $stmt->fetch();
            
            $this->authService->logAuditEvent([
                'actor_id' => $this->request->user['user_id'],
                'actor_type' => 'admin',
                'role' => $this->request->user['role'],
                'action' => 'update',
                'entity' => 'exam_results',
                'entity_id' => $id,
                'before_data' => $oldResult,
                'after_data' => $newResult,
                'ip_address' => $this->request->getClientIp(),
                'user_agent' => $this->request->getUserAgent()
            ]);
            
            return $this->response->success($newResult, 'Result updated successfully');
            
        } catch (\Exception $e) {
            return $this->response->error('Failed to update result: ' . $e->getMessage(), 500);
        }
    }
    
    public function destroy(string $id): Response
    {
        $stmt = $this->db->prepare("
            SELECT er.*, s.full_name as student_name, e.title as exam_title
            FROM exam_results er
            LEFT JOIN students s ON er.student_id = s.id
            LEFT JOIN exams e ON er.exam_id = e.id
            WHERE er.id = ?
        ");
        $stmt->execute([$id]);
        $result = $stmt->fetch();
        
        if (!$result) {
            return $this->response->error('Result not found', 404);
        }
        
        try {
            $stmt = $this->db->prepare("DELETE FROM exam_results WHERE id = ?");
            $stmt->execute([$id]);
            
            $this->authService->logAuditEvent([
                'actor_id' => $this->request->user['user_id'],
                'actor_type' => 'admin',
                'role' => $this->request->user['role'],
                'action' => 'delete',
                'entity' => 'exam_results',
                'entity_id' => $id,
                'before_data' => $result,
                'ip_address' => $this->request->getClientIp(),
                'user_agent' => $this->request->getUserAgent()
            ]);
            
            return $this->response->success(null, 'Result deleted successfully');
            
        } catch (\Exception $e) {
            return $this->response->error('Failed to delete result: ' . $e->getMessage(), 500);
        }
    }
    
    public function bulkDelete(): Response
    {
        $data = $this->request->all();
        
        if (!isset($data['result_ids']) || !is_array($data['result_ids']) || empty($data['result_ids'])) {
            return $this->response->error('No result IDs provided', 400);
        }
        
        $resultIds = array_map('intval', $data['result_ids']);
        $resultIds = array_filter($resultIds);
        
        if (empty($resultIds)) {
            return $this->response->error('Invalid result IDs provided', 400);
        }
        
        try {
            $this->db->beginTransaction();
            
            $placeholders = str_repeat('?,', count($resultIds) - 1) . '?';
            
            $stmt = $this->db->prepare("
                SELECT er.*, s.full_name as student_name, e.title as exam_title
                FROM exam_results er
                LEFT JOIN students s ON er.student_id = s.id
                LEFT JOIN exams e ON er.exam_id = e.id
                WHERE er.id IN ($placeholders)
            ");
            $stmt->execute($resultIds);
            $results = $stmt->fetchAll();
            
            $stmt = $this->db->prepare("DELETE FROM exam_results WHERE id IN ($placeholders)");
            $stmt->execute($resultIds);
            
            foreach ($results as $result) {
                $this->authService->logAuditEvent([
                    'actor_id' => $this->request->user['user_id'],
                    'actor_type' => 'admin',
                    'role' => $this->request->user['role'],
                    'action' => 'bulk_delete',
                    'entity' => 'exam_results',
                    'entity_id' => $result['id'],
                    'before_data' => $result,
                    'ip_address' => $this->request->getClientIp(),
                    'user_agent' => $this->request->getUserAgent()
                ]);
            }
            
            $this->db->commit();
            
            return $this->response->success([
                'deleted_count' => count($results),
                'deleted_results' => $results
            ], count($results) . ' result(s) deleted successfully');
            
        } catch (\Exception $e) {
            $this->db->rollBack();
            return $this->response->error('Failed to delete results: ' . $e->getMessage(), 500);
        }
    }
    
    public function import(): Response
    {
        // Debug: Log the request details
        error_log('Import request received');
        error_log('POST data: ' . print_r($_POST, true));
        error_log('FILES data: ' . print_r($_FILES, true));
        error_log('Request method: ' . $_SERVER['REQUEST_METHOD']);
        error_log('Content type: ' . ($_SERVER['CONTENT_TYPE'] ?? 'not set'));
        
        if (!isset($_FILES['file']) || $_FILES['file']['error'] !== UPLOAD_ERR_OK) {
            error_log('File upload error. FILES: ' . print_r($_FILES, true));
            return $this->response->error('No file uploaded or upload error', 400);
        }
        
        $file = $_FILES['file'];
        $allowedTypes = ['text/csv', 'application/csv', 'text/plain', 'application/vnd.ms-excel', 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet'];
        $maxSize = 10 * 1024 * 1024; // 10MB
        
        if ($file['size'] > $maxSize) {
            return $this->response->error('File size exceeds 10MB limit', 400);
        }
        
        $fileExt = strtolower(pathinfo($file['name'], PATHINFO_EXTENSION));
        if (!in_array($fileExt, ['csv', 'xlsx'])) {
            return $this->response->error('Only CSV and Excel files are allowed', 400);
        }
        
        try {
            $rows = [];
            
            if ($fileExt === 'csv') {
                $handle = fopen($file['tmp_name'], 'r');
                if (!$handle) {
                    return $this->response->error('Failed to read uploaded file', 500);
                }
                
                $headers = fgetcsv($handle);
                if (!$headers) {
                    fclose($handle);
                    return $this->response->error('Invalid CSV file - no headers found', 400);
                }
                
                while (($row = fgetcsv($handle)) !== false) {
                    if (!empty(array_filter($row))) {
                        // Ensure row has same number of elements as headers
                        $headerCount = count($headers);
                        $rowCount = count($row);
                        
                        if ($rowCount < $headerCount) {
                            // Pad row with empty strings
                            $row = array_pad($row, $headerCount, '');
                        } elseif ($rowCount > $headerCount) {
                            // Truncate row to match headers
                            $row = array_slice($row, 0, $headerCount);
                        }
                        
                        $rows[] = array_combine($headers, $row);
                    }
                }
                fclose($handle);
            }
            
            // Validate headers - support both old and new format
            $requiredHeaders = ['score'];
            $optionalHeaders = ['student_name', 'student_id', 'studentId', 'national_id', 'nationalId', 'login_code'];
            $allValidHeaders = array_merge($requiredHeaders, $optionalHeaders);
            
            if (empty($rows)) {
                return $this->response->error('No valid data rows found in file', 400);
            }
            
            $headers = array_keys($rows[0]);
            $invalidHeaders = array_diff($headers, $allValidHeaders);
            if (!empty($invalidHeaders)) {
                return $this->response->error('Invalid headers: ' . implode(', ', $invalidHeaders) . '. Valid headers: ' . implode(', ', $allValidHeaders), 400);
            }
            
            $missingRequired = array_diff($requiredHeaders, $headers);
            if (!empty($missingRequired)) {
                return $this->response->error('Missing required headers: ' . implode(', ', $missingRequired), 400);
            }
            
            // Must have at least one way to identify students
            $studentIdFields = ['student_id', 'studentId', 'login_code'];
            $hasStudentId = !empty(array_intersect($studentIdFields, $headers));
            if (!$hasStudentId) {
                return $this->response->error('Must include at least one student identifier: student_id, studentId, or login_code', 400);
            }
            
            // Get exam ID from request
            $examId = $this->request->get('exam_id');
            if (!$examId) {
                return $this->response->error('Exam ID is required', 400);
            }
            
            // Verify exam exists
            $stmt = $this->db->prepare("SELECT * FROM exams WHERE id = ?");
            $stmt->execute([$examId]);
            $exam = $stmt->fetch();
            
            if (!$exam) {
                return $this->response->error('Exam not found', 422);
            }
            
            $results = [
                'total_rows' => count($rows),
                'successful' => 0,
                'failed' => 0,
                'errors' => [],
                'imported_results' => []
            ];
            
            $rowNumber = 1;
            
            foreach ($rows as $rowData) {
                $rowNumber++;
                
                try {
                    $this->db->beginTransaction();
                    
                    if (!is_numeric(trim($rowData['score']))) {
                        throw new \Exception('Score is required and must be numeric');
                    }
                    
                    $score = (float)trim($rowData['score']);
                    
                    // Determine student identification method
                    $studentId = null;
                    $studentQuery = "SELECT * FROM students WHERE ";
                    $studentParams = [];
                    
                    if (!empty($rowData['student_id'])) {
                        $studentId = (int)trim($rowData['student_id']);
                        $studentQuery .= "id = ?";
                        $studentParams[] = $studentId;
                    } elseif (!empty($rowData['studentId'])) {
                        $studentId = (int)trim($rowData['studentId']);
                        $studentQuery .= "id = ?";
                        $studentParams[] = $studentId;
                    } elseif (!empty($rowData['login_code'])) {
                        $studentQuery .= "login_code = ?";
                        $studentParams[] = trim($rowData['login_code']);
                    } else {
                        throw new \Exception('Student identifier (student_id, studentId, or login_code) is required');
                    }
                    
                    // Add national_id validation if provided
                    if (!empty($rowData['national_id']) || !empty($rowData['nationalId'])) {
                        $nationalId = trim($rowData['national_id'] ?? $rowData['nationalId']);
                        $studentQuery .= " AND national_id = ?";
                        $studentParams[] = $nationalId;
                    }
                    
                    $stmt = $this->db->prepare($studentQuery);
                    $stmt->execute($studentParams);
                    $student = $stmt->fetch();
                    
                    if (!$student) {
                        throw new \Exception('Student not found with provided identifier');
                    }
                    
                    // Validate student is enrolled in this exam's year/grade
                    $stmt = $this->db->prepare("
                        SELECT en.id
                        FROM enrollments en
                        WHERE en.student_id = ? AND en.year_id = ? AND en.grade_id = ? AND en.status = 'active'
                    ");
                    $stmt->execute([$student['id'], $exam['year_id'], $exam['grade_id']]);
                    if (!$stmt->fetch()) {
                        throw new \Exception('Student not enrolled in appropriate grade/year for this exam');
                    }
                    
                    // Validate score range (0 to max_score, min_score is just for pass/fail logic)
                    if ($score < 0 || $score > $exam['max_score']) {
                        throw new \Exception("Score must be between 0 and {$exam['max_score']}");
                    }
                    
                    // Check for existing result
                    $stmt = $this->db->prepare("SELECT id FROM exam_results WHERE exam_id = ? AND student_id = ?");
                    $stmt->execute([$examId, $student['id']]);
                    if ($stmt->fetch()) {
                        throw new \Exception('Result already exists for this exam and student');
                    }
                    
                    // Calculate percentage
                    $percentage = ($score / $exam['max_score']) * 100;
                    
                    // Insert result
                    $stmt = $this->db->prepare("
                        INSERT INTO exam_results (exam_id, student_id, score, percentage) 
                        VALUES (?, ?, ?, ?)
                    ");
                    
                    $stmt->execute([
                        $examId,
                        $student['id'],
                        $score,
                        round($percentage, 2)
                    ]);
                    
                    $resultId = $this->db->lastInsertId();
                    
                    $this->db->commit();
                    
                    $results['successful']++;
                    $results['imported_results'][] = [
                        'result_id' => $resultId,
                        'student_id' => $student['id'],
                        'student_name' => $student['full_name'],
                        'score' => $score,
                        'percentage' => round($percentage, 2),
                        'row' => $rowNumber
                    ];
                    
                    $this->authService->logAuditEvent([
                        'actor_id' => $this->request->user['user_id'],
                        'actor_type' => 'admin',
                        'role' => $this->request->user['role'],
                        'action' => 'import',
                        'entity' => 'exam_results',
                        'entity_id' => $resultId,
                        'after_data' => ['exam_id' => $examId, 'student_id' => $student['id'], 'score' => $score, 'percentage' => round($percentage, 2)],
                        'ip_address' => $this->request->getClientIp(),
                        'user_agent' => $this->request->getUserAgent()
                    ]);
                    
                } catch (\Exception $e) {
                    $this->db->rollBack();
                    $results['failed']++;
                    $results['errors'][] = [
                        'row' => $rowNumber,
                        'data' => $rowData,
                        'error' => $e->getMessage()
                    ];
                }
            }
            
            $message = sprintf(
                'Results import completed: %d total, %d successful, %d failed',
                $results['total_rows'],
                $results['successful'],
                $results['failed']
            );
            
            return $this->response->success($results, $message);
            
        } catch (\Exception $e) {
            return $this->response->error('Import failed: ' . $e->getMessage(), 500);
        }
    }
    
    public function getByExam(string $examId): Response
    {
        // Get exam results for a specific exam with all student details
        $stmt = $this->db->prepare("SELECT * FROM exams WHERE id = ?");
        $stmt->execute([$examId]);
        $exam = $stmt->fetch();
        
        if (!$exam) {
            return $this->response->error('Exam not found', 404);
        }
        
        // Add class filter if provided
        $whereConditions = ['en.year_id = ?', 'en.grade_id = ?', 'en.status = ?', 's.enabled = 1'];
        $whereParams = [$exam['year_id'], $exam['grade_id'], 'active'];
        
        if ($classId = $this->request->get('class_id')) {
            $whereConditions[] = 'en.class_id = ?';
            $whereParams[] = $classId;
        }
        
        $whereClause = implode(' AND ', $whereConditions);
        
        $sql = "SELECT s.id as student_id, s.full_name as student_name, s.login_code,
                       er.id as result_id, er.score, er.percentage,
                       pb.label as band_label, pb.color_hex as band_color,
                       c.name as class_name,
                       CASE WHEN er.id IS NOT NULL THEN 'completed' ELSE 'pending' END as status
                FROM enrollments en
                LEFT JOIN students s ON en.student_id = s.id
                LEFT JOIN classes c ON en.class_id = c.id
                LEFT JOIN exam_results er ON er.student_id = s.id AND er.exam_id = ?
                LEFT JOIN percentage_bands pb ON (er.percentage >= pb.start_percent AND er.percentage <= pb.end_percent AND pb.enabled = 1)
                WHERE {$whereClause}
                ORDER BY c.name, s.full_name";
        
        $stmt = $this->db->prepare($sql);
        $stmt->execute(array_merge([$examId], $whereParams));
        $studentResults = $stmt->fetchAll();
        
        return $this->response->success([
            'exam' => $exam,
            'student_results' => $studentResults,
            'summary' => [
                'total_students' => count($studentResults),
                'completed' => count(array_filter($studentResults, fn($r) => $r['status'] === 'completed')),
                'pending' => count(array_filter($studentResults, fn($r) => $r['status'] === 'pending'))
            ]
        ]);
    }
    
    public function downloadTemplate(string $examId)
    {
        // Get exam details with related info
        $stmt = $this->db->prepare("
            SELECT e.*, ay.name as year_name, t.name as term_name, g.title as grade_title
            FROM exams e
            LEFT JOIN academic_years ay ON e.year_id = ay.id
            LEFT JOIN terms t ON e.term_id = t.id
            LEFT JOIN grades g ON e.grade_id = g.id
            WHERE e.id = ?
        ");
        $stmt->execute([$examId]);
        $exam = $stmt->fetch();
        
        if (!$exam) {
            // Return JSON error for AJAX request
            http_response_code(404);
            header('Content-Type: application/json');
            echo json_encode(['error' => 'Exam not found']);
            exit;
        }
        
        // Get all enrolled students for this exam's grade/year with class info
        $sql = "SELECT s.id as student_id, s.full_name as student_name, s.login_code, 
                       s.national_id, c.name as class_name,
                       er.score as current_score
                FROM enrollments en
                LEFT JOIN students s ON en.student_id = s.id
                LEFT JOIN classes c ON en.class_id = c.id
                LEFT JOIN exam_results er ON er.student_id = s.id AND er.exam_id = ?
                WHERE en.year_id = ? AND en.grade_id = ? AND en.status = 'active' 
                  AND s.enabled = 1 AND s.id IS NOT NULL
                ORDER BY c.name, s.full_name";
        
        $stmt = $this->db->prepare($sql);
        $stmt->execute([$examId, $exam['year_id'], $exam['grade_id']]);
        $students = $stmt->fetchAll();
        
        if (empty($students)) {
            // Return JSON error for AJAX request
            http_response_code(404);
            header('Content-Type: application/json');
            echo json_encode(['error' => 'No enrolled students found for this exam']);
            exit;
        }
        
        // Generate CSV content
        $csvContent = "student_name,student_id,national_id,login_code,score\n";
        
        foreach ($students as $student) {
            $csvContent .= sprintf(
                "%s,%s,%s,%s,%s\n",
                '"' . str_replace('"', '""', $student['student_name'] ?? 'Unknown') . '"',
                $student['student_id'] ?? '',
                $student['national_id'] ?? '',
                $student['login_code'] ?? '',
                $student['current_score'] ?? ''
            );
        }
        
        // Set headers for CSV download - clean output buffer first
        if (ob_get_level()) {
            ob_end_clean();
        }
        
        $filename = sprintf(
            'results_template_%s_%s_%s_%s.csv',
            preg_replace('/[^a-zA-Z0-9_-]/', '_', $exam['title']),
            preg_replace('/[^a-zA-Z0-9_-]/', '_', $exam['year_name'] ?? 'unknown'),
            preg_replace('/[^a-zA-Z0-9_-]/', '_', $exam['term_name'] ?? 'unknown'),
            date('Y-m-d')
        );
        
        header('Content-Type: text/csv; charset=utf-8');
        header('Content-Disposition: attachment; filename="' . $filename . '"');
        header('Content-Length: ' . strlen($csvContent));
        header('Cache-Control: no-cache, must-revalidate');
        header('Pragma: no-cache');
        
        echo $csvContent;
        exit;
    }
    
    public function debugTemplate(string $examId): Response
    {
        // Get exam details with related info
        $stmt = $this->db->prepare("
            SELECT e.*, ay.name as year_name, t.name as term_name, g.title as grade_title
            FROM exams e
            LEFT JOIN academic_years ay ON e.year_id = ay.id
            LEFT JOIN terms t ON e.term_id = t.id
            LEFT JOIN grades g ON e.grade_id = g.id
            WHERE e.id = ?
        ");
        $stmt->execute([$examId]);
        $exam = $stmt->fetch();
        
        if (!$exam) {
            return $this->response->error('Exam not found', 404);
        }
        
        // Get all enrolled students for this exam's grade/year with class info
        $sql = "SELECT s.id as student_id, s.full_name as student_name, s.login_code, 
                       s.national_id, c.name as class_name,
                       er.score as current_score
                FROM enrollments en
                LEFT JOIN students s ON en.student_id = s.id
                LEFT JOIN classes c ON en.class_id = c.id
                LEFT JOIN exam_results er ON er.student_id = s.id AND er.exam_id = ?
                WHERE en.year_id = ? AND en.grade_id = ? AND en.status = 'active' 
                  AND s.enabled = 1 AND s.id IS NOT NULL
                ORDER BY c.name, s.full_name";
        
        $stmt = $this->db->prepare($sql);
        $stmt->execute([$examId, $exam['year_id'], $exam['grade_id']]);
        $students = $stmt->fetchAll();
        
        return $this->response->success([
            'exam' => $exam,
            'students' => $students,
            'student_count' => count($students),
            'sql' => $sql,
            'params' => [$examId, $exam['year_id'], $exam['grade_id']]
        ]);
    }
    
    public function bulkStore(): Response
    {
        $data = $this->validate([
            'results' => 'required|array'
        ]);
        
        // exam_id is optional - if provided, all results are for that exam
        // if not provided, each result must have its own exam_id
        $globalExamId = $data['exam_id'] ?? null;
        
        if (empty($data['results'])) {
            return $this->response->error('No results provided', 400);
        }
        
        try {
            $this->db->beginTransaction();
            
            $results = [
                'total_results' => count($data['results']),
                'successful' => 0,
                'failed' => 0,
                'errors' => [],
                'created_results' => []
            ];
            
            foreach ($data['results'] as $index => $resultData) {
                try {
                    // Determine exam ID - either global or individual
                    $examId = $globalExamId ?? ($resultData['exam_id'] ?? null);
                    
                    if (!$examId || !isset($resultData['student_id']) || !isset($resultData['score'])) {
                        throw new \Exception('Exam ID, Student ID and score are required');
                    }
                    
                    $studentId = (int)$resultData['student_id'];
                    $score = (float)$resultData['score'];
                    
                    // Get exam details for this specific exam
                    $stmt = $this->db->prepare("SELECT * FROM exams WHERE id = ?");
                    $stmt->execute([$examId]);
                    $exam = $stmt->fetch();
                    
                    if (!$exam) {
                        throw new \Exception("Exam with ID {$examId} not found");
                    }
                    
                    // Validate score range (0 to max_score, min_score is just for pass/fail logic)
                    if ($score < 0 || $score > $exam['max_score']) {
                        throw new \Exception("Score must be between 0 and {$exam['max_score']} for exam '{$exam['title']}'");
                    }
                    
                    // Check for existing result
                    $stmt = $this->db->prepare("SELECT id FROM exam_results WHERE exam_id = ? AND student_id = ?");
                    $stmt->execute([$examId, $studentId]);
                    if ($stmt->fetch()) {
                        throw new \Exception("Result already exists for this student and exam '{$exam['title']}'");
                    }
                    
                    // Calculate percentage
                    $percentage = ($score / $exam['max_score']) * 100;
                    
                    // Insert result
                    $stmt = $this->db->prepare("
                        INSERT INTO exam_results (exam_id, student_id, score, percentage) 
                        VALUES (?, ?, ?, ?)
                    ");
                    
                    $stmt->execute([
                        $examId,
                        $studentId,
                        $score,
                        round($percentage, 2)
                    ]);
                    
                    $resultId = $this->db->lastInsertId();
                    
                    $results['successful']++;
                    $results['created_results'][] = [
                        'result_id' => $resultId,
                        'exam_id' => $examId,
                        'student_id' => $studentId,
                        'score' => $score,
                        'percentage' => round($percentage, 2)
                    ];
                    
                } catch (\Exception $e) {
                    $results['failed']++;
                    $results['errors'][] = [
                        'index' => $index,
                        'exam_id' => $resultData['exam_id'] ?? $globalExamId,
                        'student_id' => $resultData['student_id'] ?? null,
                        'error' => $e->getMessage()
                    ];
                }
            }
            
            $this->db->commit();
            
            $message = sprintf(
                'Bulk result creation completed: %d total, %d successful, %d failed',
                $results['total_results'],
                $results['successful'],
                $results['failed']
            );
            
            return $this->response->success($results, $message);
            
        } catch (\Exception $e) {
            $this->db->rollBack();
            return $this->response->error('Bulk result creation failed: ' . $e->getMessage(), 500);
        }
    }
}