<?php

namespace App\Controllers;

use App\Core\Controller;
use App\Core\Database;
use App\Core\Response;
use App\Services\AuthService;
use PDO;

class SettingsController extends Controller
{
    private PDO $db;
    private AuthService $authService;
    
    public function __construct($request)
    {
        parent::__construct($request);
        $this->db = Database::getConnection();
        $this->authService = new AuthService();
    }
    
    /**
     * Get all settings
     */
    public function index(): Response
    {
        try {
            // Get all settings from different tables
            $settings = [];
            
            // Academic Visibility Settings
            $stmt = $this->db->prepare("
                SELECT av.*, ay.name as current_year_name 
                FROM academic_visibility av 
                LEFT JOIN academic_years ay ON av.current_year_id = ay.id 
                LIMIT 1
            ");
            $stmt->execute();
            $settings['academic_visibility'] = $stmt->fetch() ?: null;
            
            // Display Options
            $stmt = $this->db->prepare("SELECT * FROM display_options LIMIT 1");
            $stmt->execute();
            $settings['display_options'] = $stmt->fetch() ?: null;
            
            // Pass Rules
            $stmt = $this->db->prepare("SELECT * FROM pass_rules LIMIT 1");
            $stmt->execute();
            $settings['pass_rules'] = $stmt->fetch() ?: null;
            
            // Security Policy
            $stmt = $this->db->prepare("SELECT * FROM security_policy LIMIT 1");
            $stmt->execute();
            $settings['security_policy'] = $stmt->fetch() ?: null;
            
            // Grade Costs
            $stmt = $this->db->prepare("
                SELECT gc.*, ay.name as year_name, g.title as grade_title 
                FROM grade_costs gc
                LEFT JOIN academic_years ay ON gc.year_id = ay.id
                LEFT JOIN grades g ON gc.grade_id = g.id
                WHERE gc.enabled = 1
                ORDER BY ay.name DESC, g.title
            ");
            $stmt->execute();
            $settings['grade_costs'] = $stmt->fetchAll();
            
            // Get available academic years for dropdowns
            $stmt = $this->db->prepare("SELECT id, name FROM academic_years ORDER BY name DESC");
            $stmt->execute();
            $settings['academic_years'] = $stmt->fetchAll();
            
            // Get available grades for dropdowns
            $stmt = $this->db->prepare("SELECT id, title FROM grades ORDER BY title");
            $stmt->execute();
            $settings['grades'] = $stmt->fetchAll();
            
            // Percentage Bands
            $stmt = $this->db->prepare("
                SELECT * FROM percentage_bands 
                ORDER BY sort_order, start_percent DESC
            ");
            $stmt->execute();
            $settings['percentage_bands'] = $stmt->fetchAll();
            
            // Grade Result Visibility
            $stmt = $this->db->prepare("
                SELECT grv.*, g.title as grade_title, ay.name as year_name
                FROM grade_result_visibility grv
                JOIN grades g ON grv.grade_id = g.id
                JOIN academic_years ay ON grv.year_id = ay.id
                ORDER BY g.title, ay.name DESC
            ");
            $stmt->execute();
            $settings['grade_result_visibility'] = $stmt->fetchAll();
            
            // School Settings
            $stmt = $this->db->prepare("SELECT * FROM school_settings LIMIT 1");
            $stmt->execute();
            $settings['school_settings'] = $stmt->fetch() ?: null;
            
            return $this->response->success($settings);
            
        } catch (\Exception $e) {
            return $this->response->error('Failed to load settings: ' . $e->getMessage(), 500);
        }
    }
    
    /**
     * Update settings
     */
    public function update(): Response
    {
        $data = $this->request->all();
        
        try {
            $this->db->beginTransaction();
            
            // Update Academic Visibility
            if (isset($data['academic_visibility'])) {
                $av = $data['academic_visibility'];
                $this->updateAcademicVisibility($av);
            }
            
            // Update Display Options
            if (isset($data['display_options'])) {
                $do = $data['display_options'];
                $this->updateDisplayOptions($do);
            }
            
            // Update Pass Rules
            if (isset($data['pass_rules'])) {
                $pr = $data['pass_rules'];
                $this->updatePassRules($pr);
            }
            
            // Update Security Policy
            if (isset($data['security_policy'])) {
                $sp = $data['security_policy'];
                $this->updateSecurityPolicy($sp);
            }
            
            // Update Grade Costs
            if (isset($data['grade_costs'])) {
                $gc = $data['grade_costs'];
                $this->updateGradeCosts($gc);
            }
            
            // Update Percentage Bands
            if (isset($data['percentage_bands'])) {
                $pb = $data['percentage_bands'];
                $this->updatePercentageBands($pb);
            }
            
            // Update Grade Result Visibility
            if (isset($data['grade_result_visibility'])) {
                $grv = $data['grade_result_visibility'];
                $this->updateGradeResultVisibility($grv);
            }
            
            // Update School Settings
            if (isset($data['school_settings'])) {
                $ss = $data['school_settings'];
                $this->updateSchoolSettings($ss);
            }
            
            $this->db->commit();
            
            // Log audit event
            $this->authService->logAuditEvent([
                'actor_id' => $this->request->user['user_id'],
                'actor_type' => 'admin',
                'role' => $this->request->user['role'],
                'action' => 'update',
                'entity' => 'settings',
                'entity_id' => null,
                'after_data' => $data,
                'ip_address' => $this->request->getClientIp(),
                'user_agent' => $this->request->getUserAgent()
            ]);
            
            return $this->response->success(null, 'Settings updated successfully');
            
        } catch (\Exception $e) {
            $this->db->rollBack();
            return $this->response->error('Failed to update settings: ' . $e->getMessage(), 500);
        }
    }
    
    private function updateAcademicVisibility($data)
    {
        // Check if record exists
        $stmt = $this->db->prepare("SELECT id FROM academic_visibility LIMIT 1");
        $stmt->execute();
        $exists = $stmt->fetch();
        
        if ($exists) {
            $stmt = $this->db->prepare("
                UPDATE academic_visibility SET 
                    current_year_id = ?, 
                    first_term_visible = ?, 
                    first_term_publish_at = ?, 
                    second_term_visible = ?, 
                    second_term_publish_at = ?
                WHERE id = ?
            ");
            $stmt->execute([
                $data['current_year_id'],
                (int)$data['first_term_visible'],
                $data['first_term_publish_at'] ?: null,
                (int)$data['second_term_visible'],
                $data['second_term_publish_at'] ?: null,
                $exists['id']
            ]);
        } else {
            $stmt = $this->db->prepare("
                INSERT INTO academic_visibility 
                (current_year_id, first_term_visible, first_term_publish_at, second_term_visible, second_term_publish_at) 
                VALUES (?, ?, ?, ?, ?)
            ");
            $stmt->execute([
                $data['current_year_id'],
                (int)$data['first_term_visible'],
                $data['first_term_publish_at'] ?: null,
                (int)$data['second_term_visible'],
                $data['second_term_publish_at'] ?: null
            ]);
        }
    }
    
    private function updateDisplayOptions($data)
    {
        $stmt = $this->db->prepare("SELECT id FROM display_options LIMIT 1");
        $stmt->execute();
        $exists = $stmt->fetch();
        
        if ($exists) {
            $stmt = $this->db->prepare("
                UPDATE display_options SET 
                    display_mode = ?, 
                    percent_rounding = ?
                WHERE id = ?
            ");
            $stmt->execute([
                $data['display_mode'],
                (int)$data['percent_rounding'],
                $exists['id']
            ]);
        } else {
            $stmt = $this->db->prepare("
                INSERT INTO display_options (display_mode, percent_rounding) 
                VALUES (?, ?)
            ");
            $stmt->execute([
                $data['display_mode'],
                (int)$data['percent_rounding']
            ]);
        }
    }
    
    private function updatePassRules($data)
    {
        $stmt = $this->db->prepare("SELECT id FROM pass_rules LIMIT 1");
        $stmt->execute();
        $exists = $stmt->fetch();
        
        if ($exists) {
            $stmt = $this->db->prepare("
                UPDATE pass_rules SET 
                    per_subject_threshold_mode = ?, 
                    subject_pass_min_percent = ?, 
                    overall_pass_rule = ?, 
                    overall_pass_min_percent = ?, 
                    pass_condition = ?
                WHERE id = ?
            ");
            $stmt->execute([
                $data['per_subject_threshold_mode'],
                $data['subject_pass_min_percent'] ?: null,
                $data['overall_pass_rule'],
                $data['overall_pass_min_percent'] ?: null,
                $data['pass_condition'],
                $exists['id']
            ]);
        } else {
            $stmt = $this->db->prepare("
                INSERT INTO pass_rules 
                (per_subject_threshold_mode, subject_pass_min_percent, overall_pass_rule, overall_pass_min_percent, pass_condition) 
                VALUES (?, ?, ?, ?, ?)
            ");
            $stmt->execute([
                $data['per_subject_threshold_mode'],
                $data['subject_pass_min_percent'] ?: null,
                $data['overall_pass_rule'],
                $data['overall_pass_min_percent'] ?: null,
                $data['pass_condition']
            ]);
        }
    }
    
    private function updateSecurityPolicy($data)
    {
        $stmt = $this->db->prepare("SELECT id FROM security_policy LIMIT 1");
        $stmt->execute();
        $exists = $stmt->fetch();
        
        if ($exists) {
            $stmt = $this->db->prepare("
                UPDATE security_policy SET 
                    mask_national_id_in_admin = ?, 
                    student_login_requires_unique_and_national_id = ?, 
                    session_timeout_minutes = ?
                WHERE id = ?
            ");
            $stmt->execute([
                (int)$data['mask_national_id_in_admin'],
                (int)$data['student_login_requires_unique_and_national_id'],
                (int)$data['session_timeout_minutes'],
                $exists['id']
            ]);
        } else {
            $stmt = $this->db->prepare("
                INSERT INTO security_policy 
                (mask_national_id_in_admin, student_login_requires_unique_and_national_id, session_timeout_minutes) 
                VALUES (?, ?, ?)
            ");
            $stmt->execute([
                (int)$data['mask_national_id_in_admin'],
                (int)$data['student_login_requires_unique_and_national_id'],
                (int)$data['session_timeout_minutes']
            ]);
        }
    }
    
    private function updateGradeCosts($gradeCosts)
    {
        // Log the received data for debugging
        error_log('Grade costs received: ' . json_encode($gradeCosts));
        
        // Disable all existing grade costs first
        $stmt = $this->db->prepare("UPDATE grade_costs SET enabled = 0");
        $stmt->execute();
        
        // Insert or update the provided grade costs
        foreach ($gradeCosts as $cost) {
            error_log('Processing cost: ' . json_encode($cost));
            
            if (empty($cost['year_id']) || empty($cost['grade_id']) || !isset($cost['fee_amount'])) {
                error_log('Skipping cost due to missing data');
                continue;
            }
            
            try {
                $stmt = $this->db->prepare("
                    INSERT INTO grade_costs (year_id, grade_id, fee_amount, enabled) 
                    VALUES (?, ?, ?, 1)
                    ON DUPLICATE KEY UPDATE 
                        fee_amount = VALUES(fee_amount),
                        enabled = 1
                ");
                $stmt->execute([
                    (int)$cost['year_id'],
                    (int)$cost['grade_id'],
                    (float)$cost['fee_amount']
                ]);
                error_log('Successfully processed cost for year_id: ' . $cost['year_id'] . ', grade_id: ' . $cost['grade_id']);
            } catch (\Exception $e) {
                error_log('Error processing cost: ' . $e->getMessage());
            }
        }
    }
    
    private function updatePercentageBands($percentageBands)
    {
        // Clear existing bands
        $stmt = $this->db->prepare("DELETE FROM percentage_bands");
        $stmt->execute();
        
        // Insert new bands
        foreach ($percentageBands as $index => $band) {
            if (!isset($band['start_percent']) || !isset($band['end_percent']) || 
                empty($band['label']) || empty($band['color_hex'])) {
                continue;
            }
            
            $stmt = $this->db->prepare("
                INSERT INTO percentage_bands 
                (start_percent, end_percent, label, color_hex, enabled, sort_order) 
                VALUES (?, ?, ?, ?, ?, ?)
            ");
            $stmt->execute([
                (float)$band['start_percent'],
                (float)$band['end_percent'],
                $band['label'],
                $band['color_hex'],
                isset($band['enabled']) ? (int)$band['enabled'] : 1,
                $index + 1
            ]);
        }
    }
    
    private function updateGradeResultVisibility($gradeVisibility)
    {
        foreach ($gradeVisibility as $visibility) {
            if (!isset($visibility['grade_id']) || !isset($visibility['year_id'])) {
                continue;
            }
            
            $resultsVisible = isset($visibility['results_visible']) ? (int)$visibility['results_visible'] : 1;
            
            $stmt = $this->db->prepare("
                INSERT INTO grade_result_visibility 
                (grade_id, year_id, results_visible, hide_reason) 
                VALUES (?, ?, ?, ?)
                ON DUPLICATE KEY UPDATE 
                    results_visible = VALUES(results_visible),
                    hide_reason = VALUES(hide_reason),
                    updated_at = CURRENT_TIMESTAMP
            ");
            $stmt->execute([
                (int)$visibility['grade_id'],
                (int)$visibility['year_id'],
                $resultsVisible,
                $visibility['hide_reason'] ?? null
            ]);
        }
    }
    
    private function updateSchoolSettings($data)
    {
        // Check if record exists
        $stmt = $this->db->prepare("SELECT id FROM school_settings LIMIT 1");
        $stmt->execute();
        $exists = $stmt->fetch();
        
        if ($exists) {
            $stmt = $this->db->prepare("
                UPDATE school_settings SET 
                    school_name = ?, 
                    student_portal_url = ?,
                    updated_at = CURRENT_TIMESTAMP
                WHERE id = ?
            ");
            $stmt->execute([
                $data['school_name'] ?? '',
                $data['student_portal_url'] ?? '',
                $exists['id']
            ]);
        } else {
            $stmt = $this->db->prepare("
                INSERT INTO school_settings (school_name, student_portal_url) 
                VALUES (?, ?)
            ");
            $stmt->execute([
                $data['school_name'] ?? '',
                $data['student_portal_url'] ?? ''
            ]);
        }
    }
}