<?php

namespace App\Controllers;

use App\Core\Controller;
use App\Core\Database;
use App\Core\Response;
use App\Services\AuthService;
use PDO;

class StudentController extends Controller
{
    private PDO $db;
    private AuthService $authService;
    
    public function __construct($request)
    {
        parent::__construct($request);
        $this->db = Database::getConnection();
        $this->authService = new AuthService();
    }
    
    public function index(): Response
    {
        $pagination = $this->getPaginationParams();
        $sort = $this->getSortParams(['s.id', 's.full_name', 's.national_id', 's.created_at', 'ay.name', 'g.title', 'c.name']);
        
        // Build base query with joins
        $whereConditions = ['1=1'];
        $params = [];
        
        // Filters
        if ($name = $this->request->get('name')) {
            $whereConditions[] = 's.full_name LIKE ?';
            $params[] = "%{$name}%";
        }
        
        if ($nationalId = $this->request->get('national_id')) {
            $whereConditions[] = 's.national_id LIKE ?';
            $params[] = "%{$nationalId}%";
        }
        
        if ($gender = $this->request->get('gender')) {
            $whereConditions[] = 's.gender = ?';
            $params[] = $gender;
        }
        
        if ($enabled = $this->request->get('enabled')) {
            $whereConditions[] = 's.enabled = ?';
            $params[] = (int)$enabled;
        }
        
        // New enrollment filters - need to check if any enrollment filters are applied
        $hasEnrollmentFilters = false;
        
        if ($yearId = $this->request->get('year_id')) {
            $whereConditions[] = 'e.year_id = ?';
            $params[] = (int)$yearId;
            $hasEnrollmentFilters = true;
        }
        
        if ($gradeId = $this->request->get('grade_id')) {
            $whereConditions[] = 'e.grade_id = ?';
            $params[] = (int)$gradeId;
            $hasEnrollmentFilters = true;
        }
        
        if ($classId = $this->request->get('class_id')) {
            $whereConditions[] = 'e.class_id = ?';
            $params[] = (int)$classId;
            $hasEnrollmentFilters = true;
        }
        
        if ($enrollmentStatus = $this->request->get('enrollment_status')) {
            $whereConditions[] = 'e.status = ?';
            $params[] = $enrollmentStatus;
            $hasEnrollmentFilters = true;
        }
        
        // Enrolled only filter - only show students with active enrollments
        if ($this->request->get('enrolled_only')) {
            $whereConditions[] = 'e.status = ?';
            $params[] = 'active';
            $hasEnrollmentFilters = true;
        }
        
        // General search parameter
        if ($search = $this->request->get('q')) {
            $whereConditions[] = '(s.full_name LIKE ? OR s.id LIKE ? OR s.login_code LIKE ? OR s.national_id LIKE ?)';
            $searchTerm = "%{$search}%";
            $params = array_merge($params, [$searchTerm, $searchTerm, $searchTerm, $searchTerm]);
        }
        
        $whereClause = implode(' AND ', $whereConditions);
        $orderClause = $sort['sortBy'] ? "ORDER BY {$sort['sortBy']} {$sort['sortDir']}" : 'ORDER BY s.id DESC';
        
        // Determine JOIN type based on filters
        $joinType = $hasEnrollmentFilters ? 'INNER JOIN' : 'LEFT JOIN';
        
        // Update the where clause to use latest_e instead of e for enrollment filters
        $countWhereClause = str_replace(['e.year_id', 'e.grade_id', 'e.class_id', 'e.status'], 
                                       ['latest_e.year_id', 'latest_e.grade_id', 'latest_e.class_id', 'latest_e.status'], 
                                       $whereClause);
        
        // Get total count
        $countSql = "SELECT COUNT(DISTINCT s.id) 
                     FROM students s 
                     {$joinType} (
                         SELECT e1.*
                         FROM enrollments e1
                         INNER JOIN (
                             SELECT student_id, MAX(created_at) as max_created
                             FROM enrollments
                             GROUP BY student_id
                         ) e2 ON e1.student_id = e2.student_id AND e1.created_at = e2.max_created
                     ) latest_e ON s.id = latest_e.student_id 
                     LEFT JOIN academic_years ay ON latest_e.year_id = ay.id 
                     LEFT JOIN grades g ON latest_e.grade_id = g.id 
                     LEFT JOIN classes c ON latest_e.class_id = c.id 
                     WHERE {$countWhereClause}";
        
        $countStmt = $this->db->prepare($countSql);
        $countStmt->execute($params);
        $totalRows = $countStmt->fetchColumn();
        
        // Update the where clause to use latest_e instead of e for enrollment filters
        $updatedWhereClause = str_replace(['e.year_id', 'e.grade_id', 'e.class_id', 'e.status'], 
                                         ['latest_e.year_id', 'latest_e.grade_id', 'latest_e.class_id', 'latest_e.status'], 
                                         $whereClause);
        
        // Get paginated results with enrollment info (using subquery to get most recent enrollment)
        $sql = "SELECT s.id, s.login_code, s.full_name, s.national_id, s.gender, s.birthdate, s.enabled, s.created_at,
                       latest_e.year_id, latest_e.grade_id, latest_e.class_id, latest_e.status as enrollment_status,
                       ay.name as year_name,
                       g.title as grade_title, g.code as grade_code,
                       c.name as class_name
                FROM students s 
                {$joinType} (
                    SELECT e1.*
                    FROM enrollments e1
                    INNER JOIN (
                        SELECT student_id, MAX(created_at) as max_created
                        FROM enrollments
                        GROUP BY student_id
                    ) e2 ON e1.student_id = e2.student_id AND e1.created_at = e2.max_created
                ) latest_e ON s.id = latest_e.student_id 
                LEFT JOIN academic_years ay ON latest_e.year_id = ay.id 
                LEFT JOIN grades g ON latest_e.grade_id = g.id 
                LEFT JOIN classes c ON latest_e.class_id = c.id 
                WHERE {$updatedWhereClause} 
                {$orderClause} 
                LIMIT ? OFFSET ?";
        
        $stmt = $this->db->prepare($sql);
        $stmt->execute([...$params, $pagination['perPage'], $pagination['offset']]);
        $students = $stmt->fetchAll();
        
        // Mask national ID if policy requires
        $securityStmt = $this->db->prepare("SELECT mask_national_id_in_admin FROM security_policy LIMIT 1");
        $securityStmt->execute();
        $maskNationalId = $securityStmt->fetchColumn();
        
        if ($maskNationalId) {
            foreach ($students as &$student) {
                if ($student['national_id']) {
                    $student['national_id'] = str_repeat('*', strlen($student['national_id']) - 4) . 
                                            substr($student['national_id'], -4);
                }
            }
        }
        
        $meta = [
            'page' => $pagination['page'],
            'per_page' => $pagination['perPage'],
            'total_rows' => $totalRows,
            'total_pages' => ceil($totalRows / $pagination['perPage']),
            'sort_by' => $sort['sortBy'],
            'sort_dir' => $sort['sortDir']
        ];
        
        return $this->response->paginated($students, $meta);
    }
    
    public function show(string $id): Response
    {
        $stmt = $this->db->prepare(
            "SELECT s.*, 
                    e.year_id, e.grade_id, e.class_id, e.status as enrollment_status,
                    ay.name as year_name, g.title as grade_title, c.name as class_name
             FROM students s
             LEFT JOIN enrollments e ON s.id = e.student_id
             LEFT JOIN academic_years ay ON e.year_id = ay.id
             LEFT JOIN grades g ON e.grade_id = g.id
             LEFT JOIN classes c ON e.class_id = c.id
             WHERE s.id = ?"
        );
        
        $stmt->execute([$id]);
        $student = $stmt->fetch();
        
        if (!$student) {
            return $this->response->error('Student not found', 404);
        }
        
        return $this->response->success($student);
    }
    
    public function store(): Response
    {
        $data = $this->validate([
            'full_name' => 'required|min:2|max:200',
            'national_id' => 'max:50',
            'gender' => 'max:1',
            'birthdate' => 'max:10'
        ]);
        
        // Generate unique login code
        $loginCode = $this->authService->generateLoginCode();
        
        // Check for duplicate national ID
        if (!empty($data['national_id'])) {
            $stmt = $this->db->prepare("SELECT id FROM students WHERE national_id = ?");
            $stmt->execute([$data['national_id']]);
            if ($stmt->fetch()) {
                return $this->response->error('National ID already exists', 422);
            }
        }
        
        try {
            $stmt = $this->db->prepare(
                "INSERT INTO students (login_code, full_name, national_id, gender, birthdate) 
                 VALUES (?, ?, ?, ?, ?)"
            );
            
            $stmt->execute([
                $loginCode,
                $data['full_name'],
                $data['national_id'] ?? null,
                $data['gender'] ?? null,
                $data['birthdate'] ?? null
            ]);
            
            $studentId = $this->db->lastInsertId();
            
            // Get created student
            $stmt = $this->db->prepare("SELECT * FROM students WHERE id = ?");
            $stmt->execute([$studentId]);
            $student = $stmt->fetch();
            
            // Log audit event
            $this->authService->logAuditEvent([
                'actor_id' => $this->request->user['user_id'],
                'actor_type' => 'admin',
                'role' => $this->request->user['role'],
                'action' => 'create',
                'entity' => 'students',
                'entity_id' => $studentId,
                'after_data' => $student,
                'ip_address' => $this->request->getClientIp(),
                'user_agent' => $this->request->getUserAgent()
            ]);
            
            return $this->response->success($student, 'Student created successfully');
            
        } catch (\Exception $e) {
            return $this->response->error('Failed to create student: ' . $e->getMessage(), 500);
        }
    }
    
    public function update(string $id): Response
    {
        $data = $this->validate([
            'full_name' => 'required|min:2|max:200',
            'national_id' => 'max:50',
            'gender' => 'max:1',
            'birthdate' => 'max:10',
            'enabled' => 'required'
        ]);
        
        // Check if enrollment data is provided
        $yearId = $this->request->get('year_id');
        $gradeId = $this->request->get('grade_id');
        $classId = $this->request->get('class_id');
        $updateEnrollment = ($yearId !== null) || ($gradeId !== null) || ($classId !== null);
        
        if ($updateEnrollment) {
            // Validate enrollment data only if at least one field is provided
            $enrollmentData = [];
            if ($yearId !== null) $enrollmentData['year_id'] = (int)$yearId;
            if ($gradeId !== null) $enrollmentData['grade_id'] = (int)$gradeId;
            if ($classId !== null) $enrollmentData['class_id'] = (int)$classId;
            
            // Ensure all required fields are present if any enrollment data is provided
            if (empty($enrollmentData['year_id']) || empty($enrollmentData['grade_id']) || empty($enrollmentData['class_id'])) {
                return $this->response->error('All enrollment fields (year_id, grade_id, class_id) are required when updating enrollment', 422);
            }
        }
        
        // Get existing student for audit
        $stmt = $this->db->prepare(
            "SELECT s.*, 
                    e.year_id, e.grade_id, e.class_id, e.status as enrollment_status,
                    ay.name as year_name, g.title as grade_title, c.name as class_name
             FROM students s
             LEFT JOIN enrollments e ON s.id = e.student_id
             LEFT JOIN academic_years ay ON e.year_id = ay.id
             LEFT JOIN grades g ON e.grade_id = g.id
             LEFT JOIN classes c ON e.class_id = c.id
             WHERE s.id = ?"
        );
        $stmt->execute([$id]);
        $oldStudent = $stmt->fetch();
        
        if (!$oldStudent) {
            return $this->response->error('Student not found', 404);
        }
        
        // Check for duplicate national ID (excluding current student)
        if (!empty($data['national_id'])) {
            $stmt = $this->db->prepare("SELECT id FROM students WHERE national_id = ? AND id != ?");
            $stmt->execute([$data['national_id'], $id]);
            if ($stmt->fetch()) {
                return $this->response->error('National ID already exists', 422);
            }
        }
        
        try {
            $this->db->beginTransaction();
            
            // Update student
            $stmt = $this->db->prepare(
                "UPDATE students 
                 SET full_name = ?, national_id = ?, gender = ?, birthdate = ?, enabled = ? 
                 WHERE id = ?"
            );
            
            $stmt->execute([
                $data['full_name'],
                $data['national_id'] ?? null,
                $data['gender'] ?? null,
                $data['birthdate'] ?? null,
                $data['enabled'] ? 1 : 0,
                $id
            ]);
            
            // Handle enrollment update if provided
            if ($updateEnrollment) {
                // Check if enrollment exists
                $stmt = $this->db->prepare("SELECT id FROM enrollments WHERE student_id = ?");
                $stmt->execute([$id]);
                $existingEnrollment = $stmt->fetch();
                
                if ($existingEnrollment) {
                    // Update existing enrollment
                    $stmt = $this->db->prepare(
                        "UPDATE enrollments 
                         SET year_id = ?, grade_id = ?, class_id = ? 
                         WHERE student_id = ?"
                    );
                    $stmt->execute([
                        $enrollmentData['year_id'],
                        $enrollmentData['grade_id'],
                        $enrollmentData['class_id'],
                        $id
                    ]);
                } else {
                    // Create new enrollment
                    $stmt = $this->db->prepare(
                        "INSERT INTO enrollments (student_id, year_id, grade_id, class_id, status) 
                         VALUES (?, ?, ?, ?, 'active')"
                    );
                    $stmt->execute([
                        $id,
                        $enrollmentData['year_id'],
                        $enrollmentData['grade_id'],
                        $enrollmentData['class_id']
                    ]);
                }
            }
            
            // Get updated student with enrollment info
            $stmt = $this->db->prepare(
                "SELECT s.*, 
                        e.year_id, e.grade_id, e.class_id, e.status as enrollment_status,
                        ay.name as year_name, g.title as grade_title, c.name as class_name
                 FROM students s
                 LEFT JOIN enrollments e ON s.id = e.student_id
                 LEFT JOIN academic_years ay ON e.year_id = ay.id
                 LEFT JOIN grades g ON e.grade_id = g.id
                 LEFT JOIN classes c ON e.class_id = c.id
                 WHERE s.id = ?"
            );
            $stmt->execute([$id]);
            $newStudent = $stmt->fetch();
            
            $this->db->commit();
            
            // Log audit event
            $this->authService->logAuditEvent([
                'actor_id' => $this->request->user['user_id'],
                'actor_type' => 'admin',
                'role' => $this->request->user['role'],
                'action' => $updateEnrollment ? 'update_with_enrollment' : 'update',
                'entity' => 'students',
                'entity_id' => $id,
                'before_data' => $oldStudent,
                'after_data' => $newStudent,
                'ip_address' => $this->request->getClientIp(),
                'user_agent' => $this->request->getUserAgent()
            ]);
            
            $message = $updateEnrollment ? 'Student and enrollment updated successfully' : 'Student updated successfully';
            
            return $this->response->success($newStudent, $message);
            
        } catch (\Exception $e) {
            $this->db->rollBack();
            return $this->response->error('Failed to update student: ' . $e->getMessage(), 500);
        }
    }
    
    public function destroy(string $id): Response
    {
        // Get student for audit before deletion
        $stmt = $this->db->prepare("SELECT * FROM students WHERE id = ?");
        $stmt->execute([$id]);
        $student = $stmt->fetch();
        
        if (!$student) {
            return $this->response->error('Student not found', 404);
        }
        
        // Get related data for audit
        $stmt = $this->db->prepare("SELECT COUNT(*) FROM enrollments WHERE student_id = ?");
        $stmt->execute([$id]);
        $enrollmentCount = $stmt->fetchColumn();
        
        try {
            $this->db->beginTransaction();
            
            // Delete related enrollments first
            if ($enrollmentCount > 0) {
                // Get enrollments for audit
                $stmt = $this->db->prepare("SELECT * FROM enrollments WHERE student_id = ?");
                $stmt->execute([$id]);
                $enrollments = $stmt->fetchAll();
                
                // Delete enrollments
                $stmt = $this->db->prepare("DELETE FROM enrollments WHERE student_id = ?");
                $stmt->execute([$id]);
                
                // Log enrollment deletions
                foreach ($enrollments as $enrollment) {
                    $this->authService->logAuditEvent([
                        'actor_id' => $this->request->user['user_id'],
                        'actor_type' => 'admin',
                        'role' => $this->request->user['role'],
                        'action' => 'cascade_delete',
                        'entity' => 'enrollments',
                        'entity_id' => $enrollment['id'],
                        'before_data' => $enrollment,
                        'ip_address' => $this->request->getClientIp(),
                        'user_agent' => $this->request->getUserAgent()
                    ]);
                }
            }
            
            // Add any other related data deletions here (exam_results, payments, etc.)
            // Example: DELETE FROM exam_results WHERE student_id = ?
            // Example: DELETE FROM payments WHERE student_id = ?
            
            // Finally, delete the student
            $stmt = $this->db->prepare("DELETE FROM students WHERE id = ?");
            $stmt->execute([$id]);
            
            $this->db->commit();
            
            // Log main deletion event
            $this->authService->logAuditEvent([
                'actor_id' => $this->request->user['user_id'],
                'actor_type' => 'admin',
                'role' => $this->request->user['role'],
                'action' => 'delete_with_cascade',
                'entity' => 'students',
                'entity_id' => $id,
                'before_data' => $student,
                'metadata' => ['deleted_enrollments' => $enrollmentCount],
                'ip_address' => $this->request->getClientIp(),
                'user_agent' => $this->request->getUserAgent()
            ]);
            
            $message = $enrollmentCount > 0 
                ? "Student and {$enrollmentCount} related enrollment(s) deleted successfully"
                : 'Student deleted successfully';
                
            return $this->response->success(null, $message);
            
        } catch (\Exception $e) {
            $this->db->rollBack();
            return $this->response->error('Failed to delete student: ' . $e->getMessage(), 500);
        }
    }
    
    public function import(): Response
    {
        if (!isset($_FILES['file']) || $_FILES['file']['error'] !== UPLOAD_ERR_OK) {
            return $this->response->error('No file uploaded or upload error', 400);
        }
        
        $file = $_FILES['file'];
        $allowedTypes = ['text/csv', 'application/csv', 'text/plain'];
        $maxSize = 5 * 1024 * 1024; // 5MB
        
        if ($file['size'] > $maxSize) {
            return $this->response->error('File size exceeds 5MB limit', 400);
        }
        
        if (!in_array($file['type'], $allowedTypes) && !str_ends_with($file['name'], '.csv')) {
            return $this->response->error('Only CSV files are allowed', 400);
        }
        
        try {
            $handle = fopen($file['tmp_name'], 'r');
            if (!$handle) {
                return $this->response->error('Failed to read uploaded file', 500);
            }
            
            $headers = fgetcsv($handle);
            if (!$headers) {
                fclose($handle);
                return $this->response->error('Invalid CSV file - no headers found', 400);
            }
            
            // Validate headers
            $requiredHeaders = ['full_name'];
            $optionalHeaders = ['national_id', 'gender', 'birthdate'];
            $allValidHeaders = array_merge($requiredHeaders, $optionalHeaders);
            
            $invalidHeaders = array_diff($headers, $allValidHeaders);
            if (!empty($invalidHeaders)) {
                fclose($handle);
                return $this->response->error('Invalid headers: ' . implode(', ', $invalidHeaders), 400);
            }
            
            $missingRequired = array_diff($requiredHeaders, $headers);
            if (!empty($missingRequired)) {
                fclose($handle);
                return $this->response->error('Missing required headers: ' . implode(', ', $missingRequired), 400);
            }
            
            $results = [
                'total_rows' => 0,
                'successful' => 0,
                'failed' => 0,
                'errors' => [],
                'imported_students' => []
            ];
            
            $rowNumber = 1; // Start from 1 (header is row 0)
            
            while (($row = fgetcsv($handle)) !== false) {
                $rowNumber++;
                $results['total_rows']++;
                
                // Skip empty rows
                if (empty(array_filter($row))) {
                    continue;
                }
                
                $studentData = array_combine($headers, $row);
                
                try {
                    // Validate required fields
                    if (empty(trim($studentData['full_name']))) {
                        throw new \Exception('Full name is required');
                    }
                    
                    // Clean and validate data
                    $cleanData = [
                        'full_name' => trim($studentData['full_name']),
                        'national_id' => isset($studentData['national_id']) ? trim($studentData['national_id']) : null,
                        'gender' => isset($studentData['gender']) ? strtoupper(trim($studentData['gender'])) : null,
                        'birthdate' => isset($studentData['birthdate']) ? trim($studentData['birthdate']) : null
                    ];
                    
                    // Validate gender
                    if ($cleanData['gender'] && !in_array($cleanData['gender'], ['M', 'F'])) {
                        throw new \Exception('Gender must be M (Male) or F (Female)');
                    }
                    
                    // Validate birthdate format
                    if ($cleanData['birthdate'] && !preg_match('/^\d{4}-\d{2}-\d{2}$/', $cleanData['birthdate'])) {
                        throw new \Exception('Birthdate must be in YYYY-MM-DD format');
                    }
                    
                    // Check for duplicate national ID
                    if (!empty($cleanData['national_id'])) {
                        $stmt = $this->db->prepare("SELECT id FROM students WHERE national_id = ?");
                        $stmt->execute([$cleanData['national_id']]);
                        if ($stmt->fetch()) {
                            throw new \Exception('National ID already exists: ' . $cleanData['national_id']);
                        }
                    }
                    
                    // Generate unique login code
                    $loginCode = $this->authService->generateLoginCode();
                    
                    // Insert student
                    $stmt = $this->db->prepare(
                        "INSERT INTO students (login_code, full_name, national_id, gender, birthdate) 
                         VALUES (?, ?, ?, ?, ?)"
                    );
                    
                    $stmt->execute([
                        $loginCode,
                        $cleanData['full_name'],
                        $cleanData['national_id'] ?: null,
                        $cleanData['gender'] ?: null,
                        $cleanData['birthdate'] ?: null
                    ]);
                    
                    $studentId = $this->db->lastInsertId();
                    $results['successful']++;
                    $results['imported_students'][] = [
                        'id' => $studentId,
                        'login_code' => $loginCode,
                        'full_name' => $cleanData['full_name'],
                        'row' => $rowNumber
                    ];
                    
                    // Log audit event
                    $this->authService->logAuditEvent([
                        'actor_id' => $this->request->user['user_id'],
                        'actor_type' => 'admin',
                        'role' => $this->request->user['role'],
                        'action' => 'import',
                        'entity' => 'students',
                        'entity_id' => $studentId,
                        'after_data' => array_merge($cleanData, ['login_code' => $loginCode]),
                        'ip_address' => $this->request->getClientIp(),
                        'user_agent' => $this->request->getUserAgent()
                    ]);
                    
                } catch (\Exception $e) {
                    $results['failed']++;
                    $results['errors'][] = [
                        'row' => $rowNumber,
                        'data' => $studentData,
                        'error' => $e->getMessage()
                    ];
                }
            }
            
            fclose($handle);
            
            $message = sprintf(
                'Import completed: %d total, %d successful, %d failed',
                $results['total_rows'],
                $results['successful'],
                $results['failed']
            );
            
            return $this->response->success($results, $message);
            
        } catch (\Exception $e) {
            return $this->response->error('Import failed: ' . $e->getMessage(), 500);
        }
    }
    
    public function bulkDelete(): Response
    {
        $data = $this->request->all();
        
        if (!isset($data['student_ids']) || !is_array($data['student_ids']) || empty($data['student_ids'])) {
            return $this->response->error('No student IDs provided', 400);
        }
        
        $studentIds = array_map('intval', $data['student_ids']);
        $studentIds = array_filter($studentIds); // Remove any non-numeric values
        
        if (empty($studentIds)) {
            return $this->response->error('Invalid student IDs provided', 400);
        }
        
        try {
            $this->db->beginTransaction();
            
            // Get students for audit before deletion
            $placeholders = str_repeat('?,', count($studentIds) - 1) . '?';
            $stmt = $this->db->prepare("SELECT * FROM students WHERE id IN ($placeholders)");
            $stmt->execute($studentIds);
            $students = $stmt->fetchAll();
            
            if (empty($students)) {
                $this->db->rollBack();
                return $this->response->error('No students found with provided IDs', 404);
            }
            
            // Get enrollment counts for each student
            $stmt = $this->db->prepare("SELECT student_id, COUNT(*) as enrollment_count FROM enrollments WHERE student_id IN ($placeholders) GROUP BY student_id");
            $stmt->execute($studentIds);
            $enrollmentCounts = $stmt->fetchAll(PDO::FETCH_KEY_PAIR);
            
            $totalEnrollmentsDeleted = 0;
            
            // Delete related enrollments for each student
            foreach ($studentIds as $studentId) {
                if (isset($enrollmentCounts[$studentId]) && $enrollmentCounts[$studentId] > 0) {
                    // Get enrollments for audit
                    $stmt = $this->db->prepare("SELECT * FROM enrollments WHERE student_id = ?");
                    $stmt->execute([$studentId]);
                    $enrollments = $stmt->fetchAll();
                    
                    // Delete enrollments
                    $stmt = $this->db->prepare("DELETE FROM enrollments WHERE student_id = ?");
                    $stmt->execute([$studentId]);
                    
                    $totalEnrollmentsDeleted += count($enrollments);
                    
                    // Log enrollment deletions
                    foreach ($enrollments as $enrollment) {
                        $this->authService->logAuditEvent([
                            'actor_id' => $this->request->user['user_id'],
                            'actor_type' => 'admin',
                            'role' => $this->request->user['role'],
                            'action' => 'bulk_cascade_delete',
                            'entity' => 'enrollments',
                            'entity_id' => $enrollment['id'],
                            'before_data' => $enrollment,
                            'ip_address' => $this->request->getClientIp(),
                            'user_agent' => $this->request->getUserAgent()
                        ]);
                    }
                }
                
                // Add any other related data deletions here
                // Example: DELETE FROM exam_results WHERE student_id = ?
                // Example: DELETE FROM payments WHERE student_id = ?
            }
            
            // Delete students
            $stmt = $this->db->prepare("DELETE FROM students WHERE id IN ($placeholders)");
            $stmt->execute($studentIds);
            
            $deletedCount = $stmt->rowCount();
            
            $this->db->commit();
            
            // Log audit events for each deleted student
            foreach ($students as $student) {
                $this->authService->logAuditEvent([
                    'actor_id' => $this->request->user['user_id'],
                    'actor_type' => 'admin',
                    'role' => $this->request->user['role'],
                    'action' => 'bulk_delete_with_cascade',
                    'entity' => 'students',
                    'entity_id' => $student['id'],
                    'before_data' => $student,
                    'metadata' => ['deleted_enrollments' => $enrollmentCounts[$student['id']] ?? 0],
                    'ip_address' => $this->request->getClientIp(),
                    'user_agent' => $this->request->getUserAgent()
                ]);
            }
            
            $message = $totalEnrollmentsDeleted > 0 
                ? "Successfully deleted {$deletedCount} students and {$totalEnrollmentsDeleted} related enrollments"
                : "Successfully deleted {$deletedCount} students";
            
            return $this->response->success([
                'deleted_count' => $deletedCount,
                'requested_count' => count($studentIds),
                'deleted_enrollments' => $totalEnrollmentsDeleted
            ], $message);
            
        } catch (\Exception $e) {
            return $this->response->error('Failed to delete students: ' . $e->getMessage(), 500);
        }
    }
    
    public function storeWithEnrollment(): Response
    {
        $data = $this->validate([
            'full_name' => 'required|min:2|max:200',
            'national_id' => 'max:50',
            'gender' => 'max:1',
            'birthdate' => 'max:10',
            'year_id' => 'required|numeric',
            'grade_id' => 'required|numeric',
            'class_id' => 'required|numeric'
        ]);
        
        // Generate unique login code
        $loginCode = $this->authService->generateLoginCode();
        
        // Check for duplicate national ID
        if (!empty($data['national_id'])) {
            $stmt = $this->db->prepare("SELECT id FROM students WHERE national_id = ?");
            $stmt->execute([$data['national_id']]);
            if ($stmt->fetch()) {
                return $this->response->error('National ID already exists', 422);
            }
        }
        
        try {
            $this->db->beginTransaction();
            
            // Create student
            $stmt = $this->db->prepare(
                "INSERT INTO students (login_code, full_name, national_id, gender, birthdate) 
                 VALUES (?, ?, ?, ?, ?)"
            );
            
            $stmt->execute([
                $loginCode,
                $data['full_name'],
                $data['national_id'] ?? null,
                $data['gender'] ?? null,
                $data['birthdate'] ?? null
            ]);
            
            $studentId = $this->db->lastInsertId();
            
            // Create enrollment
            $stmt = $this->db->prepare(
                "INSERT INTO enrollments (student_id, year_id, grade_id, class_id, status) 
                 VALUES (?, ?, ?, ?, 'active')"
            );
            
            $stmt->execute([
                $studentId,
                $data['year_id'],
                $data['grade_id'],
                $data['class_id']
            ]);
            
            $enrollmentId = $this->db->lastInsertId();
            
            // Get created student with enrollment info
            $stmt = $this->db->prepare(
                "SELECT s.*, 
                        e.id as enrollment_id, e.year_id, e.grade_id, e.class_id, e.status as enrollment_status,
                        ay.name as year_name,
                        g.title as grade_title, g.code as grade_code,
                        c.name as class_name
                 FROM students s
                 LEFT JOIN enrollments e ON s.id = e.student_id
                 LEFT JOIN academic_years ay ON e.year_id = ay.id
                 LEFT JOIN grades g ON e.grade_id = g.id
                 LEFT JOIN classes c ON e.class_id = c.id
                 WHERE s.id = ?"
            );
            $stmt->execute([$studentId]);
            $result = $stmt->fetch();
            
            $this->db->commit();
            
            // Log audit events
            $this->authService->logAuditEvent([
                'actor_id' => $this->request->user['user_id'],
                'actor_type' => 'admin',
                'role' => $this->request->user['role'],
                'action' => 'create_with_enrollment',
                'entity' => 'students',
                'entity_id' => $studentId,
                'after_data' => $result,
                'ip_address' => $this->request->getClientIp(),
                'user_agent' => $this->request->getUserAgent()
            ]);
            
            return $this->response->success($result, 'Student created and enrolled successfully');
            
        } catch (\Exception $e) {
            $this->db->rollBack();
            return $this->response->error('Failed to create student with enrollment: ' . $e->getMessage(), 500);
        }
    }
    
    public function importWithEnrollment(): Response
    {
        if (!isset($_FILES['file']) || $_FILES['file']['error'] !== UPLOAD_ERR_OK) {
            return $this->response->error('No file uploaded or upload error', 400);
        }
        
        $file = $_FILES['file'];
        $allowedTypes = ['text/csv', 'application/csv', 'text/plain'];
        $maxSize = 5 * 1024 * 1024; // 5MB
        
        if ($file['size'] > $maxSize) {
            return $this->response->error('File size exceeds 5MB limit', 400);
        }
        
        if (!in_array($file['type'], $allowedTypes) && !str_ends_with($file['name'], '.csv')) {
            return $this->response->error('Only CSV files are allowed', 400);
        }
        
        // Get enrollment data from request
        $requestData = $this->request->all();
        if (!isset($requestData['year_id']) || !isset($requestData['grade_id']) || !isset($requestData['class_id'])) {
            return $this->response->error('Year, Grade, and Class are required for enrollment', 400);
        }
        
        $yearId = (int)$requestData['year_id'];
        $gradeId = (int)$requestData['grade_id'];
        $classId = (int)$requestData['class_id'];
        
        try {
            $handle = fopen($file['tmp_name'], 'r');
            if (!$handle) {
                return $this->response->error('Failed to read uploaded file', 500);
            }
            
            $headers = fgetcsv($handle);
            if (!$headers) {
                fclose($handle);
                return $this->response->error('Invalid CSV file - no headers found', 400);
            }
            
            // Validate headers
            $requiredHeaders = ['full_name'];
            $optionalHeaders = ['national_id', 'gender', 'birthdate'];
            $allValidHeaders = array_merge($requiredHeaders, $optionalHeaders);
            
            $invalidHeaders = array_diff($headers, $allValidHeaders);
            if (!empty($invalidHeaders)) {
                fclose($handle);
                return $this->response->error('Invalid headers: ' . implode(', ', $invalidHeaders), 400);
            }
            
            $missingRequired = array_diff($requiredHeaders, $headers);
            if (!empty($missingRequired)) {
                fclose($handle);
                return $this->response->error('Missing required headers: ' . implode(', ', $missingRequired), 400);
            }
            
            $results = [
                'total_rows' => 0,
                'successful' => 0,
                'failed' => 0,
                'errors' => [],
                'enrolled_students' => []
            ];
            
            $rowNumber = 1;
            
            while (($row = fgetcsv($handle)) !== false) {
                $rowNumber++;
                $results['total_rows']++;
                
                // Skip empty rows
                if (empty(array_filter($row))) {
                    continue;
                }
                
                $studentData = array_combine($headers, $row);
                
                try {
                    $this->db->beginTransaction();
                    
                    // Validate required fields
                    if (empty(trim($studentData['full_name']))) {
                        throw new \Exception('Full name is required');
                    }
                    
                    // Clean and validate data
                    $cleanData = [
                        'full_name' => trim($studentData['full_name']),
                        'national_id' => isset($studentData['national_id']) ? trim($studentData['national_id']) : null,
                        'gender' => isset($studentData['gender']) ? strtoupper(trim($studentData['gender'])) : null,
                        'birthdate' => isset($studentData['birthdate']) ? trim($studentData['birthdate']) : null
                    ];
                    
                    // Validate gender
                    if ($cleanData['gender'] && !in_array($cleanData['gender'], ['M', 'F'])) {
                        throw new \Exception('Gender must be M (Male) or F (Female)');
                    }
                    
                    // Validate birthdate format
                    if ($cleanData['birthdate'] && !preg_match('/^\d{4}-\d{2}-\d{2}$/', $cleanData['birthdate'])) {
                        throw new \Exception('Birthdate must be in YYYY-MM-DD format');
                    }
                    
                    // Check for duplicate national ID
                    if (!empty($cleanData['national_id'])) {
                        $stmt = $this->db->prepare("SELECT id FROM students WHERE national_id = ?");
                        $stmt->execute([$cleanData['national_id']]);
                        if ($stmt->fetch()) {
                            throw new \Exception('National ID already exists: ' . $cleanData['national_id']);
                        }
                    }
                    
                    // Generate unique login code
                    $loginCode = $this->authService->generateLoginCode();
                    
                    // Insert student
                    $stmt = $this->db->prepare(
                        "INSERT INTO students (login_code, full_name, national_id, gender, birthdate) 
                         VALUES (?, ?, ?, ?, ?)"
                    );
                    
                    $stmt->execute([
                        $loginCode,
                        $cleanData['full_name'],
                        $cleanData['national_id'] ?: null,
                        $cleanData['gender'] ?: null,
                        $cleanData['birthdate'] ?: null
                    ]);
                    
                    $studentId = $this->db->lastInsertId();
                    
                    // Create enrollment
                    $stmt = $this->db->prepare(
                        "INSERT INTO enrollments (student_id, year_id, grade_id, class_id, status) 
                         VALUES (?, ?, ?, ?, 'active')"
                    );
                    
                    $stmt->execute([$studentId, $yearId, $gradeId, $classId]);
                    
                    $this->db->commit();
                    
                    $results['successful']++;
                    $results['enrolled_students'][] = [
                        'id' => $studentId,
                        'login_code' => $loginCode,
                        'full_name' => $cleanData['full_name'],
                        'row' => $rowNumber,
                        'year_id' => $yearId,
                        'grade_id' => $gradeId,
                        'class_id' => $classId
                    ];
                    
                    // Log audit event
                    $this->authService->logAuditEvent([
                        'actor_id' => $this->request->user['user_id'],
                        'actor_type' => 'admin',
                        'role' => $this->request->user['role'],
                        'action' => 'import_with_enrollment',
                        'entity' => 'students',
                        'entity_id' => $studentId,
                        'after_data' => array_merge($cleanData, [
                            'login_code' => $loginCode,
                            'year_id' => $yearId,
                            'grade_id' => $gradeId,
                            'class_id' => $classId
                        ]),
                        'ip_address' => $this->request->getClientIp(),
                        'user_agent' => $this->request->getUserAgent()
                    ]);
                    
                } catch (\Exception $e) {
                    $this->db->rollBack();
                    $results['failed']++;
                    $results['errors'][] = [
                        'row' => $rowNumber,
                        'data' => $studentData,
                        'error' => $e->getMessage()
                    ];
                }
            }
            
            fclose($handle);
            
            $message = sprintf(
                'Import with enrollment completed: %d total, %d successful, %d failed',
                $results['total_rows'],
                $results['successful'],
                $results['failed']
            );
            
            return $this->response->success($results, $message);
            
        } catch (\Exception $e) {
            return $this->response->error('Import failed: ' . $e->getMessage(), 500);
        }
    }

    public function getStudentExams(string $studentId): Response
    {
        try {
            // Get student details with enrollment
            $stmt = $this->db->prepare("
                SELECT s.id, s.full_name, 
                       e.year_id, e.grade_id,
                       ay.name as year_name,
                       g.title as grade_title
                FROM students s
                LEFT JOIN enrollments e ON s.id = e.student_id AND e.status = 'active'
                LEFT JOIN academic_years ay ON e.year_id = ay.id
                LEFT JOIN grades g ON e.grade_id = g.id
                WHERE s.id = ? AND s.enabled = 1
            ");
            $stmt->execute([$studentId]);
            $student = $stmt->fetch();
            
            if (!$student) {
                return $this->response->error('Student not found or not enrolled', 404);
            }
            
            if (!$student['year_id'] || !$student['grade_id']) {
                return $this->response->error('Student has no active enrollment', 422);
            }
            
            // Get all exams for this student's grade/year with current results and percentage bands
            $stmt = $this->db->prepare("
                SELECT e.id, e.title, e.exam_date, e.min_score, e.max_score,
                       e.year_id, e.grade_id, e.term_id, e.subject_id,
                       ay.name as year_name,
                       t.name as term_name,
                       g.title as grade_title,
                       sub.name as subject_name,
                       er.id as result_id,
                       er.score as current_score,
                       er.percentage as current_percentage,
                       pb.label as band_label,
                       pb.color_hex as band_color
                FROM exams e
                LEFT JOIN academic_years ay ON e.year_id = ay.id
                LEFT JOIN terms t ON e.term_id = t.id
                LEFT JOIN grades g ON e.grade_id = g.id
                LEFT JOIN subjects sub ON e.subject_id = sub.id
                LEFT JOIN exam_results er ON e.id = er.exam_id AND er.student_id = ?
                LEFT JOIN percentage_bands pb ON (er.percentage >= pb.start_percent AND er.percentage <= pb.end_percent AND pb.enabled = 1)
                WHERE e.year_id = ? AND e.grade_id = ? AND e.enabled = 1
                ORDER BY e.exam_date DESC, sub.name, e.title
            ");
            $stmt->execute([$studentId, $student['year_id'], $student['grade_id']]);
            $exams = $stmt->fetchAll();
            
            return $this->response->success($exams);
            
        } catch (\Exception $e) {
            return $this->response->error('Failed to load student exams: ' . $e->getMessage(), 500);
        }
    }
    
    public function importStudentsWithIndividualEnrollment(): Response
    {
        if (!isset($_FILES['file']) || $_FILES['file']['error'] !== UPLOAD_ERR_OK) {
            return $this->response->error('No file uploaded or upload error', 400);
        }
        
        $file = $_FILES['file'];
        $allowedTypes = ['text/csv', 'application/csv', 'text/plain'];
        $maxSize = 5 * 1024 * 1024; // 5MB
        
        if ($file['size'] > $maxSize) {
            return $this->response->error('File size exceeds 5MB limit', 400);
        }
        
        if (!in_array($file['type'], $allowedTypes) && !str_ends_with($file['name'], '.csv')) {
            return $this->response->error('Only CSV files are allowed', 400);
        }
        
        try {
            $handle = fopen($file['tmp_name'], 'r');
            if (!$handle) {
                return $this->response->error('Failed to read uploaded file', 500);
            }
            
            $headers = fgetcsv($handle);
            if (!$headers) {
                fclose($handle);
                return $this->response->error('Invalid CSV file - no headers found', 400);
            }
            
            // Validate headers - both student and enrollment data required
            $requiredHeaders = ['full_name', 'year_name', 'grade_code', 'class_code'];
            $optionalHeaders = ['national_id', 'gender', 'birthdate'];
            $allValidHeaders = array_merge($requiredHeaders, $optionalHeaders);
            
            $invalidHeaders = array_diff($headers, $allValidHeaders);
            if (!empty($invalidHeaders)) {
                fclose($handle);
                return $this->response->error('Invalid headers: ' . implode(', ', $invalidHeaders), 400);
            }
            
            $missingRequired = array_diff($requiredHeaders, $headers);
            if (!empty($missingRequired)) {
                fclose($handle);
                return $this->response->error('Missing required headers: ' . implode(', ', $missingRequired), 400);
            }
            
            $results = [
                'total_rows' => 0,
                'successful' => 0,
                'failed' => 0,
                'errors' => [],
                'enrolled_students' => []
            ];
            
            $rowNumber = 1;
            
            while (($row = fgetcsv($handle)) !== false) {
                $rowNumber++;
                $results['total_rows']++;
                
                // Skip empty rows
                if (empty(array_filter($row))) {
                    continue;
                }
                
                $studentData = array_combine($headers, $row);
                
                try {
                    $this->db->beginTransaction();
                    
                    // Validate required fields
                    if (empty(trim($studentData['full_name']))) {
                        throw new \Exception('Full name is required');
                    }
                    if (empty(trim($studentData['year_name']))) {
                        throw new \Exception('Year name is required');
                    }
                    if (empty(trim($studentData['grade_code']))) {
                        throw new \Exception('Grade code is required');
                    }
                    if (empty(trim($studentData['class_code']))) {
                        throw new \Exception('Class code is required');
                    }
                    
                    // Clean student data
                    $cleanData = [
                        'full_name' => trim($studentData['full_name']),
                        'national_id' => isset($studentData['national_id']) ? trim($studentData['national_id']) : null,
                        'gender' => isset($studentData['gender']) ? strtoupper(trim($studentData['gender'])) : null,
                        'birthdate' => isset($studentData['birthdate']) ? trim($studentData['birthdate']) : null
                    ];
                    
                    // Clean enrollment data
                    $enrollmentData = [
                        'year_name' => trim($studentData['year_name']),
                        'grade_code' => trim($studentData['grade_code']),
                        'class_code' => trim($studentData['class_code'])
                    ];
                    
                    // Validate gender
                    if ($cleanData['gender'] && !in_array($cleanData['gender'], ['M', 'F'])) {
                        throw new \Exception('Gender must be M (Male) or F (Female)');
                    }
                    
                    // Validate birthdate format
                    if ($cleanData['birthdate'] && !preg_match('/^\d{4}-\d{2}-\d{2}$/', $cleanData['birthdate'])) {
                        throw new \Exception('Birthdate must be in YYYY-MM-DD format');
                    }
                    
                    // Check for duplicate national ID
                    if (!empty($cleanData['national_id'])) {
                        $stmt = $this->db->prepare("SELECT id FROM students WHERE national_id = ?");
                        $stmt->execute([$cleanData['national_id']]);
                        if ($stmt->fetch()) {
                            throw new \Exception('National ID already exists: ' . $cleanData['national_id']);
                        }
                    }
                    
                    // Resolve enrollment IDs
                    // Get academic year ID
                    $stmt = $this->db->prepare("SELECT id FROM academic_years WHERE name = ?");
                    $stmt->execute([$enrollmentData['year_name']]);
                    $year = $stmt->fetch();
                    if (!$year) {
                        throw new \Exception('Academic year not found: ' . $enrollmentData['year_name']);
                    }
                    $yearId = $year['id'];
                    
                    // Get grade ID
                    $stmt = $this->db->prepare("SELECT id FROM grades WHERE code = ?");
                    $stmt->execute([$enrollmentData['grade_code']]);
                    $grade = $stmt->fetch();
                    if (!$grade) {
                        throw new \Exception('Grade not found: ' . $enrollmentData['grade_code']);
                    }
                    $gradeId = $grade['id'];
                    
                    // Get class ID (classes are global in current system)
                    $stmt = $this->db->prepare("SELECT id FROM classes WHERE name = ? AND enabled = 1");
                    $stmt->execute([$enrollmentData['class_code']]);
                    $class = $stmt->fetch();
                    if (!$class) {
                        throw new \Exception('Class not found: ' . $enrollmentData['class_code'] . '. Available classes should be created first in the Classes section.');
                    }
                    $classId = $class['id'];
                    
                    // Generate unique login code
                    $loginCode = $this->authService->generateLoginCode();
                    
                    // Insert student
                    $stmt = $this->db->prepare(
                        "INSERT INTO students (login_code, full_name, national_id, gender, birthdate) 
                         VALUES (?, ?, ?, ?, ?)"
                    );
                    
                    $stmt->execute([
                        $loginCode,
                        $cleanData['full_name'],
                        $cleanData['national_id'] ?: null,
                        $cleanData['gender'] ?: null,
                        $cleanData['birthdate'] ?: null
                    ]);
                    
                    $studentId = $this->db->lastInsertId();
                    
                    // Create enrollment
                    $stmt = $this->db->prepare(
                        "INSERT INTO enrollments (student_id, year_id, grade_id, class_id, status) 
                         VALUES (?, ?, ?, ?, 'active')"
                    );
                    
                    $stmt->execute([$studentId, $yearId, $gradeId, $classId]);
                    
                    $this->db->commit();
                    
                    $results['successful']++;
                    $results['enrolled_students'][] = [
                        'id' => $studentId,
                        'login_code' => $loginCode,
                        'full_name' => $cleanData['full_name'],
                        'row' => $rowNumber,
                        'year_name' => $enrollmentData['year_name'],
                        'grade_code' => $enrollmentData['grade_code'],
                        'class_code' => $enrollmentData['class_code']
                    ];
                    
                    // Log audit event
                    $this->authService->logAuditEvent([
                        'actor_id' => $this->request->user['user_id'],
                        'actor_type' => 'admin',
                        'role' => $this->request->user['role'],
                        'action' => 'import_individual_enrollment',
                        'entity' => 'students',
                        'entity_id' => $studentId,
                        'after_data' => array_merge($cleanData, $enrollmentData, [
                            'login_code' => $loginCode,
                            'year_id' => $yearId,
                            'grade_id' => $gradeId,
                            'class_id' => $classId
                        ]),
                        'ip_address' => $this->request->getClientIp(),
                        'user_agent' => $this->request->getUserAgent()
                    ]);
                    
                } catch (\Exception $e) {
                    if ($this->db->inTransaction()) {
                        $this->db->rollBack();
                    }
                    
                    $results['failed']++;
                    $results['errors'][] = [
                        'row' => $rowNumber,
                        'error' => $e->getMessage(),
                        'data' => $studentData
                    ];
                }
            }
            
            fclose($handle);
            
            return $this->response->success($results, 
                "Import completed. {$results['successful']} students imported successfully, {$results['failed']} failed."
            );
            
        } catch (\Exception $e) {
            return $this->response->error('Failed to process import: ' . $e->getMessage(), 500);
        }
    }
    
    public function downloadEnrollmentTemplate(): Response
    {
        $templatePath = __DIR__ . '/../../templates/students_with_enrollment_template.csv';
        
        if (!file_exists($templatePath)) {
            return $this->response->error('Template file not found', 404);
        }
        
        // Set headers for file download
        header('Content-Type: text/csv');
        header('Content-Disposition: attachment; filename="students_enrollment_template.csv"');
        header('Content-Length: ' . filesize($templatePath));
        
        // Output the file
        readfile($templatePath);
        exit;
    }
}