<?php

namespace App\Controllers;

use App\Core\Controller;
use App\Core\Database;
use App\Core\Response;
use PDO;

class StudentPortalController extends Controller
{
    private PDO $db;
    
    public function __construct($request)
    {
        parent::__construct($request);
        $this->db = Database::getConnection();
    }
    
    public function profile(): Response
    {
        $studentId = $this->request->user['user_id'];
        $yearId = $this->request->user['year_id'];
        
        $stmt = $this->db->prepare(
            "SELECT s.id, s.full_name, s.national_id, s.gender, s.birthdate,
                    ay.name as year_name, g.title as grade_title, c.name as class_code,
                    e.status as enrollment_status
             FROM students s
             JOIN enrollments e ON s.id = e.student_id
             JOIN academic_years ay ON e.year_id = ay.id
             JOIN grades g ON e.grade_id = g.id
             JOIN classes c ON e.class_id = c.id
             WHERE s.id = ? AND e.year_id = ?"
        );
        
        $stmt->execute([$studentId, $yearId]);
        $profile = $stmt->fetch();
        
        if (!$profile) {
            return $this->response->error('Profile not found', 404);
        }
        
        // Mask national ID if policy requires
        $securityStmt = $this->db->prepare("SELECT mask_national_id_in_admin FROM security_policy LIMIT 1");
        $securityStmt->execute();
        $maskNationalId = $securityStmt->fetchColumn();
        
        if ($maskNationalId && $profile['national_id']) {
            $profile['national_id'] = str_repeat('*', strlen($profile['national_id']) - 4) . 
                                    substr($profile['national_id'], -4);
        }
        
        return $this->response->success($profile);
    }
    
    public function results(): Response
    {
        $studentId = $this->request->user['user_id'];
        $yearId = $this->request->user['year_id'];
        $gradeId = $this->request->user['grade_id'];
        
        // Check if results are visible for this grade
        $gradeVisibilityStmt = $this->db->prepare("
            SELECT results_visible, hide_reason 
            FROM grade_result_visibility 
            WHERE grade_id = ? AND year_id = ?
        ");
        $gradeVisibilityStmt->execute([$gradeId, $yearId]);
        $gradeVisibility = $gradeVisibilityStmt->fetch();
        
        
        // If a record exists and results are not visible, deny access
        if ($gradeVisibility && !$gradeVisibility['results_visible']) {
            return $this->response->error(
                $gradeVisibility['hide_reason'] ?: 'Results are not available for your grade level at this time.',
                403
            );
        }
        
        // If no record exists, default to visible (allow access)
        
        // Check if student has paid required grade cost
        $paymentCheckStmt = $this->db->prepare("
            SELECT gc.fee_amount, COALESCE(SUM(p.amount), 0) as total_paid
            FROM grade_costs gc
            LEFT JOIN enrollments e ON gc.grade_id = ? AND gc.year_id = ?
            LEFT JOIN payments p ON e.id = p.enrollment_id AND e.student_id = ?
            WHERE gc.grade_id = ? AND gc.year_id = ? AND gc.enabled = 1
            GROUP BY gc.fee_amount
        ");
        $paymentCheckStmt->execute([$gradeId, $yearId, $studentId, $gradeId, $yearId]);
        $paymentCheck = $paymentCheckStmt->fetch();
        
        if ($paymentCheck && $paymentCheck['fee_amount'] > 0) {
            $feeAmount = (float)$paymentCheck['fee_amount'];
            $totalPaid = (float)$paymentCheck['total_paid'];
            
            if ($totalPaid < $feeAmount) {
                $remaining = $feeAmount - $totalPaid;
                return $this->response->error(
                    "Results are not available until grade fees are paid in full. Outstanding balance: $" . number_format($remaining, 2),
                    402 // Payment Required
                );
            }
        }
        
        // Get academic visibility settings
        $visibilityStmt = $this->db->prepare(
            "SELECT first_term_visible, first_term_publish_at, 
                    second_term_visible, second_term_publish_at
             FROM academic_visibility 
             WHERE current_year_id = ?"
        );
        $visibilityStmt->execute([$yearId]);
        $visibility = $visibilityStmt->fetch();
        
        // Get display options
        $displayStmt = $this->db->prepare("SELECT display_mode, percent_rounding FROM display_options LIMIT 1");
        $displayStmt->execute();
        $displayOptions = $displayStmt->fetch();
        
        // Get percentage bands
        $bandsStmt = $this->db->prepare(
            "SELECT start_percent, end_percent, label, color_hex 
             FROM percentage_bands 
             WHERE enabled = 1 
             ORDER BY sort_order"
        );
        $bandsStmt->execute();
        $bands = $bandsStmt->fetchAll();
        
        // Get pass rules
        $passRulesStmt = $this->db->prepare("SELECT * FROM pass_rules LIMIT 1");
        $passRulesStmt->execute();
        $passRules = $passRulesStmt->fetch();
        
        $results = [];
        
        // Check each term visibility
        for ($termId = 1; $termId <= 2; $termId++) {
            $termVisible = $termId === 1 ? $visibility['first_term_visible'] : $visibility['second_term_visible'];
            $publishAt = $termId === 1 ? $visibility['first_term_publish_at'] : $visibility['second_term_publish_at'];
            
            // Check if term should be visible
            if (!$termVisible || ($publishAt && strtotime($publishAt) > time())) {
                continue;
            }
            
            // Get term results
            $termResults = $this->getTermResults($studentId, $yearId, $gradeId, $termId, $displayOptions, $bands, $passRules);
            
            if (!empty($termResults['subjects'])) {
                $results[$termId] = $termResults;
            }
        }
        
        return $this->response->success([
            'results' => $results,
            'display_options' => $displayOptions,
            'bands' => $bands
        ]);
    }
    
    private function getTermResults(int $studentId, int $yearId, int $gradeId, int $termId, array $displayOptions, array $bands, array $passRules): array
    {
        // Get exam results for this term
        $stmt = $this->db->prepare(
            "SELECT s.name as subject_name, er.score, e.max_score,
                    ROUND((er.score / e.max_score * 100), ?) as percentage
             FROM exam_results er
             JOIN exams e ON er.exam_id = e.id
             JOIN subjects s ON e.subject_id = s.id
             WHERE er.student_id = ? 
             AND e.year_id = ? 
             AND e.term_id = ? 
             AND e.grade_id = ?
             AND e.enabled = 1
             ORDER BY s.name"
        );
        
        $stmt->execute([$displayOptions['percent_rounding'], $studentId, $yearId, $termId, $gradeId]);
        $examResults = $stmt->fetchAll();
        
        if (empty($examResults)) {
            return [];
        }
        
        $subjects = [];
        $totalScore = 0;
        $totalMaxScore = 0;
        $passedSubjects = 0;
        
        foreach ($examResults as $result) {
            $percentage = $result['percentage'];
            $band = $this->getBandForPercentage($percentage, $bands);
            
            $subject = [
                'subject' => $result['subject_name'],
                'percentage' => $percentage,
                'band_label' => $band['label'] ?? 'Unknown',
                'band_color' => $band['color_hex'] ?? '#666666'
            ];
            
            // Add score if display mode includes it
            if (in_array($displayOptions['display_mode'], ['color_score', 'color_score_percent'])) {
                $subject['score'] = $result['score'];
                $subject['max_score'] = $result['max_score'];
            }
            
            // Check if subject passed
            if ($this->isSubjectPassed($percentage, $gradeId, $result['subject_name'], $passRules)) {
                $passedSubjects++;
            }
            
            $subjects[] = $subject;
            $totalScore += $result['score'];
            $totalMaxScore += $result['max_score'];
        }
        
        // Calculate overall statistics
        $averagePercent = $totalMaxScore > 0 ? round(($totalScore / $totalMaxScore) * 100, $displayOptions['percent_rounding']) : 0;
        $overallBand = $this->getBandForPercentage($averagePercent, $bands);
        
        // Determine pass/fail status
        $termPassed = $this->isTermPassed($passedSubjects, count($subjects), $averagePercent, $passRules);
        
        return [
            'subjects' => $subjects,
            'total_score' => $displayOptions['display_mode'] !== 'color_only' ? $totalScore : null,
            'total_max_score' => $displayOptions['display_mode'] !== 'color_only' ? $totalMaxScore : null,
            'average_percent' => $averagePercent,
            'term_pass_fail' => $termPassed ? 'Pass' : 'Fail',
            'term_grade_label' => $overallBand['label'] ?? 'Unknown',
            'term_grade_color' => $overallBand['color_hex'] ?? '#666666'
        ];
    }
    
    private function getBandForPercentage(float $percentage, array $bands): ?array
    {
        foreach ($bands as $band) {
            if ($percentage >= $band['start_percent'] && $percentage <= $band['end_percent']) {
                return $band;
            }
        }
        return null;
    }
    
    private function isSubjectPassed(float $percentage, int $gradeId, string $subjectName, array $passRules): bool
    {
        if ($passRules['per_subject_threshold_mode'] === 'fixed') {
            $minPercent = $passRules['subject_pass_min_percent'] ?? 50; // Default to 50% if not set
            return $percentage >= $minPercent;
        }
        
        // Use grade-subject minimum
        $stmt = $this->db->prepare(
            "SELECT gsr.min_score, gsr.max_score 
             FROM grade_subject_rules gsr
             JOIN subjects s ON gsr.subject_id = s.id
             WHERE gsr.grade_id = ? AND s.name = ? AND gsr.enabled = 1"
        );
        $stmt->execute([$gradeId, $subjectName]);
        $rule = $stmt->fetch();
        
        if ($rule) {
            $minPercent = ($rule['min_score'] / $rule['max_score']) * 100;
            return $percentage >= $minPercent;
        }
        
        return false;
    }
    
    private function isTermPassed(int $passedSubjects, int $totalSubjects, float $averagePercent, array $passRules): bool
    {
        $allSubjectsPassed = $passedSubjects === $totalSubjects;
        $overallPassed = true;
        
        if ($passRules['overall_pass_rule'] === 'avgAtLeast') {
            $overallPassed = $averagePercent >= $passRules['overall_pass_min_percent'];
        }
        
        switch ($passRules['pass_condition']) {
            case 'allSubjectsOrOverall':
                return $allSubjectsPassed || $overallPassed;
            case 'allSubjectsAndOverall':
                return $allSubjectsPassed && $overallPassed;
            case 'allSubjectsOnly':
                return $allSubjectsPassed;
            case 'overallOnly':
                return $overallPassed;
            default:
                return $allSubjectsPassed;
        }
    }
}