<?php

namespace App\Controllers;

use App\Core\Controller;
use App\Core\Database;
use App\Core\Response;
use App\Services\AuthService;
use PDO;

class SubjectController extends Controller
{
    private PDO $db;
    private AuthService $authService;
    
    public function __construct($request)
    {
        parent::__construct($request);
        $this->db = Database::getConnection();
        $this->authService = new AuthService();
    }
    
    public function index(): Response
    {
        $pagination = $this->getPaginationParams();
        $sort = $this->getSortParams(['id', 'name', 'enabled', 'created_at']);
        
        $whereConditions = ['1=1'];
        $params = [];
        
        // Filters
        if ($name = $this->request->get('name')) {
            $whereConditions[] = 'name LIKE ?';
            $params[] = "%{$name}%";
        }
        
        if (isset($_GET['enabled']) && $_GET['enabled'] !== '') {
            $whereConditions[] = 'enabled = ?';
            $params[] = (int)$this->request->get('enabled');
        }
        
        $whereClause = implode(' AND ', $whereConditions);
        $orderClause = $sort['sortBy'] ? "ORDER BY {$sort['sortBy']} {$sort['sortDir']}" : 'ORDER BY name ASC';
        
        // Get total count
        $countStmt = $this->db->prepare("SELECT COUNT(*) FROM subjects WHERE {$whereClause}");
        $countStmt->execute($params);
        $totalRows = $countStmt->fetchColumn();
        
        // Get results
        $sql = "SELECT id, name, enabled, created_at 
                FROM subjects 
                WHERE {$whereClause} 
                {$orderClause} 
                LIMIT ? OFFSET ?";
        
        $stmt = $this->db->prepare($sql);
        $stmt->execute([...$params, $pagination['perPage'], $pagination['offset']]);
        $subjects = $stmt->fetchAll();
        
        $meta = [
            'page' => $pagination['page'],
            'per_page' => $pagination['perPage'],
            'total_rows' => $totalRows,
            'total_pages' => ceil($totalRows / $pagination['perPage']),
            'sort_by' => $sort['sortBy'],
            'sort_dir' => $sort['sortDir']
        ];
        
        return $this->response->paginated($subjects, $meta);
    }
    
    public function show(string $id): Response
    {
        $stmt = $this->db->prepare("SELECT * FROM subjects WHERE id = ?");
        $stmt->execute([$id]);
        $subject = $stmt->fetch();
        
        if (!$subject) {
            return $this->response->error('Subject not found', 404);
        }
        
        return $this->response->success($subject);
    }
    
    public function store(): Response
    {
        $data = $this->validate([
            'name' => 'required|min:2|max:100',
            'enabled' => 'required'
        ]);
        
        // Check for duplicate name
        $stmt = $this->db->prepare("SELECT id FROM subjects WHERE name = ?");
        $stmt->execute([$data['name']]);
        if ($stmt->fetch()) {
            return $this->response->error('Subject name already exists', 422);
        }
        
        try {
            $stmt = $this->db->prepare(
                "INSERT INTO subjects (name, enabled) VALUES (?, ?)"
            );
            
            $stmt->execute([
                $data['name'],
                (int)$data['enabled']
            ]);
            
            $subjectId = $this->db->lastInsertId();
            
            // Get created subject
            $stmt = $this->db->prepare("SELECT * FROM subjects WHERE id = ?");
            $stmt->execute([$subjectId]);
            $subject = $stmt->fetch();
            
            // Log audit event
            if (isset($this->request->user)) {
                $this->authService->logAuditEvent([
                    'actor_id' => $this->request->user['user_id'],
                    'actor_type' => 'admin',
                    'role' => $this->request->user['role'],
                    'action' => 'create',
                    'entity' => 'subjects',
                    'entity_id' => $subjectId,
                    'after_data' => $subject,
                    'ip_address' => $this->request->getClientIp(),
                    'user_agent' => $this->request->getUserAgent()
                ]);
            }
            
            return $this->response->success($subject, 'Subject created successfully');
            
        } catch (\Exception $e) {
            return $this->response->error('Failed to create subject: ' . $e->getMessage(), 500);
        }
    }
    
    public function update(string $id): Response
    {
        $data = $this->validate([
            'name' => 'required|min:2|max:100',
            'enabled' => 'required'
        ]);
        
        // Get existing subject for audit
        $stmt = $this->db->prepare("SELECT * FROM subjects WHERE id = ?");
        $stmt->execute([$id]);
        $oldSubject = $stmt->fetch();
        
        if (!$oldSubject) {
            return $this->response->error('Subject not found', 404);
        }
        
        // Check for duplicate name (excluding current subject)
        $stmt = $this->db->prepare("SELECT id FROM subjects WHERE name = ? AND id != ?");
        $stmt->execute([$data['name'], $id]);
        if ($stmt->fetch()) {
            return $this->response->error('Subject name already exists', 422);
        }
        
        try {
            $stmt = $this->db->prepare(
                "UPDATE subjects SET name = ?, enabled = ? WHERE id = ?"
            );
            
            $stmt->execute([
                $data['name'],
                (int)$data['enabled'],
                $id
            ]);
            
            // Get updated subject
            $stmt = $this->db->prepare("SELECT * FROM subjects WHERE id = ?");
            $stmt->execute([$id]);
            $newSubject = $stmt->fetch();
            
            // Log audit event
            if (isset($this->request->user)) {
                $this->authService->logAuditEvent([
                    'actor_id' => $this->request->user['user_id'],
                    'actor_type' => 'admin',
                    'role' => $this->request->user['role'],
                    'action' => 'update',
                    'entity' => 'subjects',
                    'entity_id' => $id,
                    'before_data' => $oldSubject,
                    'after_data' => $newSubject,
                    'ip_address' => $this->request->getClientIp(),
                    'user_agent' => $this->request->getUserAgent()
                ]);
            }
            
            return $this->response->success($newSubject, 'Subject updated successfully');
            
        } catch (\Exception $e) {
            return $this->response->error('Failed to update subject: ' . $e->getMessage(), 500);
        }
    }
    
    public function destroy(string $id): Response
    {
        // Check for dependent exams
        $stmt = $this->db->prepare("SELECT COUNT(*) FROM exams WHERE subject_id = ?");
        $stmt->execute([$id]);
        $examCount = $stmt->fetchColumn();
        
        if ($examCount > 0) {
            return $this->response->error('Cannot delete subject with existing exams', 422);
        }
        
        // Check for dependent grade subject rules
        $stmt = $this->db->prepare("SELECT COUNT(*) FROM grade_subject_rules WHERE subject_id = ?");
        $stmt->execute([$id]);
        $ruleCount = $stmt->fetchColumn();
        
        if ($ruleCount > 0) {
            return $this->response->error('Cannot delete subject with existing grade subject rules', 422);
        }
        
        // Get subject for audit
        $stmt = $this->db->prepare("SELECT * FROM subjects WHERE id = ?");
        $stmt->execute([$id]);
        $subject = $stmt->fetch();
        
        if (!$subject) {
            return $this->response->error('Subject not found', 404);
        }
        
        try {
            $stmt = $this->db->prepare("DELETE FROM subjects WHERE id = ?");
            $stmt->execute([$id]);
            
            // Log audit event
            if (isset($this->request->user)) {
                $this->authService->logAuditEvent([
                    'actor_id' => $this->request->user['user_id'],
                    'actor_type' => 'admin',
                    'role' => $this->request->user['role'],
                    'action' => 'delete',
                    'entity' => 'subjects',
                    'entity_id' => $id,
                    'before_data' => $subject,
                    'ip_address' => $this->request->getClientIp(),
                    'user_agent' => $this->request->getUserAgent()
                ]);
            }
            
            return $this->response->success(null, 'Subject deleted successfully');
            
        } catch (\Exception $e) {
            return $this->response->error('Failed to delete subject: ' . $e->getMessage(), 500);
        }
    }
}