<?php

namespace App\Controllers;

use App\Core\Controller;
use App\Core\Database;
use App\Core\Response;
use App\Services\AuthService;
use PDO;

class TermController extends Controller
{
    private PDO $db;
    private AuthService $authService;
    
    public function __construct($request)
    {
        parent::__construct($request);
        $this->db = Database::getConnection();
        $this->authService = new AuthService();
    }
    
    public function index(): Response
    {
        $pagination = $this->getPaginationParams();
        $sort = $this->getSortParams(['id', 'name', 'sort_order', 'visible_to_students', 'created_at']);
        
        $whereConditions = ['1=1'];
        $params = [];
        
        // Filters
        if ($name = $this->request->get('name')) {
            $whereConditions[] = 'name LIKE ?';
            $params[] = "%{$name}%";
        }
        
        if (isset($_GET['visible_to_students']) && $_GET['visible_to_students'] !== '') {
            $whereConditions[] = 'visible_to_students = ?';
            $params[] = (int)$this->request->get('visible_to_students');
        }
        
        $whereClause = implode(' AND ', $whereConditions);
        $orderClause = $sort['sortBy'] ? "ORDER BY {$sort['sortBy']} {$sort['sortDir']}" : 'ORDER BY sort_order ASC';
        
        // Get total count
        $countStmt = $this->db->prepare("SELECT COUNT(*) FROM terms WHERE {$whereClause}");
        $countStmt->execute($params);
        $totalRows = $countStmt->fetchColumn();
        
        // Get results
        $sql = "SELECT id, name, sort_order, visible_to_students, publish_at, created_at 
                FROM terms 
                WHERE {$whereClause} 
                {$orderClause} 
                LIMIT ? OFFSET ?";
        
        $stmt = $this->db->prepare($sql);
        $stmt->execute([...$params, $pagination['perPage'], $pagination['offset']]);
        $terms = $stmt->fetchAll();
        
        $meta = [
            'page' => $pagination['page'],
            'per_page' => $pagination['perPage'],
            'total_rows' => $totalRows,
            'total_pages' => ceil($totalRows / $pagination['perPage']),
            'sort_by' => $sort['sortBy'],
            'sort_dir' => $sort['sortDir']
        ];
        
        return $this->response->paginated($terms, $meta);
    }
    
    public function show(string $id): Response
    {
        $stmt = $this->db->prepare("SELECT * FROM terms WHERE id = ?");
        $stmt->execute([$id]);
        $term = $stmt->fetch();
        
        if (!$term) {
            return $this->response->error('Term not found', 404);
        }
        
        return $this->response->success($term);
    }
    
    public function store(): Response
    {
        $data = $this->validate([
            'name' => 'required|min:2|max:50',
            'sort_order' => 'required|numeric',
            'visible_to_students' => 'required',
            'publish_at' => 'max:19'
        ]);
        
        // Check for duplicate name
        $stmt = $this->db->prepare("SELECT id FROM terms WHERE name = ?");
        $stmt->execute([$data['name']]);
        if ($stmt->fetch()) {
            return $this->response->error('Term name already exists', 422);
        }
        
        // Check for duplicate sort order
        $stmt = $this->db->prepare("SELECT id FROM terms WHERE sort_order = ?");
        $stmt->execute([$data['sort_order']]);
        if ($stmt->fetch()) {
            return $this->response->error('Sort order already exists', 422);
        }
        
        try {
            $stmt = $this->db->prepare(
                "INSERT INTO terms (name, sort_order, visible_to_students, publish_at) VALUES (?, ?, ?, ?)"
            );
            
            $stmt->execute([
                $data['name'],
                $data['sort_order'],
                (int)$data['visible_to_students'],
                $data['publish_at'] ?: null
            ]);
            
            $termId = $this->db->lastInsertId();
            
            // Get created term
            $stmt = $this->db->prepare("SELECT * FROM terms WHERE id = ?");
            $stmt->execute([$termId]);
            $term = $stmt->fetch();
            
            // Log audit event
            if (isset($this->request->user)) {
                $this->authService->logAuditEvent([
                    'actor_id' => $this->request->user['user_id'],
                    'actor_type' => 'admin',
                    'role' => $this->request->user['role'],
                    'action' => 'create',
                    'entity' => 'terms',
                    'entity_id' => $termId,
                    'after_data' => $term,
                    'ip_address' => $this->request->getClientIp(),
                    'user_agent' => $this->request->getUserAgent()
                ]);
            }
            
            return $this->response->success($term, 'Term created successfully');
            
        } catch (\Exception $e) {
            return $this->response->error('Failed to create term: ' . $e->getMessage(), 500);
        }
    }
    
    public function update(string $id): Response
    {
        $data = $this->validate([
            'name' => 'required|min:2|max:50',
            'sort_order' => 'required|numeric',
            'visible_to_students' => 'required',
            'publish_at' => 'max:19'
        ]);
        
        // Get existing term for audit
        $stmt = $this->db->prepare("SELECT * FROM terms WHERE id = ?");
        $stmt->execute([$id]);
        $oldTerm = $stmt->fetch();
        
        if (!$oldTerm) {
            return $this->response->error('Term not found', 404);
        }
        
        // Check for duplicate name (excluding current term)
        $stmt = $this->db->prepare("SELECT id FROM terms WHERE name = ? AND id != ?");
        $stmt->execute([$data['name'], $id]);
        if ($stmt->fetch()) {
            return $this->response->error('Term name already exists', 422);
        }
        
        // Check for duplicate sort order (excluding current term)
        $stmt = $this->db->prepare("SELECT id FROM terms WHERE sort_order = ? AND id != ?");
        $stmt->execute([$data['sort_order'], $id]);
        if ($stmt->fetch()) {
            return $this->response->error('Sort order already exists', 422);
        }
        
        try {
            $stmt = $this->db->prepare(
                "UPDATE terms SET name = ?, sort_order = ?, visible_to_students = ?, publish_at = ? WHERE id = ?"
            );
            
            $stmt->execute([
                $data['name'],
                $data['sort_order'],
                (int)$data['visible_to_students'],
                $data['publish_at'] ?: null,
                $id
            ]);
            
            // Get updated term
            $stmt = $this->db->prepare("SELECT * FROM terms WHERE id = ?");
            $stmt->execute([$id]);
            $newTerm = $stmt->fetch();
            
            // Log audit event
            if (isset($this->request->user)) {
                $this->authService->logAuditEvent([
                    'actor_id' => $this->request->user['user_id'],
                    'actor_type' => 'admin',
                    'role' => $this->request->user['role'],
                    'action' => 'update',
                    'entity' => 'terms',
                    'entity_id' => $id,
                    'before_data' => $oldTerm,
                    'after_data' => $newTerm,
                    'ip_address' => $this->request->getClientIp(),
                    'user_agent' => $this->request->getUserAgent()
                ]);
            }
            
            return $this->response->success($newTerm, 'Term updated successfully');
            
        } catch (\Exception $e) {
            return $this->response->error('Failed to update term: ' . $e->getMessage(), 500);
        }
    }
    
    public function destroy(string $id): Response
    {
        // Check for dependent exams
        $stmt = $this->db->prepare("SELECT COUNT(*) FROM exams WHERE term_id = ?");
        $stmt->execute([$id]);
        $examCount = $stmt->fetchColumn();
        
        if ($examCount > 0) {
            return $this->response->error('Cannot delete term with existing exams', 422);
        }
        
        // Get term for audit
        $stmt = $this->db->prepare("SELECT * FROM terms WHERE id = ?");
        $stmt->execute([$id]);
        $term = $stmt->fetch();
        
        if (!$term) {
            return $this->response->error('Term not found', 404);
        }
        
        try {
            $stmt = $this->db->prepare("DELETE FROM terms WHERE id = ?");
            $stmt->execute([$id]);
            
            // Log audit event
            if (isset($this->request->user)) {
                $this->authService->logAuditEvent([
                    'actor_id' => $this->request->user['user_id'],
                    'actor_type' => 'admin',
                    'role' => $this->request->user['role'],
                    'action' => 'delete',
                    'entity' => 'terms',
                    'entity_id' => $id,
                    'before_data' => $term,
                    'ip_address' => $this->request->getClientIp(),
                    'user_agent' => $this->request->getUserAgent()
                ]);
            }
            
            return $this->response->success(null, 'Term deleted successfully');
            
        } catch (\Exception $e) {
            return $this->response->error('Failed to delete term: ' . $e->getMessage(), 500);
        }
    }
}