<?php

namespace App\Controllers;

use App\Core\Controller;
use App\Core\Database;
use App\Core\Response;
use App\Services\AuthService;
use PDO;

class UserController extends Controller
{
    private PDO $db;
    private AuthService $authService;
    
    public function __construct($request)
    {
        parent::__construct($request);
        $this->db = Database::getConnection();
        $this->authService = new AuthService();
    }
    
    public function index(): Response
    {
        // Check if user has permission to manage users (admin only)
        if (!$this->authService->hasPermission($this->request->user['role'], 'read', 'users')) {
            return $this->response->error('Insufficient permissions', 403);
        }
        
        $pagination = $this->getPaginationParams();
        $sort = $this->getSortParams(['id', 'username', 'full_name', 'role', 'enabled', 'last_login', 'created_at']);
        
        // Build filters
        $whereConditions = ['1=1'];
        $params = [];
        
        if ($username = $this->request->get('username')) {
            $whereConditions[] = 'username LIKE ?';
            $params[] = "%{$username}%";
        }
        
        if ($fullName = $this->request->get('full_name')) {
            $whereConditions[] = 'full_name LIKE ?';
            $params[] = "%{$fullName}%";
        }
        
        if ($role = $this->request->get('role')) {
            $whereConditions[] = 'role = ?';
            $params[] = $role;
        }
        
        if ($enabled = $this->request->get('enabled')) {
            $whereConditions[] = 'enabled = ?';
            $params[] = (int)$enabled;
        }
        
        // General search
        if ($search = $this->request->get('q')) {
            $whereConditions[] = '(username LIKE ? OR full_name LIKE ?)';
            $searchTerm = "%{$search}%";
            $params = array_merge($params, [$searchTerm, $searchTerm]);
        }
        
        $whereClause = implode(' AND ', $whereConditions);
        $orderClause = $sort['sortBy'] ? "ORDER BY {$sort['sortBy']} {$sort['sortDir']}" : 'ORDER BY id DESC';
        
        // Get total count
        $countStmt = $this->db->prepare("SELECT COUNT(*) FROM admin_users WHERE {$whereClause}");
        $countStmt->execute($params);
        $totalRows = $countStmt->fetchColumn();
        
        // Get paginated results
        $sql = "SELECT id, username, role, full_name, enabled, last_login, created_at, updated_at
                FROM admin_users 
                WHERE {$whereClause}
                {$orderClause}
                LIMIT ? OFFSET ?";
        
        $stmt = $this->db->prepare($sql);
        $stmt->execute([...$params, $pagination['perPage'], $pagination['offset']]);
        $users = $stmt->fetchAll();
        
        $meta = [
            'page' => $pagination['page'],
            'per_page' => $pagination['perPage'],
            'total_rows' => $totalRows,
            'total_pages' => ceil($totalRows / $pagination['perPage']),
            'sort_by' => $sort['sortBy'],
            'sort_dir' => $sort['sortDir']
        ];
        
        return $this->response->paginated($users, $meta);
    }
    
    public function show($id): Response
    {
        // Check permissions
        if (!$this->authService->hasPermission($this->request->user['role'], 'read', 'users')) {
            return $this->response->error('Insufficient permissions', 403);
        }
        
        $stmt = $this->db->prepare("
            SELECT id, username, role, full_name, enabled, last_login, created_at, updated_at
            FROM admin_users 
            WHERE id = ?
        ");
        $stmt->execute([$id]);
        $user = $stmt->fetch();
        
        if (!$user) {
            return $this->response->error('User not found', 404);
        }
        
        return $this->response->success($user);
    }
    
    public function store(): Response
    {
        // Check permissions
        if (!$this->authService->hasPermission($this->request->user['role'], 'create', 'users')) {
            return $this->response->error('Insufficient permissions', 403);
        }
        
        $data = $this->validate([
            'username' => 'required|min:3|max:50',
            'password' => 'required|min:6',
            'role' => 'required|in:admin,registrar,teacher,accountant',
            'full_name' => 'required|min:2|max:200',
            'enabled' => 'boolean'
        ]);
        
        try {
            // Check if username already exists
            $stmt = $this->db->prepare("SELECT id FROM admin_users WHERE username = ?");
            $stmt->execute([$data['username']]);
            if ($stmt->fetch()) {
                return $this->response->error('Username already exists', 422);
            }
            
            // Hash password
            $passwordHash = password_hash($data['password'], PASSWORD_DEFAULT);
            
            $stmt = $this->db->prepare("
                INSERT INTO admin_users (username, password_hash, role, full_name, enabled) 
                VALUES (?, ?, ?, ?, ?)
            ");
            
            $stmt->execute([
                $data['username'],
                $passwordHash,
                $data['role'],
                $data['full_name'],
                $data['enabled'] ?? true
            ]);
            
            $userId = $this->db->lastInsertId();
            
            // Log audit event
            $this->authService->logAuditEvent([
                'actor_id' => $this->request->user['user_id'],
                'actor_type' => 'admin',
                'role' => $this->request->user['role'],
                'action' => 'create',
                'entity' => 'admin_user',
                'entity_id' => $userId,
                'after_data' => array_merge($data, ['id' => $userId]),
                'ip_address' => $this->request->getClientIp(),
                'user_agent' => $this->request->getUserAgent()
            ]);
            
            return $this->response->success(['id' => $userId], 'User created successfully', 201);
            
        } catch (\Exception $e) {
            return $this->response->error('Failed to create user: ' . $e->getMessage(), 500);
        }
    }
    
    public function update($id): Response
    {
        // Check permissions
        if (!$this->authService->hasPermission($this->request->user['role'], 'update', 'users')) {
            return $this->response->error('Insufficient permissions', 403);
        }
        
        $data = $this->validate([
            'username' => 'sometimes|min:3|max:50',
            'password' => 'sometimes|min:6',
            'role' => 'sometimes|in:admin,registrar,teacher,accountant',
            'full_name' => 'sometimes|min:2|max:200',
            'enabled' => 'sometimes|boolean'
        ]);
        
        try {
            // Get existing user
            $stmt = $this->db->prepare("SELECT * FROM admin_users WHERE id = ?");
            $stmt->execute([$id]);
            $existingUser = $stmt->fetch();
            
            if (!$existingUser) {
                return $this->response->error('User not found', 404);
            }
            
            // Prevent users from disabling themselves
            if ($id == $this->request->user['user_id'] && isset($data['enabled']) && !$data['enabled']) {
                return $this->response->error('You cannot disable your own account', 422);
            }
            
            // Check username uniqueness if changing
            if (isset($data['username']) && $data['username'] !== $existingUser['username']) {
                $stmt = $this->db->prepare("SELECT id FROM admin_users WHERE username = ? AND id != ?");
                $stmt->execute([$data['username'], $id]);
                if ($stmt->fetch()) {
                    return $this->response->error('Username already exists', 422);
                }
            }
            
            // Build update query
            $updateFields = [];
            $updateParams = [];
            
            foreach (['username', 'role', 'full_name', 'enabled'] as $field) {
                if (isset($data[$field])) {
                    $updateFields[] = "{$field} = ?";
                    $updateParams[] = $data[$field];
                }
            }
            
            // Handle password separately
            if (isset($data['password'])) {
                $updateFields[] = "password_hash = ?";
                $updateParams[] = password_hash($data['password'], PASSWORD_DEFAULT);
            }
            
            if (empty($updateFields)) {
                return $this->response->error('No valid fields to update', 422);
            }
            
            $updateParams[] = $id;
            
            $sql = "UPDATE admin_users SET " . implode(', ', $updateFields) . " WHERE id = ?";
            $stmt = $this->db->prepare($sql);
            $stmt->execute($updateParams);
            
            // Log audit event
            $this->authService->logAuditEvent([
                'actor_id' => $this->request->user['user_id'],
                'actor_type' => 'admin',
                'role' => $this->request->user['role'],
                'action' => 'update',
                'entity' => 'admin_user',
                'entity_id' => $id,
                'before_data' => $existingUser,
                'after_data' => $data,
                'ip_address' => $this->request->getClientIp(),
                'user_agent' => $this->request->getUserAgent()
            ]);
            
            return $this->response->success(null, 'User updated successfully');
            
        } catch (\Exception $e) {
            return $this->response->error('Failed to update user: ' . $e->getMessage(), 500);
        }
    }
    
    public function destroy($id): Response
    {
        // Check permissions
        if (!$this->authService->hasPermission($this->request->user['role'], 'delete', 'users')) {
            return $this->response->error('Insufficient permissions', 403);
        }
        
        try {
            // Get existing user
            $stmt = $this->db->prepare("SELECT * FROM admin_users WHERE id = ?");
            $stmt->execute([$id]);
            $user = $stmt->fetch();
            
            if (!$user) {
                return $this->response->error('User not found', 404);
            }
            
            // Prevent users from deleting themselves
            if ($id == $this->request->user['user_id']) {
                return $this->response->error('You cannot delete your own account', 422);
            }
            
            // Check if user has created any audit logs (has dependents)
            $stmt = $this->db->prepare("SELECT COUNT(*) FROM audit_log WHERE actor_id = ? AND actor_type = 'admin'");
            $stmt->execute([$id]);
            $auditCount = $stmt->fetchColumn();
            
            if ($auditCount > 0) {
                return $this->response->error('Cannot delete user with audit history. Consider disabling instead.', 422);
            }
            
            // Delete the user
            $stmt = $this->db->prepare("DELETE FROM admin_users WHERE id = ?");
            $stmt->execute([$id]);
            
            // Log audit event
            $this->authService->logAuditEvent([
                'actor_id' => $this->request->user['user_id'],
                'actor_type' => 'admin',
                'role' => $this->request->user['role'],
                'action' => 'delete',
                'entity' => 'admin_user',
                'entity_id' => $id,
                'before_data' => $user,
                'ip_address' => $this->request->getClientIp(),
                'user_agent' => $this->request->getUserAgent()
            ]);
            
            return $this->response->success(null, 'User deleted successfully');
            
        } catch (\Exception $e) {
            return $this->response->error('Failed to delete user: ' . $e->getMessage(), 500);
        }
    }
    
    public function toggleStatus($id): Response
    {
        // Check permissions
        if (!$this->authService->hasPermission($this->request->user['role'], 'update', 'users')) {
            return $this->response->error('Insufficient permissions', 403);
        }
        
        try {
            $stmt = $this->db->prepare("SELECT * FROM admin_users WHERE id = ?");
            $stmt->execute([$id]);
            $user = $stmt->fetch();
            
            if (!$user) {
                return $this->response->error('User not found', 404);
            }
            
            // Prevent users from disabling themselves
            if ($id == $this->request->user['user_id'] && $user['enabled']) {
                return $this->response->error('You cannot disable your own account', 422);
            }
            
            $newStatus = !$user['enabled'];
            
            $stmt = $this->db->prepare("UPDATE admin_users SET enabled = ? WHERE id = ?");
            $stmt->execute([$newStatus, $id]);
            
            // Log audit event
            $this->authService->logAuditEvent([
                'actor_id' => $this->request->user['user_id'],
                'actor_type' => 'admin',
                'role' => $this->request->user['role'],
                'action' => $newStatus ? 'enable' : 'disable',
                'entity' => 'admin_user',
                'entity_id' => $id,
                'before_data' => $user,
                'after_data' => ['enabled' => $newStatus],
                'ip_address' => $this->request->getClientIp(),
                'user_agent' => $this->request->getUserAgent()
            ]);
            
            $action = $newStatus ? 'enabled' : 'disabled';
            return $this->response->success(null, "User {$action} successfully");
            
        } catch (\Exception $e) {
            return $this->response->error('Failed to update user status: ' . $e->getMessage(), 500);
        }
    }
}