<?php

namespace App\Core;

abstract class Controller
{
    protected Request $request;
    protected Response $response;
    
    public function __construct(Request $request)
    {
        $this->request = $request;
        $this->response = new Response();
    }
    
    protected function validate(array $rules): array
    {
        $errors = [];
        $data = $this->request->all();
        
        foreach ($rules as $field => $rule) {
            $ruleArray = explode('|', $rule);
            $value = $data[$field] ?? null;
            
            foreach ($ruleArray as $singleRule) {
                $error = $this->validateField($field, $value, $singleRule);
                if ($error) {
                    $errors[$field][] = $error;
                }
            }
        }
        
        if (!empty($errors)) {
            throw new ValidationException($errors);
        }
        
        return $data;
    }
    
    private function validateField(string $field, mixed $value, string $rule): ?string
    {
        if ($rule === 'required' && (is_null($value) || $value === '')) {
            return "The $field field is required.";
        }
        
        if ($rule === 'email' && $value && !filter_var($value, FILTER_VALIDATE_EMAIL)) {
            return "The $field field must be a valid email.";
        }
        
        if (str_starts_with($rule, 'min:')) {
            $min = (int) substr($rule, 4);
            if ($value && strlen($value) < $min) {
                return "The $field field must be at least $min characters.";
            }
        }
        
        if (str_starts_with($rule, 'max:')) {
            $max = (int) substr($rule, 4);
            if ($value && strlen($value) > $max) {
                return "The $field field must not exceed $max characters.";
            }
        }
        
        if ($rule === 'numeric' && $value && !is_numeric($value)) {
            return "The $field field must be numeric.";
        }
        
        return null;
    }
    
    protected function getPaginationParams(): array
    {
        $page = max(1, (int) $this->request->get('page', 1));
        $perPage = min(500, max(10, (int) $this->request->get('per_page', 50)));
        $offset = ($page - 1) * $perPage;
        
        return compact('page', 'perPage', 'offset');
    }
    
    protected function getSortParams(array $allowed = []): array
    {
        $sortBy = $this->request->get('sort_by');
        $sortDir = $this->request->get('sort_dir', 'asc');
        
        if ($sortBy && !in_array($sortBy, $allowed)) {
            $sortBy = null;
        }
        
        if (!in_array($sortDir, ['asc', 'desc'])) {
            $sortDir = 'asc';
        }
        
        return compact('sortBy', 'sortDir');
    }
}

class ValidationException extends \Exception
{
    private array $errors;
    
    public function __construct(array $errors)
    {
        $this->errors = $errors;
        parent::__construct('Validation failed');
    }
    
    public function getErrors(): array
    {
        return $this->errors;
    }
}