<?php

namespace App\Core;

class Request
{
    private array $data;
    private array $files;
    protected array $headers;
    private string $method;
    private string $uri;
    public ?array $user = null;
    
    public function __construct()
    {
        $this->method = $_SERVER['REQUEST_METHOD'];
        // Use PATH_INFO if available, otherwise parse from REQUEST_URI
        $this->uri = $_SERVER['PATH_INFO'] ?? parse_url($_SERVER['REQUEST_URI'], PHP_URL_PATH);
        $this->headers = $this->getAllHeaders();
        $this->files = $_FILES;
        
        // Parse request data based on content type
        $this->parseRequestData();
    }
    
    private function parseRequestData(): void
    {
        $contentType = $this->getHeader('Content-Type') ?? '';
        
        if (str_contains($contentType, 'application/json') || $_SERVER['REQUEST_METHOD'] === 'POST') {
            $input = file_get_contents('php://input');
            $decoded = json_decode($input, true);
            if ($decoded !== null) {
                $this->data = $decoded;
            } else {
                $this->data = array_merge($_GET, $_POST);
            }
        } else {
            $this->data = array_merge($_GET, $_POST);
        }
    }
    
    private function getAllHeaders(): array
    {
        $headers = [];
        foreach ($_SERVER as $key => $value) {
            if (str_starts_with($key, 'HTTP_')) {
                // Convert to proper header case (e.g., HTTP_AUTHORIZATION -> Authorization)
                $header = str_replace(' ', '-', ucwords(str_replace('_', ' ', strtolower(substr($key, 5)))));
                $headers[$header] = $value;
            }
        }
        // Also check CONTENT_TYPE directly
        if (isset($_SERVER['CONTENT_TYPE'])) {
            $headers['Content-Type'] = $_SERVER['CONTENT_TYPE'];
        }
        return $headers;
    }
    
    public function get(string $key, mixed $default = null): mixed
    {
        return $this->data[$key] ?? $default;
    }
    
    public function all(): array
    {
        return $this->data;
    }
    
    public function has(string $key): bool
    {
        return isset($this->data[$key]);
    }
    
    public function getMethod(): string
    {
        return $this->method;
    }
    
    public function getUri(): string
    {
        return $this->uri;
    }
    
    public function getHeader(string $name): ?string
    {
        // Check direct name match first
        if (isset($this->headers[$name])) {
            return $this->headers[$name];
        }
        
        // Check case-insensitive match
        $normalizedName = strtolower($name);
        foreach ($this->headers as $headerName => $headerValue) {
            if (strtolower($headerName) === $normalizedName) {
                return $headerValue;
            }
        }
        
        return null;
    }
    
    public function getFile(string $key): ?array
    {
        return $this->files[$key] ?? null;
    }
    
    public function getBearerToken(): ?string
    {
        $authorization = $this->getHeader('Authorization') ?? $this->getHeader('AUTHORIZATION');
        if (!$authorization) {
            // Check server variable directly
            $authorization = $_SERVER['HTTP_AUTHORIZATION'] ?? '';
        }
        
        if ($authorization && str_starts_with($authorization, 'Bearer ')) {
            return substr($authorization, 7);
        }
        return null;
    }
    
    public function getClientIp(): string
    {
        return $_SERVER['HTTP_X_FORWARDED_FOR'] ?? 
               $_SERVER['HTTP_CLIENT_IP'] ?? 
               $_SERVER['REMOTE_ADDR'] ?? 
               'unknown';
    }
    
    public function getUserAgent(): string
    {
        return $_SERVER['HTTP_USER_AGENT'] ?? '';
    }
}