<?php

namespace App\Core;

class Response
{
    private int $statusCode = 200;
    private array $headers = [];
    private mixed $data = null;
    
    public function __construct(mixed $data = null, int $statusCode = 200)
    {
        if ($data !== null) {
            $this->json($data, $statusCode);
        }
    }
    
    public function json(mixed $data, int $statusCode = 200): self
    {
        $this->data = $data;
        $this->statusCode = $statusCode;
        $this->setHeader('Content-Type', 'application/json');
        return $this;
    }
    
    public function success(mixed $data = null, string $message = 'Success'): self
    {
        return $this->json([
            'success' => true,
            'message' => $message,
            'data' => $data
        ]);
    }
    
    public function error(string $message, int $statusCode = 400, mixed $errors = null): self
    {
        return $this->json([
            'success' => false,
            'message' => $message,
            'errors' => $errors
        ], $statusCode);
    }
    
    public function paginated(array $data, array $meta): self
    {
        return $this->json([
            'success' => true,
            'data' => $data,
            'meta' => $meta
        ]);
    }
    
    public function setHeader(string $name, string $value): self
    {
        $this->headers[$name] = $value;
        return $this;
    }
    
    public function setStatus(int $statusCode): self
    {
        $this->statusCode = $statusCode;
        return $this;
    }
    
    public function getData(): mixed
    {
        return $this->data;
    }
    
    public function send(): void
    {
        http_response_code($this->statusCode);
        
        foreach ($this->headers as $name => $value) {
            header("$name: $value");
        }
        
        if ($this->data !== null) {
            if (isset($this->headers['Content-Type']) && 
                $this->headers['Content-Type'] === 'application/json') {
                echo json_encode($this->data, JSON_UNESCAPED_UNICODE);
            } else {
                echo $this->data;
            }
        }
    }
    
    public static function redirect(string $url, int $statusCode = 302): void
    {
        http_response_code($statusCode);
        header("Location: $url");
        exit;
    }
}