<?php

namespace App\Core;

class Router
{
    private array $routes = [];
    private array $middlewares = [];
    
    public function get(string $path, callable|string $handler, array $middlewares = []): void
    {
        $this->addRoute('GET', $path, $handler, $middlewares);
    }
    
    public function post(string $path, callable|string $handler, array $middlewares = []): void
    {
        $this->addRoute('POST', $path, $handler, $middlewares);
    }
    
    public function put(string $path, callable|string $handler, array $middlewares = []): void
    {
        $this->addRoute('PUT', $path, $handler, $middlewares);
    }
    
    public function delete(string $path, callable|string $handler, array $middlewares = []): void
    {
        $this->addRoute('DELETE', $path, $handler, $middlewares);
    }
    
    private function addRoute(string $method, string $path, callable|string $handler, array $middlewares): void
    {
        $pattern = $this->convertToPattern($path);
        $this->routes[] = [
            'method' => $method,
            'pattern' => $pattern,
            'path' => $path,
            'handler' => $handler,
            'middlewares' => $middlewares
        ];
    }
    
    private function convertToPattern(string $path): string
    {
        // Convert {id} to named capture groups
        $pattern = preg_replace('/\{([^}]+)\}/', '(?P<$1>[^/]+)', $path);
        return '#^' . $pattern . '$#';
    }
    
    public function dispatch(string $method, string $uri): array
    {
        // Remove query string
        $uri = parse_url($uri, PHP_URL_PATH);
        
        foreach ($this->routes as $route) {
            if ($route['method'] === $method && preg_match($route['pattern'], $uri, $matches)) {
                // Extract parameters
                $params = array_filter($matches, 'is_string', ARRAY_FILTER_USE_KEY);
                
                return [
                    'handler' => $route['handler'],
                    'params' => $params,
                    'middlewares' => $route['middlewares']
                ];
            }
        }
        
        throw new \Exception('Route not found', 404);
    }
    
    public function addMiddleware(callable|string $middleware): void
    {
        $this->middlewares[] = $middleware;
    }
    
    public function getMiddlewares(): array
    {
        return $this->middlewares;
    }
}