<?php

namespace App\Middleware;

use App\Core\Request;
use App\Core\Response;
use App\Services\JWTService;

class AuthMiddleware
{
    private JWTService $jwtService;
    
    public function __construct()
    {
        $this->jwtService = new JWTService();
    }
    
    public function __invoke(Request $request): void
    {
        $token = $request->getBearerToken();
        
        if (!$token) {
            throw new \Exception('Authentication required', 401);
        }
        
        $payload = $this->jwtService->decode($token);
        
        if (!$payload) {
            throw new \Exception('Invalid or expired token', 401);
        }
        
        // Store user data in request for controllers to access
        $request->user = $payload;
    }
}

class AdminAuthMiddleware extends AuthMiddleware
{
    public function __invoke(Request $request): void
    {
        parent::__invoke($request);
        
        if ($request->user['type'] !== 'admin') {
            throw new \Exception('Admin access required', 403);
        }
    }
}

class StudentAuthMiddleware extends AuthMiddleware
{
    public function __invoke(Request $request): void
    {
        parent::__invoke($request);
        
        if ($request->user['type'] !== 'student') {
            throw new \Exception('Student access required', 403);
        }
    }
}

class PermissionMiddleware
{
    private string $entity;
    private string $action;
    
    public function __construct(string $entity, string $action)
    {
        $this->entity = $entity;
        $this->action = $action;
    }
    
    public function __invoke(Request $request): void
    {
        if (!isset($request->user)) {
            throw new \Exception('Authentication required', 401);
        }
        
        if ($request->user['type'] !== 'admin') {
            throw new \Exception('Admin access required', 403);
        }
        
        $authService = new \App\Services\AuthService();
        
        if (!$authService->hasPermission($request->user['role'], $this->action, $this->entity)) {
            throw new \Exception('Insufficient permissions', 403);
        }
    }
}