<?php

namespace App\Services;

use App\Core\Database;
use PDO;

class AuthService
{
    private PDO $db;
    
    public function __construct()
    {
        $this->db = Database::getConnection();
    }
    
    public function authenticateAdmin(string $username, string $password): ?array
    {
        $stmt = $this->db->prepare(
            "SELECT id, username, password_hash, role, full_name, enabled 
             FROM admin_users 
             WHERE username = ? AND enabled = 1"
        );
        
        $stmt->execute([$username]);
        $user = $stmt->fetch();
        
        if (!$user || !password_verify($password, $user['password_hash'])) {
            return null;
        }
        
        // Update last login
        $updateStmt = $this->db->prepare(
            "UPDATE admin_users SET last_login = NOW() WHERE id = ?"
        );
        $updateStmt->execute([$user['id']]);
        
        return [
            'id' => $user['id'],
            'username' => $user['username'],
            'role' => $user['role'],
            'full_name' => $user['full_name']
        ];
    }
    
    public function authenticateStudent(string $loginCode, string $nationalId): ?array
    {
        // Get current academic year
        $yearStmt = $this->db->prepare(
            "SELECT current_year_id FROM academic_visibility LIMIT 1"
        );
        $yearStmt->execute();
        $currentYear = $yearStmt->fetch();
        
        if (!$currentYear) {
            throw new \Exception('No current academic year configured');
        }
        
        // Find student with both login code and national ID
        $stmt = $this->db->prepare(
            "SELECT s.id, s.login_code, s.full_name, s.national_id, 
                    e.year_id, e.grade_id, e.class_id,
                    g.title as grade_title, c.name as class_code,
                    ay.name as year_name
             FROM students s
             JOIN enrollments e ON s.id = e.student_id
             JOIN grades g ON e.grade_id = g.id
             JOIN classes c ON e.class_id = c.id
             JOIN academic_years ay ON e.year_id = ay.id
             WHERE s.login_code = ? 
             AND s.national_id = ? 
             AND s.enabled = 1
             AND e.year_id = ?
             AND e.status = 'active'"
        );
        
        $stmt->execute([$loginCode, $nationalId, $currentYear['current_year_id']]);
        $student = $stmt->fetch();
        
        if (!$student) {
            return null;
        }
        
        return [
            'id' => $student['id'],
            'login_code' => $student['login_code'],
            'full_name' => $student['full_name'],
            'national_id' => $student['national_id'],
            'year_id' => $student['year_id'],
            'grade_id' => $student['grade_id'],
            'class_id' => $student['class_id'],
            'grade_title' => $student['grade_title'],
            'class_code' => $student['class_code'],
            'year_name' => $student['year_name']
        ];
    }
    
    public function generateLoginCode(): string
    {
        do {
            $code = str_pad((string) random_int(10000000, 99999999), 8, '0', STR_PAD_LEFT);
            
            // Check if code is unique
            $stmt = $this->db->prepare("SELECT id FROM students WHERE login_code = ?");
            $stmt->execute([$code]);
        } while ($stmt->fetch());
        
        return $code;
    }
    
    public function logAuditEvent(array $data): void
    {
        $stmt = $this->db->prepare(
            "INSERT INTO audit_log (actor_id, actor_type, role, action, entity, entity_id, 
                                   before_data, after_data, ip_address, user_agent) 
             VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?)"
        );
        
        $stmt->execute([
            $data['actor_id'] ?? null,
            $data['actor_type'],
            $data['role'] ?? null,
            $data['action'],
            $data['entity'],
            $data['entity_id'] ?? null,
            isset($data['before_data']) ? json_encode($data['before_data']) : null,
            isset($data['after_data']) ? json_encode($data['after_data']) : null,
            $data['ip_address'] ?? null,
            $data['user_agent'] ?? null
        ]);
    }
    
    public function hasPermission(string $role, string $action, string $entity): bool
    {
        $permissions = [
            'admin' => ['*'], // Admin can do everything
            'registrar' => [
                // Students & Enrollments
                'students.create', 'students.read', 'students.update', 'students.delete',
                'enrollments.create', 'enrollments.read', 'enrollments.update', 'enrollments.delete',
                // Master data needed for student management
                'academic_years.read', 'grades.read', 'classes.read', 'classes.create', 'classes.update', 'classes.delete',
                // Import capabilities
                'imports.create'
            ],
            'teacher' => [
                // Academic management
                'exams.create', 'exams.read', 'exams.update', 'exams.delete',
                'exam_results.create', 'exam_results.read', 'exam_results.update', 'exam_results.delete',
                // Need to see students and academic structure
                'students.read', 'grades.read', 'classes.read', 'subjects.read',
                'grade_subject_rules.read', 'grade_subject_rules.create', 'grade_subject_rules.update', 'grade_subject_rules.delete'
            ],
            'accountant' => [
                // Financial management
                'payments.create', 'payments.read', 'payments.update', 'payments.delete',
                'grade_costs.read', 'grade_costs.create', 'grade_costs.update', 'grade_costs.delete',
                // Need to see students and basic structure
                'students.read', 'grades.read', 'classes.read'
            ]
        ];
        
        $userPermissions = $permissions[$role] ?? [];
        
        // Check for wildcard permission
        if (in_array('*', $userPermissions)) {
            return true;
        }
        
        // Check specific permission
        $permission = "$entity.$action";
        return in_array($permission, $userPermissions);
    }
}