// Language Manager for Student Portal
class StudentLanguageManager {
    constructor() {
        this.currentLang = localStorage.getItem('student_language') || 'en';
        this.translations = {
            'en': translations_en,
            'ar': translations_ar
        };
        this.rtlLanguages = ['ar'];
        
        this.init();
    }
    
    init() {
        this.applyLanguage();
        this.setupLanguageSwitcher();
        
        // Ensure event listeners are working
        setTimeout(() => {
            this.ensureEventListeners();
        }, 100);
    }
    
    // Get translation for a key with parameter substitution
    t(key, params = {}) {
        let translation = this.translations[this.currentLang][key];
        if (!translation) {
            console.warn(`Translation missing for key: ${key}`);
            return key;
        }
        
        // Replace parameters like {name}
        for (const [param, value] of Object.entries(params)) {
            translation = translation.replace(new RegExp(`\\{${param}\\}`, 'g'), value);
        }
        
        return translation;
    }
    
    // Change language
    setLanguage(lang) {
        if (this.translations[lang]) {
            this.currentLang = lang;
            localStorage.setItem('student_language', lang);
            this.applyLanguage();
            this.translatePage();
            
            // Re-setup event listeners to ensure they still work
            this.ensureEventListeners();
            
            // Reload the current page content to apply translations
            if (window.studentApp && typeof window.studentApp.reloadCurrentTab === 'function') {
                setTimeout(() => {
                    window.studentApp.reloadCurrentTab();
                }, 200);
            }
        }
    }
    
    // Apply language settings (direction, fonts, etc.)
    applyLanguage() {
        const isRTL = this.rtlLanguages.includes(this.currentLang);
        const html = document.documentElement;
        
        // Set direction and language
        html.setAttribute('dir', isRTL ? 'rtl' : 'ltr');
        html.setAttribute('lang', this.currentLang);
        
        // Add/remove classes
        document.body.classList.toggle('rtl', isRTL);
        document.body.classList.toggle('arabic', this.currentLang === 'ar');
        
        // Update document title
        document.title = this.t('nav.student_portal') + ' - School Management System';
        
        // Load RTL styles if needed
        if (isRTL) {
            this.loadRTLStyles();
        } else {
            this.removeRTLStyles();
        }
    }
    
    // Load RTL styles for Arabic
    loadRTLStyles() {
        // Remove existing RTL styles first
        this.removeRTLStyles();
        
        // Add Bootstrap RTL
        const rtlBootstrap = document.createElement('link');
        rtlBootstrap.id = 'rtl-bootstrap';
        rtlBootstrap.rel = 'stylesheet';
        rtlBootstrap.href = 'https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.rtl.min.css';
        document.head.appendChild(rtlBootstrap);
        
        // Add custom RTL styles
        const customRTL = document.createElement('style');
        customRTL.id = 'custom-rtl-student';
        customRTL.textContent = `
            /* Arabic font styling */
            .arabic {
                font-family: 'Segoe UI', 'Tahoma', 'Noto Sans Arabic', 'Arabic UI Text', sans-serif;
                font-feature-settings: 'liga' 1;
            }
            
            .arabic h1, .arabic h2, .arabic h3, .arabic h4, .arabic h5, .arabic h6 {
                font-family: 'Segoe UI', 'Tahoma', 'Noto Sans Arabic', 'Arabic UI Text', sans-serif;
                font-weight: 600;
            }
            
            .arabic .navbar-brand,
            .arabic .btn,
            .arabic .nav-link,
            .arabic .form-label,
            .arabic .form-control {
                font-family: 'Segoe UI', 'Tahoma', 'Noto Sans Arabic', 'Arabic UI Text', sans-serif;
            }
            
            /* Minimal RTL adjustments - only what's necessary */
            html[dir="rtl"] .me-2 {
                margin-right: 0 !important;
                margin-left: 0.5rem !important;
            }
            
            html[dir="rtl"] .me-3 {
                margin-right: 0 !important;
                margin-left: 1rem !important;
            }
            
            html[dir="rtl"] .ms-auto {
                margin-left: 0 !important;
                margin-right: auto !important;
            }
            
            /* Preserve click functionality - don't interfere with nav elements */
            html[dir="rtl"] .nav-pills,
            html[dir="rtl"] .nav-pills .nav-link,
            html[dir="rtl"] .term-tab {
                pointer-events: auto !important;
                cursor: pointer !important;
                position: relative !important;
                z-index: 10 !important;
            }
            
            /* Keep elements clickable and centered */
            html[dir="rtl"] .text-center,
            html[dir="rtl"] .justify-content-center,
            html[dir="rtl"] .profile-section,
            html[dir="rtl"] .welcome-banner,
            html[dir="rtl"] .login-container {
                text-align: center !important;
            }
            
            /* Language switcher positioning - ALWAYS in same position */
            .language-switcher {
                position: fixed !important;
                top: 20px !important;
                right: 20px !important;
                left: auto !important;
                z-index: 9999 !important;
                background: rgba(255, 255, 255, 0.95) !important;
                backdrop-filter: blur(10px) !important;
                border-radius: 25px !important;
                padding: 10px 15px !important;
                box-shadow: 0 4px 20px rgba(0, 0, 0, 0.15) !important;
                border: 2px solid rgba(102, 126, 234, 0.3) !important;
                min-width: 120px !important;
            }
            
            /* Keep language switcher in SAME position even in RTL */
            html[dir="rtl"] .language-switcher {
                position: fixed !important;
                top: 20px !important;
                right: 20px !important;
                left: auto !important;
                z-index: 9999 !important;
            }
            
            .language-switcher select {
                border: none;
                background: transparent;
                font-size: 14px;
                font-weight: 500;
                color: #495057;
                cursor: pointer;
                font-family: inherit;
            }
            
            .language-switcher select:focus {
                outline: none;
                box-shadow: none;
            }
            
            /* Ensure proper text alignment */
            .rtl .text-center {
                text-align: center !important;
            }
            
            .rtl .text-start {
                text-align: right !important;
            }
            
            .rtl .text-end {
                text-align: left !important;
            }
            
            /* Fix flex direction for specific components */
            .rtl .d-flex.align-items-center {
                flex-direction: row-reverse;
            }
            
            .rtl .navbar-text.d-flex.align-items-center {
                flex-direction: row-reverse;
            }
            
            /* Welcome banner adjustments */
            .rtl .welcome-banner {
                text-align: center;
            }
            
            /* Profile adjustments */
            .rtl .profile-section {
                text-align: center;
            }
            
            .rtl .avatar-circle {
                margin: 0 auto;
            }
        `;
        document.head.appendChild(customRTL);
    }
    
    // Remove RTL styles
    removeRTLStyles() {
        const rtlBootstrap = document.getElementById('rtl-bootstrap');
        const customRTL = document.getElementById('custom-rtl-student');
        
        if (rtlBootstrap) rtlBootstrap.remove();
        if (customRTL) customRTL.remove();
        
        // Reset any RTL classes
        document.body.classList.remove('rtl', 'arabic');
    }
    
    // Setup language switcher
    setupLanguageSwitcher() {
        // Remove existing switcher
        const existing = document.getElementById('student-language-switcher');
        if (existing) existing.remove();
        
        // Create language switcher
        const switcher = document.createElement('div');
        switcher.id = 'student-language-switcher';
        switcher.className = 'language-switcher';
        
        // Force consistent positioning with inline styles as backup
        switcher.style.cssText = `
            position: fixed !important;
            top: 20px !important;
            right: 20px !important;
            left: auto !important;
            z-index: 9999 !important;
            background: rgba(255, 255, 255, 0.95) !important;
            backdrop-filter: blur(10px) !important;
            border-radius: 25px !important;
            padding: 10px 15px !important;
            box-shadow: 0 4px 20px rgba(0, 0, 0, 0.15) !important;
            border: 2px solid rgba(102, 126, 234, 0.3) !important;
            min-width: 120px !important;
        `;
        
        switcher.innerHTML = `
            <select onchange="studentLangManager.setLanguage(this.value)" style="border: none; background: transparent; font-size: 14px; font-weight: 500; color: #495057; cursor: pointer; font-family: inherit; outline: none;">
                <option value="en" ${this.currentLang === 'en' ? 'selected' : ''}>🇺🇸 English</option>
                <option value="ar" ${this.currentLang === 'ar' ? 'selected' : ''}>🇸🇦 العربية</option>
            </select>
        `;
        
        document.body.appendChild(switcher);
    }
    
    // Translate all elements with data-translate attribute
    translatePage() {
        // Translate elements with data-translate attribute
        const elements = document.querySelectorAll('[data-translate]');
        elements.forEach(element => {
            const key = element.getAttribute('data-translate');
            const params = element.hasAttribute('data-translate-params') 
                ? JSON.parse(element.getAttribute('data-translate-params')) 
                : {};
            element.textContent = this.t(key, params);
        });
        
        // Translate placeholders
        const placeholders = document.querySelectorAll('[data-translate-placeholder]');
        placeholders.forEach(element => {
            const key = element.getAttribute('data-translate-placeholder');
            element.setAttribute('placeholder', this.t(key));
        });
        
        // Translate titles
        const titles = document.querySelectorAll('[data-translate-title]');
        titles.forEach(element => {
            const key = element.getAttribute('data-translate-title');
            element.setAttribute('title', this.t(key));
        });
        
        // Translate aria-labels
        const ariaLabels = document.querySelectorAll('[data-translate-aria-label]');
        ariaLabels.forEach(element => {
            const key = element.getAttribute('data-translate-aria-label');
            element.setAttribute('aria-label', this.t(key));
        });
    }
    
    // Get current language
    getCurrentLanguage() {
        return this.currentLang;
    }
    
    // Check if current language is RTL
    isRTL() {
        return this.rtlLanguages.includes(this.currentLang);
    }
    
    // Format numbers for current locale
    formatNumber(number) {
        if (this.currentLang === 'ar') {
            // Convert to Arabic-Indic numerals
            return number.toString().replace(/[0-9]/g, function(w) {
                var arabicNumerals = ['٠','١','٢','٣','٤','٥','٦','٧','٨','٩'];
                return arabicNumerals[+w];
            });
        }
        return number.toString();
    }
    
    // Get appropriate date format for current language
    formatDate(dateString) {
        if (!dateString) return '';
        
        const date = new Date(dateString);
        const options = {
            year: 'numeric',
            month: 'long',
            day: 'numeric'
        };
        
        if (this.currentLang === 'ar') {
            return date.toLocaleDateString('ar-SA', options);
        }
        
        return date.toLocaleDateString('en-US', options);
    }
    
    // Ensure event listeners are working after language change
    ensureEventListeners() {
        // Re-bind tab click events
        const tabButtons = document.querySelectorAll('[data-tab]');
        tabButtons.forEach(button => {
            button.style.pointerEvents = 'auto';
            button.style.cursor = 'pointer';
            button.style.position = 'relative';
            button.style.zIndex = '10';
            
            // Make sure the button is clickable
            if (!button.hasAttribute('data-listener-attached')) {
                button.addEventListener('click', function(e) {
                    e.preventDefault();
                    e.stopPropagation();
                    const tab = this.getAttribute('data-tab');
                    if (window.studentApp && typeof window.studentApp.switchTab === 'function') {
                        window.studentApp.switchTab(tab);
                    }
                });
                button.setAttribute('data-listener-attached', 'true');
            }
        });
        
        // Ensure language switcher is also working and positioned correctly
        const languageSwitcherContainer = document.querySelector('.language-switcher');
        const languageSwitcher = document.querySelector('.language-switcher select');
        
        if (languageSwitcherContainer) {
            // Force consistent positioning regardless of RTL
            languageSwitcherContainer.style.cssText = `
                position: fixed !important;
                top: 20px !important;
                right: 20px !important;
                left: auto !important;
                z-index: 9999 !important;
                background: rgba(255, 255, 255, 0.95) !important;
                backdrop-filter: blur(10px) !important;
                border-radius: 25px !important;
                padding: 10px 15px !important;
                box-shadow: 0 4px 20px rgba(0, 0, 0, 0.15) !important;
                border: 2px solid rgba(102, 126, 234, 0.3) !important;
                min-width: 120px !important;
            `;
        }
        
        if (languageSwitcher) {
            languageSwitcher.style.pointerEvents = 'auto';
            languageSwitcher.style.cursor = 'pointer';
            languageSwitcher.style.zIndex = '9999';
        }
    }
}

// Initialize language manager
const studentLangManager = new StudentLanguageManager();