// Student Portal Application
class StudentApp {
    constructor() {
        // Auto-detect API base path based on environment
        this.apiBase = window.location.hostname.includes('localhost') || window.location.hostname.includes('127.0.0.1') 
            ? '/student1/api' 
            : '/api';
        this.token = localStorage.getItem('student_token');
        this.student = null;
        this.currentTab = 'profile';
        
        this.init();
    }
    
    init() {
        // Initialize language manager integration
        window.studentApp = this;
        
        // Load school settings first
        this.loadSchoolSettings();
        
        // Check if already logged in
        if (this.token) {
            this.verifyToken();
        } else {
            this.showLoginScreen();
        }
        
        this.bindEvents();
        
        // Apply initial translations
        setTimeout(() => {
            this.applyTranslations();
        }, 100);
    }
    
    bindEvents() {
        // Login form
        document.getElementById('loginForm').addEventListener('submit', (e) => {
            e.preventDefault();
            this.login();
        });
        
        // Tab navigation
        document.addEventListener('click', (e) => {
            if (e.target.matches('[data-tab]')) {
                const tab = e.target.getAttribute('data-tab');
                this.switchTab(tab);
            }
        });
        
        // Logout
        document.getElementById('logoutBtn').addEventListener('click', (e) => {
            e.preventDefault();
            this.logout();
        });
        
        // Format login code input
        document.getElementById('loginCode').addEventListener('input', (e) => {
            e.target.value = e.target.value.replace(/[^0-9]/g, '').substring(0, 8);
        });
    }
    
    async login() {
        const loginCode = document.getElementById('loginCode').value.trim();
        const nationalId = document.getElementById('nationalId').value.trim();
        const errorDiv = document.getElementById('loginError');
        
        // Validation
        if (!loginCode || !nationalId) {
            this.showError(errorDiv, this.t('login.error'));
            return;
        }
        
        if (loginCode.length !== 8) {
            this.showError(errorDiv, this.t('login.error'));
            return;
        }
        
        try {
            const response = await this.apiCall('POST', '/auth/student/login', {
                login_code: loginCode,
                national_id: nationalId
            });
            
            if (response.success) {
                this.token = response.data.token;
                this.student = response.data.student;
                localStorage.setItem('student_token', this.token);
                this.showMainApp();
            } else {
                this.showError(errorDiv, this.t('login.error'));
            }
        } catch (error) {
            this.showError(errorDiv, this.t('login.error'));
        }
    }
    
    async verifyToken() {
        try {
            const response = await this.apiCall('GET', '/auth/me');
            if (response.success && response.data.type === 'student') {
                this.student = response.data;
                this.showMainApp();
            } else {
                this.logout();
            }
        } catch (error) {
            this.logout();
        }
    }
    
    logout() {
        this.token = null;
        this.student = null;
        localStorage.removeItem('student_token');
        this.showLoginScreen();
    }
    
    showLoginScreen() {
        const loginScreen = document.getElementById('loginScreen');
        const mainApp = document.getElementById('mainApp');
        
        loginScreen.style.display = 'flex';
        loginScreen.style.visibility = 'visible';
        loginScreen.style.opacity = '1';
        
        mainApp.style.display = 'none';
        mainApp.style.visibility = 'hidden';
        mainApp.style.opacity = '0';
        
        // Clear form
        document.getElementById('loginCode').value = '';
        document.getElementById('nationalId').value = '';
        
        // Clear any error messages
        const errorDiv = document.getElementById('loginError');
        if (errorDiv) {
            errorDiv.style.display = 'none';
        }
    }
    
    showMainApp() {
        const loginScreen = document.getElementById('loginScreen');
        const mainApp = document.getElementById('mainApp');
        
        // Completely hide login screen
        loginScreen.style.display = 'none';
        loginScreen.style.visibility = 'hidden';
        loginScreen.style.opacity = '0';
        loginScreen.style.position = 'absolute';
        loginScreen.style.left = '-9999px';
        
        // Show main app
        mainApp.style.display = 'block';
        mainApp.style.visibility = 'visible';
        mainApp.style.opacity = '1';
        mainApp.style.position = 'relative';
        mainApp.style.left = 'auto';
        
        // Force a repaint
        mainApp.offsetHeight;
        
        // Update student info
        document.getElementById('studentInfo').textContent = this.student.full_name;
        
        // Show welcome banner
        const welcomeBanner = document.getElementById('welcomeBanner');
        const welcomeName = document.getElementById('welcomeName');
        if (welcomeName) {
            welcomeName.textContent = this.student.full_name.split(' ')[0]; // First name only
            
            // Update welcome message with translation
            const welcomeTitle = welcomeBanner.querySelector('h3');
            if (welcomeTitle) {
                welcomeTitle.innerHTML = `<i class="bi bi-emoji-smile me-2"></i>${this.t('welcome.title', {name: this.student.full_name.split(' ')[0]})}`;
            }
            const welcomeSubtitle = welcomeBanner.querySelector('p');
            if (welcomeSubtitle) {
                welcomeSubtitle.textContent = this.t('welcome.subtitle');
            }
            
            welcomeBanner.style.display = 'block';
            
            // Hide banner after 5 seconds
            setTimeout(() => {
                if (welcomeBanner) {
                    welcomeBanner.style.transition = 'all 0.5s ease';
                    welcomeBanner.style.opacity = '0';
                    setTimeout(() => {
                        welcomeBanner.style.display = 'none';
                    }, 500);
                }
            }, 5000);
        }
        
        // Load initial content
        this.switchTab('profile');
    }
    
    switchTab(tab) {
        // Update tab buttons
        document.querySelectorAll('.term-tab').forEach(btn => {
            btn.classList.remove('active');
        });
        document.querySelector(`[data-tab="${tab}"]`).classList.add('active');
        
        // Show/hide tab content
        document.querySelectorAll('.tab-content').forEach(content => {
            content.style.display = 'none';
        });
        document.getElementById(`${tab}Tab`).style.display = 'block';
        
        this.currentTab = tab;
        
        // Load tab content
        if (tab === 'profile') {
            this.loadProfile();
        } else if (tab === 'results') {
            this.loadResults();
        } else if (tab === 'stats') {
            this.loadStatistics();
        }
    }
    
    async loadProfile() {
        const loadingDiv = document.querySelector('#profileTab .loading');
        const contentDiv = document.getElementById('profileContent');
        
        loadingDiv.style.display = 'block';
        contentDiv.innerHTML = '';
        
        try {
            const response = await this.apiCall('GET', '/student/profile');
            
            if (response.success) {
                const profile = response.data;
                
                contentDiv.innerHTML = `
                    <div style="position: relative; z-index: 10;">
                        <div class="profile-avatar mb-4">
                            <div class="avatar-circle">
                                <i class="bi bi-person-fill"></i>
                            </div>
                            <div class="avatar-glow"></div>
                        </div>
                        <h1 class="profile-name fw-bold mb-2">${profile.full_name}</h1>
                        <div class="profile-subtitle mb-3">
                            <span class="grade-tag">
                                <i class="bi bi-mortarboard me-2"></i>${profile.grade_title}
                            </span>
                            <span class="class-tag">
                                <i class="bi bi-people me-2"></i>Class ${profile.class_code}
                            </span>
                        </div>
                        <div class="profile-stats-mini">
                            <div class="mini-stat">
                                <i class="bi bi-calendar-range"></i>
                                <span>${profile.year_name}</span>
                            </div>
                            <div class="mini-stat">
                                <i class="bi bi-shield-check"></i>
                                <span>${profile.enrollment_status}</span>
                            </div>
                        </div>
                    </div>
                `;
                
                // Show enhanced info cards
                const academicInfo = document.getElementById('academicInfo');
                academicInfo.innerHTML = `
                    <div class="profile-cards-grid">
                        <div class="profile-info-card personal-card">
                            <div class="card-header-modern">
                                <div class="card-icon personal-icon">
                                    <i class="bi bi-person-badge"></i>
                                </div>
                                <h5 class="card-title-modern">Personal Information</h5>
                            </div>
                            <div class="card-content-modern">
                                <div class="info-row-modern">
                                    <div class="info-label">
                                        <i class="bi bi-hash"></i>
                                        <span>Student ID</span>
                                    </div>
                                    <div class="info-value">${profile.id}</div>
                                </div>
                                <div class="info-row-modern">
                                    <div class="info-label">
                                        <i class="bi bi-credit-card"></i>
                                        <span>National ID</span>
                                    </div>
                                    <div class="info-value">${profile.national_id || 'Not provided'}</div>
                                </div>
                                <div class="info-row-modern">
                                    <div class="info-label">
                                        <i class="bi bi-gender-${profile.gender === 'M' ? 'male' : 'female'}"></i>
                                        <span>Gender</span>
                                    </div>
                                    <div class="info-value">${profile.gender === 'M' ? 'Male' : 'Female'}</div>
                                </div>
                                ${profile.birthdate ? `
                                <div class="info-row-modern">
                                    <div class="info-label">
                                        <i class="bi bi-cake"></i>
                                        <span>Birth Date</span>
                                    </div>
                                    <div class="info-value">${new Date(profile.birthdate).toLocaleDateString()}</div>
                                </div>
                                ` : ''}
                            </div>
                        </div>
                        
                        <div class="profile-info-card academic-card">
                            <div class="card-header-modern">
                                <div class="card-icon academic-icon">
                                    <i class="bi bi-book"></i>
                                </div>
                                <h5 class="card-title-modern">Academic Information</h5>
                            </div>
                            <div class="card-content-modern">
                                <div class="info-row-modern">
                                    <div class="info-label">
                                        <i class="bi bi-calendar-range"></i>
                                        <span>Academic Year</span>
                                    </div>
                                    <div class="info-value">${profile.year_name}</div>
                                </div>
                                <div class="info-row-modern">
                                    <div class="info-label">
                                        <i class="bi bi-mortarboard"></i>
                                        <span>Grade Level</span>
                                    </div>
                                    <div class="info-value">${profile.grade_title}</div>
                                </div>
                                <div class="info-row-modern">
                                    <div class="info-label">
                                        <i class="bi bi-people"></i>
                                        <span>Class Section</span>
                                    </div>
                                    <div class="info-value">Class ${profile.class_code}</div>
                                </div>
                                <div class="info-row-modern">
                                    <div class="info-label">
                                        <i class="bi bi-shield-check"></i>
                                        <span>Enrollment Status</span>
                                    </div>
                                    <div class="info-value">
                                        <span class="status-badge active-status">${profile.enrollment_status}</span>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                `;
                academicInfo.style.display = 'block';
            } else {
                contentDiv.innerHTML = `<div class="alert alert-danger">${response.message}</div>`;
            }
        } catch (error) {
            contentDiv.innerHTML = `<div class="alert alert-danger">Error loading profile</div>`;
        } finally {
            loadingDiv.style.display = 'none';
        }
    }
    
    async loadResults() {
        const loadingDiv = document.querySelector('#resultsTab .loading');
        const contentDiv = document.getElementById('resultsContent');
        
        loadingDiv.style.display = 'block';
        contentDiv.innerHTML = '';
        
        try {
            const response = await this.apiCall('GET', '/student/results');
            
            if (response.success) {
                const data = response.data;
                
                if (Object.keys(data.results).length === 0) {
                    contentDiv.innerHTML = `
                        <div class="empty-state">
                            <div class="empty-icon">
                                <i class="bi bi-graph-up"></i>
                            </div>
                            <h3 class="empty-title">No Results Available</h3>
                            <p class="empty-subtitle">Your academic results will appear here once they are published by your teachers.</p>
                            <div class="empty-decoration">
                                <div class="floating-shape shape-1"></div>
                                <div class="floating-shape shape-2"></div>
                                <div class="floating-shape shape-3"></div>
                            </div>
                        </div>
                    `;
                } else {
                    contentDiv.innerHTML = this.renderModernResults(data.results, data.display_options);
                }
            } else {
                contentDiv.innerHTML = `<div class="alert alert-danger">${response.message}</div>`;
            }
        } catch (error) {
            let errorMessage = 'Error loading results';
            let errorIcon = 'bi-exclamation-triangle';
            let errorClass = 'alert-danger';
            
            // Handle specific error types
            if (error.response) {
                if (error.response.status === 403) {
                    errorMessage = error.response.data?.message || 'Results are not available for your grade level at this time.';
                    errorIcon = 'bi-eye-slash';
                    errorClass = 'alert-warning';
                } else if (error.response.status === 402) {
                    errorMessage = error.response.data?.message || 'Payment is required to view results.';
                    errorIcon = 'bi-credit-card';
                    errorClass = 'alert-info';
                } else if (error.response.data?.message) {
                    errorMessage = error.response.data.message;
                }
            }
            
            contentDiv.innerHTML = `
                <div class="empty-state">
                    <div class="empty-icon">
                        <i class="${errorIcon}"></i>
                    </div>
                    <h3 class="empty-title">Results Not Available</h3>
                    <p class="empty-subtitle">${errorMessage}</p>
                    <div class="empty-decoration">
                        <div class="floating-shape shape-1"></div>
                        <div class="floating-shape shape-2"></div>
                        <div class="floating-shape shape-3"></div>
                    </div>
                </div>
            `;
        } finally {
            loadingDiv.style.display = 'none';
        }
    }
    
    renderModernResults(results, displayOptions) {
        let html = '';
        
        // Create tabs for each term
        const termNames = {1: 'First Term', 2: 'Second Term'};
        const availableTerms = Object.keys(results);
        
        if (availableTerms.length > 1) {
            html += '<div class="modern-term-tabs">';
            availableTerms.forEach((termId, index) => {
                html += `
                    <button class="modern-term-tab ${index === 0 ? 'active' : ''}" 
                            onclick="switchResultsTab('term${termId}', this)">
                        <div class="tab-icon">
                            <i class="bi bi-calendar${termId}"></i>
                        </div>
                        <div class="tab-label">${termNames[termId]}</div>
                        <div class="tab-indicator"></div>
                    </button>
                `;
            });
            html += '</div>';
        }
        
        html += '<div class="tab-content">';
        
        availableTerms.forEach((termId, index) => {
            const termData = results[termId];
            const termName = termNames[termId];
            
            html += `
                <div class="modern-term-content ${index === 0 ? 'active' : ''}" id="term${termId}">
                    <div class="modern-results-container">
                        <div class="results-header">
                            <div class="results-title">
                                <div class="title-icon">
                                    <i class="bi bi-graph-up-arrow"></i>
                                </div>
                                <h3>${termName} Academic Results</h3>
                            </div>
                            ${this.renderModernTermSummary(termData, displayOptions)}
                        </div>
                        
                        <div class="subjects-grid">
                            ${this.renderModernSubjects(termData.subjects, displayOptions)}
                        </div>
                    </div>
                </div>
            `;
        });
        
        html += '</div>';
        
        return html;
    }
    
    renderModernSubjects(subjects, displayOptions) {
        let html = '';
        
        subjects.forEach((subject, index) => {
            const percentage = parseFloat(subject.percentage);
            const isExcellent = percentage >= 90;
            const isGood = percentage >= 70;
            
            html += `
                <div class="modern-subject-card" style="animation-delay: ${index * 0.1}s">
                    <div class="subject-header">
                        <div class="subject-name">
                            <div class="subject-icon ${isExcellent ? 'excellent' : isGood ? 'good' : 'normal'}">
                                <i class="bi bi-${this.getSubjectIcon(subject.subject)}"></i>
                            </div>
                            <h5>${subject.subject}</h5>
                        </div>
                        <div class="subject-grade-badge" style="background: ${subject.band_color}">
                            ${subject.band_label}
                        </div>
                    </div>
                    
                    <div class="subject-metrics">
                        ${displayOptions.display_mode !== 'color_only' ? `
                        <div class="metric-item">
                            <div class="metric-label">Score</div>
                            <div class="metric-value">${subject.score}/${subject.max_score}</div>
                        </div>
                        ` : ''}
                        
                        <div class="metric-item">
                            <div class="metric-label">Percentage</div>
                            <div class="metric-value percentage-value">${parseFloat(subject.percentage).toFixed(2)}%</div>
                        </div>
                    </div>
                    
                    <div class="subject-progress">
                        <div class="progress-track">
                            <div class="progress-fill" 
                                 style="width: ${percentage}%; background: ${subject.band_color}"></div>
                        </div>
                        <div class="progress-labels">
                            <span>0%</span>
                            <span>100%</span>
                        </div>
                    </div>
                    
                    ${displayOptions.display_mode !== 'color_only' ? `
                    <div class="subject-total-score" style="background: ${subject.band_color}">
                        <div class="total-score-label">Total Score</div>
                        <div class="total-score-value">${subject.score}/${subject.max_score}</div>
                    </div>
                    ` : ''}
                    
                    <div class="subject-achievement">
                        ${percentage >= 95 ? '<i class="bi bi-trophy achievement-icon gold"></i>' : 
                          percentage >= 85 ? '<i class="bi bi-award achievement-icon silver"></i>' : 
                          percentage >= 75 ? '<i class="bi bi-star achievement-icon bronze"></i>' : ''}
                    </div>
                </div>
            `;
        });
        
        return html;
    }
    
    getSubjectIcon(subjectName) {
        const icons = {
            'Arabic': 'translate',
            'English': 'chat-quote',
            'Mathematics': 'calculator',
            'Math': 'calculator',
            'Science': 'flask',
            'Physics': 'lightning',
            'Chemistry': 'droplet',
            'Biology': 'tree',
            'History': 'clock-history',
            'Geography': 'globe',
            'Art': 'palette',
            'Music': 'music-note',
            'PE': 'trophy',
            'Computer': 'laptop'
        };
        
        for (let key in icons) {
            if (subjectName.toLowerCase().includes(key.toLowerCase())) {
                return icons[key];
            }
        }
        return 'book';
    }
    
    renderModernTermSummary(termData, displayOptions) {
        const isPassed = termData.term_pass_fail === 'Pass';
        
        return `
            <div class="term-summary-modern">
                <div class="summary-cards">
                    ${displayOptions.display_mode !== 'color_only' ? `
                    <div class="summary-card total-score">
                        <div class="summary-icon">
                            <i class="bi bi-calculator"></i>
                        </div>
                        <div class="summary-content">
                            <div class="summary-label">Total Score</div>
                            <div class="summary-value">${termData.total_score}<span class="max-score">/${termData.total_max_score}</span></div>
                        </div>
                    </div>
                    ` : ''}
                    
                    <div class="summary-card average">
                        <div class="summary-icon">
                            <i class="bi bi-graph-up"></i>
                        </div>
                        <div class="summary-content">
                            <div class="summary-label">Average</div>
                            <div class="summary-value">${parseFloat(termData.average_percent).toFixed(2)}<span class="unit">%</span></div>
                        </div>
                    </div>
                    
                    <div class="summary-card grade">
                        <div class="summary-icon" style="color: ${termData.term_grade_color}">
                            <i class="bi bi-award"></i>
                        </div>
                        <div class="summary-content">
                            <div class="summary-label">Overall Grade</div>
                            <div class="summary-value grade-value" style="color: ${termData.term_grade_color}">
                                ${termData.term_grade_label}
                            </div>
                        </div>
                    </div>
                    
                    <div class="summary-card status ${isPassed ? 'passed' : 'failed'}">
                        <div class="summary-icon">
                            <i class="bi bi-${isPassed ? 'check-circle' : 'x-circle'}"></i>
                        </div>
                        <div class="summary-content">
                            <div class="summary-label">Status</div>
                            <div class="summary-value status-value">${termData.term_pass_fail}</div>
                        </div>
                    </div>
                </div>
                
                <div class="overall-progress">
                    <div class="progress-header">
                        <span>Overall Performance</span>
                        <span class="progress-percentage">${parseFloat(termData.average_percent).toFixed(2)}%</span>
                    </div>
                    <div class="progress-bar-container">
                        <div class="progress-bar-bg">
                            <div class="progress-bar-fill" 
                                 style="width: ${termData.average_percent}%; background: linear-gradient(90deg, ${termData.term_grade_color}, ${termData.term_grade_color}cc)"></div>
                        </div>
                    </div>
                </div>
            </div>
        `;
    }
    
    async loadStatistics() {
        const loadingDiv = document.querySelector('#statsTab .loading');
        
        loadingDiv.style.display = 'block';
        
        try {
            const response = await this.apiCall('GET', '/student/results');
            
            if (response.success) {
                const data = response.data;
                this.calculateAndDisplayStats(data.results);
            } else {
                this.displayStatsError('Unable to load statistics');
            }
        } catch (error) {
            this.displayStatsError('Error loading statistics');
        } finally {
            loadingDiv.style.display = 'none';
        }
    }
    
    calculateAndDisplayStats(results) {
        if (!results || Object.keys(results).length === 0) {
            this.displayStatsError('No results available for statistics');
            return;
        }
        
        let allSubjects = [];
        let termAverages = [];
        let totalScores = 0;
        let totalMaxScores = 0;
        
        // Collect data from all terms
        Object.values(results).forEach(termData => {
            if (termData.subjects) {
                allSubjects = allSubjects.concat(termData.subjects);
                termAverages.push(parseFloat(termData.average_percent) || 0);
            }
            totalScores += parseFloat(termData.total_score) || 0;
            totalMaxScores += parseFloat(termData.total_max_score) || 0;
        });
        
        // Calculate overall average
        const overallAverage = totalMaxScores > 0 ? ((totalScores / totalMaxScores) * 100).toFixed(1) : 0;
        
        // Find best subject
        let bestSubject = null;
        let bestPercentage = 0;
        allSubjects.forEach(subject => {
            const percentage = parseFloat(subject.percentage) || 0;
            if (percentage > bestPercentage) {
                bestPercentage = percentage;
                bestSubject = subject;
            }
        });
        
        // Calculate improvement (if multiple terms)
        let improvement = 'N/A';
        if (termAverages.length >= 2) {
            const diff = termAverages[termAverages.length - 1] - termAverages[0];
            improvement = diff >= 0 ? `+${diff.toFixed(1)}%` : `${diff.toFixed(1)}%`;
        }
        
        // Determine grade level
        const gradeLevel = this.determineGradeLevel(overallAverage);
        
        // Update stats display
        document.getElementById('overallAverage').textContent = `${overallAverage}%`;
        document.getElementById('bestSubject').textContent = bestSubject ? bestSubject.subject : 'N/A';
        document.getElementById('bestSubjectScore').textContent = bestSubject ? `Score: ${bestSubject.percentage}%` : 'Score: --';
        document.getElementById('improvement').textContent = improvement;
        document.getElementById('gradeLevel').textContent = gradeLevel;
        
        // Generate achievements
        this.generateAchievements(overallAverage, bestSubject, allSubjects);
    }
    
    determineGradeLevel(average) {
        if (average >= 90) return 'Excellent';
        if (average >= 80) return 'Very Good';
        if (average >= 70) return 'Good';
        if (average >= 60) return 'Satisfactory';
        return 'Needs Improvement';
    }
    
    generateAchievements(overallAverage, bestSubject, allSubjects) {
        const achievementsDiv = document.getElementById('achievements');
        const achievements = [];
        
        // Performance achievements
        if (overallAverage >= 90) {
            achievements.push('🎯 Outstanding Performance');
        } else if (overallAverage >= 80) {
            achievements.push('🌟 Excellent Student');
        } else if (overallAverage >= 70) {
            achievements.push('👍 Good Performance');
        }
        
        // Subject achievements
        if (bestSubject && parseFloat(bestSubject.percentage) >= 95) {
            achievements.push(`🏆 ${bestSubject.subject} Master`);
        }
        
        // Consistency achievements
        const consistentSubjects = allSubjects.filter(s => parseFloat(s.percentage) >= 75).length;
        if (consistentSubjects >= allSubjects.length * 0.8) {
            achievements.push('📈 Consistent Performer');
        }
        
        // Perfect score achievements
        const perfectScores = allSubjects.filter(s => parseFloat(s.percentage) === 100).length;
        if (perfectScores > 0) {
            achievements.push(`💯 Perfect Score x${perfectScores}`);
        }
        
        // Display achievements
        if (achievements.length === 0) {
            achievementsDiv.innerHTML = '<p class="text-muted">Keep working hard to earn achievements!</p>';
        } else {
            achievementsDiv.innerHTML = achievements.map(achievement => 
                `<span class="achievement-badge">${achievement}</span>`
            ).join(' ');
        }
    }
    
    displayStatsError(message) {
        document.getElementById('overallAverage').textContent = '--';
        document.getElementById('bestSubject').textContent = '--';
        document.getElementById('bestSubjectScore').textContent = 'Score: --';
        document.getElementById('improvement').textContent = '--';
        document.getElementById('gradeLevel').textContent = '--';
        document.getElementById('achievements').innerHTML = `<p class="text-muted">${message}</p>`;
    }
    
    async apiCall(method, endpoint, data = null) {
        const config = {
            method,
            headers: {
                'Content-Type': 'application/json'
            }
        };
        
        if (this.token) {
            config.headers.Authorization = `Bearer ${this.token}`;
        }
        
        if (data) {
            config.body = JSON.stringify(data);
        }
        
        const response = await fetch(this.apiBase + endpoint, config);
        
        if (!response.ok) {
            let errorData = null;
            try {
                errorData = await response.json();
            } catch (e) {
                // If response isn't JSON, use default error
                errorData = { message: `HTTP ${response.status}` };
            }
            
            const error = new Error(errorData.message || `HTTP ${response.status}`);
            error.response = {
                status: response.status,
                data: errorData
            };
            throw error;
        }
        
        return await response.json();
    }
    
    showError(element, message) {
        element.textContent = message;
        element.style.display = 'block';
        setTimeout(() => {
            element.style.display = 'none';
        }, 7000);
    }
    
    // Language support methods
    applyTranslations() {
        if (typeof studentLangManager !== 'undefined') {
            studentLangManager.translatePage();
        }
    }
    
    reloadCurrentTab() {
        this.applyTranslations();
        switch(this.currentTab) {
            case 'profile':
                this.loadProfile();
                break;
            case 'results':
                this.loadResults();
                break;
            case 'stats':
                this.loadStatistics();
                break;
        }
    }
    
    // Helper method to get translated text
    t(key, params = {}) {
        if (typeof studentLangManager !== 'undefined') {
            return studentLangManager.t(key, params);
        }
        return key; // fallback to key if language manager not available
    }
    
    async loadSchoolSettings() {
        try {
            const response = await fetch(`${this.apiBase}/school/settings`);
            if (response.ok) {
                const data = await response.json();
                if (data.success && data.data) {
                    const settings = data.data;
                    
                    // Update page title
                    if (settings.school_name) {
                        document.getElementById('pageTitle').textContent = `${settings.school_name} - Student Portal`;
                        document.getElementById('portalTitle').textContent = settings.school_name;
                    }
                }
            }
        } catch (error) {
            console.error('Error loading school settings:', error);
        }
    }
}

// Global function for tab switching
function switchResultsTab(targetId, clickedTab) {
    // Remove active class from all tabs
    document.querySelectorAll('.modern-term-tab').forEach(tab => {
        tab.classList.remove('active');
    });
    
    // Add active class to clicked tab
    clickedTab.classList.add('active');
    
    // Hide all term content
    document.querySelectorAll('.modern-term-content').forEach(content => {
        content.classList.remove('active');
    });
    
    // Show target content
    const targetContent = document.getElementById(targetId);
    if (targetContent) {
        targetContent.classList.add('active');
    }
}

// Initialize app when DOM is loaded
document.addEventListener('DOMContentLoaded', () => {
    window.studentApp = new StudentApp();
});